/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : PMS5003 Sensor Interface         
 */
 
//=================================================================================================
// PMS5003 Interface ET-ESP32 WROVER MODULE V1
//=================================================================================================
// VCC           <--  (+)5V
// GND           <--  GND
// PMS5003_SET   <--  ESP32_IO33(GPIO:Output)
// PMS5003_RX    <--  ESP32_IO13(TX1)
// PMS5003_TX    -->  ESP32_IO14(RX1)  
// PMS5003_RESET <--  ESP32_IO32(GPIO:Output)
//=================================================================================================


//=================================================================================================
#include <HardwareSerial.h>
#include <Wire.h> 
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialPMS5003_RX_PIN    14
#define SerialPMS5003_TX_PIN    13
#define SerialPMS5003 Serial1                                                                     // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define PMS5003_SET_PIN       33                                                                  // ESP32-WROVER :IO33
#define PMS5003_RESET_PIN     32                                                                  // ESP32-WROVER :IO32
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(PMS5003_SET_PIN,OUTPUT);                                                               // SET(1:Normal Working Mode, 0:Sleep Mode)
  pinMode(PMS5003_RESET_PIN,OUTPUT);                                                             // RESET(1:Run Mode, 0:Reset)
  digitalWrite(PMS5003_SET_PIN,HIGH);                                                            // Normal Working Mode(0:Sleep Mode)
  digitalWrite(PMS5003_RESET_PIN,HIGH);                                                          // Normal Operation(0:Reset)
  //===============================================================================================
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialPMS5003.begin(9600, SERIAL_8N1, SerialPMS5003_RX_PIN, SerialPMS5003_TX_PIN);
  while(!SerialPMS5003);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Demo Test PMS5003");
  //===============================================================================================
  
}

struct pms5003data 
{
  uint16_t framelen;
  uint16_t pm10_standard, pm25_standard, pm100_standard;
  uint16_t pm10_env, pm25_env, pm100_env;
  uint16_t particles_03um, particles_05um, particles_10um, particles_25um, particles_50um, particles_100um;
  uint16_t unused;
  uint16_t checksum;
};

struct pms5003data data;
    
void loop() 
{
  if (readPMSdata(&SerialPMS5003)) 
  {
    // reading data was successful!
    SerialDebug.println();
    SerialDebug.println("---------------------------------------");
    SerialDebug.println("Concentration Units (standard)");
    SerialDebug.print("PM 1.0: "); Serial.print(data.pm10_standard);
    SerialDebug.print("\t\tPM 2.5: "); Serial.print(data.pm25_standard);
    SerialDebug.print("\t\tPM 10: "); Serial.println(data.pm100_standard);
    SerialDebug.println("---------------------------------------");
    SerialDebug.println("Concentration Units (environmental)");
    SerialDebug.print("PM 1.0: "); Serial.print(data.pm10_env);
    SerialDebug.print("\t\tPM 2.5: "); Serial.print(data.pm25_env);
    SerialDebug.print("\t\tPM 10: "); Serial.println(data.pm100_env);
    SerialDebug.println("---------------------------------------");
    SerialDebug.print("Particles > 0.3um / 0.1L air:"); Serial.println(data.particles_03um);
    SerialDebug.print("Particles > 0.5um / 0.1L air:"); Serial.println(data.particles_05um);
    SerialDebug.print("Particles > 1.0um / 0.1L air:"); Serial.println(data.particles_10um);
    SerialDebug.print("Particles > 2.5um / 0.1L air:"); Serial.println(data.particles_25um);
    SerialDebug.print("Particles > 5.0um / 0.1L air:"); Serial.println(data.particles_50um);
    SerialDebug.print("Particles > 10.0 um / 0.1L air:"); Serial.println(data.particles_100um);
    SerialDebug.println("---------------------------------------");
  }
}

boolean readPMSdata(Stream *s) 
{
  if(! s->available()) 
  {
    return false;
  }
  
  // Read a byte at a time until we get to the special '0x42' start-byte
  if (s->peek() != 0x42) 
  {
    s->read();
    return false;
  }

  // Now read all 32 bytes
  if (s->available() < 32) 
  {
    return false;
  }
    
  uint8_t buffer[32];    
  uint16_t sum = 0;
  s->readBytes(buffer, 32);

  // get checksum ready
  for (uint8_t i=0; i<30; i++) 
  {
    sum += buffer[i];
  }

  // The data comes in endian'd, this solves it so it works on all platforms
  uint16_t buffer_u16[15];
  for (uint8_t i=0; i<15; i++) 
  {
    buffer_u16[i] = buffer[2 + i*2 + 1];
    buffer_u16[i] += (buffer[2 + i*2] << 8);
  }

  // put it into a nice struct :)
  memcpy((void *)&data, (void *)buffer_u16, 30);

  if (sum != data.checksum) 
  {
    SerialDebug.println("Checksum failure");
    return false;
  }
  // success!
  return true;
}
