/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : I2C OLED + Honeywell HPM Sensor(HPMA115S0-xxx)
 * 
 */

//=================================================================================================
// Honeywell HPM Sensor(HPMA115S0-xxx) Interface
//=================================================================================================
// 1: +3.3V       ---> ET-ESP8266-RS485 : (+)3.3V
// 2: +5V         <--- ET-ESP8266-RS485 : (+)5V
// 3: NC               
// 4: NC 
// 5: TEST
// 6: HPM TX      ---> ET-ESP8266-RS485 : D5
// 7: HPM RX      <--- ET-ESP8266-RS485 : D4   
// 8: GND         <--> ET-ESP8266-RS485 : GND
//=================================================================================================

//=================================================================================================
#include <Wire.h>
//=================================================================================================
#include "SH1106.h"
//=================================================================================================

#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//=================================================================================================
#define RS485_DIRECTION_PIN D0                                                                    // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//=================================================================================================
#define I2C_SCL1_PIN        D1                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//=================================================================================================
#define RESERVE1            D3                                                                    // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//=================================================================================================
#define SHT10_SCL2_PIN      D4                                                                    // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                    // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//=================================================================================================
#define LED_WIFI_PIN        D6                                                                    // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//=================================================================================================
#define LED_LINK_PIN        D7                                                                    // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//=================================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                    // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//=================================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//=================================================================================================

//=================================================================================================
SH1106 display(0x3c, I2C_SDA1_PIN, I2C_SCL1_PIN); 
//=================================================================================================
#define ett_logo_width  32
#define ett_logo_height 20
//=================================================================================================
const char ett_logo[] PROGMEM = 
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0xF8, 0xF7, 0xDF, 0x7F, 0x04, 0x00, 0x00, 0x00, 0xE4, 0xF7, 0xDE, 0x7B, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xF4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF4, 0x87, 0x02, 0x0A, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xE4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF8, 0x87, 0x02, 0x0A, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
//=================================================================================================

//=================================================================================================
#include "SoftwareSerial.h"
#include "HPMA115S0.h"
//=================================================================================================
SoftwareSerial HPM_Serial(D5, D4, false, 256);                                                    // RX:D5 / TX:D4
HPMA115S0 hpma115S0(HPM_Serial);                                                                  // HPM Intear Software Serial  
//=================================================================================================
unsigned int pm2_5, pm10;
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //===============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //===============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //===============================================================================================
  Serial.begin(115200);
  //===============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================

  
  //===============================================================================================
  // Initialising the UI will init the display too.
  //===============================================================================================
  display.init();
  display.flipScreenVertically();
  display.setFont(ArialMT_Plain_10);
  display.setTextAlignment(TEXT_ALIGN_LEFT);
  //===============================================================================================
  display.clear();
  display.drawXbm(0, 0, ett_logo_width, ett_logo_height,  ett_logo);
  display.drawString(30, 0,  " ETT CO.,LTD.  ");
  display.drawString(30, 10, " Demo PM Sensor");
  display.drawString(0, 20, "========================");
  display.drawString(0, 30, "PM Sensor HPMA115S0");
  display.drawString(0, 40, "PM 2.5: ??? ug/m3");
  display.drawString(0, 50, "PM 10 : ??? ug/m3");
  display.display();
  //===============================================================================================

  //===============================================================================================
  HPM_Serial.begin(9600);
  delay(5000);
  Serial.println("Starting...");
  hpma115S0.Init();
  hpma115S0.StartParticleMeasurement();
  Serial.println("Ready...");
  //===============================================================================================
}

void loop() 
{
  if(hpma115S0.ReadParticleMeasurement(&pm2_5, &pm10)) 
  {
    Serial.println("PM 2.5: " + String(pm2_5) + " ug/m3");
    Serial.println("PM 10 : " + String(pm10)  + " ug/m3");
    
    //=============================================================================================
    // Start of Display OLED
    //=============================================================================================
    display.clear();
    
    display.drawXbm(0, 0, ett_logo_width, ett_logo_height,  ett_logo);
    display.drawString(30, 0,  " ETT CO.,LTD.  ");
    display.drawString(30, 10, " Demo PM Sensor");
    
    display.drawString(0, 20, "========================");
    display.drawString(0, 30, "PM Sensor HPMA115S0");
    
    String spm2_5 = "PM 2.5: " + String(pm2_5) + " ug/m3";
    display.drawString(0, 40, spm2_5);
    
    String spm10 =  "PM 10 : " + String(pm10)  + " ug/m3";
    display.drawString(0, 50, spm10);
    
    display.display();
    //=============================================================================================
    // End of Display OLED
    //=============================================================================================
  }
  delay(2000);
}


