/*
 * Demo ET-MEGA32U4-RS485 Hardware Board
 * MCU    : ATMEGA32U4
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * ADC    : Analog#0 : A;[0..3]
 *        : Analog#1 : A4
 *        : Analog#2 : A5
 * Output : Link(LED Active High:D13)
 *        : Relay0(Active Low:D6)
 *        : Relay1(Active Low:D7)
 * Input  : RS485 ID0:D8
 *        : RS485 ID1:D9
 *        : RS485 ID2:D10
 *        : RS485 ID3:D11
 *        : RS485 ID4:D12
 * 1-Wire : DS18B20(1-Wire:D5)
 * 
 * Demo   : Modbus RTU Relay8/In8
 */
//=================================================================================================
#include "ETT_ModbusRTU.h"
//=================================================================================================
 
//=================================================================================================
// USART RS232/RS485
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
#define SerialRS485 Serial1                                                                       // Serial1(D1=TXD,D0=RXD)
//=================================================================================================
const int RS485_DIRECTION_PIN =     4;                                                            // RS485 TXD Enable,Disable
const int RS485_RXD_SELECT    =     LOW;
const int RS485_TXD_SELECT    =     HIGH;
//=================================================================================================
const int RS485_ID0_PIN       =     8;                                                            // Slave ID LSB
const int RS485_ID1_PIN       =     9;         
const int RS485_ID2_PIN       =     10;        
const int RS485_ID3_PIN       =     11;        
const int RS485_ID4_PIN       =     12;                                                           // Slave ID MSB
//=================================================================================================
const int LED_LINK_PIN        =     13;
const int LED_OFF             =     LOW;
const int LED_ON              =     HIGH;
//=================================================================================================
const int INTERNAL_RELAY0_PIN =     6;
//=================================================================================================
const int INTERNAL_RELAY1_PIN =     7;
//=================================================================================================
const int InternalRelayOff    =     HIGH;
const int InternalRelayOn     =     LOW;
//=================================================================================================

//=================================================================================================
// data array for modbus network sharing
//=================================================================================================
uint16_t au16dataRelay[16];                                                                      // data array for modbus network sharing
uint16_t au16dataInput[16];
//=================================================================================================
uint8_t u8state;                                                                                 // machine state
uint8_t u8query;                                                                                 // pointer to message query
//=================================================================================================

/**
 *  Modbus object declaration
 *  u8id : node id = 0 for master, = 1..247 for slave
 *  u8serno : serial port (use 0 for Serial)
 *  u8txenpin : 0 for RS-232 and USB-FTDI 
 *               or any pin number > 1 for RS-485
 */
Modbus master(0,                                                                                 // node id = 0(master)
              SerialRS485,                                                                       // Serial(2)
              RS485_DIRECTION_PIN);                                                              // RS485 Modbus

/**
 * This is an structe which contains a query to an slave device
 */
modbus_t telegram[4];                                                                            // 4-Modbus Commans
unsigned long u32wait;                                                                           // Scan Rate Modbus RTU
int8_t pool_size;
//=================================================================================================


void setup() 
{
  //===============================================================================================
  pinMode(INTERNAL_RELAY0_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOff);
  pinMode(INTERNAL_RELAY1_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOff);
  //===============================================================================================
  //===============================================================================================
  SerialDebug.begin(115200);
  SerialDebug.println();
  //while(!SerialDebug);                                                                           // Wait MEGA32U4 USB Serial Complete
  //===============================================================================================
  
  //===============================================================================================
  // Initial RS485
  //===============================================================================================
  SerialRS485.begin(9600);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  
  //===============================================================================================
  // Mosbus RTU Command
  //=============================================================================================== 
  
  //===============================================================================================
  // F01 Read Relay Status
  //===============================================================================================
  // 01 01 00 00 00 08 3D CC 
  // 01 01 02 00 00 B9 FC 
  //===============================================================================================
  telegram[0].u8id = 1;                                                                          // slave address : 0x01
  telegram[0].u8fct = 1;                                                                         // Function 0x01(Read Coils)
  telegram[0].u16RegAdd = 0;                                                                     // start address in slave : 0x0000
  telegram[0].u16CoilsNo = 8;                                                                    // number of elements (registers) to read : 8
  telegram[0].au16reg = &au16dataInput[0];                                                       // pointer to a memory array in the Arduino
  //===============================================================================================

  //===============================================================================================
  // F05 Write Single Coil(OFF Relay)
  //===============================================================================================
  // 01 05 00 00 00 00 CD CA 
  // 01 05 00 00 00 00 CD CA
  //===============================================================================================
  telegram[1].u8id = 1;                                                                          // slave address : 0x01
  telegram[1].u8fct = 5;                                                                         // Function 0x05(Write Single Coil)
  telegram[1].u16RegAdd = 7;                                                                     // Address = Relay(7..0)
  telegram[1].au16reg = &au16dataRelay[0];                                                       // pointer to a memory array in the Arduino
  au16dataRelay[0] = 0;                                                                          // 0 = OFF Relay
  //===============================================================================================
  
  //===============================================================================================
  // F02 Read Input Status
  //===============================================================================================
  // 01 02 00 00 00 08 79 CC 
  // 01 02 02 08 08 BF BE 
  //===============================================================================================
  telegram[2].u8id = 1;                                                                          // slave address : 0x01
  telegram[2].u8fct = 2;                                                                         // Function 0x02(Read Discrete Input)
  telegram[2].u16RegAdd = 0;                                                                     // start address in slave : 0x0000
  telegram[2].u16CoilsNo = 8;                                                                    // number of elements (registers) to read : 8
  telegram[2].au16reg = &au16dataInput[0];                                                       // pointer to a memory array in the Arduino
  //===============================================================================================
  
  //===============================================================================================
  // F05 Write Single Coil(ON Relay)
  //===============================================================================================
  // 01 05 00 00 FF 00 8C 3A 
  // 01 05 00 00 FF 00 8C 3A 
  //===============================================================================================
  telegram[3].u8id = 1;                                                                          // slave address : 0x01
  telegram[3].u8fct = 5;                                                                         // Function 0x05(Write Single Coil)
  telegram[3].u16RegAdd = 7;                                                                     // Address = Relay(7..0)
  telegram[3].au16reg = &au16dataRelay[1];                                                       // pointer to a memory array in the Arduino
  au16dataRelay[1] = 1;                                                                          // 1 = ON Relay
  //===============================================================================================
  
  //===============================================================================================
  master.begin(SerialRS485);                                                                     // Mosbus Interface RS485
  master.setTimeOut(3500);                                                                       // if there is no answer in 3500 ms, roll over
  u32wait = millis() + 1000;                                                                     // Next Scan Rate = 1 Second
  u8state = u8query = 0;                                                                         // Wait New Message
  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-MAGA32U4 RS485...Demo Modbus RTU Relay8/In8");
  //===============================================================================================
}


//=================================================================================================
// Start of Test : MODBUS RTU RELAY8/IN8
//=================================================================================================
void loop() 
{
  switch( u8state ) 
  {
    //==============================================================================================
    case 0: 
      if (millis() > u32wait) u8state++;                                                          // wait state
    break;
    //==============================================================================================
    
    //==============================================================================================
    case 1: 
      master.query(telegram[u8query]);                                                            // send query (only once)
      u8state++;
      u8query++;
      if(u8query == 4) 
      {
        u8query = 0;                                                                              // 0,1,...,3,0
      }
    break;
    //==============================================================================================
    
    //==============================================================================================
    case 2:
      //============================================================================================
      pool_size = master.poll();
      //============================================================================================
     
      //============================================================================================
      if(master.getState() == COM_IDLE) 
      {
        //==========================================================================================
        // response from the slave was received
        //==========================================================================================
        u8state = 0;                                                                              // Wait New Message
        u32wait = millis() + 1000;                                                                // Next Scan Rate = 1Second
        //==========================================================================================
        if(pool_size > 4)                                                                         // check incoming messages
        {
          SerialDebug.print("Modbus Receive Size = ");
          SerialDebug.print(pool_size);
        
          if(u8query == 0)
          {
            //======================================================================================
            // u8query = 3++ -> 4 -> 0 (Function 0x05(Write Single Coil) Result)
            //======================================================================================
            SerialDebug.print(" -> Q0 Write Single Coil Result...OK");
            SerialDebug.println();
            //======================================================================================
          }
          else if(u8query == 1)
          {
            //======================================================================================
            // u8query = 0++ -> 1 (Function 0x01(Read Coils) Result)
            //======================================================================================
            SerialDebug.print(" -> Q1 Read Coils Result = ");
            SerialDebug.print(au16dataInput[0],BIN);    
            SerialDebug.println();
            //======================================================================================
          }
          else if(u8query == 2)
          {
            //======================================================================================
            // u8query = 1++ -> 2 (Function 0x05 Result)
            //======================================================================================
            SerialDebug.print(" -> Q2 Write Single Coil Result...OK");
            SerialDebug.println();
            //======================================================================================
          }
          else if(u8query == 3)
          {
            //======================================================================================
            // u8query = 2++ -> 3 (Function 0x02(Read Discrete Input) Result)
            //======================================================================================
            SerialDebug.print(" -> Q3 Read Discrete Input Result = ");
            SerialDebug.print(au16dataInput[0],BIN);          
            SerialDebug.println();
            //======================================================================================
          }
        }
        //==========================================================================================
      } 
      //============================================================================================
    break;
    //==============================================================================================
  }
}
//==================================================================================================
// End of Test : MODBUS RTU RELAY8/IN8
//==================================================================================================


