/*
 * Demo ET-MEGA32U4-RS485 Hardware Board
 * MCU    : ATMEGA32U4
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * ADC    : Analog#0 : A;[0..3]
 *        : Analog#1 : A4
 *        : Analog#2 : A5
 * Output : Link(LED Active High:D13)
 *        : Relay0(Active Low:D6)
 *        : Relay1(Active Low:D7)
 * Input  : RS485 ID0:D8
 *        : RS485 ID1:D9
 *        : RS485 ID2:D10
 *        : RS485 ID3:D11
 *        : RS485 ID4:D12
 * 1-Wire : DS18B20(1-Wire:D5)
 * 
 * Demo   : RS485 Control Relay
 */
 
//=================================================================================================
// USART RS232/RS485
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
#define SerialRS485 Serial1                                                                       // Serial1(D1=TXD,D0=RXD)
//=================================================================================================
const int RS485_DIRECTION_PIN =     4;                                                            // RS485 TXD Enable,Disable
const int RS485_RXD_SELECT    =     LOW;
const int RS485_TXD_SELECT    =     HIGH;
//=================================================================================================
const int RS485_ID0_PIN       =     8;                                                            // Slave ID LSB
const int RS485_ID1_PIN       =     9;         
const int RS485_ID2_PIN       =     10;        
const int RS485_ID3_PIN       =     11;        
const int RS485_ID4_PIN       =     12;                                                           // Slave ID MSB
//=================================================================================================
int SlaveAddress = 0;
char SlaveID[2];
//=================================================================================================
const int LED_LINK_PIN        =     13;
const int LED_OFF             =     LOW;
const int LED_ON              =     HIGH;
//=================================================================================================
const int INTERNAL_RELAY0_PIN =     6;
//=================================================================================================
const int INTERNAL_RELAY1_PIN =     7;
//=================================================================================================
const int InternalRelayOff    =     HIGH;
const int InternalRelayOn     =     LOW;
//=================================================================================================

//=================================================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
// <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
// <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
// <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
//=================================================================================================
static union  
{
  byte cmd_buff[9];                                                                               // 8 Byte Max
  //===============================================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
  // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
  // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
  // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
  //===============================================================================================
  struct 
  {
    //=============================================================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte relay;                         // '0':Relay[0],'1':Relay[1]
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //=============================================================================================
  }control_relay;
  //===============================================================================================
  
}rs485;
//=================================================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY0                  = '0';  
const byte      RELAY1                  = '1';
const byte      RELAY_ON                = '0';  
const byte      RELAY_OFF               = '1';
//=================================================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//=================================================================================================

//=================================================================================================
// Initial Slave Address
//=================================================================================================
void InitialSlaveAddress()
{
  //===============================================================================================
  SlaveAddress = 0;
  //===============================================================================================
  if(digitalRead(RS485_ID0_PIN) == HIGH)
  {
    SlaveAddress += 1;
  }
  if(digitalRead(RS485_ID1_PIN) == HIGH)
  {
    SlaveAddress += 2;
  }
  if(digitalRead(RS485_ID2_PIN) == HIGH)
  {
    SlaveAddress += 4;
  }
  if(digitalRead(RS485_ID3_PIN) == HIGH)
  {
    SlaveAddress += 8;
  }
  if(digitalRead(RS485_ID4_PIN) == HIGH)
  {
    SlaveAddress += 16;
  }
  //===============================================================================================
  String StringID = String(SlaveAddress,DEC);
  //===============================================================================================
  if(SlaveAddress < 10)
  {
    SlaveID[0] = '0';
    SlaveID[1] = StringID[0];
  }
  else
  {
    SlaveID[0] = StringID[0];
    SlaveID[1] = StringID[1];
  }
  //===============================================================================================
}
//=================================================================================================

void setup() 
{
  //===============================================================================================
  pinMode(INTERNAL_RELAY0_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOff);
  pinMode(INTERNAL_RELAY1_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOff);
  //===============================================================================================
  //===============================================================================================
  SerialDebug.begin(115200);
  SerialDebug.println();
  while(!SerialDebug);                                                                            // Wait MEGA32U4 USB Serial Complete
  //===============================================================================================
  
  //===============================================================================================
  // Initial RS485
  //===============================================================================================
  SerialRS485.begin(115200);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  pinMode(RS485_ID0_PIN, INPUT_PULLUP);
  pinMode(RS485_ID1_PIN, INPUT_PULLUP);
  pinMode(RS485_ID2_PIN, INPUT_PULLUP);
  pinMode(RS485_ID3_PIN, INPUT_PULLUP);
  pinMode(RS485_ID4_PIN, INPUT_PULLUP);
  InitialSlaveAddress();
  //===============================================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println();
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  // Start of Receive & Service RS485 Command
  //===============================================================================================
  if(SerialRS485.available() > 0)
  {  
    rs485_chr = SerialRS485.read();
    
    if(rs485_chr == STX_CODE)                                                                     // Verify Start of Command Header
    {
      //===========================================================================================
      // Reset Packet Frame Reset
      //===========================================================================================
      rs485_cmd_head = 1;                                                                         // Command Header Not Complete  
      rs485_cmd_cnt = 0;                                                                          // Re-Start Command Count
      for(int i=0; i<sizeof(rs485.cmd_buff); i++)
      {
        rs485.cmd_buff[i] = '\0';
      }
      rs485_cmd_ptr = &rs485.cmd_buff[0];
      //===========================================================================================
    } 

    //Start of Receive & Verify RS485 Packet Command
    if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
    { 
      *rs485_cmd_ptr = rs485_chr;                                                                 // Save Character 
      rs485_cmd_ptr++;                                                                            // Next Pointer         
      rs485_cmd_cnt++;  

      //SerialDebug.write(rs485_chr);
        
      //===========================================================================================
      // Start of Analysis & Service RS485 Command Packet   
      //===========================================================================================
      if(rs485_chr == 0x0D)
      {           
        //=========================================================================================
        // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
        // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
        // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
        // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
        // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
        //=========================================================================================
        if((rs485.control_relay.head      == STX_CODE)&&                                          // head : 0x02
        
           (rs485.control_relay.id[0]     == SlaveID[0])&&                                        // id[0]:0..9
           (rs485.control_relay.id[1]     == SlaveID[1])&&                                        // id[1]:0..9
           
           (rs485.control_relay.sep1      == SEP1)&&                                              // sep1 : ':' 
           
           ((rs485.control_relay.relay    == '0')||
            (rs485.control_relay.relay    == '1'))&&                                              // '0':Relay[0],'1':Relay[1]
           
           (rs485.control_relay.sep2      == SEP2)&&                                              // sep2 : '='
           
           ((rs485.control_relay.operate  == '0')||
            (rs485.control_relay.operate  == '1'))&&                                              // operate('0':OFF,'1':ON)
           
           (rs485.control_relay.enter     == END_CODE))                                           // Enter:0x0D = End of Packet
        {
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOff);
            
            //=====================================================================================
            // Start of RS485 Test
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[0] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
            // End of RS485 Test
            //=====================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOn);
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[0] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='1')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOff);
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[1] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='1')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOn);
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[1] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
          }
          //=======================================================================================
        }
        //=========================================================================================
        // Clear RS485 Buffer
        //=========================================================================================
        rs485_cmd_head = 0;                                                                      // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                                                       // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];   
        //=========================================================================================
      } 
      //===========================================================================================
      // End of Analysis & Service RS485 Command Packet   
      //===========================================================================================
    }
  }
  //===============================================================================================
  // End of Receive & Service RS485 Command
  //===============================================================================================

}

