/*
 * Demo ET-MEGA32U4-RS485 Hardware Board
 * MCU    : ATMEGA32U4
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * ADC    : Analog#0 : A;[0..3]
 *        : Analog#1 : A4
 *        : Analog#2 : A5
 * Output : Link(LED Active High:D13)
 *        : Relay0(Active Low:D6)
 *        : Relay1(Active Low:D7)
 * Input  : RS485 ID0:D8
 *        : RS485 ID1:D9
 *        : RS485 ID2:D10
 *        : RS485 ID3:D11
 *        : RS485 ID4:D12
 * 1-Wire : DS18B20(1-Wire:D5)
 * 
 * Demo   : System Hardware Test
 *        : Analog A0..A5   -->  Connect Volume Adjust 0..5Volt For Test
 *        : Slave Address   -->  Jumper ON/OFF ID[0..4] For Test
 *        : D5(1-Wire)      -->  Connect 1-Wires DS18B20 For Test
 *        : I2C Bus 5Pin    -->  Connect BME280 Sensor For Test
 *        : I2C 4 Pin       -->  Connect ET-I2C REL8 For Test
 *        : RS485           -->  Connect Converter RS485 & Send Command Baudrae(115200) For Test 
 *        :                      <STX><ID0><ID1><:><0><=><0><CR>   "*??:0=0"<Cr> OFF Relay[0]
 *        :                      <STX><ID0><ID1><:><0><=><1><CR>   "*??:0=1"<Cr> ON  Relay[0]
 *        :                      <STX><ID0><ID1><:><1><=><0><CR>   "*??:1=0"<Cr> OFF Relay[1]
 *        :                      <STX><ID0><ID1><:><1><=><1><CR>   "*??:1=1"<Cr> ON  Relay[1]
 */

//=================================================================================================
#include <OneWire.h>                                                                              // 1-Wire Bus
#include <DallasTemperature.h>                                                                    // DS18B20
//=================================================================================================
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>
//=================================================================================================
#include <Wire.h>                                                                                 // I2C Bus
//=================================================================================================
#include "pcf8574.h"                                                                              // PCF8574/A
//=================================================================================================
#include "SHT1x.h"
//=================================================================================================
//=================================================================================================
#include "DS3231.h"
//=================================================================================================

//=================================================================================================
// USART RS232/RS485
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
#define SerialRS485 Serial1                                                                       // Serial1(D1=TXD,D0=RXD)
//=================================================================================================
const int RS485_DIRECTION_PIN =     4;                                                            // RS485 TXD Enable,Disable
const int RS485_RXD_SELECT    =     LOW;
const int RS485_TXD_SELECT    =     HIGH;
//=================================================================================================
const int RS485_ID0_PIN       =     8;                                                            // Slave ID LSB
const int RS485_ID1_PIN       =     9;         
const int RS485_ID2_PIN       =     10;        
const int RS485_ID3_PIN       =     11;        
const int RS485_ID4_PIN       =     12;                                                           // Slave ID MSB
//=================================================================================================
int SlaveAddress = 0;
char SlaveID[2];
//=================================================================================================
const int LED_LINK_PIN        =     13;
const int LED_OFF             =     LOW;
const int LED_ON              =     HIGH;
//=================================================================================================
const int ONE_WIRE_BUS        =     5;
//=================================================================================================
const int INTERNAL_RELAY0_PIN =     6;
//=================================================================================================
const int INTERNAL_RELAY1_PIN =     7;
//=================================================================================================
const int InternalRelayOff    =     HIGH;
const int InternalRelayOn     =     LOW;
//=================================================================================================

//=================================================================================================
OneWire oneWire(ONE_WIRE_BUS);                                                                    // 1-Wire
DallasTemperature ds18b20(&oneWire);                                                              // DS18B20
//=================================================================================================

//=================================================================================================
// I2C Bus : SCL = D3, SDA = D2
//=================================================================================================
const int SDA_I2C_PIN        =  2;                                                                // ATMEGA32U4:PD1(Arduino:D2:SDA)
const int SCL_I2C_PIN        =  3;                                                                // ATMEGA32U4:PD0(Arduino:D3:SCL)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP0(0x20);                                                                 // PCF8574  = 0100,000+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP0(0x38);                                                                // PCF8574A = 0111,000+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP1(0x21);                                                                 // PCF8574  = 0100,001+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP1(0x39);                                                                // PCF8574A = 0111,001+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP2(0x22);                                                                 // PCF8574  = 0100,010+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP2(0x3A);                                                                // PCF8574A = 0111,010+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP3(0x23);                                                                 // PCF8574  = 0100,011+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP3(0x3B);                                                                // PCF8574A = 0111,011+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP4(0x24);                                                                 // PCF8574  = 0100,100+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP4(0x3C);                                                                // PCF8574A = 0111,100+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP5(0x25);                                                                 // PCF8574  = 0100,101+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP5(0x3D);                                                                // PCF8574A = 0111,101+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP6(0x26);                                                                 // PCF8574  = 0100,110+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP6(0x3E);                                                                // PCF8574A = 0111,110+(0:W,1:R)
//=================================================================================================
PCF8574 PCF8574_RELAY_EXP7(0x27);                                                                 // PCF8574  = 0100,111+(0:W,1:R)
PCF8574 PCF8574A_INPUT_EXP7(0x3F);                                                                // PCF8574A = 0111,111+(0:W,1:R)
//=================================================================================================
#define ExternalRelayOn      0
#define ExternalRelayOff     1
//=================================================================================================

//=================================================================================================
bool pcf8574_relay_exp0_status  = false;
bool pcf8574a_input_exp0_status = false;
//
byte relay_external_dev0    = 0xFF;
byte input_external_dev0    = 0xFF;
//=================================================================================================
const byte RELAY0_ON_MASK   =  0xFE;                                                              // 1111 1110
const byte RELAY1_ON_MASK   =  0xFD;                                                              // 1111 1101
const byte RELAY2_ON_MASK   =  0xFB;                                                              // 1111 1011
const byte RELAY3_ON_MASK   =  0xF7;                                                              // 1111 0111
const byte RELAY4_ON_MASK   =  0xEF;                                                              // 1110 1111
const byte RELAY5_ON_MASK   =  0xDF;                                                              // 1101 1111
const byte RELAY6_ON_MASK   =  0xBF;                                                              // 1011 1111
const byte RELAY7_ON_MASK   =  0x7F;                                                              // 0111 1111
//=================================================================================================
const byte RELAY0_OFF_MASK  =  0x01;                                                              // 0000 0001
const byte RELAY1_OFF_MASK  =  0x02;                                                              // 0000 0010
const byte RELAY2_OFF_MASK  =  0x04;                                                              // 0000 0100
const byte RELAY3_OFF_MASK  =  0x08;                                                              // 0000 1000
const byte RELAY4_OFF_MASK  =  0x10;                                                              // 0001 0000
const byte RELAY5_OFF_MASK  =  0x20;                                                              // 0010 0000
const byte RELAY6_OFF_MASK  =  0x40;                                                              // 0100 0000
const byte RELAY7_OFF_MASK  =  0x80;                                                              // 1000 0000
//=================================================================================================
int RelayIndex = 0;
//=================================================================================================

//=================================================================================================
#define SEALEVELPRESSURE_HPA (1013.25)
Adafruit_BME280 bme280;                                                                           // I2C
//=================================================================================================
float bme280_temperature;
float bme280_humidity;
float bme280_pressure;  
float bme280_altitude;
bool  bme280_status;
//=================================================================================================

//=================================================================================================
// RTC:DS2231
//=================================================================================================
DS3231 clock;
RTCDateTime dt;
//=================================================================================================

//=================================================================================================
#define SHT10_SDA_BUS0_PIN  A0
#define SHT10_SCL_BUS0_PIN  A1
SHT1x SHT10_BUS0(SHT10_SDA_BUS0_PIN, SHT10_SCL_BUS0_PIN);
//=================================================================================================
#define SHT10_SDA_BUS1_PIN  A2
#define SHT10_SCL_BUS1_PIN  A3
SHT1x SHT10_BUS1(SHT10_SDA_BUS1_PIN, SHT10_SCL_BUS1_PIN);
//=================================================================================================
float sht10_dev0_temperature;
float sht10_dev0_humidity;
//=================================================================================================
float sht10_dev1_temperature;
float sht10_dev1_humidity;
//=================================================================================================
             
//=================================================================================================
//unsigned long lastGet1WireSensorTime = 0;
//unsigned long lastGetI2CSensorTime = 0;
unsigned long lastGetRTCSecondTime = 0;
//unsigned long lastSecondTime = 0;
//=================================================================================================

int sensorValue0;
int sensorValue1;
int sensorValue2;
int sensorValue3;
int sensorValue4;
int sensorValue5;
float voltage0;
float voltage1;
float voltage2;
float voltage3;
float voltage4;
float voltage5;

//=================================================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><0><=><0><CR>   *??:0=0<Cr> OFF Relay[0]
// <STX><ID0><ID1><:><0><=><1><CR>   *??:0=1<Cr> ON  Relay[0]
// <STX><ID0><ID1><:><1><=><0><CR>   *??:1=0<Cr> OFF Relay[1]
// <STX><ID0><ID1><:><1><=><1><CR>   *??:1=1<Cr> ON  Relay[1]
//=================================================================================================
static union  
{
  byte cmd_buff[9];                                                                              // 8 Byte Max
  //===============================================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
  // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
  // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
  // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
  //===============================================================================================
  struct 
  {
    //=============================================================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte relay;                         // '0':Relay[0],'1':Relay[1]
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //=============================================================================================
  }control_relay;
  //===============================================================================================
  
}rs485;
//=================================================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY0                  = '0';  
const byte      RELAY1                  = '1';
const byte      RELAY_ON                = '0';  
const byte      RELAY_OFF               = '1';
//=================================================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//=================================================================================================

//=================================================================================================
// Initial Slave Address
//=================================================================================================
void InitialSlaveAddress()
{
  //===============================================================================================
  SlaveAddress = 0;
  //===============================================================================================
  if(digitalRead(RS485_ID0_PIN) == HIGH)
  {
    SlaveAddress += 1;
  }
  if(digitalRead(RS485_ID1_PIN) == HIGH)
  {
    SlaveAddress += 2;
  }
  if(digitalRead(RS485_ID2_PIN) == HIGH)
  {
    SlaveAddress += 4;
  }
  if(digitalRead(RS485_ID3_PIN) == HIGH)
  {
    SlaveAddress += 8;
  }
  if(digitalRead(RS485_ID4_PIN) == HIGH)
  {
    SlaveAddress += 16;
  }
  //===============================================================================================
  String StringID = String(SlaveAddress,DEC);
  //===============================================================================================
  if(SlaveAddress < 10)
  {
    SlaveID[0] = '0';
    SlaveID[1] = StringID[0];
  }
  else
  {
    SlaveID[0] = StringID[0];
    SlaveID[1] = StringID[1];
  }
  //===============================================================================================
}
//=================================================================================================

void setup() 
{
  //===============================================================================================
  analogReference(DEFAULT);                                                                       // ADC Reference = +5V
  //===============================================================================================
  pinMode(INTERNAL_RELAY0_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOff);
  pinMode(INTERNAL_RELAY1_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOff);
  //===============================================================================================
  //===============================================================================================
  SerialDebug.begin(115200);
  SerialDebug.println();
  while(!SerialDebug);                                                                            // Wait MEGA32U4 USB Serial Complete
  //===============================================================================================
  SerialDebug.println("======================================");
  bme280_status = bme280.begin(0x76);
  if(!bme280_status)
  {
    SerialDebug.println("Initial BME280...Error");
  }
  SerialDebug.println("Initial BME280...Complete");
  SerialDebug.println();
  SerialDebug.println("======================================");
  //===================================================================================================
  // Initialize DS3231
  SerialDebug.println("Initialize DS3231");;
  clock.begin();
  //===================================================================================================
  clock.armAlarm1(false);
  clock.armAlarm2(false);
  clock.clearAlarm1();
  clock.clearAlarm2();
  //===================================================================================================
  // Manual (Year, Month, Day, Hour, Minute, Second)
  clock.setDateTime(2017, 6, 27, 7, 0, 59);
  //===================================================================================================

  //===============================================================================================
  ds18b20.begin();                                                                                // 1-Wire : DS18B20 Sensor
  //===============================================================================================
  
  //===============================================================================================
  // Initial RS485
  //===============================================================================================
  SerialRS485.begin(115200);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  pinMode(RS485_ID0_PIN, INPUT_PULLUP);
  pinMode(RS485_ID1_PIN, INPUT_PULLUP);
  pinMode(RS485_ID2_PIN, INPUT_PULLUP);
  pinMode(RS485_ID3_PIN, INPUT_PULLUP);
  pinMode(RS485_ID4_PIN, INPUT_PULLUP);
  InitialSlaveAddress();
  //===============================================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println();
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
}
 
void loop() 
{ 
  //======================================================================================================================================
  // Start of Receive & Service RS485 Command
  //======================================================================================================================================
  if(SerialRS485.available() > 0)
  {  
    rs485_chr = SerialRS485.read();
    
    if(rs485_chr == STX_CODE)                                                                     // Verify Start of Command Header
    {
      //===========================================================================================
      // Reset Packet Frame Reset
      //===========================================================================================
      rs485_cmd_head = 1;                                                                         // Command Header Not Complete  
      rs485_cmd_cnt = 0;                                                                          // Re-Start Command Count
      for(int i=0; i<sizeof(rs485.cmd_buff); i++)
      {
        rs485.cmd_buff[i] = '\0';
      }
      rs485_cmd_ptr = &rs485.cmd_buff[0];
      //===========================================================================================
    } 

    //Start of Receive & Verify RS485 Packet Command
    if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
    { 
      *rs485_cmd_ptr = rs485_chr;                                                                 // Save Character 
      rs485_cmd_ptr++;                                                                            // Next Pointer         
      rs485_cmd_cnt++;  

      //===========================================================================================
      // Start of Analysis & Service RS485 Command Packet   
      //===========================================================================================
      if(rs485_chr == 0x0D)
      {           
        //=========================================================================================
        // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
        // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
        // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
        // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
        // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
        //=========================================================================================
        if((rs485.control_relay.head      == STX_CODE)&&                                // head : 0x02
        
           (rs485.control_relay.id[0]     == SlaveID[0])&&                              // id[0]:0..9
           (rs485.control_relay.id[1]     == SlaveID[1])&&                              // id[1]:0..9
           
           (rs485.control_relay.sep1      == SEP1)&&                                    // sep1 : ':' 
           
           ((rs485.control_relay.relay    == '0')||
            (rs485.control_relay.relay    == '1'))&&                                    // '0':Relay[0],'1':Relay[1]
           
           (rs485.control_relay.sep2      == SEP2)&&                                    // sep2 : '='
           
           ((rs485.control_relay.operate  == '0')||
            (rs485.control_relay.operate  == '1'))&&                                    // operate('0':OFF,'1':ON)
           
           (rs485.control_relay.enter     == END_CODE))                                 // Enter:0x0D = End of Packet
        {
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOff);
            
            //=============================================================================================
            // Start of RS485 Test
            //=============================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[0] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=============================================================================================
            // End of RS485 Test
            //=============================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY0_PIN, InternalRelayOn);
            //=============================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[0] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=============================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='1')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOff);
            //=============================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[1] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=============================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='1')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY1_PIN, InternalRelayOn);
            //=============================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Relay[1] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=============================================================================================
          }
          //=======================================================================================
        }
        //=======================================================================================
        // Clear RS485 Buffer
        //=======================================================================================
        rs485_cmd_head = 0;                                                                     // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                                                      // Re-Start Command Count
        rs485.cmd_buff[0] = '\0';                                                               // Null String
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];   
        //=======================================================================================
      } 
      //===========================================================================================
      // End of Analysis & Service RS485 Command Packet   
      //===========================================================================================
    }
  }
  //======================================================================================================================================
  // End of Receive & Service RS485 Command
  //======================================================================================================================================

  //===============================================================================================
  // Start of Loop Time Service
  //===============================================================================================
  if(millis() - lastGetRTCSecondTime > 3000ul)                                                    // 1-Second
  {
    //=============================================================================================
    digitalWrite(LED_LINK_PIN, LED_ON);
    //=============================================================================================

    //=============================================================================================
    // Start of RS485 Hardware Slave ID Test
    //=============================================================================================
    InitialSlaveAddress();
    Serial.print("My RS485 Slave ID : ");
    Serial.write(SlaveID[0]);
    Serial.write(SlaveID[1]);
    Serial.println();
    //=============================================================================================
    // End of RS485 Hardware Slave ID Test
    //=============================================================================================
    
    //=============================================================================================
    // Start of I2C#1(5V) Bus RTC : DS3231 Test
    //=============================================================================================
    dt = clock.getDateTime();
    Serial.println();
    Serial.println(clock.dateFormat("d-m-Y H:i:s - l", dt));
    //=============================================================================================
    // End of I2C#1(5V) Bus RTC : DS3231 Test
    //=============================================================================================

    //=============================================================================================
    // Start of 1-Wire Bus : DS18B20 Test
    //=============================================================================================
    ds18b20.requestTemperatures();                                                                // Send the command to get temperatures
    //=============================================================================================
    SerialDebug.print("DS18B20 Temperature = ");
    SerialDebug.println(ds18b20.getTempCByIndex(0),1);                                            // ttt.t   : Result Temperature(c)
    //=============================================================================================
    // End of 1-Wire Bus : DS18B20 Test
    //=============================================================================================
    
    //=============================================================================================
    // Start of I2C#1(3.3V) Bus : ET-SENSOR BME280 Test
    //=============================================================================================
    bme280_temperature  = bme280.readTemperature();                   // *C
    bme280_humidity     = bme280.readHumidity();                      // %RH
    bme280_pressure     = bme280.readPressure() / 100.0F;             // hPa 
    bme280_altitude     = bme280.readAltitude(SEALEVELPRESSURE_HPA);  // m
    //
    SerialDebug.print("BME280 Temperature = ");
    SerialDebug.print(bme280.readTemperature());
    SerialDebug.println(" *C");
    
    SerialDebug.print("BME280 Pressure = ");
    SerialDebug.print(bme280.readPressure() / 100.0F);
    SerialDebug.println(" hPa");
    
    SerialDebug.print("BME280 Approx. Altitude = ");
    SerialDebug.print(bme280.readAltitude(SEALEVELPRESSURE_HPA));
    SerialDebug.println(" m");
    
    SerialDebug.print("BME280 Humidity = ");
    SerialDebug.print(bme280.readHumidity());
    SerialDebug.println(" %");
    //=============================================================================================
    // End of I2C#1(3.3V) Bus : ET-SENSOR BME280 Test
    //=============================================================================================

    //=============================================================================================
    // Test ADC Analog A0...A5
    //=============================================================================================
    sensorValue0 = analogRead(A0);
    sensorValue1 = analogRead(A1);
    sensorValue2 = analogRead(A2);
    sensorValue3 = analogRead(A3);
    sensorValue4 = analogRead(A4);
    sensorValue5 = analogRead(A5);
    
    voltage0 = sensorValue0 * (5.0 / 1023.0);
    voltage1 = sensorValue1 * (5.0 / 1023.0);
    voltage2 = sensorValue2 * (5.0 / 1023.0);
    voltage3 = sensorValue3 * (5.0 / 1023.0);
    voltage4 = sensorValue4 * (5.0 / 1023.0);
    voltage5 = sensorValue5 * (5.0 / 1023.0);

    Serial.print("Analog A0..A5 : ");
    Serial.print(voltage0);
    Serial.print(" , ");
    Serial.print(voltage1);
    Serial.print(" , ");
    Serial.print(voltage2);
    Serial.print(" , ");
    Serial.print(voltage3);
    Serial.print(" , ");
    Serial.print(voltage4);
    Serial.print(" , ");
    Serial.println(voltage5);
    //=============================================================================================

    
    //=============================================================================================
    // Start of External I2C Output Relay(ET-I2C REL8) Test
    //=============================================================================================
    RelayIndex++;
    RelayIndex &= 0x0F;
    //=============================================================================================
    switch(RelayIndex)
    {
      case 0:
        relay_external_dev0 &= RELAY0_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay0 : ON");
      break;
      
      case 1:
        relay_external_dev0 &= RELAY1_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay1 : ON");
      break;
      
      case 2:
        relay_external_dev0 &= RELAY2_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay2 : ON");
      break;
      
      case 3:
        relay_external_dev0 &= RELAY3_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay3 : ON");
      break;
      
      case 4:
        relay_external_dev0 &= RELAY4_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay4 : ON");
      break;
      
      case 5:
        relay_external_dev0 &= RELAY5_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay5 : ON");
      break;
      
      case 6:
        relay_external_dev0 &= RELAY6_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay6 : ON");
      break;
      
      case 7:
        relay_external_dev0 &= RELAY7_ON_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay7 : ON");
      break;

      case 8:
        relay_external_dev0 |= RELAY0_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay0 : OFF");
      break;
      
      case 9:
        relay_external_dev0 |= RELAY1_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay1 : OFF");
      break;
      
      case 10:
        relay_external_dev0 |= RELAY2_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay2 : OFF");
      break;
      
      case 11:
        relay_external_dev0 |= RELAY3_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay3 : OFF");
      break;
      
      case 12:
        relay_external_dev0 |= RELAY4_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay4 : OFF");
      break;
      
      case 13:
        relay_external_dev0 |= RELAY5_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay5 : OFF");
      break;
      
      case 14:
        relay_external_dev0 |= RELAY6_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay7 : OFF");
      break;
      
      case 15:
        relay_external_dev0 |= RELAY7_OFF_MASK;
        PCF8574_RELAY_EXP0.write8(relay_external_dev0);
        PCF8574_RELAY_EXP1.write8(relay_external_dev0);
        PCF8574_RELAY_EXP2.write8(relay_external_dev0);
        PCF8574_RELAY_EXP3.write8(relay_external_dev0);
        PCF8574_RELAY_EXP4.write8(relay_external_dev0);
        PCF8574_RELAY_EXP5.write8(relay_external_dev0);
        PCF8574_RELAY_EXP6.write8(relay_external_dev0);
        PCF8574_RELAY_EXP7.write8(relay_external_dev0);
        SerialDebug.println("External I2C Relay7 : OFF");
      break;
    }
    //=============================================================================================
    // End of External I2C Output Relay(ET-I2C REL8) Test
    //=============================================================================================
    
    //=============================================================================================
    digitalWrite(LED_LINK_PIN, LED_OFF);
    //=============================================================================================
    //=============================================================================================
    lastGetRTCSecondTime = millis();
    //=============================================================================================
  }
  //===============================================================================================
  // End of Loop Time Service
  //===============================================================================================
 
}


void checkAlarms()
{
  RTCAlarmTime a1;  
  RTCAlarmTime a2;

  if (clock.isArmed1())
  {
    a1 = clock.getAlarm1();

    SerialDebug.print("Alarm1 is triggered ");
    switch (clock.getAlarmType1())
    {
      case DS3231_EVERY_SECOND:
        SerialDebug.println("every second");
        break;
      case DS3231_MATCH_S:
        SerialDebug.print("when seconds match: ");
        SerialDebug.println(clock.dateFormat("__ __:__:s", a1));
        break;
      case DS3231_MATCH_M_S:
        SerialDebug.print("when minutes and sencods match: ");
        SerialDebug.println(clock.dateFormat("__ __:i:s", a1));
        break;
        
      case DS3231_MATCH_H_M_S:
        SerialDebug.print("when hours, minutes and seconds match: ");
        //Serial.println(clock.dateFormat("__ H:i:s", a1));
        SerialDebug.println(clock.dateFormat("H:i:s", a1));
        break;
        
      case DS3231_MATCH_DT_H_M_S:
        SerialDebug.print("when date, hours, minutes and seconds match: ");
        SerialDebug.println(clock.dateFormat("d H:i:s", a1));
        break;
      case DS3231_MATCH_DY_H_M_S:
        SerialDebug.print("when day of week, hours, minutes and seconds match: ");
        SerialDebug.println(clock.dateFormat("l H:i:s", a1));
        break;
      default: 
        SerialDebug.println("UNKNOWN RULE");
        break;
    }
  } else
  {
    SerialDebug.println("Alarm1 is disarmed.");
  }

  if (clock.isArmed2())
  {
    a2 = clock.getAlarm2();

    SerialDebug.print("Alarm2 is triggered ");
    switch (clock.getAlarmType2())
    {
      case DS3231_EVERY_MINUTE:
        SerialDebug.println("every minute");
        break;
      case DS3231_MATCH_M:
        SerialDebug.print("when minutes match: ");
        SerialDebug.println(clock.dateFormat("__ __:i:s", a2));
        break;
      case DS3231_MATCH_H_M:
        SerialDebug.print("when hours and minutes match:");
        SerialDebug.println(clock.dateFormat("__ H:i:s", a2));
        break;
      case DS3231_MATCH_DT_H_M:
        SerialDebug.print("when date, hours and minutes match: ");
        SerialDebug.println(clock.dateFormat("d H:i:s", a2));
        break;
      case DS3231_MATCH_DY_H_M:
        SerialDebug.println("when day of week, hours and minutes match: ");
        SerialDebug.print(clock.dateFormat("l H:i:s", a2));
        break;
      default: 
        SerialDebug.println("UNKNOWN RULE"); 
        break;
    }
  } else
  {
    SerialDebug.println("Alarm2 is disarmed.");
  }
}



