/*
 * Demo ET-MEGA32U4-RS485 Hardware Board
 * MCU    : ATMEGA32U4
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * ADC    : Analog#0 : A;[0..3]
 *        : Analog#1 : A4
 *        : Analog#2 : A5
 * Output : Link(LED Active High:D13)
 *        : Relay0(Active Low:D6)
 *        : Relay1(Active Low:D7)
 * Input  : RS485 ID0:D8
 *        : RS485 ID1:D9
 *        : RS485 ID2:D10
 *        : RS485 ID3:D11
 *        : RS485 ID4:D12
 * 1-Wire : DS18B20(1-Wire:D5)
 * 
 * Demo   : I2C : OLED 1.3 I2C
 *        : HK1100C
 */

//=================================================================================================
#include "U8glib.h"
//=================================================================================================
// setup u8g object, please remove comment from one of the following constructor calls
// IMPORTANT NOTE: The following list is incomplete. The complete list of supported 
// devices with all constructor calls is here: https://github.com/olikraus/u8glib/wiki/device

//U8GLIB_SH1106_128X64 u8g(13, 11, 10, 9);  // SW SPI Com: SCK = 13, MOSI = 11, CS = 10, A0 = 9
//U8GLIB_SH1106_128X64 u8g(4, 5, 6, 7); // SW SPI Com: SCK = 4, MOSI = 5, CS = 6, A0 = 7 (new blue HalTec OLED)
U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_NONE); // I2C / TWI 
//U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_DEV_0|U8G_I2C_OPT_FAST); // Dev 0, Fast I2C / TWI
//U8GLIB_SH1106_128X64 u8g(U8G_I2C_OPT_NO_ACK); // Display which does not send ACK

const unsigned char ETT_logo [] PROGMEM = 
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1F, 0xEF, 0xFB, 0xFE,
  0x20, 0x00, 0x00, 0x00, 0x27, 0xEF, 0x7B, 0xDE, 0x28, 0x01, 0x40, 0x50, 0x28, 0x01, 0x40, 0x50,
  0x28, 0x01, 0x40, 0x50, 0x2F, 0xE1, 0x40, 0x50, 0x20, 0x01, 0x40, 0x50, 0x2F, 0xE1, 0x40, 0x50,
  0x28, 0x01, 0x40, 0x50, 0x28, 0x01, 0x40, 0x50, 0x28, 0x01, 0x40, 0x50, 0x27, 0xE1, 0x40, 0x50,
  0x20, 0x01, 0x40, 0x50, 0x1F, 0xE1, 0x40, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
//=================================================================================================


//=================================================================================================
// USART RS232/RS485
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
//=================================================================================================
//=================================================================================================
// Pressure Sensor(ADC)
//=================================================================================================
const float HK1100C_min_sensor     = 0.5;
const float HK1100C_max_sensor     = 4.5;
//=================================================================================================
const float HK1100C_min_result_bar = 0.0;
const float HK1100C_max_result_bar = 12.0;
//=================================================================================================
const float HK1100C_min_result_MPa = 0.0;
const float HK1100C_max_result_MPa = 1.2;
//=================================================================================================
const float HK1100C_min_result_psi = 0.0;
const float HK1100C_max_result_psi = 174.0452856;
//=================================================================================================
const float full_scale_adc_voltage = 5.0;                                                         // ADC 0..5VDC
const float full_scale_adc_result  = 1023.0;                                                      // ADC 10Bit 0..1023
//=================================================================================================
const int Sensor_HK1100C_Pin = A5;                                                                // ADC Interface HK1100C
int Sensor_HK1100C_Value = 0;                                                                     // value read from the pot
//=================================================================================================
float adc_volt;
float pressure;
//=================================================================================================
float hk1100c_bar;
float hk1100c_psi;
float hk1100c_MPa;
//=================================================================================================
//=================================================================================================
unsigned long lastGetHK1100CPressureTime = 0;
//=================================================================================================

void setup() 
{
  //===============================================================================================
  analogReference(DEFAULT);                                                                       // ADC Reference = +5V
  //===============================================================================================
  SerialDebug.begin(115200);                                                                      // Debug USART
  //while(!SerialDebug);                                                                            // wait for serial port to connect. Needed for Leonardo only 
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  // Start of Read Pressure Sensor HK1100C
  //===============================================================================================
  if(millis() - lastGetHK1100CPressureTime > 1000ul)                                              // 1-Second
  {
    //=============================================================================================
    // Sensor  : Pressure Transducer Sensor Oil Fuel Diesel Gas Water Air Sensor
    //=============================================================================================
    // Signal  : Red    = +5.0 VDC
    //         : Black  = 0VDC(GND)
    //         : Yellow = Output(0.5-4.5VDC)
    //=============================================================================================
    // Output  : 0.5-4.5VDC(0-12 bar, 0-1.2 MPa, 0-174.0452856 PSI)
    //=============================================================================================
    //=============================================================================================
    // 0.5 - 4.5v = 0 - 12 bar
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 12.0 / 4.0 = 3.0
    // bar = (n - 0.5) * 3.0
    //=============================================================================================

    //=============================================================================================
    // 0.5 - 4.5v = 0 - 174.0452856 psi
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 174.0452856 / 4.0 = 43.5113214
    // psi = (n - 0.5) * 43.5113214
    //=============================================================================================
  
    //=============================================================================================
    // 0.5 - 4.5v = 0 - 1.2MPa
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 1.2 / 4.0 = 0.3
    // MPa = (n - 0.5) * 0.3
    //=============================================================================================
    Sensor_HK1100C_Value = analogRead(Sensor_HK1100C_Pin);
    adc_volt = Sensor_HK1100C_Value * (full_scale_adc_voltage / full_scale_adc_result);
    //=============================================================================================
  
    //=============================================================================================
    if(adc_volt < HK1100C_min_sensor)    
    {
      pressure = HK1100C_min_sensor;                                                                // Min Scale Adjust
    }
    else if(adc_volt > HK1100C_max_sensor)
    {
      pressure = HK1100C_max_sensor;                                                                // Max Scale Adjust
    }
    else
    {
      pressure = adc_volt;                                                                          // Normal Scale
    }
    //=============================================================================================
  
    //=============================================================================================
    hk1100c_bar = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_bar - HK1100C_min_result_bar)/(HK1100C_max_sensor - HK1100C_min_sensor));
    hk1100c_psi = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_psi - HK1100C_min_result_psi)/(HK1100C_max_sensor - HK1100C_min_sensor));
    hk1100c_MPa = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_MPa - HK1100C_min_result_MPa)/(HK1100C_max_sensor - HK1100C_min_sensor));
    //=============================================================================================
  
    //=============================================================================================
    SerialDebug.print("sensor HK1100C = ");
    SerialDebug.print(adc_volt);
    SerialDebug.print(" volt \t output = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_bar);
    SerialDebug.print("\t bar \t = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_psi);
    SerialDebug.print("\t PSI \t = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_MPa);
    SerialDebug.println("\t MPa");
    //=============================================================================================

    //=============================================================================================
    // Start of Display OLED
    //=============================================================================================
    u8g.firstPage();  
    do {
    
      u8g.drawBitmapP( 0, 0, 4, 20, ETT_logo);
  
      u8g.setFont(u8g_font_6x10);

      u8g.drawStr( 40, 10, "OLED 1.3");
      u8g.drawStr( 40, 20, "I2C");
 
      u8g.setPrintPos(0, 30); 
      u8g.print("HK1100C : ");
      u8g.print(adc_volt);
      u8g.print(" Volt");
    
      u8g.setPrintPos(0, 40); 
      u8g.print("HK1100C : ");
      u8g.print(hk1100c_bar);
      u8g.print(" bar");
    
      u8g.setPrintPos(0, 50); 
      u8g.print("HK1100C : ");
      u8g.print(hk1100c_psi);
      u8g.print(" PSI");
    
      u8g.setPrintPos(0, 60); 
      u8g.print("HK1100C : ");
      u8g.print(hk1100c_MPa);
      u8g.print(" MPa");
    } while( u8g.nextPage() );
    //=============================================================================================
    // End of Display OLED
    //=============================================================================================
    
    //=============================================================================================
    lastGetHK1100CPressureTime = millis();
    //=============================================================================================
  }  
  //===============================================================================================
  // End of Read Pressure Sensor HK1100C
  //===============================================================================================
}

