/*
 * Demo ET-MEGA32U4-RS485 Hardware Board
 * MCU    : ATMEGA32U4
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * ADC    : Analog#0 : A;[0..3]
 *        : Analog#1 : A4
 *        : Analog#2 : A5
 * Output : Link(LED Active High:D13)
 *        : Relay0(Active Low:D6)
 *        : Relay1(Active Low:D7)
 * Input  : RS485 ID0:D8
 *        : RS485 ID1:D9
 *        : RS485 ID2:D10
 *        : RS485 ID3:D11
 *        : RS485 ID4:D12
 * 1-Wire : DS18B20(1-Wire:D5)
 * 
 * Demo   : ADC(Analog#3:A5) Interface Pressure Sensor : HK1100C
 */
 
//=================================================================================================
// USART RS232
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
//=================================================================================================

//=================================================================================================
// Pressure Sensor(ADC)
//=================================================================================================
const float HK1100C_min_sensor     = 0.5;
const float HK1100C_max_sensor     = 4.5;
//=================================================================================================
const float HK1100C_min_result_bar = 0.0;
const float HK1100C_max_result_bar = 12.0;
//=================================================================================================
const float HK1100C_min_result_MPa = 0.0;
const float HK1100C_max_result_MPa = 1.2;
//=================================================================================================
const float HK1100C_min_result_psi = 0.0;
const float HK1100C_max_result_psi = 174.0452856;
//=================================================================================================
const float full_scale_adc_voltage = 5.0;                                                         // ADC 0..5VDC
const float full_scale_adc_result  = 1023.0;                                                      // ADC 10Bit 0..1023
//=================================================================================================
const int Sensor_HK1100C_Pin = A5;                                                                // ADC Interface HK1100C
int Sensor_HK1100C_Value = 0;                                                                     // value read from the pot
//=================================================================================================
float adc_volt;
float pressure;
//=================================================================================================
float hk1100c_bar;
float hk1100c_psi;
float hk1100c_MPa;
//=================================================================================================
//=================================================================================================
unsigned long lastGetHK1100CPressureTime = 0;
//=================================================================================================

void setup() 
{
  //===============================================================================================
  analogReference(DEFAULT);                                                                       // ADC Reference = +5V
  //===============================================================================================
  SerialDebug.begin(115200);                                                                      // Debug USART
  while(!SerialDebug);                                                                            // wait for serial port to connect. Needed for Leonardo only 
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  // Start of Read Pressure Sensor HK1100C
  //===============================================================================================
  if(millis() - lastGetHK1100CPressureTime > 1000ul)                                              // 1-Second
  {
    //=============================================================================================
    // Sensor  : Pressure Transducer Sensor Oil Fuel Diesel Gas Water Air Sensor
    //=============================================================================================
    // Signal  : Red    = +5.0 VDC
    //         : Black  = 0VDC(GND)
    //         : Yellow = Output(0.5-4.5VDC)
    //=============================================================================================
    // Output  : 0.5-4.5VDC(0-12 bar, 0-1.2 MPa, 0-174.0452856 PSI)
    //=============================================================================================
    //=============================================================================================
    // 0.5 - 4.5v = 0 - 12 bar
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 12.0 / 4.0 = 3.0
    // bar = (n - 0.5) * 3.0
    //=============================================================================================

    //=============================================================================================
    // 0.5 - 4.5v = 0 - 174.0452856 psi
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 174.0452856 / 4.0 = 43.5113214
    // psi = (n - 0.5) * 43.5113214
    //=============================================================================================
  
    //=============================================================================================
    // 0.5 - 4.5v = 0 - 1.2MPa
    //=============================================================================================
    // 4.5 - 0.5 = 4.0
    // 1.2 / 4.0 = 0.3
    // MPa = (n - 0.5) * 0.3
    //=============================================================================================
    Sensor_HK1100C_Value = analogRead(Sensor_HK1100C_Pin);
    adc_volt = Sensor_HK1100C_Value * (full_scale_adc_voltage / full_scale_adc_result);
    //=============================================================================================
  
    //=============================================================================================
    if(adc_volt < HK1100C_min_sensor)    
    {
      pressure = HK1100C_min_sensor;                                                                // Min Scale Adjust
    }
    else if(adc_volt > HK1100C_max_sensor)
    {
      pressure = HK1100C_max_sensor;                                                                // Max Scale Adjust
    }
    else
    {
      pressure = adc_volt;                                                                          // Normal Scale
    }
    //=============================================================================================
  
    //=============================================================================================
    hk1100c_bar = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_bar - HK1100C_min_result_bar)/(HK1100C_max_sensor - HK1100C_min_sensor));
    hk1100c_psi = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_psi - HK1100C_min_result_psi)/(HK1100C_max_sensor - HK1100C_min_sensor));
    hk1100c_MPa = (pressure - HK1100C_min_sensor) * ((HK1100C_max_result_MPa - HK1100C_min_result_MPa)/(HK1100C_max_sensor - HK1100C_min_sensor));
    //=============================================================================================
  
    //=============================================================================================
    SerialDebug.print("sensor HK1100C = ");
    SerialDebug.print(adc_volt);
    SerialDebug.print(" volt \t output = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_bar);
    SerialDebug.print("\t bar \t = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_psi);
    SerialDebug.print("\t PSI \t = ");
    //=============================================================================================
    SerialDebug.print(hk1100c_MPa);
    SerialDebug.println("\t MPa");
    //=============================================================================================
    
    //=============================================================================================
    lastGetHK1100CPressureTime = millis();
    //=============================================================================================
  }  
  //===============================================================================================
  // End of Read Pressure Sensor HK1100C
  //===============================================================================================
}

