/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : QC Hardware I/O Test
 *        : I2C 4 Pin    -> SHT10
 *        : I2C 5 Pin    -> BME280
 *        : External I2C -> OLED
 *        : RS485        -> RS485 Converter Send Command Baudrate 115200BPS Control Relay
 *        :              -> "*99:0=0"<Cr> = OFF Relay
 *        :              -> "*99:0=1"<Cr> = ON  Relay
 */

//=================================================================================================
#include <Wire.h>
//=================================================================================================
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>
//=================================================================================================
#include "SH1106.h"
//=================================================================================================
#include "SHT1x.h"                                                                                // SHT10
//=================================================================================================
#include "DS3231.h"
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//=================================================================================================
#define SerialRS485 Serial                                                                       // Serial1(D1=TXD,D0=RXD)
#define RS485_DIRECTION_PIN D0                                                                   // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//=================================================================================================
#define I2C_SCL1_PIN        D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//=================================================================================================
#define RESERVE1            D3                                                                   // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//=================================================================================================
#define SHT10_SCL2_PIN      D4                                                                   // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                   // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//=================================================================================================
#define LED_WIFI_PIN        D6                                                                   // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//=================================================================================================
#define LED_LINK_PIN        D7                                                                   // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//=================================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                   // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//=================================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//=================================================================================================

//=================================================================================================
Adafruit_BME280 bme280;                 // I2C
//=================================================================================================
#define SEALEVELPRESSURE_HPA (1013.25)
float bme280_temperature;
float bme280_humidity;
float bme280_pressure;  
float bme280_altitude;
bool  bme280_status;
//=================================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=================================================================================================

//=====================================
SHT1x SHT10_BUS2(SHT10_SDA2_PIN, SHT10_SCL2_PIN);
//======================================
float sht10_bus2_temperature;
float sht10_bus2_humidity;
unsigned long lastSHT10RequestTime = 0;
//======================================

//=================================================================================================
SH1106 display(0x3c, I2C_SDA1_PIN, I2C_SCL1_PIN); 
//=================================================================================================
#define ett_logo_width  32
#define ett_logo_height 20
const char ett_logo[] PROGMEM = 
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0xF8, 0xF7, 0xDF, 0x7F, 0x04, 0x00, 0x00, 0x00, 0xE4, 0xF7, 0xDE, 0x7B, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xF4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF4, 0x87, 0x02, 0x0A, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xE4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF8, 0x87, 0x02, 0x0A, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
//=================================================================================================

//=================================================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
// <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
// <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
// <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
//=================================================================================================
static union  
{
  byte cmd_buff[9];                                                                               // 8 Byte Max
  //===============================================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
  // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
  // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
  // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
  //===============================================================================================
  struct 
  {
    //=============================================================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte relay;                         // '0':Relay[0],'1':Relay[1]
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //=============================================================================================
  }control_relay;
  //===============================================================================================
  
}rs485;
//=================================================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY0                  = '0';  
const byte      RELAY1                  = '1';
const byte      RELAY_ON                = '0';  
const byte      RELAY_OFF               = '1';
//=================================================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//=================================================================================================

//=================================================================================================
int minutes;
int seconds; 
//=================================================================================================
String TimeString = "";
//=================================================================================================

//=================================================================================================
// RTC:DS2231
//=================================================================================================
DS3231 clock;
RTCDateTime dt;
//=================================================================================================


void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //===============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //===============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //===============================================================================================
  Serial.begin(115200);
  //===============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  
  //===============================================================================================
  //Serial.println("BME280 I2C(0x76) Test");
  bme280_status = bme280.begin(0x76);
  if(!bme280_status)
  {
    //Serial.println("Initial BME280...Error");
    while(1);
  }
  //===============================================================================================
  //===============================================================================================
  clock.begin();
  clock.armAlarm1(false);
  clock.armAlarm2(false);
  clock.clearAlarm1();
  clock.clearAlarm2();
 
  // Manual (Year, Month, Day, Hour, Minute, Second)
  clock.setDateTime(2017, 1, 1, 0, 0, 0);
  //===============================================================================================
  //===============================================================================================
  display.init();
  display.flipScreenVertically();
  display.setFont(ArialMT_Plain_10);

  display.clear();
  display.drawXbm(0, 0, ett_logo_width, 20,  ett_logo);
  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.display();
  //===============================================================================================
  //===============================================================================================
  // Initial RS485
  //===============================================================================================
  SerialRS485.begin(115200);
  SerialRS485.setDebugOutput(false);  //Disable Debug on Serial
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write('9');
  SerialRS485.write('9');
  SerialRS485.println();
  SerialRS485.println("Send Command *99:0=0<Cr> For OFF Relay");
  SerialRS485.println("Send Command *99:0=1<Cr> For ON  Relay");
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
}
 
void loop() 
{ 
  //===============================================================================================
  // Start of Receive & Service RS485 Command
  //===============================================================================================
  if(SerialRS485.available() > 0)
  {  
    rs485_chr = SerialRS485.read();
    
    if(rs485_chr == STX_CODE)                                                                     // Verify Start of Command Header
    {
      //===========================================================================================
      // Reset Packet Frame Reset
      //===========================================================================================
      rs485_cmd_head = 1;                                                                         // Command Header Not Complete  
      rs485_cmd_cnt = 0;                                                                          // Re-Start Command Count
      for(int i=0; i<sizeof(rs485.cmd_buff); i++)
      {
        rs485.cmd_buff[i] = '\0';
      }
      rs485_cmd_ptr = &rs485.cmd_buff[0];
      //===========================================================================================
    } 

    //Start of Receive & Verify RS485 Packet Command
    if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
    { 
      *rs485_cmd_ptr = rs485_chr;                                                                 // Save Character 
      rs485_cmd_ptr++;                                                                            // Next Pointer         
      rs485_cmd_cnt++;  

      //SerialDebug.write(rs485_chr);
        
      //===========================================================================================
      // Start of Analysis & Service RS485 Command Packet   
      //===========================================================================================
      if(rs485_chr == 0x0D)
      {           
        //=========================================================================================
        // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
        // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
        // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
        // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
        // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
        //=========================================================================================
        if((rs485.control_relay.head      == STX_CODE)&&                                          // head : 0x02
        
           (rs485.control_relay.id[0]     == '9')&&                                               // id[0]:0..9
           (rs485.control_relay.id[1]     == '9')&&                                               // id[1]:0..9
           
           (rs485.control_relay.sep1      == SEP1)&&                                              // sep1 : ':' 
           
           ((rs485.control_relay.relay    == '0')||
            (rs485.control_relay.relay    == '1'))&&                                              // '0':Relay[0],'1':Relay[1]
           
           (rs485.control_relay.sep2      == SEP2)&&                                              // sep2 : '='
           
           ((rs485.control_relay.operate  == '0')||
            (rs485.control_relay.operate  == '1'))&&                                              // operate('0':OFF,'1':ON)
           
           (rs485.control_relay.enter     == END_CODE))                                           // Enter:0x0D = End of Packet
        {
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
            digitalWrite(LED_WIFI_PIN, LedOFF);
            //=====================================================================================
            // Start of RS485 Test
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write('9');
            SerialRS485.write('9');
            SerialRS485.println("...Relay[0] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
            // End of RS485 Test
            //=====================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
            digitalWrite(LED_WIFI_PIN, LedON);
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write('9');
            SerialRS485.write('9');
            SerialRS485.println("...Relay[0] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
          }
          //=======================================================================================
        }
        //=========================================================================================
        // Clear RS485 Buffer
        //=========================================================================================
        rs485_cmd_head = 0;                                                                      // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                                                       // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];   
        //=========================================================================================
      } 
      //===========================================================================================
      // End of Analysis & Service RS485 Command Packet   
      //===========================================================================================
    }
  }
  //===============================================================================================
  // End of Receive & Service RS485 Command
  //===============================================================================================
  
  //===============================================================================================
  // Start of Read I2C Sensor(BME280)
  //===============================================================================================
  if(millis() - lastGetI2CSensorTime > 3000ul)                        // 3-Second
  {
    //=============================================================================================
    digitalWrite(LED_LINK_PIN, LedON);
    //=============================================================================================
    bme280_temperature  = bme280.readTemperature();                   // *C
    bme280_humidity     = bme280.readHumidity();                      // %RH
    bme280_pressure     = bme280.readPressure() / 100.0F;             // hPa 
    bme280_altitude     = bme280.readAltitude(SEALEVELPRESSURE_HPA);  // m
    //
    sht10_bus2_temperature = SHT10_BUS2.readTemperatureC();
    sht10_bus2_humidity = SHT10_BUS2.readHumidity();
    //=============================================================================================
    
    //=============================================================================================
    // Start of Display OLED
    //=============================================================================================
    display.clear();
    
    display.drawXbm(0, 0, ett_logo_width, ett_logo_height,  ett_logo);
    //
    int sensorValue = analogRead(A0);
    float volt = 3.30*(float)sensorValue/1023.00;
    String svolt = "ADC : " + String(volt) + " V";
    display.drawString(40, 0, svolt);
 
    dt = clock.getDateTime();
    display.drawString(40, 10, "RTC : ");
    display.drawString(70, 10, clock.dateFormat("H:i:s", dt));

    String stemp1 = "BME280 Temp. : " + String(bme280.readTemperature()) + " *C";
    display.drawString(0, 20, stemp1);
    
    String shumi1 = "BME280 Humi. : " + String(bme280.readHumidity()) + " %RH";
    display.drawString(0, 30, shumi1);
    
    String stemp2 = "SHT10 Temp. : " + String(sht10_bus2_temperature) + " *C";
    display.drawString(0, 40, stemp2);
    
    String shumi2 = "SHT10 Humi. : " + String(sht10_bus2_humidity) + " %RH";
    display.drawString(0, 50, shumi2);
    display.display();
    //=============================================================================================
    // End of Display OLED
    //=============================================================================================
    digitalWrite(LED_LINK_PIN, LedOFF);
    //=============================================================================================
    lastGetI2CSensorTime = millis();
    //=============================================================================================
  }
}

