/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : BLYNK -> "ET-Demo-IoT"
 *        : Blynk library V0.4.8
 *        : I2C#2 -> SHT10(Temperature,Humidity)
 *        : Relay
 *        : LED_LINK = Blynk Application Connect/Disconnect   
 *        : Blynk Widget Setup
 *        : -> SHT10 Temperature    -> Visual V0  = Gauge & History Graph
 *        : -> SHT10 Humidity       -> Visual V1  = Gauge & History Graph
 *        : -> Relay ON/OFF Control -> Visual V2  = Button 
 *        : -> Wifi Signal Level    -> Visual V3  = Level-H 
 */
 
//===============================================================================================
#define BLYNK_PRINT Serial    // Comment this out to disable prints and save space
//===============================================================================================
#include <ESP8266WiFi.h>
#include <BlynkSimpleEsp8266.h>                                                                  // Blynk
//===============================================================================================
#include "SHT1x.h"                                                                               // SHT10
//===============================================================================================

//===============================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//===============================================================================================
#define RS485_DIRECTION_PIN D0                                                                   // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===============================================================================================
#define I2C_SCL1_PIN        D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//
#define BME280_SCL1_PIN     D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//
#define SHT10_SCL1_PIN      D1                                                                   // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                   // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//===============================================================================================
#define RESERVE1            D3                                                                   // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//===============================================================================================
#define SHT10_SCL2_PIN      D4                                                                   // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                   // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//===============================================================================================
#define LED_WIFI_PIN        D6                                                                   // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//===============================================================================================
#define LED_LINK_PIN        D7                                                                   // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//===============================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                   // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//===============================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//===============================================================================================

//===============================================================================================
SHT1x SHT10_BUS2(SHT10_SDA2_PIN, SHT10_SCL2_PIN);
//===============================================================================================
float sht10_bus2_temperature;
float sht10_bus2_humidity;
//===============================================================================================

//===============================================================================================
// Project    : ET-Demo-IoT :
//===============================================================================================
char auth[] = "your blynk auth token code";
char ssid[] = "your wifi ssid";
char pass[] = "your password";
//===============================================================================================
BlynkTimer timer1;
bool BlynkConnectResult;
int BlynkService = 0;
//===============================================================================================
int relayState = InternalRelayOff;
//===============================================================================================

//===============================================================================================
// Blynk Free Account = 2000 Energy Level
//===============================================================================================
#define Widget_Temperature  V0   // Temperature(SHT10)    -> Gauge(300 Energy Level)
#define Widget_Humidity     V1   // Humidity(SHT10)       -> Gauge(300 Energy Level)
                                 // V0:Temp,V1:Humi       -> History Graph(900 Energy Level)
//===============================================================================================
#define Widget_ControlRelay V2   // Relay ON/OFF Control  -> Button(200 Energy Level)
//===============================================================================================
#define Widget_WiFiSignal   V3   // Wifi Signal Level     -> Level-H(200 Energy Level)
//===============================================================================================

//===============================================================================================
int WiFiSignal;
bool isFirstConnect = true;
unsigned long lastConnectionAttempt = millis();
//===============================================================================================

//===============================================================================================
// This function is called every time Blynk gets connected to the server. 
// It’s convenient to call sync functions here.
//===============================================================================================
BLYNK_CONNECTED()                                                                                // Every time we connect to the cloud...
{
  //=============================================================================================
  digitalWrite(LED_WIFI_PIN, WifiConnect);                                                       // ON LED
  //=============================================================================================
  if(isFirstConnect) 
  {
    //===========================================================================================
    Blynk.syncAll();                                                                             // Request Blynk server to re-send latest values for all pins
    //===========================================================================================                  
   
    //===========================================================================================
    Blynk.syncVirtual(Widget_ControlRelay);                                                      // Request the latest state from the server
    //=========================================================================================== 
  
    //===========================================================================================
    isFirstConnect = false;
    //===========================================================================================
  }  
  //=============================================================================================
}
//===============================================================================================
// Every time we connect to the cloud...
//===============================================================================================

//===============================================================================================
// Button Widget Service
// Setup Button Widget Output = Visual(V?)
//===============================================================================================
BLYNK_WRITE(Widget_ControlRelay)
{
  //=============================================================================================
  int thisState = param.asInt();                                                                 // Button Widget Output Value(0,1)
  int lastState = digitalRead(INTERNAL_RELAY_PIN);                                               // Relay Pin Control Logic State
  //=============================================================================================
  if(thisState != lastState)                                                                     // If State Change = Update Pin State
  {
    digitalWrite(INTERNAL_RELAY_PIN, thisState);                                                 // Update Relay Status(ON/OFF)
  } 
  //=============================================================================================
}
//===============================================================================================
// Button Widget Service
//===============================================================================================


void BlynkServiceTimer()
{
  //=============================================================================================
  digitalWrite(LED_LINK_PIN, LedON);
  //=============================================================================================

  //=============================================================================================
  WiFiSignal = map(WiFi.RSSI(), -105, -40, 0, 100);
  //=============================================================================================
  
  switch(BlynkService)
  {
    case 0:  
      sht10_bus2_temperature = SHT10_BUS2.readTemperatureC(); 
    break;

    case 1:
      sht10_bus2_humidity = SHT10_BUS2.readHumidity();
    break;

    case 2: 
      //=========================================================================================
      // Start of Blynk Update
      //=========================================================================================
      Blynk.virtualWrite(Widget_Temperature, sht10_bus2_temperature);
      //=========================================================================================
      Blynk.virtualWrite(Widget_Humidity, sht10_bus2_humidity);
      //=========================================================================================
      Blynk.virtualWrite(Widget_WiFiSignal, WiFiSignal);
      //=========================================================================================
      Blynk.syncVirtual(Widget_ControlRelay);                                                    // Request the latest state from the server
      //=========================================================================================
    break;

    default:
    break;
  }
  
  BlynkService++;
  if(BlynkService > 2)
  {
    BlynkService = 0; 
  }
  //=============================================================================================
  digitalWrite(LED_LINK_PIN, LedOFF);
  //=============================================================================================
}

void setup() 
{
  //=============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //=============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //=============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //=============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);                                  
  //=============================================================================================
  Serial.begin(115200);
  //=============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //=============================================================================================
  
  //=============================================================================================
  sht10_bus2_temperature = SHT10_BUS2.readTemperatureC();
  sht10_bus2_humidity = SHT10_BUS2.readHumidity();
  //=============================================================================================
 
  //=============================================================================================
  // Start of Blynk Application Interface
  //=============================================================================================
  Blynk.begin(auth, ssid, pass);
  //=============================================================================================
 
  //=============================================================================================
  WiFiSignal = map(WiFi.RSSI(), -105, -40, 0, 100);
  //=============================================================================================
  
  //=============================================================================================
  timer1.setInterval(2000L, BlynkServiceTimer);                                                  // 2 Second Update Sensor
  //=============================================================================================
}

void loop() 
{ 
  //=============================================================================================
  // Start of Handle Disconnect
  //=============================================================================================
  if(WiFi.status() != WL_CONNECTED) 
  {
    digitalWrite(LED_WIFI_PIN, WifiDisconnect);                                                  // OFF LED
    if(millis() - lastConnectionAttempt > 5000ul)                                                // 5-Second
    {
      if (pass && strlen(pass))
      {
        WiFi.begin((char*)ssid, (char*)pass);
      }
      else
      {
        WiFi.begin((char*)ssid);
      }
      lastConnectionAttempt = millis();
    }
  }
  else
  {
    digitalWrite(LED_WIFI_PIN, WifiConnect);                                                     // ON LED
  }
  //=============================================================================================
  Blynk.run();
  timer1.run(); 
  //=============================================================================================
  //=============================================================================================
  // End of Handle Disconnect
  //=============================================================================================
}

