/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : Internal I2C RTC : DS3231 
 * 
 */
 
//=================================================================================================
#include <Wire.h>                    // I2C Bus
#include "DS3231.h"
//=================================================================================================
#define SerialDebug Serial                                                                        // USB Serial
//=================================================================================================

//===================================================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================
#define RS485_DIRECTION_PIN D0                                                                                    // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================================================================
#define I2C_SCL1_PIN        D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//===================================================================================================================
#define RESERVE1            D3                                                                                    // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//===================================================================================================================
#define SHT10_SCL2_PIN      D4                                                                                    // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                                    // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//===================================================================================================================
#define LED_WIFI_PIN        D6                                                                                    // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//===================================================================================================================
#define LED_LINK_PIN        D7                                                                                    // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//===================================================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                                    // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//===================================================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================


//=================================================================================================
int minutes;
int seconds; 
//=================================================================================================
String TimeString = "";
//=================================================================================================

//=================================================================================================
// RTC:DS2231
//=================================================================================================
DS3231 clock;
RTCDateTime dt;
//=================================================================================================

//===================================================================================================================
unsigned long lastSecondTime = 0;
//===================================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //===============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //===============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //===============================================================================================
  Serial.begin(115200);
  //===============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================

  
  //===============================================================================================
  Serial.println();
  Serial.println("Demo Test I2C RTC:DS3231");
  //===================================================================================================
  // Initialize DS3231
  Serial.println("Initialize DS3231");
  clock.begin();

  // Disarm alarms and clear alarms for this example, because alarms is battery backed.
  // Under normal conditions, the settings should be reset after power and restart microcontroller.
  clock.armAlarm1(false);
  clock.armAlarm2(false);
  clock.clearAlarm1();
  clock.clearAlarm2();
 
  // Manual (Year, Month, Day, Hour, Minute, Second)
  //clock.setDateTime(2014, 4, 25, 0, 0, 0);

  // Set Alarm - Every second.
  // DS3231_EVERY_SECOND is available only on Alarm1.
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm1(0, 0, 0, 0, DS3231_EVERY_SECOND);

  // Set Alarm - Every full minute.
  // DS3231_EVERY_MINUTE is available only on Alarm2.
  // setAlarm2(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm2(0, 0, 0, 0, DS3231_EVERY_MINUTE);
  
  // Set Alarm1 - Every 20s in each minute
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  //clock.setAlarm1(0, 0, 0, 20, DS3231_MATCH_S);

  // Set Alarm2 - Every 01m in each hour
  // setAlarm2(Date or Day, Hour, Minute, Mode, Armed = true)
  //clock.setAlarm2(0, 0, 1,     DS3231_MATCH_M);

  // Set Alarm - Every 01m:25s in each hour
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm1(0, 0, 1, 25, DS3231_MATCH_M_S);

  // Set Alarm - Every 01h:10m:30s in each day
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm1(0, 1, 10, 30, DS3231_MATCH_H_M_S);

  // Set Alarm - 07h:00m:00s in 25th day in month
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm1(25, 7, 0, 0, DS3231_MATCH_DT_H_M_S);

  // Set Alarm - 10h:45m:30s in every Friday (1 - Mon, 7 - Sun)
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  // clock.setAlarm1(5, 10, 40, 30, DS3231_MATCH_DY_H_M_S);

  //===================================================================================================
  // Manual (Year, Month, Day, Hour, Minute, Second)
  clock.setDateTime(2017, 6, 27, 7, 0, 59);

  // Set Alarm - Every 07h:01m:30s in each day
  // setAlarm1(Date or Day, Hour, Minute, Second, Mode, Armed = true)
  clock.setAlarm1(0, 7, 1, 30, DS3231_MATCH_H_M_S);
  
  // Check alarm settings
  checkAlarms();
  //===================================================================================================
}

void loop() 
{
  //===============================================================================================
  // Start of Read 1-Wire Temperature
  //===============================================================================================
  if(millis() - lastSecondTime > 1000ul)                         // 1-Second
  {
    //=============================================================================================
    // Start of RTC Service
    //=============================================================================================
    dt = clock.getDateTime();
    Serial.println(clock.dateFormat("d-m-Y H:i:s - l", dt));

    // Call isAlarm1(false) if you want clear alarm1 flag manualy by clearAlarm1();
    //if(clock.isAlarm1())
    if(clock.isAlarm1(false))
    {
      Serial.println("ALARM 1 TRIGGERED!");
      clock.clearAlarm1();
    }

    // Call isAlarm2(false) if you want clear alarm1 flag manualy by clearAlarm2();
    //if (clock.isAlarm2())
    //{
    //  Serial.println("ALARM 2 TRIGGERED!");
    //}
    //=============================================================================================
    // End of RTC Service
    //=============================================================================================
    
    //=============================================================================================
    lastSecondTime = millis();
    //=============================================================================================
  }
  //===============================================================================================
  // End of Read 1-Wire Temperature
  //===============================================================================================
}

void checkAlarms()
{
  RTCAlarmTime a1;  
  RTCAlarmTime a2;

  if (clock.isArmed1())
  {
    a1 = clock.getAlarm1();

    Serial.print("Alarm1 is triggered ");
    switch (clock.getAlarmType1())
    {
      case DS3231_EVERY_SECOND:
        Serial.println("every second");
        break;
      case DS3231_MATCH_S:
        Serial.print("when seconds match: ");
        Serial.println(clock.dateFormat("__ __:__:s", a1));
        break;
      case DS3231_MATCH_M_S:
        Serial.print("when minutes and sencods match: ");
        Serial.println(clock.dateFormat("__ __:i:s", a1));
        break;
        
      case DS3231_MATCH_H_M_S:
        Serial.print("when hours, minutes and seconds match: ");
        //Serial.println(clock.dateFormat("__ H:i:s", a1));
        Serial.println(clock.dateFormat("H:i:s", a1));
        break;
        
      case DS3231_MATCH_DT_H_M_S:
        Serial.print("when date, hours, minutes and seconds match: ");
        Serial.println(clock.dateFormat("d H:i:s", a1));
        break;
      case DS3231_MATCH_DY_H_M_S:
        Serial.print("when day of week, hours, minutes and seconds match: ");
        Serial.println(clock.dateFormat("l H:i:s", a1));
        break;
      default: 
        Serial.println("UNKNOWN RULE");
        break;
    }
  } else
  {
    Serial.println("Alarm1 is disarmed.");
  }

  if (clock.isArmed2())
  {
    a2 = clock.getAlarm2();

    Serial.print("Alarm2 is triggered ");
    switch (clock.getAlarmType2())
    {
      case DS3231_EVERY_MINUTE:
        Serial.println("every minute");
        break;
      case DS3231_MATCH_M:
        Serial.print("when minutes match: ");
        Serial.println(clock.dateFormat("__ __:i:s", a2));
        break;
      case DS3231_MATCH_H_M:
        Serial.print("when hours and minutes match:");
        Serial.println(clock.dateFormat("__ H:i:s", a2));
        break;
      case DS3231_MATCH_DT_H_M:
        Serial.print("when date, hours and minutes match: ");
        Serial.println(clock.dateFormat("d H:i:s", a2));
        break;
      case DS3231_MATCH_DY_H_M:
        Serial.println("when day of week, hours and minutes match: ");
        Serial.print(clock.dateFormat("l H:i:s", a2));
        break;
      default: 
        Serial.println("UNKNOWN RULE"); 
        break;
    }
  } else
  {
    Serial.println("Alarm2 is disarmed.");
  }
}


