/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : RS485 Control Relay
 * 
 */
 
//=================================================================================================
#include <Wire.h>                    // I2C Bus
//=================================================================================================
#define SerialRS485 Serial                                                                       // Serial1(D1=TXD,D0=RXD)
//================================================================================================= 

//===================================================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================
#define RS485_DIRECTION_PIN D0                                                                                    // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================================================================
#define I2C_SCL1_PIN        D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//===================================================================================================================
#define RESERVE1            D3                                                                                    // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//===================================================================================================================
#define SHT10_SCL2_PIN      D4                                                                                    // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                                    // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//===================================================================================================================
#define LED_WIFI_PIN        D6                                                                                    // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//===================================================================================================================
#define LED_LINK_PIN        D7                                                                                    // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//===================================================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                                    // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//===================================================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================

//=================================================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
// <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
// <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
// <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
//=================================================================================================
static union  
{
  byte cmd_buff[9];                                                                               // 8 Byte Max
  //===============================================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
  // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
  // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
  // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
  //===============================================================================================
  struct 
  {
    //=============================================================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte relay;                         // '0':Relay[0],'1':Relay[1]
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //=============================================================================================
  }control_relay;
  //===============================================================================================
  
}rs485;
//=================================================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY0                  = '0';  
const byte      RELAY1                  = '1';
const byte      RELAY_ON                = '0';  
const byte      RELAY_OFF               = '1';
//=================================================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //===============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //===============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //===============================================================================================
  Serial.begin(115200);
  //===============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
 
  //===============================================================================================
  // Initial RS485
  //===============================================================================================
  SerialRS485.begin(115200);
  //SerialRS485.setDebugOutput(false);  //Disable Debug on Serial
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write('9');
  SerialRS485.write('9');
  SerialRS485.println();
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
}

void loop() 
{
  //===============================================================================================
  // Start of Receive & Service RS485 Command
  //===============================================================================================
  if(SerialRS485.available() > 0)
  {  
    rs485_chr = SerialRS485.read();
    
    if(rs485_chr == STX_CODE)                                                                     // Verify Start of Command Header
    {
      //===========================================================================================
      // Reset Packet Frame Reset
      //===========================================================================================
      rs485_cmd_head = 1;                                                                         // Command Header Not Complete  
      rs485_cmd_cnt = 0;                                                                          // Re-Start Command Count
      for(int i=0; i<sizeof(rs485.cmd_buff); i++)
      {
        rs485.cmd_buff[i] = '\0';
      }
      rs485_cmd_ptr = &rs485.cmd_buff[0];
      //===========================================================================================
    } 

    //Start of Receive & Verify RS485 Packet Command
    if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
    { 
      *rs485_cmd_ptr = rs485_chr;                                                                 // Save Character 
      rs485_cmd_ptr++;                                                                            // Next Pointer         
      rs485_cmd_cnt++;  

      //SerialDebug.write(rs485_chr);
        
      //===========================================================================================
      // Start of Analysis & Service RS485 Command Packet   
      //===========================================================================================
      if(rs485_chr == 0x0D)
      {           
        //=========================================================================================
        // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
        // <STX><ID0><ID1><:><0><=><0><CR>   OFF Relay[0]
        // <STX><ID0><ID1><:><0><=><1><CR>   ON  Relay[0]
        // <STX><ID0><ID1><:><1><=><0><CR>   OFF Relay[1]
        // <STX><ID0><ID1><:><1><=><1><CR>   ON  Relay[1]
        //=========================================================================================
        if((rs485.control_relay.head      == STX_CODE)&&                                          // head : 0x02
        
           (rs485.control_relay.id[0]     == '9')&&                                               // id[0]:0..9
           (rs485.control_relay.id[1]     == '9')&&                                               // id[1]:0..9
           
           (rs485.control_relay.sep1      == SEP1)&&                                              // sep1 : ':' 
           
           ((rs485.control_relay.relay    == '0')||
            (rs485.control_relay.relay    == '1'))&&                                              // '0':Relay[0],'1':Relay[1]
           
           (rs485.control_relay.sep2      == SEP2)&&                                              // sep2 : '='
           
           ((rs485.control_relay.operate  == '0')||
            (rs485.control_relay.operate  == '1'))&&                                              // operate('0':OFF,'1':ON)
           
           (rs485.control_relay.enter     == END_CODE))                                           // Enter:0x0D = End of Packet
        {
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='0'))                                       
          {
            digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
            
            //=====================================================================================
            // Start of RS485 Test
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write('9');
            SerialRS485.write('9');
            SerialRS485.println("...Relay[0] : OFF");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
            // End of RS485 Test
            //=====================================================================================
          }
          //=======================================================================================
          //=======================================================================================
          if((rs485.control_relay.relay=='0')&&
             (rs485.control_relay.operate=='1'))                                       
          {
            digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
            //=====================================================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write('9');
            SerialRS485.write('9');
            SerialRS485.println("...Relay[0] : ON");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================================================
          }
          //=======================================================================================
        }
        //=========================================================================================
        // Clear RS485 Buffer
        //=========================================================================================
        rs485_cmd_head = 0;                                                                      // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                                                       // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];   
        //=========================================================================================
      } 
      //===========================================================================================
      // End of Analysis & Service RS485 Command Packet   
      //===========================================================================================
    }
  }
  //===============================================================================================
  // End of Receive & Service RS485 Command
  //===============================================================================================

}

