/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D0:LOW=Receive,HIGH=Send)
 * I2C#1  : SCL1=D1,SDA1=D2
 *        : BME280
 *        : RTC:DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : I2C OLED 1.3
 * I2C#2  : SCL2=D4,SDA2=D5
 *        : SHT10
 * ADC    : Analog#0(0.3V) 
 * Output : Wifi LED(D6:HIGH=ON,LOW=OFF)
 *        : Link LED(D7:HIGH=ON,LOW=OFF)
 *        : Relay0(LOW=ON,HIGH=OGFF)
 * 
 * Demo   : BME280 + OLED
 * 
 */

//=================================================================================================
#include <Wire.h>
//=================================================================================================
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>
//=================================================================================================
#include "SH1106.h"
//=================================================================================================

//===================================================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================
#define RS485_DIRECTION_PIN D0                                                                                    // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================================================================
#define I2C_SCL1_PIN        D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                                                    // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                                                    // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//===================================================================================================================
#define RESERVE1            D3                                                                                    // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
//===================================================================================================================
#define SHT10_SCL2_PIN      D4                                                                                    // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                                                    // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//===================================================================================================================
#define LED_WIFI_PIN        D6                                                                                    // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//===================================================================================================================
#define LED_LINK_PIN        D7                                                                                    // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedON               1
#define LedOFF              0
//===================================================================================================================
#define INTERNAL_RELAY_PIN  D8                                                                                    // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//===================================================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//===================================================================================================================


//=================================================================================================
Adafruit_BME280 bme280;                 // I2C
//=================================================================================================
#define SEALEVELPRESSURE_HPA (1013.25)
float bme280_temperature;
float bme280_humidity;
float bme280_pressure;  
float bme280_altitude;
bool  bme280_status;
//=================================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=================================================================================================

//=================================================================================================
SH1106 display(0x3c, I2C_SDA1_PIN, I2C_SCL1_PIN); 
//=================================================================================================
#define ett_logo_width  32
#define ett_logo_height 20
const char ett_logo[] PROGMEM = 
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
  0xF8, 0xF7, 0xDF, 0x7F, 0x04, 0x00, 0x00, 0x00, 0xE4, 0xF7, 0xDE, 0x7B, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xF4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF4, 0x87, 0x02, 0x0A, 
  0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 0x14, 0x80, 0x02, 0x0A, 
  0xE4, 0x87, 0x02, 0x0A, 0x04, 0x80, 0x02, 0x0A, 0xF8, 0x87, 0x02, 0x0A, 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
//=================================================================================================

void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //===============================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOFF);
  //===============================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, LedOFF);
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //===============================================================================================
  Serial.begin(115200);
  //===============================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //===============================================================================================
  
  //==============================================================================================
  Serial.println("BME280 I2C(0x76) Test");
  bme280_status = bme280.begin(0x76);
  if(!bme280_status)
  {
    Serial.println("Initial BME280...Error");
    while(1);
  }
  Serial.println("Initial BME280...Complete");
  Serial.println();
  //===============================================================================================
  display.init();
  display.flipScreenVertically();
  display.setFont(ArialMT_Plain_10);

  display.clear();
  display.drawXbm(0, 0, ett_logo_width, 20,  ett_logo);
  display.setTextAlignment(TEXT_ALIGN_LEFT);
  display.drawString(40, 0, "OLED 1.3");
  display.drawString(40, 10, "I2C");
  display.display();
  //===============================================================================================
}
 
void loop() 
{ 
  //===============================================================================================
  // Start of Read I2C Sensor(BME280)
  //===============================================================================================
  if(millis() - lastGetI2CSensorTime > 5000ul)                        // 5-Second
  {
    bme280_temperature  = bme280.readTemperature();                   // *C
    bme280_humidity     = bme280.readHumidity();                      // %RH
    bme280_pressure     = bme280.readPressure() / 100.0F;             // hPa 
    bme280_altitude     = bme280.readAltitude(SEALEVELPRESSURE_HPA);  // m
    //
    Serial.println();
    Serial.print("BME280 Temperature = ");
    Serial.print(bme280.readTemperature());
    Serial.println(" *C");
    
    Serial.print("BME280 Pressure = ");
    Serial.print(bme280.readPressure() / 100.0F);
    Serial.println(" hPa");
    
    Serial.print("BME280 Approx. Altitude = ");
    Serial.print(bme280.readAltitude(SEALEVELPRESSURE_HPA));
    Serial.println(" m");
    
    Serial.print("BME280 Humidity = ");
    Serial.print(bme280.readHumidity());
    Serial.println(" %RH");
    //=============================================================================================
    
    //=============================================================================================
    
    
    //=============================================================================================
    // Start of Display OLED
    //=============================================================================================
    display.clear();
    
    display.drawXbm(0, 0, ett_logo_width, ett_logo_height,  ett_logo);
    display.drawString(40, 0, "OLED 1.3");
    display.drawString(40, 10, "I2C");
    
    String stemp = "Temperature : " + String(bme280.readTemperature()) + " *C";
    display.drawString(0, 20, stemp);
    
    String spress = "Pressure : " + String(bme280.readPressure() / 100.0F) + " hPa";
    display.drawString(0, 30, spress);
    
    String salt = "Altitude : " + String(bme280.readAltitude(SEALEVELPRESSURE_HPA)) + " m";
    display.drawString(0, 40, salt);
    
    String shumi = "Humidity : " + String(bme280.readHumidity()) + " %RH";
    display.drawString(0, 50, shumi);
    display.display();
    //=============================================================================================
    // End of Display OLED
    //=============================================================================================
    
    //=============================================================================================
    lastGetI2CSensorTime = millis();
    //=============================================================================================
  }
}

