/*
 * Demo ET-ESP8266-RS485 Hardware Board
 * MCU    : ESP8266(ESP-WROOM-02)
 *        : Arduino NodeMCU 
 * RS485  : RS485 RXD:D9
 *        : RS485 TXD:D10
 *        : RS485 Direction : D0(Low=Receive,High=Send)
 * I2C    : I2C1:SCL1=D1,SDA1=D2
 *        : I2C2:SCL2=D5,SDA2=D4
 *        : BME280
 *        : DS3231
 *        : PCF8574(External Output Relay)
 *        : PCF8574A(External Input Opto)
 *        : OLED 1.3
 * ADC    : Analog#0 : ADC(0..3V)
 * Output : WiFi(LED Active High:D6)
 *        : Link(LED Active High:D7)
 *        : Relay(Active Low:D8)
 * 
 * Demo   : Demo HT16K33 Control 7 Segment Display SHT31 Temperature & Humidity 
 */

//=========================================================================================
#include <Wire.h> 
//=========================================================================================
#include "ET_HT16K33_7SEGMENT.h"
//=========================================================================================
#include <Adafruit_SHT31.h>
//=========================================================================================

//=========================================================================================
// Start of Default Hardware : ET-ESP8266-RS485
//=========================================================================================
#define RS485_DIRECTION_PIN D0                                                             // ESP-WROOM-02 :D0,ESP8266:GPIO16
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//=========================================================================================
#define I2C_SCL1_PIN        D1                                                             // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define I2C_SDA1_PIN        D2                                                             // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define BME280_SCL1_PIN     D1                                                             // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define BME280_SDA1_PIN     D2                                                             // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT10_SCL1_PIN      D1                                                             // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT10_SDA1_PIN      D2                                                             // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
#define SHT31_SCL1_PIN      D1                                                             // ESP-WROOM-02 : D1, ESP8266 : GPIO5  (***SCL***)
#define SHT31_SDA1_PIN      D2                                                             // ESP-WROOM-02 : D2, ESP8266 : GPIO4  (***SDA***)
//=========================================================================================
#define RESERVE1            D3                                                             // ESP-WROOM-02 : D3, ESP8266 : GPIO0  (***BOOT***)
#define SWITCH_PIN          D3
#define SwitchRelease       1
#define SwitchPress         0
//=========================================================================================
#define SHT10_SCL2_PIN      D4                                                             // ESP-WROOM-02 : D4, ESP8266 : GPIO2
#define SHT10_SDA2_PIN      D5                                                             // ESP-WROOM-02 : D5, ESP8266 : GPIO14 (SCK)
//=========================================================================================
#define LED_WIFI_PIN        D6                                                             // ESP-WROOM-02 : D6, ESP8266 : GPIO12 (MISO)
#define WifiConnect         1
#define WifiDisconnect      0
//=========================================================================================
#define LED_LINK_PIN        D7                                                             // ESP-WROOM-02 : D7, ESP8266 : GPIO13 (MOSI)
#define LedOn               1
#define LedOff              0
//=========================================================================================
#define INTERNAL_RELAY_PIN  D8                                                             // ESP-WROOM-02 : D8, ESP8266 : GPIO15 (SS#)
#define InternalRelayOn     1
#define InternalRelayOff    0
//=========================================================================================
// End of Default Hardware : ET-ESP8266-RS485
//=========================================================================================

//=========================================================================================
ET_HT16K33_7SEGMENT display = ET_HT16K33_7SEGMENT(0x70);
//=========================================================================================

//=========================================================================================
Adafruit_SHT31 sht31 = Adafruit_SHT31();                              // I2C
//=========================================================================================
float sht31_temperature;
float sht31_humidity;
bool  sht31_status;
//=========================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=========================================================================================

void setup() 
{
  //=======================================================================================
  // Start of Initial Default Hardware : ET-ESP8266-RS485
  //=======================================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=======================================================================================
  pinMode(LED_LINK_PIN, OUTPUT);
  digitalWrite(LED_LINK_PIN, LedOff);
  //=======================================================================================
  pinMode(LED_WIFI_PIN, OUTPUT);
  digitalWrite(LED_WIFI_PIN, WifiDisconnect);
  //=======================================================================================
  pinMode(SWITCH_PIN, INPUT);
  //=======================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=======================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                        
  //=======================================================================================
  Serial.begin(115200);
  //=======================================================================================
  // End of Initial Default Hardware : ET-ESP8266-RS485
  //=======================================================================================

  //=======================================================================================
  // Initial HT16K33 Display Control
  //=======================================================================================
  display.begin();
  //=======================================================================================

  //===============================================================================================
  // Start of Initial SHT31
  //===============================================================================================
  Serial.println();
  Serial.println("SHT31 I2C(0x44) Test");
  //===============================================================================================
  sht31_status = sht31.begin(0x44);
  if(!sht31_status)
  {
    Serial.println("Initial SHT31...Error");
    display.print(10000, DEC);                        // Print Error(number too long) : [-][-][-][-]
    display.writeDisplayPanel0();                     // Transfer Buffer to Display Panel[0] & Update to HT16K33(Lower-LSB)
    display.writeDisplayPanel1();                     // Transfer Buffer to Display Panel[1] & Update to HT16K33(Lower-MSB)
    display.writeDisplayPanel2();                     // Transfer Buffer to Display Panel[2] & Update to HT16K33(Upper-LSB)
    display.writeDisplayPanel3();                     // Transfer Buffer to Display Panel[3] & Update to HT16K33(Upper-MSB)
    while(1);
  }
  //===============================================================================================
  Serial.println("Initial SHT31...Complete");
  Serial.println();
  //===============================================================================================
  // End of Initial SHT31
  //===============================================================================================
}

void loop() 
{
  //=============================================================================================
  // Start of Update SHT31 Sensor Value
  //=============================================================================================
  if(millis() - lastGetI2CSensorTime > 1000ul)        // 1-Second
  {
    sht31_temperature  = sht31.readTemperature();     // *C
    sht31_humidity     = sht31.readHumidity();        // %RH
    
    if(! isnan(sht31_temperature)) 
    {
      Serial.print("SHT31 Temperature = ");
      Serial.print(sht31.readTemperature());
      Serial.println(" *C");

      display.print(sht31_temperature,1);             // Create Display Buffer = tt.t
      display.writeDisplayPanel0();                   // Transfer Buffer to Display Panel[0] & Update to HT16K33(Lower-LSB)
      display.writeDisplayPanel2();                   // Transfer Buffer to Display Panel[2] & Update to HT16K33(Upper-LSB)
    }
    if(! isnan(sht31_humidity)) 
    {
      Serial.print("SHT31 Humidity = ");
      Serial.print(sht31.readHumidity());
      Serial.println(" %");

      display.print(sht31_humidity,1);                // Create Display Buffer = hh.h
      display.writeDisplayPanel1();                   // Transfer Buffer to Display Panel[1] & Update to HT16K33(Lower-MSB)
      display.writeDisplayPanel3();                   // Transfer Buffer to Display Panel[3] & Update to HT16K33(Upper-MSB)
    }
    //===========================================================================================
    lastGetI2CSensorTime = millis();
    //===========================================================================================
    digitalWrite(LED_LINK_PIN, LedOff);
    //===========================================================================================
  }
  //=============================================================================================
  // End of Update SHT31 Sensor Value
  //=============================================================================================
}

