/*
 * Demo ET-DSP6 I2C Hardware Board
 * MCU        : ATMEGA32U4
 *            : Arduino Genuino Micro
 *            : Bootloader
 *            : -> .../caterina/Caterina-Genuino-Micro.hex
 *            : Fuse Bit
 *            : -> low_fuses      = 0xFF
 *            : -> high_fuses     = 0xD8
 *            : -> extended_fuses = 0xCB(0xFB)
 *            : Lock Bit
 *            : -> 0x2F(0xEF)
 * RS485      : RS485 RXD:D0
 *            : RS485 TXD:D1
 *            : RS485 Direction(D10:LOW=Receive,HIGH=Send)
 * I2C        : SCL=D3,SDA=D2
 *            : RTC:DS3231
 *            : PCF8575
 * IR Remote  : A0
 * Opto Input : A1,A2,A3
 *            : D4,D5,D6
 * Relay      : D8           
 * 
 * Demo       : RS485 Control Relay ON/OFF 
 */

//===================================================================
#include <Wire.h>                                                   // I2C Bus
//===================================================================
#include "pcf8575.h"                                                // PCF8575/A
//===================================================================

//===================================================================
#define SerialDebug Serial                                          // USB Serial
#define SerialRS485 Serial1                                         // Serial1(D1=TXD,D0=RXD)
#define SerialRS232 Serial1                                         // Serial1(D1=TXD,D0=RXD)
int inByte;
//===================================================================

//===================================================================
// Start of Default Hardware : ET-DSP6 I2C
//===================================================================
#define OPTO_INPUT0_PIN     4
#define OPTO_INPUT1_PIN     5
#define OPTO_INPUT2_PIN     6
//===================================================================
#define OPTO_INPUT3_PIN     A1
#define OPTO_INPUT4_PIN     A2
#define OPTO_INPUT5_PIN     A3
//===================================================================
#define SignalRelease       HIGH
#define SignalPress         LOW
//===================================================================

//===================================================================
#define IR_REMOTE_PIN       A0
//===================================================================

//===================================================================
#define INTERNAL_RELAY_PIN  8
//===================================================================
#define InternalRelayOff    LOW
#define InternalRelayOn     HIGH
//===================================================================
#define RS485_DIRECTION_PIN 10                                      // 0:RXD,1:TXD
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
// End of Default Hardware : ET-DSP6 I2C
//===================================================================

//===================================================================
PCF8575 PCF8575_SLAVE0(0x20);                                       // PCF8575  = 0100,000+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE1(0x21);                                       // PCF8575  = 0100,001+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE2(0x22);                                       // PCF8575  = 0100,010+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE3(0x23);                                       // PCF8575  = 0100,011+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE4(0x24);                                       // PCF8575  = 0100,100+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE5(0x25);                                       // PCF8575  = 0100,101+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE6(0x26);                                       // PCF8575  = 0100,110+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE7(0x27);                                       // PCF8575  = 0100,111+(0:W,1:R)
//===================================================================

//===================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
// <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
//===================================================================
static union  
{
  byte cmd_buff[9];                                                 // 8 Byte Max
  //=================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
  // <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
  //=================================================================
  struct 
  {
    //===============================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'99'  
    byte sep1;                          // ':'
    byte relay;                         // 'R':Relay
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //===============================================================
  }control_relay;
  //=================================================================
  
}rs485;
//===================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY                   = 'R';  
const byte      RELAY_ON                = '1';  
const byte      RELAY_OFF               = '0';
//===================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//===================================================================
char SlaveID[2] = {'0','1'};                                       // RS485 ID = '01'
//===================================================================

void setup() 
{
  //=================================================================
  // Start of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================
  SerialRS485.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  
  //=================================================================
  pinMode(OPTO_INPUT0_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT1_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT2_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT3_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT4_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT5_PIN, INPUT_PULLUP);
  //=================================================================
  //=================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
  // End of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================

  //=================================================================
  // If Select Jumper A2 : LOW = [0][1][2][3][4][5]
  //=================================================================
  PCF8575_SLAVE2.write_bcd_normal(0,1);                             // msb,lsb
  PCF8575_SLAVE1.write_bcd_normal(2,3);                             // msb,lsb
  PCF8575_SLAVE0.write_bcd_normal(4,5);                             // msb,lsb
  //=================================================================
  //=================================================================
  // If Select Jumper A2 : HIGH = [8][9][A][B][C][D]
  //=================================================================
  PCF8575_SLAVE6.write_bcd_normal(8,9);                             // msb,lsb
  PCF8575_SLAVE5.write_bcd_normal(10,11);                           // msb,lsb
  PCF8575_SLAVE4.write_bcd_normal(12,13);                           // msb,lsb
  //=================================================================
}

void loop() 
{
  //=================================================================
  // Start of RS485 Test
  //=================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.println("Send Command : *01:R=0<Cr>...OFF Relay");
  SerialRS485.println("Send Command : *01:R=1<Cr>...ON  Relay");
  SerialRS485.println("==============================================================");
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  // End of RS485 Test
  //=================================================================

  //=================================================================
  while(1)
  {
    //===============================================================
    // Start of Receive & Service RS485 Command
    //===============================================================
    if(SerialRS485.available() > 0)
    {  
      rs485_chr = SerialRS485.read();
    
      if(rs485_chr == STX_CODE)                                     // Verify Start of Command Header
      { 
        //===========================================================
        // Reset Packet Frame Reset
        //===========================================================
        rs485_cmd_head = 1;                                         // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                          // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];
        //===========================================================
      } 

      //Start of Receive & Verify RS485 Packet Command
      if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
      { 
        *rs485_cmd_ptr = rs485_chr;                                 // Save Character 
        rs485_cmd_ptr++;                                            // Next Pointer         
        rs485_cmd_cnt++;  

        //SerialDebug.write(rs485_chr);
        
        //===========================================================
        // Start of Analysis & Service RS485 Command Packet   
        //===========================================================
        if(rs485_chr == 0x0D)
        {           
          //=========================================================
          // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
          // <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
          // <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
          //=========================================================
          if((rs485.control_relay.head      == STX_CODE)&&          // head : 0x02
        
             (rs485.control_relay.id[0]     == SlaveID[0])&&        // id[0]:0..9
             (rs485.control_relay.id[1]     == SlaveID[1])&&        // id[1]:0..9
           
             (rs485.control_relay.sep1      == SEP1)&&              // sep1 : ':' 
           
             (rs485.control_relay.relay    == 'R')&&                // 'R':Relay
           
             (rs485.control_relay.sep2      == SEP2)&&              // sep2 : '='
           
             ((rs485.control_relay.operate  == '0')||
              (rs485.control_relay.operate  == '1'))&&              // operate('0':OFF,'1':ON)
           
             (rs485.control_relay.enter     == END_CODE))           // Enter:0x0D = End of Packet
          {
            //=======================================================
            if(rs485.control_relay.operate=='0')                                       
            {
              //=====================================================
              digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
              //=====================================================
              // Start of RS485 Test
              //=====================================================
              digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
              SerialRS485.print("Slave ID:");
              SerialRS485.write(SlaveID[0]);
              SerialRS485.write(SlaveID[1]);
              SerialRS485.println("...Relay : OFF");
              SerialRS485.flush();
              digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
              //=====================================================
              // End of RS485 Test
              //=====================================================
            }
            //=======================================================
            //=======================================================
            if(rs485.control_relay.operate=='1')                                       
            {
              //=====================================================
              digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
              //=====================================================
              digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
              SerialRS485.print("Slave ID:");
              SerialRS485.write(SlaveID[0]);
              SerialRS485.write(SlaveID[1]);
              SerialRS485.println("...Relay : ON");
              SerialRS485.flush();
              digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
              //=====================================================
            }
            //=======================================================
          
          }
          //=========================================================
          // Clear RS485 Buffer
          //=========================================================
          rs485_cmd_head = 0;                                       // Command Header Not Complete  
          rs485_cmd_cnt = 0;                                        // Re-Start Command Count
          for(int i=0; i<sizeof(rs485.cmd_buff); i++)
          {
            rs485.cmd_buff[i] = '\0';
          }
          rs485_cmd_ptr = &rs485.cmd_buff[0];   
          //=========================================================
        } 
        //===========================================================
        // End of Analysis & Service RS485 Command Packet   
        //===========================================================
      }
    }
    //===============================================================
    // End of Receive & Service RS485 Command
    //===============================================================
  } 
  //=================================================================

}

