/*
 * Demo ET-DSP6 I2C Hardware Board
 * MCU        : ATMEGA32U4
 *            : Arduino Genuino Micro
 *            : Bootloader
 *            : -> .../caterina/Caterina-Genuino-Micro.hex
 *            : Fuse Bit
 *            : -> low_fuses      = 0xFF
 *            : -> high_fuses     = 0xD8
 *            : -> extended_fuses = 0xCB(0xFB)
 *            : Lock Bit
 *            : -> 0x2F(0xEF)
 * RS485      : RS485 RXD:D0
 *            : RS485 TXD:D1
 *            : RS485 Direction(D10:LOW=Receive,HIGH=Send)
 * I2C        : SCL=D3,SDA=D2
 *            : RTC:DS3231
 *            : PCF8575
 * IR Remote  : A0
 * Opto Input : A1,A2,A3
 *            : D4,D5,D6
 * Relay      : D8           
 * 
 * Demo       : SPI F-RAM(FM25L16B) Read/Write 
 */

//===================================================================
#include <Wire.h>                                                   // I2C Bus
//===================================================================
#include <SPI.h>
//===================================================================
#include "ET_SPI_FRAM.h"
//===================================================================

//===================================================================
#define SerialDebug Serial                                          // USB Serial
#define SerialRS485 Serial1                                         // Serial1(D1=TXD,D0=RXD)
#define SerialRS232 Serial1                                         // Serial1(D1=TXD,D0=RXD)
//===================================================================

//===================================================================
// Start of Default Hardware : ET-DSP6 I2C
//===================================================================
#define OPTO_INPUT0_PIN     4
#define OPTO_INPUT1_PIN     5
#define OPTO_INPUT2_PIN     6
//===================================================================
#define OPTO_INPUT3_PIN     A1
#define OPTO_INPUT4_PIN     A2
#define OPTO_INPUT5_PIN     A3
//===================================================================
#define SignalRelease       HIGH
#define SignalPress         LOW
//===================================================================

//===================================================================
#define IR_REMOTE_PIN       A0
//===================================================================

//===================================================================
#define INTERNAL_RELAY_PIN  8
//===================================================================
#define InternalRelayOff    LOW
#define InternalRelayOn     HIGH
//===================================================================
#define RS485_DIRECTION_PIN 10                                      // 0:RXD,1:TXD
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
// End of Default Hardware : ET-DSP6 I2C
//===================================================================

//===================================================================
uint8_t SPI_FRAM_CS   = 9;
ET_SPI_FRAM fram = ET_SPI_FRAM(SPI_FRAM_CS);                       // use hardware SPI
//===================================================================
uint16_t          addr = 0;
//===================================================================

void setup() 
{
  //=================================================================
  // Start of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================
  SerialRS485.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  pinMode(OPTO_INPUT0_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT1_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT2_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT3_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT4_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT5_PIN, INPUT_PULLUP);
  //=================================================================
  //=================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
  // End of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================

  //=================================================================
  SerialDebug.begin(115200);                                       // Debug USART
  while(!SerialDebug);                                             // wait USB Serial Complete 
  //=================================================================
  SerialDebug.println();
  SerialDebug.println("Demo SPI FRAM(FM25L16B : 2-KByte x 8) Test");
  
  if(fram.begin()) 
  {
    SerialDebug.println("Found SPI FRAM");
  } 
  else 
  {
    SerialDebug.println("No SPI FRAM found ... check your connections\r\n");
    while (1);
  }
  //=================================================================

  //=================================================================
  // Read the first byte
  //=================================================================
  uint8_t test = fram.read8(0x00);
  SerialDebug.print("Restarted "); 
  SerialDebug.print(test); 
  SerialDebug.println(" times");
  //=================================================================

  //=================================================================
  // Test write ++
  //=================================================================
  fram.writeEnable(true);
  fram.write8(0x0, test+1);
  fram.writeEnable(false);
  //=================================================================
  fram.writeEnable(true);
  fram.write(0x1, (uint8_t *)"FTW!", 5);
  fram.writeEnable(false);
  //=================================================================

  //=================================================================
  // dump the entire 8K of memory!
  //=================================================================
  uint8_t value;
  for (uint16_t a = 0; a < 8192; a++) 
  {
    value = fram.read8(a);
    if ((a % 32) == 0) 
    {
      SerialDebug.print("\n 0x"); SerialDebug.print(a, HEX); SerialDebug.print(": ");
    }
    SerialDebug.print("0x"); 
    if (value < 0x1) 
      SerialDebug.print('0');
    SerialDebug.print(value, HEX); SerialDebug.print(" ");
  }
  //=================================================================
  
}

void loop() 
{
  
}

