/*
 * Demo ET-DSP6 I2C Hardware Board
 * MCU        : ATMEGA32U4
 *            : Arduino Genuino Micro
 *            : Bootloader
 *            : -> .../caterina/Caterina-Genuino-Micro.hex
 *            : Fuse Bit
 *            : -> low_fuses      = 0xFF
 *            : -> high_fuses     = 0xD8
 *            : -> extended_fuses = 0xCB(0xFB)
 *            : Lock Bit
 *            : -> 0x2F(0xEF)
 * RS485      : RS485 RXD:D0
 *            : RS485 TXD:D1
 *            : RS485 Direction(D10:LOW=Receive,HIGH=Send)
 * I2C        : SCL=D3,SDA=D2
 *            : RTC:DS3231
 *            : PCF8575
 * IR Remote  : A0
 * Opto Input : A1,A2,A3
 *            : D4,D5,D6
 * Relay      : D8           
 * 
 * Demo       : Clock 6 Digit HH:MM:SS 
 */

//===================================================================
#include <Wire.h>                                                   // I2C Bus
//===================================================================
#include "pcf8575.h"                                                // PCF8575/A
//===================================================================
#include "ET_DS3231.h"
//===================================================================

//===================================================================
// Start of Default Hardware : ET-DSP6 I2C
//===================================================================
#define OPTO_INPUT0_PIN     4
#define OPTO_INPUT1_PIN     5
#define OPTO_INPUT2_PIN     6
//===================================================================
#define OPTO_INPUT3_PIN     A1
#define OPTO_INPUT4_PIN     A2
#define OPTO_INPUT5_PIN     A3
//===================================================================
#define SignalRelease       HIGH
#define SignalPress         LOW
//===================================================================

//===================================================================
#define IR_REMOTE_PIN       A0
//===================================================================

//===================================================================
#define INTERNAL_RELAY_PIN  8
//===================================================================
#define InternalRelayOff    LOW
#define InternalRelayOn     HIGH
//===================================================================
#define RS485_DIRECTION_PIN 10                                      // 0:RXD,1:TXD
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
// End of Default Hardware : ET-DSP6 I2C
//===================================================================

//===================================================================
PCF8575 PCF8575_SLAVE0(0x20);                                       // PCF8575  = 0100,000+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE1(0x21);                                       // PCF8575  = 0100,001+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE2(0x22);                                       // PCF8575  = 0100,010+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE3(0x23);                                       // PCF8575  = 0100,011+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE4(0x24);                                       // PCF8575  = 0100,100+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE5(0x25);                                       // PCF8575  = 0100,101+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE6(0x26);                                       // PCF8575  = 0100,110+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE7(0x27);                                       // PCF8575  = 0100,111+(0:W,1:R)
//===================================================================

//===================================================================
// ET_DS3231.h
//===================================================================
ET_DS3231 myRTC;
DateTime myTimeNow;
//===================================================================
unsigned long lastGetSecondTime = 0;
//===================================================================

void setup() 
{
  //=================================================================
  // Start of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================
  Serial.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  pinMode(OPTO_INPUT0_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT1_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT2_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT3_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT4_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT5_PIN, INPUT_PULLUP);
  //=================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
  // End of Initial Default Hardware : ET-DSP6 I2C
  //=================================================================

  //=================================================================
  // If Select Jumper A2 : LOW = [0][1][2][3][4][5]
  //=================================================================
  PCF8575_SLAVE2.write_bcd_normal(0,1);                             // msb,lsb
  PCF8575_SLAVE1.write_bcd_normal(2,3);                             // msb,lsb
  PCF8575_SLAVE0.write_bcd_normal(4,5);                             // msb,lsb
  //=================================================================
  //=================================================================
  // If Select Jumper A2 : HIGH = [8][9][A][B][C][D]
  //=================================================================
  PCF8575_SLAVE6.write_bcd_normal(8,9);                             // msb,lsb
  PCF8575_SLAVE5.write_bcd_normal(10,11);                           // msb,lsb
  PCF8575_SLAVE4.write_bcd_normal(12,13);                           // msb,lsb
  //=================================================================
  
  //=================================================================
  myRTC.begin();
  //=================================================================
  //myRTC.adjust(DateTime(year(), month(), day(), hour(), minute(), second()));
  myRTC.adjust(DateTime(2017, 2, 23, 11, 37, 0));
  //=================================================================
  myRTC.armAlarm1(false);
  myRTC.clearAlarm1();
  //=================================================================
  myRTC.armAlarm2(false);
  myRTC.clearAlarm2();
  //=================================================================
  myRTC.setAlarm1(0, 0, 0, 0, DS3231_EVERY_SECOND);                 // Alarm Every Second
  //=================================================================
  lastGetSecondTime = millis();
  //=================================================================
}

void loop() 
{
  //=================================================================
  // 1st Half of Second Display = [H][H] : [M][M] : [S][S]
  //=================================================================
  if(myRTC.isAlarm1(false))
  {
    //===============================================================
    //digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
    //===============================================================
    myTimeNow = myRTC.now();
    //===============================================================
    // If Select Jumper A2 : LOW = [H][H]:[M][M]:[S][S]
    //===============================================================
    PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.hour()/10)%10),     // Hour(msb)
                                     (myTimeNow.hour()%10),         // Hour(lsb)
                                     false,                         // OFF : Dot MSB Hour
                                     true);                         // ON  : Dot LSB Hour
    PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     true);                         // ON  : Dot LSB Minute
    PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second                                 
    //===============================================================
    //===============================================================
    // If Select Jumper A2 : HIGH = [H][H]:[M][M]:[S][S]
    //===============================================================
    PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.hour()/10)%10),     // Hour(msb)
                                     (myTimeNow.hour()%10),         // Hour(lsb)
                                     false,                         // OFF : Dot MSB Hour
                                     true);                         // ON  : Dot LSB Hour
    PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     true);                         // ON  : Dot LSB Minute
    PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second                                 
    //===============================================================
    myRTC.clearAlarm1();
    //===============================================================
    lastGetSecondTime = millis();
    //===============================================================
  }
  //=================================================================

  //=================================================================
  // 2nd Half of Second Display = [H][H] [M][M] [S][S]
  //=================================================================
  if(millis() - lastGetSecondTime > 500ul)                          // 0.5-Second
  {
    //===============================================================
    //digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
    //===============================================================
    myTimeNow = myRTC.now();
    //===============================================================
    // If Select Jumper A2 : LOW = [H][H] [M][M] [S][S]
    //===============================================================
    PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.hour()/10)%10),     // Hour(msb)
                                     (myTimeNow.hour()%10),         // Hour(lsb)
                                     false,                         // OFF : Dot MSB Hour
                                     false);                        // OFF : Dot LSB Hour
    PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second                                
    //===============================================================
    //===============================================================
    // If Select Jumper A2 : HIGH = [H][H] [M][M] [S][S]
    //===============================================================
    PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.hour()/10)%10),     // Hour(msb)
                                     (myTimeNow.hour()%10),         // Hour(lsb)
                                     false,                         // OFF : Dot MSB Hour
                                     false);                        // OFF : Dot LSB Hour
    PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second                                
    //===============================================================
    lastGetSecondTime = millis();
    //===============================================================
  }
}

