/*
 * Demo ET-DSP4 I2C Hardware Board
 * MCU        : ATMEGA32U4
 *            : Arduino Genuino Micro
 *            : Bootloader
 *            : -> .../caterina/Caterina-Genuino-Micro.hex
 *            : Fuse Bit
 *            : -> low_fuses      = 0xFF
 *            : -> high_fuses     = 0xD8
 *            : -> extended_fuses = 0xCB(0xFB)
 *            : Lock Bit
 *            : -> 0x2F(0xEF)
 * RS485      : RS485 RXD:D0
 *            : RS485 TXD:D1
 *            : RS485 Direction(D10:LOW=Receive,HIGH=Send)
 * I2C        : SCL=D3,SDA=D2
 *            : RTC:DS3231
 *            : PCF8575
 * IR Remote  : A0
 * Opto Input : A1,A2,A3
 *            : D4,D5,D6
 * Relay      : D8           
 * 
 * Demo       : ET-DSP4 I2C Hardware Test 
 */

//===================================================================
#include <Wire.h>                                                   // I2C Bus
//===================================================================
#include "pcf8575.h"                                                // PCF8575/A
//===================================================================
#include "ET_DS3231.h"
//===================================================================
#include <IRremote.h>                                               // IR Remote 
//===================================================================

//===================================================================
#define SerialDebug Serial                                          // USB Serial
#define SerialRS485 Serial1                                         // Serial1(D1=TXD,D0=RXD)
#define SerialRS232 Serial1                                         // Serial1(D1=TXD,D0=RXD)
int inByte;
//===================================================================

//===================================================================
// Start of Default Hardware : ET-DSP4 I2C
//===================================================================
#define OPTO_INPUT0_PIN     4
#define OPTO_INPUT1_PIN     5
#define OPTO_INPUT2_PIN     6
//===================================================================
#define OPTO_INPUT3_PIN     A1
#define OPTO_INPUT4_PIN     A2
#define OPTO_INPUT5_PIN     A3
//===================================================================
#define SignalRelease       HIGH
#define SignalPress         LOW
//===================================================================

//===================================================================
#define IR_REMOTE_PIN       A0
//===================================================================

//===================================================================
#define INTERNAL_RELAY_PIN  8
//===================================================================
#define InternalRelayOff    LOW
#define InternalRelayOn     HIGH
//===================================================================
#define RS485_DIRECTION_PIN 10                                      // 0:RXD,1:TXD
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
// End of Default Hardware : ET-DSP4 I2C
//===================================================================

//===================================================================
PCF8575 PCF8575_SLAVE0(0x20);                                       // PCF8575  = 0100,000+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE1(0x21);                                       // PCF8575  = 0100,001+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE2(0x22);                                       // PCF8575  = 0100,010+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE3(0x23);                                       // PCF8575  = 0100,011+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE4(0x24);                                       // PCF8575  = 0100,100+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE5(0x25);                                       // PCF8575  = 0100,101+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE6(0x26);                                       // PCF8575  = 0100,110+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE7(0x27);                                       // PCF8575  = 0100,111+(0:W,1:R)
//===================================================================

//===================================================================
// ET_DS3231.h
//===================================================================
ET_DS3231 myRTC;
DateTime myTimeNow;
//===================================================================
unsigned long lastGetSecondTime = 0;
unsigned long lastGetDelayTime = 0;
//===================================================================

//===================================================================
// IR Remote Receiver
//===================================================================
IRrecv IR_Remote(IR_REMOTE_PIN);
decode_results IR_Result;
//===================================================================

//===================================================================
int thisInput0State;
int thisInput1State;
int thisInput2State;
int thisInput3State;
int thisInput4State;
int thisInput5State;
//===================================================================
int lastInput0State = SignalRelease;
int lastInput1State = SignalRelease;
int lastInput2State = SignalRelease;
int lastInput3State = SignalRelease;
int lastInput4State = SignalRelease;
int lastInput5State = SignalRelease;
//===================================================================

//===================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
// <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
// <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
//===================================================================
static union  
{
  byte cmd_buff[9];                                                  // 8 Byte Max
  //=================================================================
  // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
  // <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
  // <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
  //=================================================================
  struct 
  {
    //===============================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte relay;                         // 'R':Relay
    byte sep2;                          // '=' 
    byte operate;                       // '0':OFF,'1':ON
    byte enter;                         // <Cr>:0x0D
    //===============================================================
  }control_relay;
  //=================================================================
  
}rs485;
//===================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
const byte      RELAY                   = 'R';  
const byte      RELAY_ON                = '1';  
const byte      RELAY_OFF               = '0';
//===================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//===================================================================
char SlaveID[2] = {'0','1'};
//===================================================================

void setup() 
{
  //=================================================================
  // Start of Initial Default Hardware : ET-DSP4 I2C
  //=================================================================
  SerialRS485.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  SlaveID[0] = '0';
  SlaveID[1] = '1';
  //=================================================================
  pinMode(OPTO_INPUT0_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT1_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT2_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT3_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT4_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT5_PIN, INPUT_PULLUP);
  //=================================================================
  lastInput0State = SignalRelease;
  lastInput1State = SignalRelease;
  lastInput2State = SignalRelease;
  lastInput3State = SignalRelease;
  lastInput4State = SignalRelease;
  lastInput5State = SignalRelease;
  //=================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
  // End of Initial Default Hardware : ET-DSP4 I2C
  //=================================================================

  //=================================================================
  SerialDebug.begin(115200);                                       // Debug USART
  while(!SerialDebug);                                             // wait USB Serial Complete 
  //=================================================================
  print_menu_test();                                               // Print Menu Test
  //=================================================================
}

void loop() 
{
  if(SerialDebug.available() > 0)
  {
    inByte = SerialDebug.read();
    
    if(inByte == '1') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("1. Test 7-SEGMENT");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_7seg();
      //============================================================= 
    }
    
    else if(inByte == '2') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("2. Test RTC DS3231");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_rtc();
      //============================================================= 
    }
    
    else if(inByte == '3') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("3. Test OPTO-Input");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_opto();
      //============================================================= 
    }

    else if(inByte == '4') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("4. Test Relay");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_relay();
      //============================================================= 
    }
    
    else if(inByte == '5') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("5. Test RS232 & RS485");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_usart();
      //============================================================= 
    }

    else if(inByte == '6') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("6. Test IR Remote");
      SerialDebug.println("Press E(Exit)...Stop Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_ir_remote();
      //============================================================= 
    }
    
    else if((inByte == 'E') ||(inByte == 'e'))                   
    {
      print_menu_test();  
    }
  }  
}

int print_menu_test(void)
{
  //=================================================================
  // If Select Jumper A2:A1 = LOW:LOW
  //=================================================================
  PCF8575_SLAVE1.write_bcd_normal(0,1);                              // msb,lsb
  PCF8575_SLAVE0.write_bcd_normal(2,3);                              // msb,lsb
  //=================================================================
  
  //=================================================================
  // If Select Jumper A2:A1 = LOW:HIGH
  //=================================================================
  PCF8575_SLAVE3.write_bcd_normal(4,5);                              // msb,lsb
  PCF8575_SLAVE2.write_bcd_normal(6,7);                              // msb,lsb
  //=================================================================

  //=================================================================
  // If Select Jumper A2:A1 = HIGH:LOW
  //=================================================================
  PCF8575_SLAVE5.write_bcd_normal(8,9);                              // msb,lsb
  PCF8575_SLAVE4.write_bcd_normal(0x0A,0x0B);                        // msb,lsb
  //=================================================================

  //=================================================================
  // If Select Jumper A2:A1 = HIGH:HIGH
  //=================================================================
  PCF8575_SLAVE7.write_bcd_normal(0x0C,0x0D);                        // msb,lsb
  PCF8575_SLAVE6.write_bcd_normal(0x0E,0x0F);                        // msb,lsb
  //=================================================================
  
  //=================================================================
  // Start of Menu Test
  //=================================================================
  SerialDebug.println();
  SerialDebug.println("==============================================================");
  SerialDebug.println("QC Test : ET-DSP4 I2C");
  SerialDebug.println("==============================================================");
  SerialDebug.println("1. Test 7 SEGMENT");
  SerialDebug.println("2. Test RTC DS3231");
  SerialDebug.println("3. Test Input OPTO-Isolate");
  SerialDebug.println("4. Test Relay");
  SerialDebug.println("5. Test RS232 & RS485");
  SerialDebug.println("6. Test IR Remote");
  SerialDebug.println("==============================================================");
  SerialDebug.println("Select [1..6] For Test");
  SerialDebug.println("==============================================================");
  //=================================================================
  // End of Menu Test
  //================================================================= 
}

/*****************************
 * 1. Test 7 Segment 4 Digit
 ****************************/
int test_7seg(void)
{
  while(1)
  {
    //=================================================================
    // Test Display Segment : A..G,Dp
    //=================================================================
    for(uint8_t seg=0; seg<8; seg++)                                  // Seg-A,Seg-B,Seg-C,...,Seg-G,Seg-Dp
    {
      //===============================================================
      PCF8575_SLAVE0.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = LOW:LOW
      PCF8575_SLAVE1.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = LOW:LOW
      
      PCF8575_SLAVE2.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = LOW:HIGH
      PCF8575_SLAVE3.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = LOW:HIGH
      
      PCF8575_SLAVE4.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = HIGH:LOW
      PCF8575_SLAVE5.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = HIGH:LOW

      PCF8575_SLAVE6.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = HIGH:HIGH
      PCF8575_SLAVE7.write_seg((1<<(seg)),(1<<(seg)));                // A2:A1 = HIGH:HIGH
      //===============================================================
    
      //===============================================================
      lastGetDelayTime = millis();  
      //===============================================================
      while(millis() - lastGetDelayTime < 1000ul)                    // 1-Second
      {
        //=============================================================
        if(SerialDebug.available() > 0)
        {
          inByte = SerialDebug.read();
          if((inByte == 'E') ||(inByte == 'e'))                     // Exit Test
          {
            print_menu_test();
            return;
          }
        }
        //=============================================================
      }
      //===============================================================
      lastGetDelayTime = millis();
      //===============================================================
    }
    //=================================================================

    //=================================================================
    // Test Display Number : 0..9,A..F
    //=================================================================  
    for(uint16_t bcd=0; bcd<16; bcd++)                                // 0..9,A..F
    {
      //===============================================================
      PCF8575_SLAVE0.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = LOW:LOW
      PCF8575_SLAVE1.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = LOW:LOW
      
      PCF8575_SLAVE2.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = LOW:HIGH
      PCF8575_SLAVE3.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = LOW:HIGH
      
      PCF8575_SLAVE4.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = HIGH:LOW
      PCF8575_SLAVE5.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = HIGHLOW

      PCF8575_SLAVE6.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = HIGH:HIGH
      PCF8575_SLAVE7.write_bcd_normal(bcd,bcd,false,false);           // A2:A1 = HIGH:HIGH
      //===============================================================
    
      //===============================================================
      lastGetDelayTime = millis();  
      //===============================================================
      while(millis() - lastGetDelayTime < 1000ul)                    // 1-Second
      {
        //=============================================================
        if(SerialDebug.available() > 0)
        {
          inByte = SerialDebug.read();
          if((inByte == 'E') ||(inByte == 'e'))                     // Exit Test
          {
            print_menu_test();
            return;
          }
        }
        //=============================================================
      }
      //===============================================================
      lastGetDelayTime = millis();
      //===============================================================
    }
    //=================================================================
  }  
}

/***********************************
 * 2. Test RTC:DS3231 Clock:6 Demo 
 **********************************/
int test_rtc(void)
{
  //=================================================================
  myRTC.begin();
  //=================================================================
  //myRTC.adjust(DateTime(year(), month(), day(), hour(), minute(), second()));
  myRTC.adjust(DateTime(2017, 2, 23, 0, 0, 0));
  //=================================================================
  myRTC.armAlarm1(false);
  myRTC.clearAlarm1();
  //=================================================================
  myRTC.armAlarm2(false);
  myRTC.clearAlarm2();
  //=================================================================
  myRTC.setAlarm1(0, 0, 0, 0, DS3231_EVERY_SECOND);                 // Alarm Every Second
  //=================================================================
  lastGetSecondTime = millis();
  //=================================================================

  while(1)
  {
    //===============================================================
    // 1st Half of Second Display = [H][H] : [M][M] : [S][S]
    //===============================================================
    if(myRTC.isAlarm1(false))
    {
      //=============================================================
      myTimeNow = myRTC.now();
      //=============================================================
      // [H][H]:[M][M]:[S][S]
      //=============================================================
      SerialDebug.print("Time : ");
      SerialDebug.print((myTimeNow.hour()/10)%10);
      SerialDebug.print((myTimeNow.hour()%10));
      SerialDebug.print(":");
      SerialDebug.print((myTimeNow.minute()/10)%10);
      SerialDebug.print((myTimeNow.minute()%10));
      SerialDebug.print(":");
      SerialDebug.print((myTimeNow.second()/10)%10);
      SerialDebug.print((myTimeNow.second()%10));
      SerialDebug.println();
      //=============================================================
      // If Select Jumper A2:A1 = LOW:LOW 
      PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.minute()/10)%10), // Minute(msb)
                                       (myTimeNow.minute()%10),     // Minute(lsb)
                                       false,                       // OFF : Dot MSB Minute
                                       true);                       // ON  : Dot LSB Minute
      PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                       (myTimeNow.second()%10),     // Second(lsb)
                                       false,                       // OFF : Dot MSB Second
                                       false);                      // OFF : Dot LSB Second         
      //=============================================================
      // If Select Jumper A2:A1 = LOW:HIGH
      PCF8575_SLAVE3.write_bcd_normal(((myTimeNow.minute()/10)%10), // Minute(msb)
                                       (myTimeNow.minute()%10),     // Minute(lsb)
                                       false,                       // OFF : Dot MSB Minute
                                       true);                       // ON  : Dot LSB Minute
      PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                       (myTimeNow.second()%10),     // Second(lsb)
                                       false,                       // OFF : Dot MSB Second
                                       false);                      // OFF : Dot LSB Second  
      //=============================================================
      // If Select Jumper A2:A1 = HIGH:LOW 
      PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.minute()/10)%10), // Minute(msb)
                                       (myTimeNow.minute()%10),     // Minute(lsb)
                                       false,                       // OFF : Dot MSB Minute
                                       true);                       // ON  : Dot LSB Minute
      PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                       (myTimeNow.second()%10),     // Second(lsb)
                                       false,                       // OFF : Dot MSB Second
                                       false);                      // OFF : Dot LSB Second  
      //============================================================= 
      // If Select Jumper A2:A1 = HIGH:HIGH     
      PCF8575_SLAVE7.write_bcd_normal(((myTimeNow.minute()/10)%10), // Minute(msb)
                                       (myTimeNow.minute()%10),     // Minute(lsb)
                                       false,                       // OFF : Dot MSB Minute
                                       true);                       // ON  : Dot LSB Minute
      PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                       (myTimeNow.second()%10),     // Second(lsb)
                                       false,                       // OFF : Dot MSB Second
                                       false);                      // OFF : Dot LSB Second                                                      
      //=============================================================
      myRTC.clearAlarm1();
      //=============================================================
      lastGetSecondTime = millis();
      //=============================================================
    }
    //===============================================================

    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 500ul)                    // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'E') ||(inByte == 'e'))                     // Exit Test
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
    
    //===============================================================
    // 2nd Half of Second Display = [H][H] [M][M] [S][S]
    //===============================================================
    //===============================================================
    // [H][H] [M][M] [S][S]
    //===============================================================
    myTimeNow = myRTC.now();
    //===============================================================
    // If Select Jumper A2:A1 = LOW:LOW 
    PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second                                
    //=============================================================== 
    // If Select Jumper A2:A1 = LOW:HIGH 
    PCF8575_SLAVE3.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second         
    //===============================================================
    // If Select Jumper A2:A1 = HIGH:LOW 
    PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second        
    //===============================================================
    // If Select Jumper A2:A1 = HIGH:HIGH 
    PCF8575_SLAVE7.write_bcd_normal(((myTimeNow.minute()/10)%10),   // Minute(msb)
                                     (myTimeNow.minute()%10),       // Minute(lsb)
                                     false,                         // OFF : Dot MSB Minute
                                     false);                        // OFF : Dot LSB Minute
    PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.second()/10)%10),   // Second(msb)
                                     (myTimeNow.second()%10),       // Second(lsb)
                                     false,                         // OFF : Dot MSB Second
                                     false);                        // OFF : Dot LSB Second  
    //===============================================================                                                                   
  }
}

/*******************
 * 3. Test OPTO
 *******************/
int test_opto(void)
{
  while(1)
  {    
    //===============================================================
    // Check Input0 Status
    //===============================================================
    thisInput0State = digitalRead(OPTO_INPUT0_PIN);
    if(thisInput0State != lastInput0State)
    {
      //=============================================================
      SerialDebug.print("Input0 = ");
      SerialDebug.println((thisInput0State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput0State = thisInput0State;
      //=============================================================
    }
    //===============================================================

    //===============================================================
    // Check Input1 Status
    //===============================================================
    thisInput1State = digitalRead(OPTO_INPUT1_PIN);
    if(thisInput1State != lastInput1State)
    {
      //=============================================================
      SerialDebug.print("Input1 = ");
      SerialDebug.println((thisInput1State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput1State = thisInput1State;
      //=============================================================
    }
    //===============================================================

    //===============================================================
    // Check Input2 Status
    //===============================================================
    thisInput2State = digitalRead(OPTO_INPUT2_PIN);
    if(thisInput2State != lastInput2State)
    {
      //=============================================================
      SerialDebug.print("Input2 = ");
      SerialDebug.println((thisInput2State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput2State = thisInput2State;
      //=============================================================
    }
    //===============================================================

    //===============================================================
    // Check Input3 Status
    //===============================================================
    thisInput3State = digitalRead(OPTO_INPUT3_PIN);
    if(thisInput3State != lastInput3State)
    {
      //=============================================================
      SerialDebug.print("Input3 = ");
      SerialDebug.println((thisInput3State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput3State = thisInput3State;
      //=============================================================
    }
    //===============================================================

    //===============================================================
    // Check Input4 Status
    //===============================================================
    thisInput4State = digitalRead(OPTO_INPUT4_PIN);
    if(thisInput4State != lastInput4State)
    {
      //=============================================================
      SerialDebug.print("Input4 = ");
      SerialDebug.println((thisInput4State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput4State = thisInput4State;
      //=============================================================
    }
    //===============================================================

    //===============================================================
    // Check Input5 Status
    //===============================================================
    thisInput5State = digitalRead(OPTO_INPUT5_PIN);
    if(thisInput5State != lastInput5State)
    {
      //=============================================================
      SerialDebug.print("Input5 = ");
      SerialDebug.println((thisInput5State ? "Release" : "Press"));  // 0:Press, 1:Release
      //=============================================================
      //=============================================================
      lastInput5State = thisInput5State;
      //=============================================================
    }
    //===============================================================
    
    //===============================================================
    if(SerialDebug.available() > 0)
    {
      inByte = SerialDebug.read();
      if((inByte == 'E') ||(inByte == 'e'))                         // Exit Test
      {
        print_menu_test();
        return;
      }
    }
    //===============================================================
  }
}

/*******************
 * 4. Test Relay
 *******************/
int test_relay(void)
{
  while(1)
  {    
    //===============================================================
    digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
    //===============================================================
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 1000ul)                     // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'E') ||(inByte == 'e'))                       // Exit Test
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================

    //===============================================================
    digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
    //===============================================================
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 1000ul)                     // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'E') ||(inByte == 'e'))                       // Exit Test
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  }
}

/*****************************
 * 5. Test RS485 & RS485 ID
 *****************************/
int test_usart(void)
{
  //=================================================================
  // Start of RS485 Test
  //=================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.println("Send Command : *01:R=0<Cr>...OFF Relay");
  SerialRS485.println("Send Command : *01:R=1<Cr>...ON  Relay");
  SerialRS485.println("==============================================================");
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  // End of RS485 Test
  //=================================================================
  while(1)
  {
    //===============================================================
    // Start of Receive & Service RS485 Command
    //===============================================================
    if(SerialRS485.available() > 0)
    {  
      rs485_chr = SerialRS485.read();
    
      if(rs485_chr == STX_CODE)                                    // Verify Start of Command Header
      { 
        //===========================================================
        // Reset Packet Frame Reset
        //===========================================================
        rs485_cmd_head = 1;                                        // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                         // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];
        //===========================================================
      } 

      //Start of Receive & Verify RS485 Packet Command
      if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
      { 
        *rs485_cmd_ptr = rs485_chr;                                // Save Character 
        rs485_cmd_ptr++;                                           // Next Pointer         
        rs485_cmd_cnt++;  

        //SerialDebug.write(rs485_chr);
        
        //===========================================================
        // Start of Analysis & Service RS485 Command Packet   
        //===========================================================
        if(rs485_chr == 0x0D)
        {           
          //=========================================================
          // <STX><ID0><ID1><Sep1><Relay><Sep2><Operate><Enter>
          // <STX><ID0><ID1><:><R><=><0><CR>   OFF Relay
          // <STX><ID0><ID1><:><R><=><1><CR>   ON  Relay
          //=========================================================
          if((rs485.control_relay.head      == STX_CODE)&&         // head : '*'
        
             (rs485.control_relay.id[0]     == SlaveID[0])&&       // id[0]:0..9
             (rs485.control_relay.id[1]     == SlaveID[1])&&       // id[1]:0..9
           
             (rs485.control_relay.sep1      == SEP1)&&             // sep1 : ':' 
           
             (rs485.control_relay.relay    == 'R')&&               // 'R':Relay
           
             (rs485.control_relay.sep2      == SEP2)&&             // sep2 : '='
           
             ((rs485.control_relay.operate  == '0')||
              (rs485.control_relay.operate  == '1'))&&             // operate('0':OFF,'1':ON)
           
             (rs485.control_relay.enter     == END_CODE))          // Enter:0x0D = End of Packet
          {
            //=======================================================
            if(rs485.control_relay.operate=='0')                                       
            {
              //=====================================================
              digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
              //=====================================================
              // Start of RS485 Test
              //=====================================================
              digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
              SerialRS485.print("Slave ID:");
              SerialRS485.write(SlaveID[0]);
              SerialRS485.write(SlaveID[1]);
              SerialRS485.println("...Relay : OFF");
              SerialRS485.flush();
              digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
              //=====================================================
              // End of RS485 Test
              //=====================================================
            }
            //=======================================================
            //=======================================================
            if(rs485.control_relay.operate=='1')                                       
            {
              //=====================================================
              digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOn);
              //=====================================================
              digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
              SerialRS485.print("Slave ID:");
              SerialRS485.write(SlaveID[0]);
              SerialRS485.write(SlaveID[1]);
              SerialRS485.println("...Relay : ON");
              SerialRS485.flush();
              digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
              //=====================================================
            }
            //=======================================================
          
          }
          //=========================================================
          // Clear RS485 Buffer
          //=========================================================
          rs485_cmd_head = 0;                                      // Command Header Not Complete  
          rs485_cmd_cnt = 0;                                       // Re-Start Command Count
          for(int i=0; i<sizeof(rs485.cmd_buff); i++)
          {
            rs485.cmd_buff[i] = '\0';
          }
          rs485_cmd_ptr = &rs485.cmd_buff[0];   
          //=========================================================
        } 
        //===========================================================
        // End of Analysis & Service RS485 Command Packet   
        //===========================================================
      }
    }
    //===============================================================
    // End of Receive & Service RS485 Command
    //===============================================================
    
    //===============================================================
    if(SerialDebug.available() > 0)
    {
      inByte = SerialDebug.read();
      if((inByte == 'E') ||(inByte == 'e'))                        // Exit Test
      {
        print_menu_test();
        return;
      }
    }
    //===============================================================
  }
}

/***********************************
 * 6. Test IR Remote Receiver Demo 
 * IR Model -> ET-IR REMOTE KEY
 **********************************/
int test_ir_remote(void)
{
  //=================================================================
  IR_Remote.enableIRIn();                                          // Start the receiver
  //=================================================================
  SerialDebug.println();
  SerialDebug.println("============================");
  SerialDebug.println("ET-IR REMOTE KEY...Demo Test");
  SerialDebug.println("============================");
  //=================================================================
  while(1)
  {
    if(IR_Remote.decode(&IR_Result)) 
    {
      if(IR_Result.value != 0xFFFFFFFF)
      {
        //Serial.print(results.value, HEX);
        switch(IR_Result.value)
        {
          case 0xFFA25D:
            SerialDebug.println("PWR");
          break;
        
          case 0xFFE21D:
            SerialDebug.println("MENU");
          break;

          case 0xFF22DD:
            SerialDebug.println("TEST");
          break;

          case 0xFF02FD:
            SerialDebug.println("+");
          break;

          case 0xFFC23D:
            SerialDebug.println("Back");
          break;

          case 0xFFE01F:
            SerialDebug.println("|<<");
          break;

          case 0xFFA857:
            SerialDebug.println(">");
          break;

          case 0xFF906F:
            SerialDebug.println(">>|");
          break;

          case 0xFF6897:
            SerialDebug.println("0");
          break;

          case 0xFF9867:
            SerialDebug.println("-");
          break;

          case 0xFFB04F:
            SerialDebug.println("C");
          break;

          case 0xFF30CF:
            SerialDebug.println("1");
          break;

          case 0xFF18E7:
            SerialDebug.println("2");
          break;

          case 0xFF7A85:
            SerialDebug.println("3");
          break;

          case 0xFF10EF:
            SerialDebug.println("4");
          break;

          case 0xFF38C7:
            SerialDebug.println("5");
          break;

          case 0xFF5AA5:
            SerialDebug.println("6");
          break;

          case 0xFF42BD:
            SerialDebug.println("7");
          break;

          case 0xFF4AB5:
            SerialDebug.println("8");
          break;

          case 0xFF52AD:
            SerialDebug.println("9");
          break;
        }
      }
      IR_Remote.resume(); // Receive the next value
    }
    //delay(100);
    
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 100ul)                    // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'E') ||(inByte == 'e'))                     // Exit Test
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  } //while
} 




