/*
 * Demo ET-DSP4 I2C Hardware Board
 * MCU        : ATMEGA32U4
 *            : Arduino Genuino Micro
 *            : Bootloader
 *            : -> .../caterina/Caterina-Genuino-Micro.hex
 *            : Fuse Bit
 *            : -> low_fuses      = 0xFF
 *            : -> high_fuses     = 0xD8
 *            : -> extended_fuses = 0xCB(0xFB)
 *            : Lock Bit
 *            : -> 0x2F(0xEF)
 * RS485      : RS485 RXD:D0
 *            : RS485 TXD:D1
 *            : RS485 Direction(D10:LOW=Receive,HIGH=Send)
 * I2C        : SCL=D3,SDA=D2
 *            : RTC:DS3231
 *            : PCF8575
 * IR Remote  : A0
 * Opto Input : A1,A2,A3
 *            : D4,D5,D6
 * Relay      : D8           
 * 
 * Demo       : Demo IR Remote Receive(ET-IR REMOTE KEY) 
 */

//===================================================================
#include <Wire.h>                                                   // I2C Bus
//===================================================================
#include "pcf8575.h"                                                // PCF8575/A
//===================================================================
#include <IRremote.h>                                               // IR Remote 
//===================================================================

//===================================================================
#define SerialDebug Serial                                          // USB Serial
#define SerialRS485 Serial1                                         // Serial1(D1=TXD,D0=RXD)
#define SerialRS232 Serial1                                         // Serial1(D1=TXD,D0=RXD)
int inByte;
//===================================================================

//===================================================================
// Start of Default Hardware : ET-DSP4 I2C
//===================================================================
#define OPTO_INPUT0_PIN     4
#define OPTO_INPUT1_PIN     5
#define OPTO_INPUT2_PIN     6
//===================================================================
#define OPTO_INPUT3_PIN     A1
#define OPTO_INPUT4_PIN     A2
#define OPTO_INPUT5_PIN     A3
//===================================================================
#define SignalRelease       HIGH
#define SignalPress         LOW
//===================================================================

//===================================================================
#define IR_REMOTE_PIN       A0
//===================================================================

//===================================================================
#define INTERNAL_RELAY_PIN  8
//===================================================================
#define InternalRelayOff    LOW
#define InternalRelayOn     HIGH
//===================================================================
#define RS485_DIRECTION_PIN 10                                      // 0:RXD,1:TXD
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
// End of Default Hardware : ET-DSP4 I2C
//===================================================================

//===================================================================
PCF8575 PCF8575_SLAVE0(0x20);                                       // PCF8575  = 0100,000+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE1(0x21);                                       // PCF8575  = 0100,001+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE2(0x22);                                       // PCF8575  = 0100,010+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE3(0x23);                                       // PCF8575  = 0100,011+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE4(0x24);                                       // PCF8575  = 0100,100+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE5(0x25);                                       // PCF8575  = 0100,101+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE6(0x26);                                       // PCF8575  = 0100,110+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE7(0x27);                                       // PCF8575  = 0100,111+(0:W,1:R)
//===================================================================

//===================================================================
// IR Remote Receiver
//===================================================================
IRrecv IR_Remote(IR_REMOTE_PIN);
decode_results IR_Result;
//===================================================================

void setup() 
{
  //=================================================================
  // Start of Initial Default Hardware : ET-DSP4 I2C
  //=================================================================
  SerialRS485.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  pinMode(OPTO_INPUT0_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT1_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT2_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT3_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT4_PIN, INPUT_PULLUP);
  pinMode(OPTO_INPUT5_PIN, INPUT_PULLUP);
  //=================================================================
  //=================================================================
  pinMode(INTERNAL_RELAY_PIN, OUTPUT);
  digitalWrite(INTERNAL_RELAY_PIN, InternalRelayOff);
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
  // End of Initial Default Hardware : ET-DSP4 I2C
  //=================================================================

  //=================================================================
  // If Select Jumper A2:A1 = LOW:LOW   = [0][1][2][3]
  //=================================================================
  PCF8575_SLAVE1.write_bcd_normal(0,1);                             // msb,lsb
  PCF8575_SLAVE0.write_bcd_normal(2,3);                             // msb,lsb
  //=================================================================
  // If Select Jumper A2:A1 = LOW:HIGH  = [4][5][6][7]
  PCF8575_SLAVE3.write_bcd_normal(4,5);                             // msb,lsb
  PCF8575_SLAVE2.write_bcd_normal(6,7);                             // msb,lsb
  //=================================================================
  // If Select Jumper A2:A1 = HIGH:LOW = [8][9][A][B]
  PCF8575_SLAVE5.write_bcd_normal(8,9);                             // msb,lsb
  PCF8575_SLAVE4.write_bcd_normal(10,11);                           // msb,lsb
  //=================================================================
  // If Select Jumper A2:A1 = HIGH:HIGH = [C][D][E][F]
  PCF8575_SLAVE7.write_bcd_normal(12,13);                           // msb,lsb
  PCF8575_SLAVE6.write_bcd_normal(14,15);                           // msb,lsb
  //=================================================================

  //=================================================================
  SerialDebug.begin(115200);                                       // Debug USART
  //while(!SerialDebug);                                             // wait USB Serial Complete 
  //=================================================================
  //=================================================================
  IR_Remote.enableIRIn();                                             // Start the receiver
  //=================================================================
  SerialDebug.println();
  SerialDebug.println("============================");
  SerialDebug.println("ET-IR REMOTE KEY...Demo Test");
  SerialDebug.println("============================");
  //=================================================================
}

void loop() 
{
  if(IR_Remote.decode(&IR_Result)) 
  {
    if(IR_Result.value != 0xFFFFFFFF)
    {
      //Serial.print(results.value, HEX);
      // If Select Jumper A2:A1 = LOW:LOW
      PCF8575_SLAVE1.write_bcd_normal((IR_Result.value>>12)&0x0F,(IR_Result.value>>8)&0x0F);          // msb,lsb
      PCF8575_SLAVE0.write_bcd_normal((IR_Result.value>>4)&0x0F,IR_Result.value&0x0F);                // msb,lsb
      // If Select Jumper A2:A1 = LOW:HIGH
      PCF8575_SLAVE3.write_bcd_normal((IR_Result.value>>12)&0x0F,(IR_Result.value>>8)&0x0F);          // msb,lsb
      PCF8575_SLAVE2.write_bcd_normal((IR_Result.value>>4)&0x0F,IR_Result.value&0x0F);                // msb,lsb
      // If Select Jumper A2:A1 = HIGH:LOW
      PCF8575_SLAVE5.write_bcd_normal((IR_Result.value>>12)&0x0F,(IR_Result.value>>8)&0x0F);          // msb,lsb
      PCF8575_SLAVE4.write_bcd_normal((IR_Result.value>>4)&0x0F,IR_Result.value&0x0F);                // msb,lsb
      // If Select Jumper A2:A1 = HIGH:HIGH
      PCF8575_SLAVE7.write_bcd_normal((IR_Result.value>>12)&0x0F,(IR_Result.value>>8)&0x0F);          // msb,lsb
      PCF8575_SLAVE6.write_bcd_normal((IR_Result.value>>4)&0x0F,IR_Result.value&0x0F);                // msb,lsb
      //
      switch(IR_Result.value)
      {
        case 0xFFA25D:
          SerialDebug.println("PWR");
        break;
        
        case 0xFFE21D:
          SerialDebug.println("MENU");
        break;

        case 0xFF22DD:
          SerialDebug.println("TEST");
        break;

        case 0xFF02FD:
          SerialDebug.println("+");
        break;

        case 0xFFC23D:
          SerialDebug.println("Back");
        break;

        case 0xFFE01F:
          SerialDebug.println("|<<");
        break;

        case 0xFFA857:
          SerialDebug.println(">");
        break;

        case 0xFF906F:
          SerialDebug.println(">>|");
        break;

        case 0xFF6897:
          SerialDebug.println("0");
        break;

        case 0xFF9867:
          SerialDebug.println("-");
        break;

        case 0xFFB04F:
          SerialDebug.println("C");
        break;

        case 0xFF30CF:
          SerialDebug.println("1");
        break;

        case 0xFF18E7:
          SerialDebug.println("2");
        break;

        case 0xFF7A85:
          SerialDebug.println("3");
        break;

        case 0xFF10EF:
          SerialDebug.println("4");
        break;

        case 0xFF38C7:
          SerialDebug.println("5");
        break;

        case 0xFF5AA5:
          SerialDebug.println("6");
        break;

        case 0xFF42BD:
          SerialDebug.println("7");
        break;

        case 0xFF4AB5:
          SerialDebug.println("8");
        break;

        case 0xFF52AD:
          SerialDebug.println("9");
        break;
      }
    }
    IR_Remote.resume(); // Receive the next value
  }
  delay(100);
}

