/*
 * Demo ET-BIG7x2-I2C Master Hardware Board
 * MCU    : ATMEGA32U4(PRO-MICRO)
 *        : Arduino Leonado
 *        : Bootloader
 *        : -> .../caterina/Caterina-Leonardo.hex
 *        : Fuse Bit
 *        : -> low_fuses      = 0xFF
 *        : -> high_fuses     = 0xD8
 *        : -> extended_fuses = 0xCB(0xFB)
 *        : Lock Bit
 *        : -> 0x2F(0xEF)
 * RS485  : RS485 RXD:D0
 *        : RS485 TXD:D1
 *        : RS485 Direction(D4:LOW=Receive,HIGH=Send)
 * I2C    : SCL=D3,SDA=D2
 *        : RTC:DS3231
 *        : PCF8575
 * ADC    : Analog#0 : A;[0..3]
 * Input  : RS485 ID0:D5
 *        : RS485 ID1:D6
 *        : RS485 ID2:D7
 *        : RS485 ID3:D8
 *        : RS485 ID4:D9
 * 1-Wire : DS18B20(1-Wire:D10)
 * 
 * Demo   : Display RS485-ID[00..31] on 7-Segment
 */

#include <Wire.h>                                                   // I2C Bus
#include "pcf8575.h"                                                // PCF8575
#include "ET_DS3231.h"
//===================================================================
#define I2C_SDA_PIN         2                                       // D2
#define I2C_SCL_PIN         3                                       // D3
//===================================================================
//===================================================================
#define SerialDebug         Serial                                  // USB Serial
#define SerialRS485         Serial1                                 // Serial1(D1=TXD,D0=RXD)
int inByte;
//===================================================================
//===================================================================
#define RS485_DIRECTION_PIN 4                                       // RS485 TXD Enable,Disable
#define RS485_RXD_SELECT    LOW
#define RS485_TXD_SELECT    HIGH
//===================================================================
//===================================================================
#define RS485_ID0_PIN       5                                       // Slave ID LSB
#define RS485_ID1_PIN       6         
#define RS485_ID2_PIN       7        
#define RS485_ID3_PIN       8        
#define RS485_ID4_PIN       9                                       // Slave ID MSB
//===================================================================
int NewAddressID = 0;
int OldAddressID = 0;
//===================================================================

//===================================================================
ET_DS3231 myRTC;
DateTime myTimeNow;
//===================================================================
unsigned long lastGetDelayTime = 0;
//===================================================================

int ledPin[] = {10,A0,A1,A2,A3};


//===================================================================
PCF8575 PCF8575_SLAVE0(0x20);                                       // PCF8575  = 0100,000+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE1(0x21);                                       // PCF8575  = 0100,001+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE2(0x22);                                       // PCF8575  = 0100,010+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE3(0x23);                                       // PCF8575  = 0100,011+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE4(0x24);                                       // PCF8575  = 0100,100+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE5(0x25);                                       // PCF8575  = 0100,101+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE6(0x26);                                       // PCF8575  = 0100,110+(0:W,1:R)
//===================================================================
PCF8575 PCF8575_SLAVE7(0x27);                                       // PCF8575  = 0100,111+(0:W,1:R)
//===================================================================

//===================================================================
// Start of RS485 Command Variable
// <STX><ID0><ID1><Sep1><cmd><Sep2><data[2]><Enter>
//===================================================================
static union  
{
  byte cmd_buff[10];                                                  // 8 Byte Max
  //=================================================================
  // <STX><ID0><ID1><Sep1><cmd><Sep2><data[2]><Enter>
  // <STX><ID0><ID1><:><D><=><dd><CR>  
  //=================================================================
  struct 
  {
    //===============================================================
    byte head;                          // <STX>:'*'                                                                       
    byte id[2];                         // ID[0:1}:'00'..'31'  
    byte sep1;                          // ':'
    byte cmd;                           // 'D':Display
    byte sep2;                          // '=' 
    byte dat[2];                        // Display Data : '00-99'
    byte enter;                         // <Cr>:0x0D
    //===============================================================
  }control_display;
  //=================================================================
  
}rs485;
//===================================================================
const byte      STX_CODE                = '*';
const byte      END_CODE                = 0x0D;
const byte      SEP1                    = ':';  
const byte      SEP2                    = '=';
//===================================================================
static byte *rs485_cmd_ptr;
static byte rs485_chr;
static byte rs485_cmd_head = 0;
static byte rs485_cmd_cnt = 0;  
//===================================================================
char SlaveID[2];
//===================================================================
bool exit_test = false;

//===================================================================
// Initial Slave Address
//===================================================================
int GetMyAddressID()
{
  //=================================================================
  int SlaveAddress = 0;
  //=================================================================
  if(digitalRead(RS485_ID0_PIN) == HIGH)
  {
    SlaveAddress += 1;
  }
  if(digitalRead(RS485_ID1_PIN) == HIGH)
  {
    SlaveAddress += 2;
  }
  if(digitalRead(RS485_ID2_PIN) == HIGH)
  {
    SlaveAddress += 4;
  }
  if(digitalRead(RS485_ID3_PIN) == HIGH)
  {
    SlaveAddress += 8;
  }
  if(digitalRead(RS485_ID4_PIN) == HIGH)
  {
    SlaveAddress += 16;
  }
  //=================================================================
  SlaveID[0] = ((SlaveAddress/10)%10)+'0';
  SlaveID[1] = (SlaveAddress%10)+'0';
  //=================================================================
  return SlaveAddress;
  //=================================================================
}
//===================================================================

//===================================================================
void setup() 
{
  //=================================================================
  for (int i=0; i<sizeof(ledPin)/sizeof(int); i++)
  {
    pinMode(ledPin[i],OUTPUT);
    digitalWrite(ledPin[i],LOW);
  }
  //=================================================================
  
  //=================================================================
  Wire.begin();                                                     // Initial I2C Bus
  //=================================================================
 
  //=================================================================
  // Initial RS232(USB Serial)
  //=================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);                                             // wait USB Serial Ready
  //=================================================================
  
  //=================================================================
  // Initial RS485
  //=================================================================
  SerialRS485.begin(115200);
  //=================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  pinMode(RS485_ID0_PIN, INPUT_PULLUP);
  pinMode(RS485_ID1_PIN, INPUT_PULLUP);
  pinMode(RS485_ID2_PIN, INPUT_PULLUP);
  pinMode(RS485_ID3_PIN, INPUT_PULLUP);
  pinMode(RS485_ID4_PIN, INPUT_PULLUP);
  //=================================================================
  //=================================================================
  NewAddressID = GetMyAddressID();
  OldAddressID = NewAddressID;
  //=================================================================
  
  //=================================================================
  PCF8575_SLAVE0.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE1.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE2.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE3.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE4.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE5.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE6.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  PCF8575_SLAVE7.write_seg(0x00,0x00);                              // Clear Display(Blank) 
  //=================================================================
  print_menu_test();
  //=================================================================
}

void loop() 
{
  if(SerialDebug.available() > 0)
  {
    inByte = SerialDebug.read();
    
    if(inByte == '1') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("1. Test 7 SEGMENT & SLAVE Device ID:[00..07]");
      SerialDebug.println("   After Blink Change Device ID Jumper For Test Device ID");
      SerialDebug.println("   Press <E>....Exit Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_7seg();
      //============================================================= 
    }
    
    else if(inByte == '2') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("2. Test RS485 SLAVE ID:[00..31]");
      SerialDebug.println("   Change RS485 ID Jumper For Test Hardware ID");
      SerialDebug.println("   Press <E>....Exit Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_rs485_setID();
      //============================================================= 
    }
    
    else if(inByte == '3') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("3. Test RS485 Send & Receive");
      SerialDebug.println("   Send <*><IDH><IDL><:><D><=><DSP1><DSP2><CR>");
      SerialDebug.println("   Press <E>....Exit Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_rs485_data();
      //============================================================= 
    }

    else if(inByte == '4') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("4. Test RTC DS3231");
      SerialDebug.println("   Press <E>....Exit Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_rtc();
      //============================================================= 
    }
    
    else if(inByte == '5') 
    {
      //=============================================================
      SerialDebug.println();
      SerialDebug.println("==============================================================");
      SerialDebug.println("5. Test GPIO (D10,A0,A1,A2,A3)");
      SerialDebug.println("   Press <E>....Exit Test");
      SerialDebug.println("==============================================================");
      //=============================================================  
      test_gpio();
      //============================================================= 
    }
  }
}

int print_menu_test(void)
{
  //=================================================================
  NewAddressID = GetMyAddressID();
  OldAddressID = NewAddressID;
  //=================================================================
  //=================================================================  
  PCF8575_SLAVE0.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE1.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE2.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE3.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE4.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE5.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE6.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE7.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  //=================================================================
     
  //=================================================================
  // Start of RS485 Test
  //=================================================================
  SerialDebug.println();
  SerialDebug.println("==============================================================");
  SerialDebug.println("QC Test ET-BIG 7X2 I2C MASTER");
  SerialDebug.print("My RS485 Address ID = ");
  SerialDebug.print((NewAddressID/10)%10);
  SerialDebug.println(NewAddressID%10);
  SerialDebug.println("==============================================================");
  SerialDebug.println("1. Test 7 SEGMENT & SLAVE Device ID:[00..07]");
  SerialDebug.println("2. Test RS485 SLAVE ID:[00..31]");
  SerialDebug.println("3. Test RS485 Send & Receive Data");
  SerialDebug.println("4. Test RTC DS3231");
  SerialDebug.println("5. Test GPIO (D0,A0,A1,A2,A3)");
  SerialDebug.println("==============================================================");
  SerialDebug.println("Select [1..5] For Test");
  SerialDebug.println("==============================================================");
  //=================================================================
  // End of RS485 Test
  //================================================================= 
}

/*
 * Test 7 Segment Display & Slave Device ID
 */
int test_7seg(void)
{
  //=================================================================
  // Test Display Segment : A..G,Dp
  //=================================================================
  for(uint8_t seg=0; seg<8; seg++)                                  // Seg-A,Seg-B,Seg-C,...,Seg-G,Seg-Dp
  {
    //===============================================================
    PCF8575_SLAVE0.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE1.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE2.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE3.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE4.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE5.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE6.write_seg((1<<(seg)),(1<<(seg)));  
    PCF8575_SLAVE7.write_seg((1<<(seg)),(1<<(seg)));  
    //===============================================================
    
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 1000ul)                    // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'e')||(inByte == 'E'))
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  }
  //=================================================================

  //=================================================================
  // Test Display Number : 0..9,A..F
  //=================================================================  
  for(uint16_t bcd=0; bcd<16; bcd++)                                // 0..9,A..F
  {
    //===============================================================
    PCF8575_SLAVE0.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE1.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE2.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE3.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE4.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE5.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE6.write_bcd_normal(bcd,bcd,false,false); 
    PCF8575_SLAVE7.write_bcd_normal(bcd,bcd,false,false); 
    //===============================================================
    
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 1000ul)                    // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
       if((inByte == 'e')||(inByte == 'E'))
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  }
  //=================================================================

  //=================================================================
  // Start of Test Device ID
  //=================================================================
  while(1)
  {
    //===============================================================
    // Test Display Device Slave ID : 00,01,02,03,04,05,06,07
    //===============================================================  
    PCF8575_SLAVE0.write_bcd_normal(0,0,false,false); 
    PCF8575_SLAVE1.write_bcd_normal(0,1,false,false); 
    PCF8575_SLAVE2.write_bcd_normal(0,2,false,false); 
    PCF8575_SLAVE3.write_bcd_normal(0,3,false,false); 
    PCF8575_SLAVE4.write_bcd_normal(0,4,false,false); 
    PCF8575_SLAVE5.write_bcd_normal(0,5,false,false); 
    PCF8575_SLAVE6.write_bcd_normal(0,0,false,false); 
    PCF8575_SLAVE7.write_bcd_normal(0,7,false,false); 
    //===============================================================
    
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 500ul)                     // 0.5-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'e')||(inByte == 'E'))
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
    
    //===============================================================
    // Test Display Device Slave ID : 00,01,02,03,04,05,06,07
    //===============================================================  
    PCF8575_SLAVE0.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE1.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE2.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE3.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE4.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE5.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE6.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    PCF8575_SLAVE7.write_seg(0x00,0x00);                            // Clear Display(Blank) 
    //===============================================================
    
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 500ul)                     // 0.5-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'e')||(inByte == 'E'))
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  }    
  //=================================================================
  // End of Test Device ID
  //=================================================================
}

/*
 * Test RS485 & RS485 ID
 */
int test_rs485_setID(void)
{
  //=================================================================
  NewAddressID = GetMyAddressID();
  OldAddressID = NewAddressID;
  //=================================================================
  //=================================================================  
  PCF8575_SLAVE0.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE1.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE2.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE3.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE4.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE5.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE6.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  PCF8575_SLAVE7.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
  //=================================================================
  
  //=================================================================
  // Start of RS485 Test
  //=================================================================
  SerialDebug.println("ET-BIX7x2 I2C RS485");
  SerialDebug.print("My Address ID = ");
  SerialDebug.print((NewAddressID/10)%10);
  SerialDebug.println(NewAddressID%10);
  SerialDebug.println("Change RS485 ID Jumper For Test Hardware ID");
  SerialDebug.println("Press <E>....Exit Test");
  SerialDebug.println("==============================================================");
  //=================================================================
  // End of RS485 Test
  //=================================================================
  while(1)
  {
    //===============================================================
    if(SerialDebug.available() > 0)
    {
      inByte = SerialDebug.read();
      if((inByte == 'e')||(inByte == 'E'))
      {
        print_menu_test();
        return;
      }
    }
    //===============================================================
    
    NewAddressID = GetMyAddressID();
    if(NewAddressID != OldAddressID)
    {
      //=============================================================  
      PCF8575_SLAVE0.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE1.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE2.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE3.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE4.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE5.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE6.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      PCF8575_SLAVE7.write_bcd_normal(((NewAddressID/10)%10),(NewAddressID%10),false,false); 
      //=============================================================
      OldAddressID = NewAddressID;
      //=============================================================
      //=============================================================
      // Start of RS485 Test
      //=============================================================
      SerialDebug.println("ET-BIG 7X2 I2C RS485");
      SerialDebug.print("My Address ID = ");
      SerialDebug.print((NewAddressID/10)%10);
      SerialDebug.println(NewAddressID%10);
      SerialDebug.println("Change RS485 ID Jumper For Test Hardware ID");
      SerialDebug.println("Press <E>....Exit Test");
      //=============================================================
      // End of RS485 Test
      //=============================================================
    }
  }
}

/*******************************
 * 5. Test RS485 Send & Receive
 *******************************/
int test_rs485_data(void)
{
  //=================================================================
  NewAddressID = GetMyAddressID();
  OldAddressID = NewAddressID;
  //=================================================================
  
  //=================================================================
  // Start of RS485 Test
  //=================================================================
  digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.print("My RS485 Slave ID:");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println();
  SerialRS485.println("==============================================================");
  SerialRS485.print("Send Command : *");
  SerialRS485.write(SlaveID[0]);
  SerialRS485.write(SlaveID[1]);
  SerialRS485.println(":D=dd<Cr>...dd='00..99'");
  SerialRS485.println("==============================================================");
  SerialRS485.flush();
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //=================================================================
  // End of RS485 Test
  //=================================================================
  while(1)
  {
    //===============================================================
    // Start of Receive & Service RS485 Command
    //===============================================================
    if(SerialRS485.available() > 0)
    {  
      rs485_chr = SerialRS485.read();
    
      if(rs485_chr == STX_CODE)                                    // Verify Start of Command Header
      { 
        //===========================================================
        // Reset Packet Frame Reset
        //===========================================================
        rs485_cmd_head = 1;                                        // Command Header Not Complete  
        rs485_cmd_cnt = 0;                                         // Re-Start Command Count
        for(int i=0; i<sizeof(rs485.cmd_buff); i++)
        {
          rs485.cmd_buff[i] = '\0';
        }
        rs485_cmd_ptr = &rs485.cmd_buff[0];
        //===========================================================
      } 

      //Start of Receive & Verify RS485 Packet Command
      if((rs485_cmd_head == 1) && (rs485_cmd_cnt <=(sizeof(rs485.cmd_buff))))
      { 
        *rs485_cmd_ptr = rs485_chr;                                // Save Character 
        rs485_cmd_ptr++;                                           // Next Pointer         
        rs485_cmd_cnt++;  

        //===========================================================
        // Start of Analysis & Service RS485 Command Packet   
        //===========================================================
        if(rs485_chr == 0x0D)
        {         
          //=========================================================
          // <STX><ID0><ID1><Sep1><cmd><Sep2><data[2]><Enter>
          // <STX><ID0><ID1><:><D><=><??><CR>  
          //=========================================================
          if((rs485.control_display.head      == STX_CODE)&&         // head : '*'
        
             (rs485.control_display.id[0]     == SlaveID[0])&&       // id[0]:0..9
             (rs485.control_display.id[1]     == SlaveID[1])&&       // id[1]:0..9
           
             (rs485.control_display.sep1      == SEP1)&&             // sep1 : ':' 
           
             (rs485.control_display.cmd       == 'D')&&               // 'D':Display
           
             (rs485.control_display.sep2      == SEP2)&&             // sep2 : '='
           
             (rs485.control_display.enter     == END_CODE))          // Enter:0x0D = End of Packet
          {
            //=====================================================
            PCF8575_SLAVE0.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE1.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE2.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE3.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE4.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false);  
            PCF8575_SLAVE5.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE6.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            PCF8575_SLAVE7.write_bcd_normal((rs485.control_display.dat[0]&0x0F),(rs485.control_display.dat[1]&0x0F),false,false); 
            //=====================================================
              
            //=====================================================
            // Start of RS485 Test
            //=====================================================
            digitalWrite(RS485_DIRECTION_PIN, RS485_TXD_SELECT);
            SerialRS485.print("Slave ID:");
            SerialRS485.write(SlaveID[0]);
            SerialRS485.write(SlaveID[1]);
            SerialRS485.println("...Update Display....Complete");
            SerialRS485.flush();
            digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
            //=====================================================
            // End of RS485 Test
            //=====================================================
          }
          //=========================================================
          // Clear RS485 Buffer
          //=========================================================
          rs485_cmd_head = 0;                                      // Command Header Not Complete  
          rs485_cmd_cnt = 0;                                       // Re-Start Command Count
          for(int i=0; i<sizeof(rs485.cmd_buff); i++)
          {
            rs485.cmd_buff[i] = '\0';
          }
          rs485_cmd_ptr = &rs485.cmd_buff[0];   
          //=========================================================
        } 
        //===========================================================
        // End of Analysis & Service RS485 Command Packet   
        //===========================================================
      }
    }
    //===============================================================
    // End of Receive & Service RS485 Command
    //===============================================================
    
    //===============================================================
    if(SerialDebug.available() > 0)
    {
      inByte = SerialDebug.read();
      if((inByte == 'E') ||(inByte == 'e'))                        // Exit Test
      {
        print_menu_test();
        return;
      }
    }
    //===============================================================
  }
}

/*
 * Test RTC DS3231
 */
int test_rtc(void)
{
  //=================================================================
  myRTC.begin();
  //=================================================================
  //myRTC.adjust(DateTime(year(), month(), day(), hour(), minute(), second()));
  myRTC.adjust(DateTime(2017, 2, 23, 11, 37, 0));
  //=================================================================
  myRTC.armAlarm1(false);
  myRTC.clearAlarm1();
  //=================================================================
  myRTC.armAlarm2(false);
  myRTC.clearAlarm2();
  //=================================================================
  
  //=================================================================
  myRTC.setAlarm1(0, 0, 0, 0, DS3231_EVERY_SECOND);                 // Alarm Every Second
  //=================================================================

  while(1)
  {
    //===============================================================
    if(myRTC.isAlarm1(false))
    {
      //=============================================================
      myTimeNow = myRTC.now();
      PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second
      PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second         
      PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second   
      PCF8575_SLAVE3.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second  
      PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second    
      PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second  
      PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second      
      PCF8575_SLAVE7.write_bcd_normal(((myTimeNow.second()/10)%10),  // Second(msb)
                                       (myTimeNow.second()%10),      // Second(lsb)
                                       false,                        // OFF : Dot Minute
                                       true);                        // ON  : Dot Second                                                                                                                                                                                                            
      //=============================================================
      myRTC.clearAlarm1();
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();  
    //===============================================================
    while(millis() - lastGetDelayTime < 500ul)                    // 1-Second
    {
      //=============================================================
      if(SerialDebug.available() > 0)
      {
        inByte = SerialDebug.read();
        if((inByte == 'e')||(inByte == 'E'))
        {
          print_menu_test();
          return;
        }
      }
      //=============================================================
    }
    //===============================================================
    lastGetDelayTime = millis();
    //===============================================================
  
    //===============================================================
    myTimeNow = myRTC.now();
    PCF8575_SLAVE0.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second
    PCF8575_SLAVE1.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second    
    PCF8575_SLAVE2.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second   
    PCF8575_SLAVE3.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second
    PCF8575_SLAVE4.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second  
    PCF8575_SLAVE5.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second     
    PCF8575_SLAVE6.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second       
    PCF8575_SLAVE7.write_bcd_normal(((myTimeNow.second()/10)%10), // Second(msb)
                                    (myTimeNow.second()%10),      // Second(lsb)
                                    false,                        // OFF : Dot Minute
                                    false);                       // OFF : Dot Second                                                                                                                                                                                                           
    //================================================================
  }
 
}

/*
 * Test GPIO
 */
int test_gpio(void)
{
  //=================================================================
  for (int i=0; i<sizeof(ledPin)/sizeof(int); i++)
  {
    pinMode(ledPin[i],OUTPUT);
    digitalWrite(ledPin[i],LOW);
  }
  //=================================================================

  //=================================================================
  while(1)
  {
    //===============================================================
    for (int i=0; i<sizeof(ledPin)/sizeof(int); i++)
    {
      //=============================================================
      digitalWrite(ledPin[i],HIGH);
      //=============================================================
      //=============================================================
      lastGetDelayTime = millis();  
      //=============================================================
      while(millis() - lastGetDelayTime < 500ul)                     // 0.5-Second
      {
        //===========================================================
        if(SerialDebug.available() > 0)
        {
          inByte = SerialDebug.read();
          if((inByte == 'e')||(inByte == 'E'))
          {
            print_menu_test();
            return;
          }
        }
        //===========================================================
      }
      //=============================================================
      lastGetDelayTime = millis();
      //=============================================================
    }
    //===============================================================

    //===============================================================
    for (int i=0; i<sizeof(ledPin)/sizeof(int); i++)
    {
      //=============================================================
      digitalWrite(ledPin[i],LOW);
      //=============================================================
      //=============================================================
      lastGetDelayTime = millis();  
      //=============================================================
      while(millis() - lastGetDelayTime < 500ul)                     // 0.5-Second
      {
        //===========================================================
        if(SerialDebug.available() > 0)
        {
          inByte = SerialDebug.read();
          if((inByte == 'e')||(inByte == 'E'))
          {
            print_menu_test();
            return;
          }
        }
        //===========================================================
      }
      //=============================================================
      lastGetDelayTime = millis();
      //=============================================================
    }
    //===============================================================
  }
  //=================================================================
}


