/************************************************/
/* Demo Program For ET-BASE PIC40/46K80(ICSP)   */
/* MCU Control : PIC18F46K80                    */
/*             : Run 64MHz(Internal 16MHz+PLL)  */
/* Compiler    : PIC-CCS Version 4.120          */
/* Function    : Demo ET-SPI DSP7SEG Board      */
/************************************************/
// STLED316S Panel Display Control(7 x 7Segment)       
// DIN       = RE0(SPI:MOSI)
// CLK       = RE1(SPI:SCK)
// STB       = RE2(SPI:SS#)

#include <18F46K80.h>
#include <stdlib.h>

// Fuses: PIC18F46K80(CCS Compiler)
//////// Fuses: VREGSLEEP_SW,VREGSLEEP,INTRC_LP,INTRC_HP,SOSC_LOW,SOSC_DIG
//////// Fuses: SOSC_HIGH,NOXINST,XINST,LP,XT,HSH,HSM,ECH,ECH_IO,RC,RC_IO
//////// Fuses: INTRC_IO,INTRC,ECM,ECM_IO,ECL,ECL_IO,NOPLLEN,PLLEN,NOFCMEN
//////// Fuses: FCMEN,NOIESO,IESO,PUT,NOPUT,NOBROWNOUT,BROWNOUT_SW
//////// Fuses: BROWNOUT_NOSL,BROWNOUT,BORV30,BORV27,BORV20,BORV18,BORM_LOW
//////// Fuses: BORM_MED,BORM_HIGH,ZPBORM,NOWDT,WDT_NOSLEEP,WDT_SW,WDT,WDT1
//////// Fuses: WDT2,WDT4,WDT8,WDT16,WDT32,WDT64,WDT128,WDT256,WDT512
//////// Fuses: WDT1024,WDT2048,WDT4096,WDT8192,WDT16384,WDT32768,WDT65536
//////// Fuses: WDT131072,WDT262144,WDT524288,WDT1048576,CANC,CANB,MSSPMSK5
//////// Fuses: MSSPMSK7,NOMCLR,MCLR,NOSTVREN,STVREN,BBSIZ1K,BBSIZ2K,DEBUG
//////// Fuses: NODEBUG,PROTECT,NOPROTECT,CPB,NOCPB,CPD,NOCPD,WRT,NOWRT,WRTC
//////// Fuses: NOWRTC,WRTB,NOWRTB,WRTD,NOWRTD,EBTR,NOEBTR,EBTRB,NOEBTRB

//Select Jumper RA6,RA7(XTAL/GPIO) = GPIO
#fuses INTRC_IO,PLLEN,NOWDT,NOPROTECT,NODEBUG,SOSC_DIG                        // Ineternal 16MHz + PLL(4) = 64MHz 
#use delay(clock=64000000)

//STLED316S Interface
#define  STLED316S_DIN_PIN       PIN_E0                                       // DIN                            
#define  STLED316S_CLK_PIN       PIN_E1                                       // CLK
#define  STLED316S_STB_PIN       PIN_E2                                       // STB

#define  stled316s_din_dir()     output_drive(STLED316S_DIN_PIN)              // DIN = Output
#define  stled316s_din_set()     output_bit(STLED316S_DIN_PIN,1)              // DIN = 1    
#define  stled316s_din_clr()     output_bit(STLED316S_DIN_PIN,0)              // DIN = 0

#define  stled316s_clk_dir()     output_drive(STLED316S_CLK_PIN)              // CLK = Output
#define  stled316s_clk_set()     output_bit(STLED316S_CLK_PIN,1)              // CLK = 1    
#define  stled316s_clk_clr()     output_bit(STLED316S_CLK_PIN,0)              // CLK = 0

#define  stled316s_stb_dir()     output_drive(STLED316S_STB_PIN)              // STB = Output
#define  stled316s_stb_set()     output_bit(STLED316S_STB_PIN,1)              // STB = 1    
#define  stled316s_stb_clr()     output_bit(STLED316S_STB_PIN,0)              // STB = 0

// STLED316S Command
// Bit   = 7-6-5-4-3-2-1-0
// 7     = Reserve(0)
// 6     = Direction(0=Write,1=Read) 
// 5     = Cursor Address(0=Auto Increase,1=Fixed)
// 4-3   = Bank(Page)
//       = Page0(00):Write Segment Display
//       = Page1(01):Write LED Display
//       = Page2(10):Setup Segment Brightness 
//       = Page3(11):Setup LED Brightness
// 2-1-0 = Address(0..5)

// Write 7-Segment Display Command(00100+Address[5..0])
// Command Code = 0x20..0x25
// Option Data  = Segment Display Code
// 0(Reserve)+0(Write)+1(Fixed)+00(Page0)+Address(???) 
// -> Address 0x00 Display Segment Digit[1]
// -> Address 0x01 Display Segment Digit[2]
// -> Address 0x02 Display Segment Digit[3]
// -> Address 0x03 Display Segment Digit[4]
// -> Address 0x04 Display Segment Digit[5]
// -> Address 0x05 Display Segment Digit[6]
#define  WRITE_SEG_DISPLAY       0x20                                         // Write Segment Display                           
#define  DIG2                    0x00
#define  DIG3                    0x01
#define  DIG4                    0x02
#define  DIG5                    0x03
#define  DIG6                    0x04
#define  DIG7                    0x05
   
// Write LED Display Command(00101000)
// Command Code = 0x28
// Option Data  = LED Display Code
// 0(Reserve)+0(Write)+1(Fixed)+01(Page1)+000(Address=0x00) 
// Command Code = 001+01+000(Fixed Address=0x00 Only) 
// 001(Write Fixed)+01(Page)+000(Address)
#define  WRITE_LED_DISPLAY       0x28                                         // Write LED Display
#define  DIG1                    0x00

// Display ON Command(00101101) 
// Command Code = 0x2D
// Option Data  = 0x00
// 0(Reserve)+0(Write)+1(Fixed)+01(Page1)+Address(101) 
#define  WRITE_DISPLAY_ON        0x2D                                         // ON  Display Command

// Display OFF Command(00101110)
// Command Code = 0x2E
// Option Data  = 0x00
// 0(Reserve)+0(Write)+1(Fixed)+01(Page1)+Address(110) 
#define  WRITE_DISPLAY_OFF       0x2E                                         // OFF Display Command

// Write Config Display Command (00110000)
// Command Code = 0x30
// Option Data  = Config Value
// 0(Reserve)+0(Write)+1(Fixed)+10(Page2)+Address(0x00) 
#define  WRITE_CONFIG            0x30                                         // Write Config Display
// Option Data of Config Code(11111101)
// - Bit[7..5] = All Digit Constant Brightness Level[0..7]
// - Bit[4..3] = Display Mode 00:Variable Bright,11=Constant Bright
// - Bit[2..0] = Number of Digit Scan Display[0..5]
#define  CONSTANT_BRIGHT         0xFD                                         // Scan All Digit+Constant Bright
#define  VARIABLE_BRIGHT         0xE5                                         // Scan All Digit+Variable Bright

// Setup Segment Brightness Level Command(00110+Address[0x01..0x03])
// Command Code = 0x31..0x33
// Option Data  = Brightness Level Code
// 0(Reserve)+0(Write)+1(Fixed)+10(Page2)+Address(???)
// -> Address 0x01 Setup Brightness Seg Digit[2:1]
// -> Address 0x02 Setup Brightness Seg Digit[4:3]
// -> Address 0x03 Setup Brightness Seg Digit[6:5]
#define  WRITE_SEG_BRIGHTNESS    0x30                                         // Setup Segment Brightness
#define  DIG_SEG2_SEG1           0x01      
#define  DIG_SEG4_SEG3           0x02
#define  DIG_SEG6_SEG5           0x03

// Setup LED Brightness Level Command(00111+Address[0..3])
// Command Code = 0x38..0x3B
// Option Data  = Brightness Level Code
// 0(Reserve)+0(Write)+1(Fixed)+11(Page3)+Address(???)
// -> Address 0x00 Setup Brightness LED Digit[2:1]
// -> Address 0x01 Setup Brightness LED Digit[4:3]
// -> Address 0x02 Setup Brightness LED Digit[6:5]
// -> Address 0x03 Setup Brightness LED Digit[8:7]
#define  WRITE_LED_BRIGHTNESS    0x38                                         // Setup LED Brightness
#define  DIG_LED2_LED1           0x00      
#define  DIG_LED4_LED3           0x01
#define  DIG_LED6_LED5           0x02
#define  DIG_LED8_LED7           0x03

/************************/
/* 7 Segment Code Table */
/************************/
const unsigned char tab_7seg[] = 
{
  0x3F,0x06,0x5B,0x4F,                                                        // 0  1  2  3
  0x66,0x6D,0x7D,0x07,                                                        // 4  5  6  7
  0x7F,0x6F,0x77,0x7C,                                                        // 8  9  A  B
  0x39,0x5E,0x79,0x71,                                                        // C  D  E  F
          
  0x40,0x00,                                                                  // -  Space
                    
  // Segment A..G,Dp
  0x01,0x02,0x04,0x08,                                                        // Seg-A Seg-B Seg-C Seg-D  
  0x10,0x20,0x40,0x80                                                         // Seg-E Seg-F Seg-G Seg-Dp
};

// 7-Segment Code Pointer
#define  NUM0                    0
#define  NUM1                    1
#define  NUM2                    2
#define  NUM3                    3
#define  NUM4                    4
#define  NUM5                    5
#define  NUM6                    6
#define  NUM7                    7
#define  NUM8                    8
#define  NUM9                    9
#define  CHAR_A                  10
#define  CHAR_B                  11
#define  CHAR_C                  12
#define  CHAR_D                  13
#define  CHAR_E                  14
#define  CHAR_F                  15
#define  CHAR_SUB                16
#define  CHAR_SPACE              17
#define  SEG_A                   18
#define  SEG_B                   19
#define  SEG_C                   20
#define  SEG_D                   21
#define  SEG_E                   22
#define  SEG_F                   23
#define  SEG_G                   24
#define  SEG_Dp                  25

/* pototype  section */
void stled316s_init(void);
void send_seg_display(unsigned char seg_digit,unsigned char seg_code);
void send_hex_display(unsigned char seg_digit,unsigned char hex_code);
void setup_brightness_display(unsigned char bright_level);
void stled316s_send_byte(unsigned char seg_command,unsigned char seg_data);

/* Main Program Start Here */
void main(void)
{ 
  setup_oscillator(OSC_16MHZ|OSC_PLL_ON);                                     // Sets the Internal Oscillator to 64MHz (PLL Enabled)
  stled316s_init();                                                           // Initial STLED316S
  
  while(true)
  {    
    send_seg_display(0,6);                                                    // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7  
    setup_brightness_display(0);
    delay_ms(500);

    send_seg_display(0,5);                                                    // Display Digit[0] = 1
    send_seg_display(1,6);                                                    // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(1);
    delay_ms(500);

    send_seg_display(0,4);                                                    // Display Digit[0] = 1
    send_seg_display(1,5);                                                    // Display Digit[1] = 2
    send_seg_display(2,6);                                                    // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(2);
    delay_ms(500);

    send_seg_display(0,3);                                                    // Display Digit[0] = 1
    send_seg_display(1,4);                                                    // Display Digit[1] = 2
    send_seg_display(2,5);                                                    // Display Digit[2] = 3
    send_seg_display(3,6);                                                    // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(3); 
    delay_ms(500);
    
    send_seg_display(0,2);                                                    // Display Digit[0] = 1
    send_seg_display(1,3);                                                    // Display Digit[1] = 2
    send_seg_display(2,4);                                                    // Display Digit[2] = 3
    send_seg_display(3,5);                                                    // Display Digit[3] = 4
    send_seg_display(4,6);                                                    // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(4); 
    delay_ms(500);
    
    send_seg_display(0,1);                                                    // Display Digit[0] = 1
    send_seg_display(1,2);                                                    // Display Digit[1] = 2
    send_seg_display(2,3);                                                    // Display Digit[2] = 3
    send_seg_display(3,4);                                                    // Display Digit[3] = 4
    send_seg_display(4,5);                                                    // Display Digit[4] = 5
    send_seg_display(5,6);                                                    // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(5); 
    delay_ms(500);
    
    send_seg_display(0,0);                                                    // Display Digit[0] = 1
    send_seg_display(1,1);                                                    // Display Digit[1] = 2
    send_seg_display(2,2);                                                    // Display Digit[2] = 3
    send_seg_display(3,3);                                                    // Display Digit[3] = 4
    send_seg_display(4,4);                                                    // Display Digit[4] = 5
    send_seg_display(5,5);                                                    // Display Digit[5] = 6
    send_seg_display(6,6);                                                    // Display Digit[6] = 7
    setup_brightness_display(6); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,0);                                                    // Display Digit[1] = 2
    send_seg_display(2,1);                                                    // Display Digit[2] = 3
    send_seg_display(3,2);                                                    // Display Digit[3] = 4
    send_seg_display(4,3);                                                    // Display Digit[4] = 5
    send_seg_display(5,4);                                                    // Display Digit[5] = 6
    send_seg_display(6,5);                                                    // Display Digit[6] = 7
    setup_brightness_display(7); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,0);                                                    // Display Digit[2] = 3
    send_seg_display(3,1);                                                    // Display Digit[3] = 4
    send_seg_display(4,2);                                                    // Display Digit[4] = 5
    send_seg_display(5,3);                                                    // Display Digit[5] = 6
    send_seg_display(6,4);                                                    // Display Digit[6] = 7
    setup_brightness_display(6); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,0);                                                    // Display Digit[3] = 4
    send_seg_display(4,1);                                                    // Display Digit[4] = 5
    send_seg_display(5,2);                                                    // Display Digit[5] = 6
    send_seg_display(6,3);                                                    // Display Digit[6] = 7
    setup_brightness_display(5); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,0);                                                    // Display Digit[4] = 5
    send_seg_display(5,1);                                                    // Display Digit[5] = 6
    send_seg_display(6,2);                                                    // Display Digit[6] = 7
    setup_brightness_display(4); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,0);                                                    // Display Digit[5] = 6
    send_seg_display(6,1);                                                    // Display Digit[6] = 7
    setup_brightness_display(3); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,0);                                                    // Display Digit[6] = 7
    setup_brightness_display(2); 
    delay_ms(500);
    
    send_seg_display(0,CHAR_SPACE);                                           // Display Digit[0] = 1
    send_seg_display(1,CHAR_SPACE);                                           // Display Digit[1] = 2
    send_seg_display(2,CHAR_SPACE);                                           // Display Digit[2] = 3
    send_seg_display(3,CHAR_SPACE);                                           // Display Digit[3] = 4
    send_seg_display(4,CHAR_SPACE);                                           // Display Digit[4] = 5
    send_seg_display(5,CHAR_SPACE);                                           // Display Digit[5] = 6
    send_seg_display(6,CHAR_SPACE);                                           // Display Digit[6] = 7
    setup_brightness_display(1); 
    delay_ms(500);
  }
}

/*****************************/
/* Initial GLCD 122x32 Pixel */
/*   MPU(80) Interface Mode  */
/*****************************/
void stled316s_init(void) 
{    
  stled316s_din_dir();                                                        // Control Signal
  stled316s_clk_dir();
  stled316s_stb_dir();
  stled316s_din_set();                                                        // Standby DIN
  stled316s_clk_set();                                                        // Standby CLK  
  stled316s_stb_set();                                                        // Standby STB
    
  stled316s_send_byte(WRITE_CONFIG,VARIABLE_BRIGHT);                          // Config Scan All Digit+Variable Brightness
  //stled316s_send_byte(WRITE_CONFIG,CONSTANT_BRIGHT);
  setup_brightness_display(7);                                                // Setup Brightness Level = 7(Max)         
  stled316s_send_byte(WRITE_DISPLAY_ON,0x00);                                 // Display ON   
}

/***********************************************/
/* Write Digit Segment Display By Segment Code */
/* [DIG6][DIG5][DIG4][DIG3][DIG2][DIG1][DIG0]  */
/* Input : seg_digit[6..0]                     */
/*       : seg_code[0..25]                     */ 
/***********************************************/
void send_seg_display(unsigned char seg_digit,unsigned char seg_code)
{
  unsigned char seg_data;

  seg_data = tab_7seg[seg_code];

  switch(seg_digit)
  { 
    case 0x00 : stled316s_send_byte(WRITE_LED_DISPLAY|DIG1,seg_data);         // Display Digit[0] = STLED316S Segment:DIG1(LED:Pin5)
                break;
    
    case 0x01 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG2,seg_data);         // Display Digit[1] = STLED316S Segment:DIG2(Pin:6)
                break;
    
    case 0x02 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG3,seg_data);         // Display Digit[2] = STLED316S Segment:DIG3(Pin:7)
                break;
    
    case 0x03 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG4,seg_data);         // Display Digit[3] = STLED316S Segment:DIG4(Pin:8)
                break;
     
    case 0x04 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG5,seg_data);         // Display Digit[4] = STLED316S Segment:DIG5(Pin:10)    
                break;
    
    case 0x05 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG6,seg_data);         // Display Digit[5] = STLED316S Segment:DIG6(Pin:11)
                break;
    
    case 0x06 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG7,seg_data);         // Display Digit[6] = STLED316S Segment:DIG7(Pin:12)
                break;
  } 
}

/***********************************************/
/* Write Digit Segment Display By BCD Hex Code */
/* [DIG6][DIG5][DIG4][DIG3][DIG2][DIG1][DIG0]  */
/* Input : seg_digit[6..0]                     */
/*       : seg_data[0x00..0xFF]                */ 
/***********************************************/
void send_hex_display(unsigned char seg_digit,unsigned char hex_code)
{
  switch(seg_digit)
  {    
    case 0x00 : stled316s_send_byte(WRITE_LED_DISPLAY|DIG1,hex_code);         // Display Digit[0] = STLED316S Segment:DIG1(LED:Pin5)
                break;
    
    case 0x01 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG2,hex_code);         // Display Digit[1] = STLED316S Segment:DIG2(Pin:6)
                break;
    
    case 0x02 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG3,hex_code);         // Display Digit[2] = STLED316S Segment:DIG3(Pin:7)
                break;
    
    case 0x03 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG4,hex_code);         // Display Digit[3] = STLED316S Segment:DIG4(Pin:8)
                break;
      
    case 0x04 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG5,hex_code);         // Display Digit[4] = STLED316S Segment:DIG5(Pin:10)
                break;
    
    case 0x05 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG6,hex_code);         // Display Digit[5] = STLED316S Segment:DIG6(Pin:11)
                break;
    
    case 0x06 : stled316s_send_byte(WRITE_SEG_DISPLAY|DIG7,hex_code);         // Display Digit[6] = STLED316S Segment:DIG7(Pin:12)
                break;
  } 
}
/**********************************/
/* Setup Brightness Level Display */
/* Input : bright_level[0..7]     */
/**********************************/
void setup_brightness_display(unsigned char bright_level)
{
  unsigned char bright_code;

  switch(bright_level)
  {    
    case 0x00 : bright_code = 0x00;                                           // Level[0] = 1/16 of PWM
                break;
    
    case 0x01 : bright_code = 0x11;                                           // Level[1] = 2/16 of PWM
                break;
    
    case 0x02 : bright_code = 0x22;                                           // Level[2] = 4/16 of PWM
                break;
    
    case 0x03 : bright_code = 0x33;                                           // Level[3] = 10/16 of PWM
                break;   
    
    case 0x04 : bright_code = 0x44;                                           // Level[4] = 11/16 of PWM
                break;

    case 0x05 : bright_code = 0x55;                                           // Level[5] = 12/16 of PWM
                break;
    
    case 0x06 : bright_code = 0x66;                                           // Level[6] = 13/16 of PWM
                break;

    case 0x07 : bright_code = 0x77;                                           // Level[7] = 14/16 of PWM
                break;

    default   : bright_code = 0x77;                                           // Level[7] = 14/16 of PWM
  } 
  
  //Update All Digit Brightness Level
  stled316s_send_byte(WRITE_SEG_BRIGHTNESS|DIG_SEG2_SEG1,bright_code);        // Setup Segment Brightness Level Digit[2],Digit[1]  
  stled316s_send_byte(WRITE_SEG_BRIGHTNESS|DIG_SEG4_SEG3,bright_code);        // Setup Segment Brightness Level Digit[4],Digit[3] 
  stled316s_send_byte(WRITE_SEG_BRIGHTNESS|DIG_SEG6_SEG5,bright_code);        // Setup Segment Brightness Level Digit[6],Digit[5]  
  
  //8 Bit LED = Segment Digit[0] of ET-SPI DSP7SEG
  stled316s_send_byte(WRITE_LED_BRIGHTNESS|DIG_LED2_LED1,bright_code);        // Setup LED(Bit) Brightness Level LED[2],LED[1]   
  stled316s_send_byte(WRITE_LED_BRIGHTNESS|DIG_LED4_LED3,bright_code);        // Setup LED(Bit) Brightness Level LED[4],LED[3] 
  stled316s_send_byte(WRITE_LED_BRIGHTNESS|DIG_LED6_LED5,bright_code);        // Setup LED(Bit) Brightness Level LED[6],LED[5]    
  stled316s_send_byte(WRITE_LED_BRIGHTNESS|DIG_LED8_LED7,bright_code);        // Setup LED(Bit) Brightness Level LED[8],LED[7]  
}

/*****************************/
/* Write STLED316S Byte Data */
/* 16 Bit SPI Shift Protocol */
/* - LSB First(1.00 MHz Max) */
/* - Shift on Rising Clock   */
/* - Fisrt Byte  = Command   */
/* - Second Byte = Data      */
/*****************************/
void stled316s_send_byte(unsigned char seg_command,unsigned char seg_data) 
{ 
  unsigned char Bit = 0;                                                      // Bit Counter
  
  //Start Command  
  stled316s_stb_clr();                                                        // Start Communication
  delay_us(1);

  //Send Command(LSB First)
  for (Bit = 0; Bit < 8; Bit++)                                               // 8 Bit Write
  {
    stled316s_clk_clr();                                                      // Standby SCLK   
    delay_us(1);
    
    //Generate Signal DIN Data
    if(seg_command&0x01)                                                      // LSB First
    {
     stled316s_din_set();  
    }
    else 
    {
     stled316s_din_clr();
    }

    //Shift Bit DIN Signal    
    delay_us(1);
    stled316s_clk_set();                                                      // Strobe Signal Bit(DIN)
    delay_us(1);
    seg_command >>= 1;                                                        // Next Bit Data    
  }   
  
  //Send Data(LSB First)
  for (Bit = 0; Bit < 8; Bit++)                                               // 8 Bit Write
  {
    stled316s_clk_clr();                                                      // Standby SCLK   
    delay_us(1);
    
    //Generate Signal DIN Data
    if(seg_data&0x01)                                                         // LSB First
    {
     stled316s_din_set();  
    }
    else 
    {
     stled316s_din_clr();
    }
    
    //Shift Bit DIN Signal    
    delay_us(1);
    stled316s_clk_set();                                                      // Strobe Signal Bit(DIN)
    delay_us(1);
    seg_data >>= 1;                                                           // Next Bit Data    
  }   
  
  //Latch Command
  delay_us(1);
  stled316s_stb_set();                                                        // Stop Communication
}

