;/**************************************/;
;/* Example Program For "ET-DSP4"      */;
;/* Files Name  : DSP4_EX1.ASM         */;
;/* CPU Control : AT89C2051            */;
;/* Display Unit: Segment2.3"(74HC595) */;
;/* Assembler   : SXA51                */;
;/**************************************/;
;
;/** 74HC595 Display Control
SERIAL      EQU     P1.0                    ; Serial Input pin
LATCH       EQU     P1.1                    ; Latch Clock pin
SHIFT       EQU     P1.2                    ; Shift Clock pin
;
;/* Port For Control 93X46 */;
EEP_DIN     EQU     P1.3                    ; Data Input of EEPROM
EEP_DOUT    EQU     ACC.3                   ; Data Output of EEPROM
EEP_CLOCK   EQU     P1.4                    ; Serial clock
EEP_CS      EQU     P1.5                    ; Chip select of EEPROM
PORT_EEP    EQU     P1                      ; Port Data I/O EEPROM
;
;/** RTC & WATCH-DOG
RTC_CLOCK   EQU     P1.4                    ; Serial Clock of RTC-DS1202
RTC_RESET   EQU     P1.5                    ; Reset RTC-DS1202
RTC_DATA    EQU     P1.6                    ; Data I/O of RTC-DS1202
RTC_READ    EQU     ACC.6                   ; Data Output From RTC
WATCH_DOG   EQU     P1.7                    ; Reset Watchdog Signal
DIR_DATA    EQU     P3.7                    ; RS485 Direction : 0 = RX,1=TX
;
;/************************/;
;/* RTC Register Address */;
;/************************/;
;
SECOND      EQU     00H                     ; Second 00-59
MINUTE      EQU     01H                     ; Minute 00-59
HOUR        EQU     02H                     ; Hour 01-12 or 00-23
DATE        EQU     03H                     ; Date 01-28/29 or 01-30 or 01-31
MONTH       EQU     04H                     ; Month 01-12
DAY         EQU     05H                     ; Day 01-07
YEAR        EQU     06H                     ; Year 00-99
CTRL_RTC    EQU     07H                     ; Write Protect Enable/Disable
CLK_BRST    EQU     1FH                     ; RTC Burst Mode
;
;/* Data Buffer
ORG         20H
FLAG_BUF:   DS      1                       ; Flag Status
DSP_BUFF:   DS      4                       ; Display Buffer (Decimal)
SEC_CNT:    DS      1                       ; Time (1-Second) Buffer Count
ADDRESS:    DS      1
DATA:       DS      1
COUNT:      DS      1
LINE:       DS      1
EEP_BUFF:   DS      2
STACK:      DS      32                      ; Stack System
;
KEY_PRES    EQU     FLAG_BUF.0              ; Scann Key
KEY_STAS    EQU     FLAG_BUF.1
STAS_DSP    EQU     FLAG_BUF.2              ; Status Display
UPDATE      EQU     FLAG_BUF.3
DOT         EQU     FLAG_BUF.4

            ORG     0000H                   ; Reset Vector of CPU
            LJMP    MAIN
            ;
            ORG     000BH                   ; Timer0 Vector Interupt
            LJMP    TIMER_INT
            ;
MAIN:       MOV     SP,#STACK               ; Stack
            MOV     P1,#0                   ; Clear Port
            ;
INIT_SER:   MOV     A,#0FDH                 ; Set baud rate 9600
            MOV     TH1,A
            MOV     TL1,A
            MOV     TMOD,#00100001B         ; Timer1 Mode2 & Timer0 Mode1
            MOV     SCON,#01010000B         ; Serial mode 1
            CLR     ES                      ; Disable serial interupt
            CLR     ET1                     ; Disable timer1 interupt
            SETB    TR1                     ; Set timer1 control
            CLR     EA                      ; Disable Global Interupt
            ;
            CLR     DOT
            MOV     DSP_BUFF+0,#0
            MOV     DSP_BUFF+1,#1
            MOV     DSP_BUFF+2,#2
            MOV     DSP_BUFF+3,#3
            ACALL   DISPLAY                 ; Display 4-Digit
            ;
MENUTEST:   LCALL   PRINT_SER
            DB      0CH,0DH,0AH
            DB      'Test Hardware  ET-DSP4',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      '[1]. Test Serial Port RS232 & RS422',0DH,0AH
            DB      '[2]. Test RTC DS1202',0DH,0AH
            DB      '[3]. Test EEPROM 93C46',0DH,0AH
            DB      '[4]. Test Display Segment',0DH,0AH
            DB      'Select Test ...',00H
            ;
TEST00:     LCALL   RST_WATCH
            JNB     RI,TEST00               ; Wait Data From Com Port
            MOV     A,SBUF                  ; Get Data From Com Port
            CLR     RI
TEST01:     CJNE    A,#'1',TEST02
            LJMP    TEST_232
            ;
TEST02:     CJNE    A,#'2',TEST03
            LJMP    TEST_RTC
            ;
TEST03:     CJNE    A,#'3',TEST04
            LJMP    TEST_EEP
            ;
TEST04:     CJNE    A,#'4',TEST00
            CLR     DOT
            ;
TEST_DSP:   MOV     COUNT,#17
            MOV     DATA,#0
LOOP_DSP:   MOV     DSP_BUFF+0,DATA
            MOV     DSP_BUFF+1,DATA
            MOV     DSP_BUFF+2,DATA
            MOV     DSP_BUFF+3,DATA
            LCALL   DISPLAY
            ;
            MOV     R1,#5
DSP_DLY0:   MOV     R2,#0
DSP_DLY1:   MOV     R3,#0
            DJNZ    R3,$
            LCALL   RST_WATCH               ; Reset Watch-Dog
            DJNZ    R2,DSP_DLY1
            DJNZ    R1,DSP_DLY0
            ;
            INC     DATA
            DJNZ    COUNT,LOOP_DSP
            SJMP    TEST_DSP

TEST_232:   LCALL   PRINT_SER
            DB      0CH,0DH,0AH
            DB      'Press Any Key For Test Serial Port',0DH,0AH,00H
            ;
LOOP_232:   LCALL   RST_WATCH
            JNB     RI,LOOP_232             ; Wait Data From Com Port
            MOV     A,SBUF                  ; Get Data From Com Port
            CLR     RI
            ;
            CJNE    A,#0DH,ECHO_232
            MOV     A,#0
            PUSH    ACC
            PUSH    ACC
            RET
            ;
ECHO_232:   MOV     SBUF,A                  ; Echo Data
            JNB     TI,$
            CLR     TI
            SJMP    LOOP_232

TEST_RTC:   LCALL   DISABLE                 ; Write Protect OFF
            MOV     A,#SECOND
            LCALL   READ_RTC
            CLR     ACC.7                   ; Disable Clock Halt Flag
            MOV     B,A
            MOV     A,#SECOND
            LCALL   WRITE_RTC
            LCALL   ENABLE                  ; Write Protect ON
            ;
            MOV     TMOD,#00000001B         ; Timer0 Mode1 (16Bit)
            MOV     SEC_CNT,#50             ; 0.5-Second Counter (10mS x 50)
            MOV     TH0,#0DBH               ; 10 mSec Period
            MOV     TL0,#0FFH
            SETB    ET0                     ; Enable Timer0 Interupt
            SETB    TR0                     ; Start Timer Count
            SETB    EA                      ; Enable Global Interupt
            ;
LOOP_RTC:   LCALL   RST_WATCH
            JB      UPDATE,DSP_TIME
            SJMP    LOOP_RTC
            ;
DSP_TIME:   MOV     A,#MINUTE               ; Get Minute Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+0,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+1,A
            ;
            JNB     STAS_DSP,DSP_TIM1
            SETB    DOT                     ; "."
            SJMP    DSP_TIM2
DSP_TIM1:   CLR     DOT                     ; Blank
            ;
DSP_TIM2:   MOV     A,#SECOND               ; Get Second Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+2,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+3,A
            LCALL   DISPLAY
            CLR     UPDATE
            SJMP    LOOP_RTC


TEST_EEP:   SETB    TR0
            MOV     P1,#0FFH
            CLR     EEP_CS                  ; CS Active = "1"
            CLR     EEP_CLOCK               ; 1st Clock = Rising Edge
            LCALL   READ_WORD
            ;
            LCALL   PRINT_SER
            DB      0CH,0DH,0AH,0DH,0AH
            DB      'Hardware  ET-DSP4',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      'Test Serial eeprom 93xx46 (128x8)'
            DB      0DH,0AH,0DH,0AH,00H
            ;
            MOV     EEP_BUFF+0,TH0
            MOV     EEP_BUFF+1,TL0
            ;
            LCALL   PRINT_SER
            DB      'Write Address 00H = ',00H
            MOV     A,EEP_BUFF+0
            LCALL   TX_2ASC
            MOV     B,EEP_BUFF+0            ; Data To Write eeprom
            MOV     A,#00H                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read Address 00H = ',00H
            MOV     A,#00H                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            ;
            LCALL   PRINT_SER
            DB      0DH,0AH,0DH,0AH
            DB      'Write Address 7FH = ',00H
            MOV     A,EEP_BUFF+1
            LCALL   TX_2ASC
            MOV     B,EEP_BUFF+1            ; Data To Write eeprom
            MOV     A,#7FH                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read eeprom Address 7FH = ',00H
            MOV     A,#7FH                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            MOV     A,#0DH
            LCALL   TX_BYTE
            MOV     A,#0AH
            LCALL   TX_BYTE
            ;
            MOV     A,#0
            PUSH    ACC
            PUSH    ACC
            RET

;/***************************/;
;/* Display Segment 4-Digit */;
;/* Input = DSP_BUFF 4-Byte */;
;/***************************/;
;
DISPLAY:    MOV     A,DSP_BUFF+0            ; Display : 1000
            MOV     DPTR,#TAB_SEG           ; Point to Table Display
            MOVC    A,@A+DPTR               ; Convert to 7-Segment Code
            ACALL   SER_OUT                 ; Shift Data to 74HC595
            ;
            MOV     A,DSP_BUFF+1            ; Display : 100
            MOV     DPTR,#TAB_SEG           ; Point to Table Display
            MOVC    A,@A+DPTR               ; Convert to 7-Segment Code
            JNB     DOT,NOT_DOT
            SETB    ACC.0
NOT_DOT:    ACALL   SER_OUT                 ; Shift Data to 74HC595
            ;
            MOV     A,DSP_BUFF+2            ; Display : 10
            MOV     DPTR,#TAB_SEG           ; Point to Table Display
            MOVC    A,@A+DPTR               ; Convert to 7-Segment Code
            ACALL   SER_OUT                 ; Shift Data to 74HC595
            ;
            MOV     A,DSP_BUFF+3            ; Display : 1
            MOV     DPTR,#TAB_SEG           ; Point to Table Display
            MOVC    A,@A+DPTR               ; Convert to 7-Segment Code
            ACALL   SER_OUT                 ; Shift Data to 74HC595
            ;
            SETB    LATCH                   ; Latch Data at Rising Edge
            CLR     LATCH
            RET

;/*************************/;
;/* Shift Data to 74HC595 */;
;/* Input : ACC           */;
;/*************************/;
;
SER_OUT:    ACALL   RST_WATCH               ; Restart Watch-Dog
            MOV     R2,#8                   ; Counter data ShiftOut 1 Byte
            CLR     CY
SHF_OUT:    RLC     A                       ; Shift Out data D0..D7
            MOV     SERIAL,C
            SETB    SHIFT                   ; Shift Clock at Rising Edge
            CLR     SHIFT
            DJNZ    R2,SHF_OUT              ; Loop Send This Byte
            RET

;/*********************/;
;/*  Table 7-Segment  */;
;/*  Code Number 0-9  */;
;/*  1 = ON ,0 = OFF  */;
;/* MSB..........LSB  */;
;/* A B C D E F G DP  */;
;/**********************;
;
TAB_SEG:    DB      0FCH,060H,0DAH          ; 0 1 2
            DB      0F2H,066H,0B6H          ; 3 4 5
            DB      0BEH,0E0H,0FEH          ; 6 7 8
            DB      0F6H,0EEH,03EH          ; 9 A B
            DB      09CH,07AH,09EH          ; C D E
            DB      08EH,001H,000H          ; F Dp Blank

;/****************************/;
;/* Read Ram From RTC-DS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RAM:   ACALL   RST_WATCH               ; Restart Watch-Dog
            ACALL   START_RST               ; Start Communication
            MOV     R2,#8
            RL      A                       ; Shift A0 to A1
            SETB    ACC.7                   ; Command Status
            SETB    ACC.6                   ; Ram Accress
            SETB    ACC.0                   ; Read Command
            CLR     CY
READ_RAM1:  CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C              ; Prepare Port
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,READ_RAM1            ; Repeat Send Read-Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RAM2:  SETB    RTC_DATA
            CLR     RTC_CLOCK               ; Read data D0..D7 (Data)
            MOV     A,P1
            MOV     C,RTC_READ              ; Read data out bit
            MOV     A,B
            RRC     A                       ; Read D0..D7
            MOV     B,A                     ; Backup Data
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,READ_RAM2
            MOV     A,B                     ; Data Output 1-Byte
            ACALL   STOP_RST                ; Disable Communication
            RET

;/***************************/;
;/* Write Ram to RTC-DS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RAM:  ACALL   RST_WATCH               ; Restart Watch-Dog
            ACALL   START_RST               ; Enable Communication
            MOV     R2,#8
            RL      A                       ; Shift A0 to A1
            SETB    ACC.7                   ; Command Status
            SETB    ACC.6                   ; Ram Accress
            CLR     ACC.0                   ; Write Command
            CLR     CY
WRITE_RAM1: CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Send D0..D7 (Write-Command)
            MOV     RTC_DATA,C
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,WRITE_RAM1           ; Repeat write Command
            ;
            MOV     A,B                     ; Get Data For Send
            MOV     R2,#8
            CLR     CY
WRITE_RAM2: CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Write D0..D7
            MOV     RTC_DATA,C
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,WRITE_RAM2           ; Repeat write Data
            ACALL   STOP_RST                ; Disable Communication
            RET

;/****************************/;
;/* Read Data From RTCDS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RTC:   ACALL   RST_WATCH               ; Restart Watch-Dog
            ACALL   START_RST               ; Start Communication
            MOV     R2,#8
            RL      A                       ; Shift A0 to A1
            SETB    ACC.7                   ; Command Status
            CLR     ACC.6                   ; Ram Accress
            SETB    ACC.0                   ; Read Command
            CLR     CY
READ_RTC1:  CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C              ; Prepare Port
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,READ_RTC1            ; Repeat write Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RTC2:  SETB    RTC_DATA
            CLR     RTC_CLOCK               ; Read data D0..D7 (Data)
            MOV     A,P1
            MOV     C,RTC_READ              ; Read data out bit
            MOV     A,B
            RRC     A                       ; Read D0..D7
            MOV     B,A                     ; Backup Data
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,READ_RTC2
            MOV     A,B                     ; Data Output 1-Byte
            ACALL   STOP_RST                ; Disable Communication
            RET

;/***************************/;
;/* Write Data to RTCDS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RTC:  ACALL   RST_WATCH               ; Restart Watch-Dog
            ACALL   START_RST               ; Enable Communication
            MOV     R2,#8
            RL      A                       ; Shift A0 to A1
            SETB    ACC.7                   ; Command Status
            CLR     ACC.6                   ; Ram Accress
            CLR     ACC.0                   ; Write Command
            CLR     CY
WRITE_RTC1: CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Write D0..D7 (Command)
            MOV     RTC_DATA,C
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,WRITE_RTC1           ; Repeat write Command
            ;
            MOV     A,B                     ; Get Data
            MOV     R2,#8
            CLR     CY
WRITE_RTC2: CLR     RTC_CLOCK               ; Falling Clock
            RRC     A                       ; Write D0..D7
            MOV     RTC_DATA,C
            SETB    RTC_CLOCK               ; Rising Clock
            DJNZ    R2,WRITE_RTC2           ; Repeat write Data
            ACALL   STOP_RST                ; Disable Communication
            RET

;/*************************/;
;/* Write Protect Disable */;
;/* Now You Can Write RTC */;
;/*************************/;
;
DISABLE:    PUSH    ACC
            PUSH    B
            MOV     A,#CTRL_RTC
            MOV     B,#00H                  ; Disable Write-Protect
            ACALL   WRITE_RTC
            POP     B
            POP     ACC
            RET

;/************************/;
;/* Write Protect Enable */;
;/* You Can't Write RTC  */;
;/************************/;
;
ENABLE:     PUSH    ACC
            PUSH    B
            MOV     A,#CTRL_RTC
            MOV     B,#80H                  ; Enable Write-Protect
            ACALL   WRITE_RTC
            POP     B
            POP     ACC
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/* For Start Communicate */;
;/*************************/;
;
START_RST:  CLR     RTC_CLOCK               ; Start RTC-Clock
            SETB    RTC_RESET               ; Enable Communication
            NOP
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/*  For Stop Communicate */;
;/*************************/;
;
STOP_RST:   SETB    RTC_CLOCK               ; Stop RTC-Clock
            CLR     RTC_RESET               ; Disable Communication
            NOP
            RET

;/*************************/;
;/* Send 1-Byte to RS-232 */;
;/* Input   : ACC         */;
;/* Output  : Serial port */;
;/*************************/;
;
TX_BYTE:    ACALL   RST_WATCH               ; Restart Watch-Dog
            CLR     TI                      ; Makesure New Send
            MOV     SBUF,A
            JNB     TI,$                    ; Wait TX Data Ready
            CLR     TI
            RET

;/****************************/;
;/* Receive Data From RS-232 */;
;/* Input   :  Serial Port   */;
;/* Output  :  ACC           */;
;/****************************/;
;
RX_BYTE:    LCALL   RST_WATCH
            JNB     RI,RX_BYTE               ; Wait RX Data Ready
            CLR     RI
            MOV     A,SBUF                   ; Get RX Data
            RET

;/*********************/;
;/** Reset Watch-Dog **/;
;/*********************/;
;
RST_WATCH:  CLR     WATCH_DOG               ; Reset Watch-Dog
            SETB    WATCH_DOG               ; Enable Watch-Dog
            RET

;************************************
;*  Print Data data to Serial Port  *
;*  to display buffer.              *
;*  Usage    : LCALL PRINT_SER      *
;*           : DB   'xxxx',00       *
;*  Register : ACC                  *
;*  Note     : last byte must be 00 *
;************************************
;
PRINT_SER:  POP     DPH
            POP     DPL
PRINT1:     CLR     A
            MOVC    A,@A+DPTR
            CJNE    A,#00H,PRINT2
            SJMP    PRINT3
PRINT2:     LCALL   TX_BYTE
            INC     DPTR
            SJMP    PRINT1
PRINT3:     PUSH    DPL
            PUSH    DPH
            RET

;******************************
;*    Send 2 byte of ASCII    *
;*  Input : ACC (HEX)         *
;*  Example : A = A3H         *
;*          : send 41H,33H    *
;******************************
;
TX_2ASC:    LCALL   HEX_2ASC
            LCALL   TX_BYTE
            MOV     A,B
            LCALL   TX_BYTE
            RET

;******************************
;*   Convert HEX to ASCII     *
;* Input    :  A              *
;* Output   :  A (high-byte)  *
;*          :  B (low-byte)   *
;******************************
;
HEX_2ASC:   PUSH    ACC
            LCALL   HEX_TO_ASC
            MOV     B,A              ; B = Lo byte
            POP     ACC
            SWAP    A
            LCALL   HEX_TO_ASC
            RET

;********************************
;*  Convert hex (0-F) to ASCII  *
;*  Input   : ACC (HEX code)    *
;*  Output  : ACC (ASCII code)  *
;********************************
;
HEX_TO_ASC: PUSH    PSW
            CLR     CY
            ANL     A,#00001111B
            CJNE    A,#09H,HTOA1
            SETB    CY
HTOA1:      JNB     CY,HTOA2       ;> 0AH
            ORL     A,#30H         ;< 0AH
            SJMP    END_HTOA
HTOA2:      SUBB    A,#09H         ;> 0AH
            ORL     A,#40H
END_HTOA:   POP     PSW
            RET

;/***********************************/;
;/* Subroutine for EEPROM 93AA46    */;
;/* 8-Bits Organization (128x8)     */;
;/* Reference to Microchips Data Ic */;
;/* EWEN  : Erase/Write Enable      */;
;/* EWDS  : Erase/Write Disable     */;
;/* WRITE : Write Data 8-Bit 1-Byte */;
;/* ERASE : Erase Data 8-Bit 1-Byte */;
;/* READ_WORD  : Read 8-Bit 1-Byte  */;
;/* WRITE_WORD : Write 8-Bit 1-Byte */;
;/***********************************/;
;
;/**********************/;
;/*    EWEN Command    */;
;/* Erase/Write Enable */;
;/**********************/;
;
EWEN:       LCALL   RST_WATCH
            CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Enable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS                  ; Standby
            RET

;/***********************/;
;/*     EWDS Command    */;
;/* Erase/Write Disable */;
;/***********************/;
;
EWDS:       LCALL   RST_WATCH
            CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Disable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS
            RET

;/*********************************/;
;/* WRITE DATA TO EEPROM (93AA46) */;
;/* Input    : ACC  = Address     */;
;/*          : B    = Data(8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
WRITE:      LCALL   RST_WATCH
            CLR     EEP_DIN
            SETB    EEP_CS                  ; Write command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
WREEP1:     RLC     A                       ; Write  address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP1               ; Repeat write address
            ;
            MOV     R2,#8                   ; Write data 8 bit in B
            MOV     A,B
WREEP2:     RLC     A                       ; Write data D7..D0
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP2
            ;
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCSL
            SETB    EEP_CS
WR_STAT:    LCALL   CLOCK                   ; wait ready state
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     WR_STAT                 ; Repeat if Busy
            CLR     EEP_CS                  ; OK ready So Standby
            RET

;/**************************/;
;/*        Erase byte      */;
;/* Input : ACC = Address  */;
;/* Reg.  : ACC,R2         */;
;/**************************/;
;
ERASE:      LCALL   RST_WATCH
            CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RAS1:       RLC     A                       ; Write Address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RAS1
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCLS
            SETB    EEP_CS
BUSY:       LCALL   CLOCK
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     BUSY                    ; Loop if Busy
            CLR     EEP_CS                  ; OK Ready So Standby
            RET

;/*********************************/;
;/* READ DATA FROM EEPROM(93AA46) */;
;/* Input    : ACC = Address      */;
;/* Output   : B   = Data (8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
READ_WORD:  LCALL   RST_WATCH
            CLR     EEP_DIN
            SETB    EEP_CS                  ; Read Command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RDEEP1:     RLC     A                       ; Write address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RDEEP1               ; Repeat write address
            CLR     EEP_DIN
            MOV     R1,#0
            MOV     R2,#8
RDEEP2:     LCALL   CLOCK                   ; Read data D0..D7
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT              ; Read data out bit
            MOV     A,R1
            RLC     A
            MOV     R1,A
            DJNZ    R2,RDEEP2
            MOV     B,R1                    ; 1st Byte
            MOV     DPH,#0                  ; Dont Care 2nd Byte
            CLR     EEP_CS
            RET

;/***************************/;
;/* Write 'WORD' To EEPROM  */;
;/* Input : B    = Data     */;
;/*       : ACC  = Address  */;
;/* Reg.  : ACC,B,R2        */;
;/***************************/;
;
WRITE_WORD: PUSH    ACC
            LCALL   EWEN                    ; Erase/Write Enable
            POP     ACC                     ; Get address
            PUSH    ACC                     ; Save address
            LCALL   ERASE                   ; Erase before write
            POP     ACC                     ; Get address
            LCALL   WRITE                   ; Write
            LCALL   EWDS                    ; Erase/Write Disable
            RET

;/******************/;
;/*  Serial clock  */;
;/******************/;
;
CLOCK:      LCALL   RST_WATCH
            SETB    EEP_CLOCK               ; SK = hi
            NOP
            NOP
            NOP
            NOP
            CLR     EEP_CLOCK               ; SK = low
            NOP
            NOP
            NOP
            NOP
            RET

;/******************************/;
;/* Timer 0 : 1 Second Service */;
;/*    Interupt Every 10 mSec  */;
;/* Auto Reload Every 1 Second */;
;/******************************/;
;
TIMER_INT:  MOV     TH0,#0DBH               ; 10 mSec Period Auto Reload
            MOV     TL0,#0FFH
            DJNZ    SEC_CNT+0,END_TIME      ; Check Time = 0.5 Second?
            MOV     SEC_CNT+0,#50           ; 10mSec x 50 = 0.5 Second
            CPL     STAS_DSP                ; Toggle Status Display
            SETB    UPDATE                  ; Update Time
END_TIME:   RETI

            END





























