;/***********************************/;
;/* Demo Software For Mini ET-REM8  */;
;/* Files Name  : DSP8TEST.ASM      */;
;/* Board & CPU : AT89C2051         */;
;/*  (Mini Remote Display 8-Digit)  */;
;/* Assembler   : SXA51             */;
;/* Function    : Test System Board */;
;/***********************************/;
;
;/* Port For Control MAX7219 */;
MAX_DATA    EQU     P1.4                    ; Data Input
MAX_CLOCK   EQU     P1.5                    ; Clock Input
MAX_LOAD    EQU     P1.6                    ; Load Input
;
;/* Port For Control DS1202 */;
RTC_RESET   EQU     P1.0                    ; RTC Reset/CS
RTC_DATA    EQU     P1.1                    ; RTC Data I/O
RTC_CLOCK   EQU     P1.2                    ; RTC Clock
RST_WATCH   EQU     P1.7                    ; Watchdog Reset
;
;/* Port For Control 93X46 */;
EEP_DIN     EQU     P1.1                    ; Data Input of EEPROM
EEP_DOUT    EQU     ACC.1                   ; Data Output of EEPROM
EEP_CLOCK   EQU     P1.2                    ; Serial clock
EEP_CS      EQU     P1.3                    ; Chip select of EEPROM
PORT_EEP    EQU     P1                      ; Port Data I/O EEPROM
;
SECOND      EQU     00H                     ; Second 00-59
MINUTE      EQU     01H                     ; Minute 00-59
HOUR        EQU     02H                     ; Hour 01-12 or 00-23
DATE        EQU     03H                     ; Data 01-28/29 or 01-30 or 01-31
MONTH       EQU     04H                     ; Month 01-12
DAY         EQU     05H                     ; Day 01-07
YEAR        EQU     06H                     ; Year 00-99
CTRL_RTC    EQU     07H                     ; Write Protect Enable/Disable

            ORG     20H                     ; Internal RAM of CPU
FLAG_BUF:   DS      1                       ; Flag Status
ROW:        DS      1
COLUM:      DS      1
DSP_BUFF:   DS      8
SEC_CNT:    DS      1                       ; Time (1-Second) Buffer Count
ADDRESS:    DS      1
DATA:       DS      1
COUNT:      DS      1
LINE:       DS      1
EEP_BUFF:   DS      2
STACK:      DS      32                      ; Stack Area
;
KEY_PRES    EQU     FLAG_BUF.0              ; Scann Key
KEY_STAS    EQU     FLAG_BUF.1
STAS_DSP    EQU     FLAG_BUF.2              ; Status Display
UPDATE      EQU     FLAG_BUF.3

            ORG     0000H                   ; Reset Vector of CPU
            LJMP    MAIN
            ;
            ORG     000BH                   ; Timer0 Vector Interupt
            LJMP    TIMER_INT
            ;
MAIN:       MOV     SP,#STACK               ; Initial Stack 32 Byte
            MOV     FLAG_BUF,#0
            MOV     P1,#0FFH
            SETB    RTC_CLOCK               ; Initial Port RTC-DS1202
            SETB    RTC_DATA
            CLR     RTC_RESET
            CLR     EEP_CS                  ; Disable EEPROM CS
            LCALL   INIT_MAX                ; Initial MAX7219
            ;
INIT_SER:   MOV     A,#0FDH                 ; Set baud rate 9600
            MOV     TH1,A
            MOV     TL1,A
            MOV     TMOD,#00100001B         ; Timer1 Mode2 & Timer0 Mode1
            MOV     SCON,#01010000B         ; Serial mode 1
            CLR     ES                      ; Disable serial interupt
            CLR     ET1                     ; Disable timer1 interupt
            SETB    TR1                     ; Set timer1 control
            CLR     EA                      ; Disable Global Interupt
            ;
CLR_BUFF:   MOV     R2,#8
            MOV     R0,#DSP_BUFF
            MOV     A,#12H                  ; Blank
CLR_BUF1:   MOV     @R0,A                   ; Clear Display Buffer
            INC     R0
            DJNZ    R2,CLR_BUF1
            MOV     DSP_BUFF+0,#0
            MOV     DSP_BUFF+1,#1
            MOV     DSP_BUFF+2,#2
            MOV     DSP_BUFF+3,#3
            MOV     DSP_BUFF+4,#4
            MOV     DSP_BUFF+5,#5
            MOV     DSP_BUFF+6,#6
            MOV     DSP_BUFF+7,#7
            LCALL   DISPLAY                 ; Display Data to 7-Segment
            ;
MENUTEST:   LCALL   PRINT_SER
            DB      0CH,0DH,0AH
            DB      'Test Hardware  ET-REM8',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      '[1]. Test Serial Port RS232 & RS422',0DH,0AH
            DB      '[2]. Test RTC DS1202',0DH,0AH
            DB      '[3]. Test EEPROM 93C46',0DH,0AH
            DB      '[4]. Test Display Segment',0DH,0AH
            DB      'Select Test ...',00H
            ;
TEST00:     JNB     RI,$                    ; Wait Data From Com Port
            MOV     A,SBUF                  ; Get Data From Com Port
            CLR     RI
TEST01:     CJNE    A,#'1',TEST02
            CLR     EA
            LJMP    TEST_232
            ;
TEST02:     CJNE    A,#'2',TEST03
            LJMP    TEST_RTC
            ;
TEST03:     CJNE    A,#'3',TEST04
            LJMP    TEST_EEP
            ;
TEST04:     CJNE    A,#'4',TEST00
            CLR     EA
            LJMP    TEST_DSP

TEST_232:   LCALL   PRINT_SER
            DB      0CH,0DH,0AH
            DB      'Press Any Key For Test Serial Port',0DH,0AH,00H
            ;
LOOP_232:   JNB     RI,$                    ; Wait Data From Com Port
            MOV     A,SBUF                  ; Get Data From Com Port
            CLR     RI
            ;
            CJNE    A,#0DH,ECHO_232
            MOV     A,#0
            PUSH    ACC
            PUSH    ACC
            RET
            ;
ECHO_232:   MOV     SBUF,A                  ; Echo Data
            JNB     TI,$
            CLR     TI
            SJMP    LOOP_232

TEST_EEP:   SETB    TR0
            MOV     P1,#0FFH
            CLR     RTC_RESET
            CLR     EEP_CS                  ; CS Active = "1"
            CLR     EEP_CLOCK               ; 1st Clock = Rising Edge
            LCALL   READ_WORD
            ;
            LCALL   PRINT_SER
            DB      0CH,0DH,0AH,0DH,0AH
            DB      'Hardware  ET-REM8',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      'Test Serial eeprom 93xx46 (128x8)'
            DB      0DH,0AH,0DH,0AH,00H
            ;
            MOV     EEP_BUFF+0,TH0
            MOV     EEP_BUFF+1,TL0
            ;
            LCALL   PRINT_SER
            DB      'Write Address 00H = ',00H
            MOV     A,EEP_BUFF+0
            LCALL   TX_2ASC
            MOV     B,EEP_BUFF+0            ; Data To Write eeprom
            MOV     A,#00H                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read Address 00H = ',00H
            MOV     A,#00H                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            ;
            LCALL   PRINT_SER
            DB      0DH,0AH,0DH,0AH
            DB      'Write Address 7FH = ',00H
            MOV     A,EEP_BUFF+1
            LCALL   TX_2ASC
            MOV     B,EEP_BUFF+1            ; Data To Write eeprom
            MOV     A,#7FH                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read eeprom Address 7FH = ',00H
            MOV     A,#7FH                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            MOV     A,#0DH
            LCALL   TX_BYTE
            MOV     A,#0AH
            LCALL   TX_BYTE
            ;
            MOV     A,#0
            PUSH    ACC
            PUSH    ACC
            RET

TEST_DSP:   MOV     COUNT,#17
            MOV     DATA,#0
LOOP_DSP:   MOV     DSP_BUFF+0,DATA
            MOV     DSP_BUFF+1,DATA
            MOV     DSP_BUFF+2,DATA
            MOV     DSP_BUFF+3,DATA
            MOV     DSP_BUFF+4,DATA
            MOV     DSP_BUFF+5,DATA
            MOV     DSP_BUFF+6,DATA
            MOV     DSP_BUFF+7,DATA
            LCALL   DISPLAY
            ;
            MOV     R1,#5
DSP_DLY0:   MOV     R2,#0
DSP_DLY1:   MOV     R3,#0
            DJNZ    R3,$
            DJNZ    R2,DSP_DLY1
            DJNZ    R1,DSP_DLY0
            ;
            INC     DATA
            DJNZ    COUNT,LOOP_DSP
            SJMP    TEST_DSP

TEST_RTC:   LCALL   DISABLE                 ; Write Protect OFF
            MOV     A,#SECOND
            LCALL   READ_RTC
            CLR     ACC.7                   ; Disable Clock Halt Flag
            MOV     B,A
            MOV     A,#SECOND
            LCALL   WRITE_RTC
            LCALL   ENABLE                  ; Write Protect ON
            ;
            MOV     TMOD,#00000001B         ; Timer0 Mode1 (16Bit)
            MOV     SEC_CNT,#50             ; 0.5-Second Counter (10mS x 50)
            MOV     TH0,#0DBH               ; 10 mSec Period
            MOV     TL0,#0FFH
            SETB    ET0                     ; Enable Timer0 Interupt
            SETB    TR0                     ; Start Timer Count
            SETB    EA                      ; Enable Global Interupt
            ;
LOOP:       JB      UPDATE,DSP_TIME
            SJMP    LOOP
            ;
DSP_TIME:   MOV     A,#HOUR                 ; Get Hour Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+0,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+1,A
            ;
            JNB     STAS_DSP,DSP_TIM1
            MOV     DSP_BUFF+2,#11H         ; "-"
            SJMP    DSP_TIM2
DSP_TIM1:   MOV     DSP_BUFF+2,#12H         ; Blank
            ;
DSP_TIM2:   MOV     A,#MINUTE               ; Get Minute Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+3,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+4,A
            ;
            JNB     STAS_DSP,DSP_TIM3
            MOV     DSP_BUFF+5,#11H         ; "-"
            SJMP    DSP_TIM4
DSP_TIM3:   MOV     DSP_BUFF+5,#12H         ; Blank
            ;
DSP_TIM4:   MOV     A,#SECOND               ; Get Second Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+6,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+7,A
            LCALL   DISPLAY
            CLR     UPDATE
            SJMP    LOOP


;/********************/;
;/* Initial MAX7219  */;
;/********************/;
;
INIT_MAX:   MOV     DPTR,#0A0FH             ; Intensity (Max)
            LCALL   SHF_MAX
            MOV     DPTR,#0B07H             ; Scan Limit (8digit)
            LCALL   SHF_MAX
            MOV     DPTR,#0900H             ; Set Mode (No Decode BCD)
            LCALL   SHF_MAX
            MOV     DPTR,#0C01H             ; Not Shutdown (Normal Opr.)
            LCALL   SHF_MAX
            MOV     DPTR,#0F00H             ; Not Display test (Normal Opr.)
            LCALL   SHF_MAX
            RET

;/*************************/;
;/* Shift Data to MAX7219 */;
;/* Input : DPH = Address */;
;/*       : DPL = Data    */;
;/* Reg.  : DPTR,R2,ACC   */;
;/*************************/;
;
SHF_MAX:    MOV     R2,#8                   ; Counter Address out
            MOV     A,DPH
            CLR     CY
            CLR     MAX_CLOCK
            CLR     MAX_LOAD
SHF_MAX1:   RLC     A                       ; Shift Out D15..D8
            MOV     MAX_DATA,C
            SETB    MAX_CLOCK               ; Shift clock
            CLR     MAX_CLOCK
            DJNZ    R2,SHF_MAX1
            MOV     R2,#8                   ; Counter data out
            MOV     A,DPL
            CLR     CY
SHF_MAX2:   RLC     A                       ; Shift Out D7..D0
            MOV     MAX_DATA,C
            SETB    MAX_CLOCK               ; Shift Clock
            CLR     MAX_CLOCK
            DJNZ    R2,SHF_MAX2
            SETB    MAX_LOAD                ; Latch Data
            CLR     MAX_LOAD
            RET

;/*************************/;
;/* Table of Display Code */;
;/*    Dp A B C D E F G   */;
;/*    D7 ...........D0   */;
;/*************************/;
;
TAB_SEG:    DB      7EH,30H,6DH,79H         ; 0 1 2 3
            DB      33H,5BH,5FH,70H         ; 4 5 6 7
            DB      7FH,7BH,77H,1FH         ; 8 9 A B
            DB      4EH,3DH,4FH,47H         ; C D E F
            DB      80H,01H,00H             ; - Blank

;/*****************************/;
;/* Display 7-Segment 8-Digit */;
;/* Input : DSP_BUFF (8-Byte) */;
;/*****************************/;
;
DISPLAY:    MOV     R7,#8
            MOV     R1,#08H                 ; Digit 1
            MOV     R0,#DSP_BUFF            ; Buffer digit1
DSP1:       MOV     A,@R0
            ANL     A,#01111111B            ; Ignore Dp
            MOV     DPTR,#TAB_SEG
            MOVC    A,@A+DPTR
            MOV     DPH,R1                  ; Address
            MOV     DPL,A                   ; Data
            MOV     A,@R0
            ANL     A,#10000000B            ; Check Dp
            ORL     A,DPL
            MOV     DPL,A
            LCALL   SHF_MAX
            INC     R0                      ; Next Buffer
            DEC     R1                      ; Next Digit
            DJNZ    R7,DSP1
            RET

;/****************************/;
;/* Read Ram From RTC-DS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RAM:   LCALL   START_RST           ; Start Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            SETB    ACC.6               ; Ram Accress
            SETB    ACC.0               ; Read Command
            CLR     CY
READ_RAM1:  LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C          ; Prepare Port
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RAM1        ; Repeat Send Read-Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RAM2:  SETB    RTC_DATA            ; Initial for Input
            NOP
            NOP
            LCALL   CLOCK_LO            ; Read data D0..D7 (Data)
            NOP                         ; Wait Data From RTC
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            MOV     C,RTC_DATA          ; Read data out bit
            MOV     A,B
            RRC     A                   ; Read D0..D7
            MOV     B,A                 ; Backup Data
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RAM2
            MOV     A,B                 ; Data Output 1-Byte
            LCALL   STOP_RST            ; Disable Communication
            RET

;/***************************/;
;/* Write Ram to RTC-DS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RAM:  LCALL   START_RST           ; Enable Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            SETB    ACC.6               ; Ram Accress
            CLR     ACC.0               ; Write Command
            CLR     CY
WRITE_RAM1: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Write-Command)
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RAM1       ; Repeat write Command
            ;
            MOV     A,B                 ; Get Data For Send
            MOV     R2,#8
            CLR     CY
WRITE_RAM2: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Write D0..D7
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI
            DJNZ    R2,WRITE_RAM2       ; Repeat write Data
            LCALL   STOP_RST            ; Disable Communication
            RET

;/****************************/;
;/* Read RTC From RTC-DS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RTC:   LCALL   START_RST           ; Start Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            CLR     ACC.6               ; Ram Accress
            SETB    ACC.0               ; Read Command
            CLR     CY
READ_RTC1:  LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C          ; Prepare Port
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Clock
            DJNZ    R2,READ_RTC1        ; Repeat write Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RTC2:  SETB    RTC_DATA            ; Initial for Input
            NOP
            NOP
            SETB    RTC_DATA            ; Initial for Input
            LCALL   CLOCK_LO            ; Read data D0..D7 (Data)
            NOP                         ; Wait Data From RTC
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            MOV     C,RTC_DATA          ; Read data out bit
            MOV     A,B
            RRC     A                   ; Read D0..D7
            MOV     B,A                 ; Backup Data
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RTC2
            MOV     A,B                 ; Data Output 1-Byte
            LCALL   STOP_RST            ; Disable Communication
            RET

;/***************************/;
;/* Write RTC to RTC-DS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RTC:  LCALL   START_RST           ; Enable Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            CLR     ACC.6               ; Ram Accress
            CLR     ACC.0               ; Write Command
            CLR     CY
WRITE_RTC1: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Write D0..D7 (Command)
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RTC1       ; Repeat write Command
            ;
            MOV     A,B                 ; Get Data
            MOV     R2,#8
            CLR     CY
WRITE_RTC2: LCALL   CLOCK_LO
            RRC     A                   ; Write D0..D7
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RTC2       ; Repeat write Data
            LCALL   STOP_RST            ; Disable Communication
            RET

;/*************************/;
;/* Write Protect Disable */;
;/* Now You Can Write RTC */;
;/* Reg.    : ACC,B,R2    */;
;/*************************/;
;
DISABLE:    MOV     A,#CTRL_RTC
            MOV     B,#00H              ; Disable Write-Protect
            LCALL   WRITE_RTC
            RET

;/************************/;
;/* Write Protect Enable */;
;/* You Can't Write RTC  */;
;/* Reg.    : ACC,B,R2   */;
;/************************/;
;
ENABLE:     MOV     A,#CTRL_RTC
            MOV     B,#80H              ; Enable Write-Protect
            LCALL   WRITE_RTC
            RET

;/*****************/;
;/*  RTC Clock    */;
;/*  Falling Edge */;
;/*****************/;
;
CLOCK_LO:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            CLR     RTC_CLOCK           ; Falling Clock
            NOP
            NOP
            NOP
            NOP
            RET

;/***************/;
;/* RTC Clock   */;
;/* Rising Edge */;
;/***************/;
;
CLOCK_HI:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            SETB    RTC_CLOCK           ; Rising Clock
            NOP
            NOP
            NOP
            NOP
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/* For Start Communicate */;
;/*************************/;
;
START_RST:  CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            CLR     RTC_RESET
            CLR     RTC_CLOCK
            NOP
            NOP
            NOP
            NOP
            SETB    RTC_RESET           ; Enable Communication
            NOP
            NOP
            NOP
            NOP
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/*  For Stop Communicate */;
;/*************************/;
;
STOP_RST:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            SETB    RTC_CLOCK
            NOP
            NOP
            NOP
            NOP
            CLR     RTC_RESET           ; Disable Communication
            NOP
            NOP
            NOP
            NOP
            RET

;/******************************/;
;/* Timer 0 : 1 Second Service */;
;/*    Interupt Every 10 mSec  */;
;/* Auto Reload Every 1 Second */;
;/******************************/;
;
TIMER_INT:  MOV     TH0,#0DBH               ; 10 mSec Period Auto Reload
            MOV     TL0,#0FFH
            DJNZ    SEC_CNT+0,END_TIME      ; Check Time = 0.5 Second?
            MOV     SEC_CNT+0,#50           ; 10mSec x 50 = 0.5 Second
            CPL     STAS_DSP                ; Toggle Status Display
            SETB    UPDATE                  ; Update Time
END_TIME:   RETI

;************************************
;*  Print Data data to Serial Port  *
;*  to display buffer.              *
;*  Usage    : LCALL PRINT_SER      *
;*           : DB   'xxxx',00       *
;*  Register : ACC                  *
;*  Note     : last byte must be 00 *
;************************************
;
PRINT_SER:  POP     DPH
            POP     DPL
PRINT1:     CLR     A
            MOVC    A,@A+DPTR
            CJNE    A,#00H,PRINT2
            SJMP    PRINT3
PRINT2:     LCALL   TX_BYTE
            INC     DPTR
            SJMP    PRINT1
PRINT3:     PUSH    DPL
            PUSH    DPH
            RET

;/********************************/;
;/*  Scan Keyboard Matrix  (4x4) */;
;/*  Output    : ACC (code 0..F) */;
;/*            : FFH (Not press) */;
;/*  Register  : ACC,B,R2,DPTR   */;
;/*  Sub. Call : KEY_DLY         */;
;/*            : KEY_NEW         */;
;/********************************/;
;
SCANKEY:    MOV     ROW,#4              ; Counter for Scan 4x4
            MOV     COLUM,#11111110B    ; Data for Scan Colum
SCAN1:      ORL     P1,#0FH             ; Set P1.0-P1.3
            MOV     A,COLUM             ; Port Scan Key
            RL      A                   ; Shift Left 2bit
            RL      A
            ANL     A,#00111100B
            ANL     P3,#11000011B
            ORL     P3,A                ; Scan Colum (P3.2-P3.5)
            MOV     A,P1
            ANL     A,#0FH              ; Check Row Only
            CJNE    A,#0FH,PRESS        ; Check row only (P1.0-P1.3)
            MOV     A,COLUM             ; Scan Next Colum
            RL      A
            MOV     COLUM,A
            DJNZ    ROW,SCAN1
            MOV     A,#0FFH             ; Not Keypress
            CLR     KEY_PRES            ; not key Press
            SJMP    END_SCAN
            ;
PRESS:      LCALL   KEY_DLY             ; Debounce key
            ORL     P1,#0FH             ; Set P1.0-P1.3
            MOV     A,COLUM             ; Port Scan Key
            RL      A                   ; Shift Left 2bit
            RL      A
            ANL     A,#00111100B
            ANL     P3,#11000011B
            ORL     P3,A                ; Scan Colum (P3.2-P3.5)
            MOV     A,P1
            ANL     A,#0FH              ; Check Row Only
            CJNE    A,#0FH,KEY_CODE     ; Check row only (P1.0-P1.3)
            SJMP    SCANKEY             ; No keybounce or noise
            ;
KEY_CODE:   LCALL   KEY_NEW
            JNB     KEY_STAS,SCAN1      ; if old key don't care
            SWAP    A
            MOV     B,A                 ; Row code (High)
            MOV     A,COLUM
            ANL     A,#0FH              ; Colum (Low Byte)
            ORL     A,B                 ; Mix Row & Colum
            MOV     B,A                 ; Row & Colum code
            ;
            MOV     DPTR,#KEY_TAB
            MOV     R2,#0FFH
CODE1:      INC     R2
            MOV     A,R2
            MOVC    A,@A+DPTR
            CJNE    A,B,CODE1
            MOV     A,R2                ; Number code 0..F
END_SCAN:   RET

;/**********************************/;
;/*     Check Old & New key        */;
;/*  KEY_STAS = 0 is old press Key */;
;/*  KEY_STAS = 1 is 1st press Key */;
;/**********************************/;
;
KEY_NEW:    CLR     KEY_STAS            ; Next time = old key
            JB      KEY_PRES,END_NEW    ; Jump if old key press
            SETB    KEY_PRES            ; Next time = old press
            SETB    KEY_STAS            ; Next time = old key
END_NEW:    RET

;/****************/;
;/*   Delay For  */;
;/* Debounce key */;
;/****************/;
;
KEY_DLY:    MOV     R6,#01FH
KEY_DL1:    MOV     R7,#0FFH
            DJNZ    R7,$
            DJNZ    R6,KEY_DL1
            RET

;/*********************/;
;/* Table of Key Code */;
;/* Col = P 3.2-3-4-5 */;
;/*       X7 XB XD XE */;
;/* Row = P 1.0-1-2-3 */;
;/*       7X BX DX EX */;
;/*********************/;
;
KEY_TAB:    DB      077H,07BH,07DH,07EH     ; Row0
            DB      0B7H,0BBH,0BDH,0BEH     ; Row1
            DB      0D7H,0DBH,0DDH,0DEH     ; Row2
            DB      0E7H,0EBH,0EDH,0EEH     ; Row3

;/*********************/;
;/* Shift Left Buffer */;
;/*   DSP_BUFF+0..7   */;
;/* MSB <- LSB <- ACC */;
;/*********************/;
;
SHF_BUFF:   PUSH    ACC
            MOV     R0,#DSP_BUFF+0
            MOV     R1,#DSP_BUFF+1
            MOV     R2,#7
SHF_BUF1:   MOV     A,@R1
            MOV     @R0,A
            INC     R0
            INC     R1
            DJNZ    R2,SHF_BUF1
            POP     ACC
            MOV     @R0,A
            RET

;/***********************************/;
;/* Subroutine for EEPROM 93AA46    */;
;/* 8-Bits Organization (128x8)     */;
;/* Reference to Microchips Data Ic */;
;/* EWEN  : Erase/Write Enable      */;
;/* EWDS  : Erase/Write Disable     */;
;/* WRITE : Write Data 8-Bit 1-Byte */;
;/* ERASE : Erase Data 8-Bit 1-Byte */;
;/* READ_WORD  : Read 8-Bit 1-Byte  */;
;/* WRITE_WORD : Write 8-Bit 1-Byte */;
;/***********************************/;
;
;/**********************/;
;/*    EWEN Command    */;
;/* Erase/Write Enable */;
;/**********************/;
;
EWEN:       CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Enable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS                  ; Standby
            RET

;/***********************/;
;/*     EWDS Command    */;
;/* Erase/Write Disable */;
;/***********************/;
;
EWDS:       CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Disable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS
            RET

;/*********************************/;
;/* WRITE DATA TO EEPROM (93AA46) */;
;/* Input    : ACC  = Address     */;
;/*          : B    = Data(8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
WRITE:      CLR     EEP_DIN
            SETB    EEP_CS                  ; Write command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
WREEP1:     RLC     A                       ; Write  address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP1               ; Repeat write address
            ;
            MOV     R2,#8                   ; Write data 8 bit in B
            MOV     A,B
WREEP2:     RLC     A                       ; Write data D7..D0
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP2
            ;
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCSL
            SETB    EEP_CS
WR_STAT:    LCALL   CLOCK                   ; wait ready state
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     WR_STAT                 ; Repeat if Busy
            CLR     EEP_CS                  ; OK ready So Standby
            RET

;/**************************/;
;/*        Erase byte      */;
;/* Input : ACC = Address  */;
;/* Reg.  : ACC,R2         */;
;/**************************/;
;
ERASE:      CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RAS1:       RLC     A                       ; Write Address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RAS1
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCLS
            SETB    EEP_CS
BUSY:       LCALL   CLOCK
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     BUSY                    ; Loop if Busy
            CLR     EEP_CS                  ; OK Ready So Standby
            RET

;/*********************************/;
;/* READ DATA FROM EEPROM(93AA46) */;
;/* Input    : ACC = Address      */;
;/* Output   : B   = Data (8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
READ_WORD:  CLR     EEP_DIN
            SETB    EEP_CS                  ; Read Command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RDEEP1:     RLC     A                       ; Write address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RDEEP1               ; Repeat write address
            CLR     EEP_DIN
            MOV     R1,#0
            MOV     R2,#8
RDEEP2:     LCALL   CLOCK                   ; Read data D0..D7
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT              ; Read data out bit
            MOV     A,R1
            RLC     A
            MOV     R1,A
            DJNZ    R2,RDEEP2
            MOV     B,R1                    ; 1st Byte
            MOV     DPH,#0                  ; Dont Care 2nd Byte
            CLR     EEP_CS
            RET

;/***************************/;
;/* Write 'WORD' To EEPROM  */;
;/* Input : B    = Data     */;
;/*       : ACC  = Address  */;
;/* Reg.  : ACC,B,R2        */;
;/***************************/;
;
WRITE_WORD: PUSH    ACC
            LCALL   EWEN                    ; Erase/Write Enable
            POP     ACC                     ; Get address
            PUSH    ACC                     ; Save address
            LCALL   ERASE                   ; Erase before write
            POP     ACC                     ; Get address
            LCALL   WRITE                   ; Write
            LCALL   EWDS                    ; Erase/Write Disable
            RET

;/******************/;
;/*  Serial clock  */;
;/******************/;
;
CLOCK:      SETB    EEP_CLOCK               ; SK = hi
            NOP
            NOP
            NOP
            NOP
            CLR     EEP_CLOCK               ; SK = low
            NOP
            NOP
            NOP
            NOP
            RET

;*************************
;* Send 1-Byte to RS-232 *
;* Input   : ACC         *
;* Output  : Serial port *
;*************************
;
TX_BYTE:    PUSH   IE
            CLR    TI
            MOV    SBUF,A
            JNB    TI,$
            CLR    TI
            POP    IE
            RET

;******************************
;*    Send 2 byte of ASCII    *
;*  Input : ACC (HEX)         *
;*  Example : A = A3H         *
;*          : send 41H,33H    *
;******************************
;
TX_2ASC:    LCALL   HEX_2ASC
            LCALL   TX_BYTE
            MOV     A,B
            LCALL   TX_BYTE
            RET

;******************************
;*   Convert HEX to ASCII     *
;* Input    :  A              *
;* Output   :  A (high-byte)  *
;*          :  B (low-byte)   *
;******************************
;
HEX_2ASC:   PUSH    ACC
            LCALL   HEX_TO_ASC
            MOV     B,A              ; B = Lo byte
            POP     ACC
            SWAP    A
            LCALL   HEX_TO_ASC
            RET

;********************************
;*  Convert hex (0-F) to ASCII  *
;*  Input   : ACC (HEX code)    *
;*  Output  : ACC (ASCII code)  *
;********************************
;
HEX_TO_ASC: PUSH    PSW
            CLR     CY
            ANL     A,#00001111B
            CJNE    A,#09H,HTOA1
            SETB    CY
HTOA1:      JNB     CY,HTOA2       ;> 0AH
            ORL     A,#30H         ;< 0AH
            SJMP    END_HTOA
HTOA2:      SUBB    A,#09H         ;> 0AH
            ORL     A,#40H
END_HTOA:   POP     PSW
            RET

            END










            






