;/**********************************/;
;/* Monitor For Display 14-Segment */;
;/* Files Name  : DSP14EX5.ASM     */;
;/* Board & CPU : AT89C51          */;
;/* (Display 14-Segment / 8-Digit) */;
;/* Assembler   : SXA51            */;
;/* Function    : System Board Test*/;
;/**********************************/;
;
;/* Port For Control Display */;
RST_WATCH   EQU     P1.0                    ; Reset Watchdog
DISPLAY1    EQU     P1.1                    ; Latch Display1
DISPLAY2    EQU     P1.2                    ; Latch Display2
GATE_DSP    EQU     P1.3                    ; Enable Display
;/* Port For Control 93LC56B */;
EEP_CS      EQU     P1.4                    ; Chip select of EEPROM
EEP_CLOCK   EQU     P1.5                    ; Serial clock
EEP_DIN     EQU     P1.6                    ; Data Input of EEPROM
EEP_DOUT    EQU     ACC.6                   ; Data Output of EEPROM
PORT_EEP    EQU     P1                      ; Port Data I/O EEPROM
;/* Port For Control DS1202 */;
RTC_CLOCK   EQU     P1.5
RTC_DATA    EQU     P1.6
RTC_RESET   EQU     P1.7
;
SPEAKER     EQU     P3.6                    ; Speaker
DIR_485     EQU     P3.7

            ORG     20H                     ; Internal RAM of CPU
FLAG_BUF:   DS      1                       ; Flag Status
SEC_CNT:    DS      1                       ; Time (1-Second) Buffer Count
DSP_BUFF:   DS      8                       ; Display Buffer
SCAN_CNT:   DS      1
SCAN_BUF:   DS      1
DIG_BUFF:   DS      1
MOVE_CNT:   DS      1
MOVE_INX:   DS      1
ADDRESS:    DS      1
DATA:       DS      1
COUNT:      DS      1
LINE:       DS      1
EEP_BUFF:   DS      2
STACK:      DS      32
;
STAS_DSP    EQU     FLAG_BUF.0              ; Status Display
UPDATE      EQU     FLAG_BUF.1
;
;/* Constant Value */;
SECOND      EQU     00H                     ; Second 00-59
MINUTE      EQU     01H                     ; Minute 00-59
HOUR        EQU     02H                     ; Hour 01-12 or 00-23
DATE        EQU     03H                     ; Data 01-28/29 or 01-30 or 01-31
MONTH       EQU     04H                     ; Month 01-12
DAY         EQU     05H                     ; Day 01-07
YEAR        EQU     06H                     ; Year 00-99
CTRL_RTC    EQU     07H                     ; Write Protect Enable/Disable
            
            ORG     0000H                   ; Reset Vector of CPU
            LJMP    MAIN
            ORG     000BH                   ; Timer0 Vector Interupt
            LJMP    SCAN_DSP
            ORG     001BH                   ; Timer1 Vector Interupt
            LJMP    TIMER_INT
            ;
MAIN:       MOV     SP,#STACK               ; Initial Stack 32 Byte
            MOV     FLAG_BUF,#0
            MOV     P0,#000H
            MOV     P1,#0FFH
            MOV     P2,#000H
            SETB    RTC_CLOCK               ; Initial Port RTC-DS1202
            SETB    RTC_DATA
            CLR     RTC_RESET
            CLR     EEP_CS                  ; Disable EEPROM CS
            ;
INIT_SER:   MOV     A,#0FDH                 ; Set Baud rate 9600
            MOV     TH1,A
            MOV     TL1,A
            MOV     TMOD,#00100001B         ; Timer1 Mode2 & Timer0 Mode1
            SETB    ET0                     ; Enable Timer0 Interupt
            CLR     ET1                     ; Disable Timer1 Interupt
            CLR     ES                      ; Disable Serial Interupt
            SETB    TR1                     ; Set timer1 control
            MOV     SCON,#01010000B         ; Serial Port Mode 1
            MOV     TH0,#0FCH               ; 1 milliSec Period For Timer0
            MOV     TL0,#066H
            MOV     SCAN_CNT,#08H           ; Scan Count = 8-Digit
            MOV     SCAN_BUF,#01H           ; Scan ULN2803 = Colum0
            MOV     DIG_BUFF,#DSP_BUFF      ; Display Buffer = DG-0
            SETB    TR0                     ; Strat Timer0
            SETB    EA                      ; Enable Global Interupt
            ;
            MOV     DSP_BUFF+0,#14          ; D
            MOV     DSP_BUFF+1,#29          ; S
            MOV     DSP_BUFF+2,#26          ; P
            MOV     DSP_BUFF+3,#1           ; 1
            MOV     DSP_BUFF+4,#4           ; 4
            MOV     DSP_BUFF+5,#29          ; S
            MOV     DSP_BUFF+6,#15          ; E
            MOV     DSP_BUFF+7,#17          ; G
            ;
MENUTEST:   LCALL   PRINT_SER
            DB      0CH,0DH,0AH
            DB      'Test Hardware DSP14SEG',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      '[1]. Test Serial Port RS232 & RS422',0DH,0AH
            DB      '[2]. Test RTC DS1202',0DH,0AH
            DB      '[3]. Test EEPROM 93C46',0DH,0AH
            DB      '[4]. Test Display Segment',0DH,0AH
            DB      'Select Test ...',00H
            ;
TEST00:     JNB     RI,$                    ; Wait Data From Com Port
            MOV     A,SBUF                  ; Get Data From Com Port
            CLR     RI
TEST01:     CJNE    A,#'1',TEST02
            LJMP    TEST_232
            ;
TEST02:     CJNE    A,#'2',TEST03
            LJMP    TEST_RTC
            ;
TEST03:     CJNE    A,#'3',TEST04
            LJMP    TEST_EEP
            ;
TEST04:     CJNE    A,#'4',TEST00
            LJMP    TEST_DSP

TEST_RTC:   MOV     TMOD,#00010001B         ; Timer0,1 Mode1 (16Bit)
            MOV     TH0,#0FCH               ; 1 milliSec Period For Timer0
            MOV     TL0,#066H
            MOV     SCAN_CNT,#08H           ; Scan Count = 8-Digit
            MOV     SCAN_BUF,#01H           ; Scan ULN2803 = Colum0
            MOV     DIG_BUFF,#DSP_BUFF      ; Display Buffer = DG-0
            SETB    ET0                     ; Enable Timer0 Interupt
            SETB    TR0                     ; Strat Timer0
            MOV     SEC_CNT,#50             ; 0.5-Second Counter (10mS x 50)
            MOV     TH1,#0DBH               ; 10 mSec Period
            MOV     TL1,#0FFH
            SETB    ET1                     ; Enable Timer1 Interupt
            SETB    TR1                     ; Start Timer1 Count
            SETB    EA                      ; Enable Global Interupt
            ;
            MOV     P1,#0FFH
            MOV     P2,#000H
            SETB    RTC_CLOCK               ; Initial Port RTC-DS1202
            SETB    RTC_DATA
            CLR     RTC_RESET
            CLR     EEP_CS                  ; Disable EEPROM CS
            MOV     A,#SECOND
            LCALL   READ_RTC
            ;
            LCALL   DISABLE                 ; Write Protect OFF
            MOV     A,#SECOND
            LCALL   READ_RTC
            CLR     ACC.7                   ; Disable Clock Halt Flag
            MOV     B,A
            MOV     A,#SECOND
            LCALL   WRITE_RTC
            ;
            MOV     A,#HOUR
            MOV     B,#00H
            LCALL   WRITE_RTC
            MOV     A,#MINUTE
            MOV     B,#00H
            LCALL   WRITE_RTC
            MOV     A,#SECOND
            MOV     B,#00H
            LCALL   WRITE_RTC
            ;
            LCALL   ENABLE                  ; Write Protect ON
            ;
LOOP:       JB      UPDATE,DSP_TIME
            SJMP    LOOP
            ;
DSP_TIME:   MOV     A,#HOUR                 ; Get Hour Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+0,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+1,A
            ;
            JNB     STAS_DSP,DSP_TIM1
            MOV     DSP_BUFF+2,#25H         ; "-"
            SJMP    DSP_TIM2
DSP_TIM1:   MOV     DSP_BUFF+2,#0AH         ; Blank
            ;
DSP_TIM2:   MOV     A,#MINUTE               ; Get Minute Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+3,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+4,A
            ;
            JNB     STAS_DSP,DSP_TIM3
            MOV     DSP_BUFF+5,#25H         ; "-"
            SJMP    DSP_TIM4
DSP_TIM3:   MOV     DSP_BUFF+5,#0AH         ; Blank
            ;
DSP_TIM4:   MOV     A,#SECOND               ; Get Second Time
            LCALL   READ_RTC
            PUSH    ACC
            SWAP    A
            ANL     A,#0FH
            MOV     DSP_BUFF+6,A
            POP     ACC
            ANL     A,#0FH
            MOV     DSP_BUFF+7,A
            CLR     UPDATE
            SJMP    LOOP

;/*****************************/;
;/* Timer0 : Service Routines */;
;/* Interupt Every 1 milliSec */;
;/* Scan Display on Interrupt */;
;/* Display 7-Segment 8-Digit */;
;/* Input : DSP_BUFF (8-Byte) */;
;/*       : DIG_BUFF (^DSP8)  */;
;/*       : SCAN_BUF (Colum)  */;
;/*       : SCAN_CNT (Count)  */;
;/*****************************/;
;
SCAN_DSP:   PUSH    PSW                     ; Save Register
            PUSH    DPH
            PUSH    DPL
            PUSH    ACC
            PUSH    B
            SETB    RS0                     ; Select Register Bank-1
            ;
            MOV     TH0,#0FCH               ; 1 milliSec Auto Reload
            MOV     TL0,#066H
            MOV     R0,DIG_BUFF             ; Buffer Display Code
            ;
SCAN_DS1:   SETB    GATE_DSP                ; Disable Data Display
            MOV     P2,#00H                 ; Disable Colum Display
            MOV     DPTR,#TAB_14SEG
            MOV     A,@R0
            MOV     B,#2                    ; 2 Byte / 1 Code
            MUL     AB
            MOV     B,A                     ; Save Pointer
            MOVC    A,@A+DPTR               ; Get High Byte From Table
            MOV     P0,A
            CLR     DISPLAY2                ; Latch High Byte Display
            SETB    DISPLAY2
            MOV     A,B                     ; Get Pointer
            INC     A                       ; Adjust to Low Byte
            MOVC    A,@A+DPTR               ; Get Low Byte From Table
            MOV     P0,A
            CLR     DISPLAY1                ; Latch Low Byte Display
            SETB    DISPLAY1
            MOV     P2,SCAN_BUF             ; Scan Display
            CLR     GATE_DSP                ; Enable Scan Display
            ;
            INC     DIG_BUFF                ; Next Digit Scan
            MOV     A,SCAN_BUF
            RL      A
            MOV     SCAN_BUF,A
            DJNZ    SCAN_CNT,SCAN_DS2
            MOV     SCAN_CNT,#08H           ; Scan Count = 8-Digit
            MOV     SCAN_BUF,#01H           ; Scan ULN2803 = Colum0
            MOV     DIG_BUFF,#DSP_BUFF      ; Display Buffer = DG-0
            ;
SCAN_DS2:   POP     B
            POP     ACC
            POP     DPL
            POP     DPH
            POP     PSW
            RETI

;/******************************/;
;/* Timer 1 0.5 Second Service */;
;/*    Interupt Every 10 mSec  */;
;/* Auto Reload Every0.5Second */;
;/******************************/;
;
TIMER_INT:  MOV     TH1,#0DBH               ; 10 mSec Period Auto Reload
            MOV     TL1,#0FFH
            DJNZ    SEC_CNT+0,END_TIME      ; Check Time = 0.5 Second?
            MOV     SEC_CNT+0,#50           ; 10mSec x 50 = 0.5 Second
            CPL     STAS_DSP                ; Toggle Status Display
            SETB    UPDATE                  ; Update Time
END_TIME:   RETI

;/*************************/;
;/*           A            /;
;/*        -------         /;
;/*      | I  J  K |       /;
;/*    F |  \ | /  | B     /;
;/*      |   ...   |       /;
;/*        -G- -H-         /;
;/*      |   ...   |       /;
;/*    E |  / | \  | C     /;
;/*      | L  M  N |       /;
;/*        -------   **    /;
;/*           D      Dp    /;
;/*                        /;
;/*************************/;
;/* Table of Display Code */;
;/* 1st Byte = High Byte  */;
;/* 2nd Byte = Low Byte   */;
;/* Hi = Dp x N M L K J I */;
;/* Lo =  H G F E D C B A */;
;/* Data D7 ...........D0 */;
;/* Data in This Table is */;
;/* Hi1,Lo1,Hi2,Lo2.....  */;
;/*************************/;
;
TAB_14SEG:  DB      00CH,03FH,000H,006H     ; 0     1
            DB      008H,08BH,000H,08FH     ; 2     3
            DB      000H,0E6H,020H,069H     ; 4     5
            DB      000H,0FDH,00CH,001H     ; 6     7
            DB      000H,0FFH,000H,0EFH     ; 8     9
            DB      000H,000H,000H,0F7H     ; Sp    A
            DB      012H,08FH,000H,039H     ; B     C
            DB      012H,00FH,000H,079H     ; D     E
            DB      000H,071H,000H,0BDH     ; F     G
            DB      000H,0F6H,012H,009H     ; H     I
            DB      000H,01EH,024H,070H     ; J     K
            DB      000H,038H,005H,036H     ; L     M
            DB      021H,036H,000H,03FH     ; N     O
            DB      000H,0F3H,020H,03FH     ; P     Q
            DB      020H,0F3H,000H,0EDH     ; R     S
            DB      012H,001H,000H,03EH     ; T     U
            DB      00CH,030H,028H,036H     ; V     W
            DB      02DH,000H,015H,000H     ; X     Y
            DB      00CH,009H,000H,0C0H     ; Z     -

;/****************************/;
;/* Read Ram From RTC-DS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RAM:   LCALL   START_RST           ; Start Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            SETB    ACC.6               ; Ram Accress
            SETB    ACC.0               ; Read Command
            CLR     CY
READ_RAM1:  LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C          ; Prepare Port
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RAM1        ; Repeat Send Read-Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RAM2:  SETB    RTC_DATA            ; Initial for Input
            NOP
            NOP
            LCALL   CLOCK_LO            ; Read data D0..D7 (Data)
            NOP                         ; Wait Data From RTC
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            MOV     C,RTC_DATA          ; Read data out bit
            MOV     A,B
            RRC     A                   ; Read D0..D7
            MOV     B,A                 ; Backup Data
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RAM2
            MOV     A,B                 ; Data Output 1-Byte
            LCALL   STOP_RST            ; Disable Communication
            RET

;/***************************/;
;/* Write Ram to RTC-DS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RAM:  LCALL   START_RST           ; Enable Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            SETB    ACC.6               ; Ram Accress
            CLR     ACC.0               ; Write Command
            CLR     CY
WRITE_RAM1: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Write-Command)
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RAM1       ; Repeat write Command
            ;
            MOV     A,B                 ; Get Data For Send
            MOV     R2,#8
            CLR     CY
WRITE_RAM2: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Write D0..D7
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI
            DJNZ    R2,WRITE_RAM2       ; Repeat write Data
            LCALL   STOP_RST            ; Disable Communication
            RET

;/****************************/;
;/* Read RTC From RTC-DS1202 */;
;/* Input    : ACC = Address */;
;/* Output   : ACC = Data    */;
;/* Reg.     : ACC,B,R2      */;
;/****************************/;
;
READ_RTC:   LCALL   START_RST           ; Start Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            CLR     ACC.6               ; Ram Accress
            SETB    ACC.0               ; Read Command
            CLR     CY
READ_RTC1:  LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Send D0..D7 (Read-Command)
            MOV     RTC_DATA,C          ; Prepare Port
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Clock
            DJNZ    R2,READ_RTC1        ; Repeat write Command
            ;
            MOV     B,#0
            MOV     R2,#8
READ_RTC2:  SETB    RTC_DATA            ; Initial for Input
            NOP
            NOP
            SETB    RTC_DATA            ; Initial for Input
            LCALL   CLOCK_LO            ; Read data D0..D7 (Data)
            NOP                         ; Wait Data From RTC
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            NOP
            MOV     C,RTC_DATA          ; Read data out bit
            MOV     A,B
            RRC     A                   ; Read D0..D7
            MOV     B,A                 ; Backup Data
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,READ_RTC2
            MOV     A,B                 ; Data Output 1-Byte
            LCALL   STOP_RST            ; Disable Communication
            RET

;/***************************/;
;/* Write RTC to RTC-DS1202 */;
;/* Input   : ACC = Address */;
;/*         : B = Data      */;
;/* Reg.    : ACC,B,R2      */;
;/***************************/;
;
WRITE_RTC:  LCALL   START_RST           ; Enable Communication
            MOV     R2,#8
            RL      A                   ; Shift A0 to A1
            SETB    ACC.7               ; Command Status
            CLR     ACC.6               ; Ram Accress
            CLR     ACC.0               ; Write Command
            CLR     CY
WRITE_RTC1: LCALL   CLOCK_LO            ; Falling Clock
            RRC     A                   ; Write D0..D7 (Command)
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RTC1       ; Repeat write Command
            ;
            MOV     A,B                 ; Get Data
            MOV     R2,#8
            CLR     CY
WRITE_RTC2: LCALL   CLOCK_LO
            RRC     A                   ; Write D0..D7
            MOV     RTC_DATA,C
            NOP                         ; Wait Bus Ready
            NOP
            NOP
            NOP
            LCALL   CLOCK_HI            ; Rising Clock
            DJNZ    R2,WRITE_RTC2       ; Repeat write Data
            LCALL   STOP_RST            ; Disable Communication
            RET

;/*************************/;
;/* Write Protect Disable */;
;/* Now You Can Write RTC */;
;/* Reg.    : ACC,B,R2    */;
;/*************************/;
;
DISABLE:    MOV     A,#CTRL_RTC
            MOV     B,#00H              ; Disable Write-Protect
            LCALL   WRITE_RTC
            RET

;/************************/;
;/* Write Protect Enable */;
;/* You Can't Write RTC  */;
;/* Reg.    : ACC,B,R2   */;
;/************************/;
;
ENABLE:     MOV     A,#CTRL_RTC
            MOV     B,#80H              ; Enable Write-Protect
            LCALL   WRITE_RTC
            RET

;/*****************/;
;/*  RTC Clock    */;
;/*  Falling Edge */;
;/*****************/;
;
CLOCK_LO:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            CLR     RTC_CLOCK           ; Falling Clock
            NOP
            NOP
            NOP
            NOP
            RET

;/***************/;
;/* RTC Clock   */;
;/* Rising Edge */;
;/***************/;
;
CLOCK_HI:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            SETB    RTC_CLOCK           ; Rising Clock
            NOP
            NOP
            NOP
            NOP
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/* For Start Communicate */;
;/*************************/;
;
START_RST:  CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            CLR     RTC_RESET
            CLR     RTC_CLOCK
            NOP
            NOP
            NOP
            NOP
            SETB    RTC_RESET           ; Enable Communication
            NOP
            NOP
            NOP
            NOP
            RET

;/*************************/;
;/* Initial Reset & Clock */;
;/*  For Stop Communicate */;
;/*************************/;
;
STOP_RST:   CLR     RST_WATCH           ; Reset Watch-Dog
            SETB    RST_WATCH           ; Enable Watch-Dog
            SETB    RTC_CLOCK
            NOP
            NOP
            NOP
            NOP
            CLR     RTC_RESET           ; Disable Communication
            NOP
            NOP
            NOP
            NOP
            RET

TEST_232:   MOV     SCAN_CNT,#08H           ; Scan Count = 8-Digit
            MOV     SCAN_BUF,#01H           ; Scan ULN2803 = Colum0
            MOV     DIG_BUFF,#DSP_BUFF      ; Display Buffer = DG-0
            SETB    TR0                     ; Strat Timer0
            SETB    EA                      ; Enable Global Interupt
            ;
            MOV     A,#'T'
            LCALL   CONV_DSP
            MOV     A,#'E'
            LCALL   CONV_DSP
            MOV     A,#'S'
            LCALL   CONV_DSP
            MOV     A,#'T'
            LCALL   CONV_DSP
            MOV     A,#' '
            LCALL   CONV_DSP
            MOV     A,#'2'
            LCALL   CONV_DSP
            MOV     A,#'3'
            LCALL   CONV_DSP
            MOV     A,#'2'
            LCALL   CONV_DSP
            ;
GET_BYTE:   JNB     RI,$                    ; Receive ASCII from RS232
            CLR     RI
            MOV     A,SBUF
            LCALL   TX_BYTE
            LCALL   CONV_DSP                ; Convert ASCII to DSP_BUFF
            SJMP    GET_BYTE

;/****************************/;
;/* Convert ASCII to Display */;
;/* Input : ACC (ASCII Code) */;
;/****************************/;
;
CONV_DSP:   CJNE    A,#20H,CHK_NUM0
            MOV     A,#0AH                  ; Display Space
            LCALL   SHF_BUFF
            SJMP    END_CONV
CHK_NUM0:   CJNE    A,#30H,CHK_NUM1
CHK_NUM1:   JC      END_CONV
            CJNE    A,#40H,CHK_NUM2
CHK_NUM2:   JNC     CHK_CHR1
            ANL     A,#0FH                  ; Convert 0-9 to Display
            LCALL   SHF_BUFF
            SJMP    END_CONV
CHK_CHR1:   CJNE    A,#'A',CHK_CHR2
CHK_CHR2:   JC      END_CONV
            CJNE    A,#'Z'+1,CHK_CHR3
CHK_CHR3:   JNC     CHK_CHR4
            CLR     CY
            SUBB    A,#36H                  ; Convert A-Z to Display
            LCALL   SHF_BUFF
            SJMP    END_CONV
CHK_CHR4:   CJNE    A,#'a',CHK_CHR5
CHK_CHR5:   JC      END_CONV
            CJNE    A,#'z'+1,CHK_CHR6
CHK_CHR6:   JNC     END_CONV
            CLR     CY
            SUBB    A,#56H                  ; Convert a-z to Display
            LCALL   SHF_BUFF
END_CONV:   RET

;/*********************/;
;/* Shift Left Buffer */;
;/*   DSP_BUFF+0..7   */;
;/* MSB <- LSB <- ACC */;
;/*********************/;
;
SHF_BUFF:   PUSH    ACC
            MOV     R0,#DSP_BUFF+0
            MOV     R1,#DSP_BUFF+1
            MOV     R2,#7
SHF_BUF1:   MOV     A,@R1
            MOV     @R0,A
            INC     R0
            INC     R1
            DJNZ    R2,SHF_BUF1
            POP     ACC
            MOV     @R0,A
            RET

TEST_DSP:   MOV     TH0,#0FCH               ; 1 milliSec Period For Timer0
            MOV     TL0,#066H
            MOV     SCAN_CNT,#08H           ; Scan Count = 8-Digit
            MOV     SCAN_BUF,#01H           ; Scan ULN2803 = Colum0
            MOV     DIG_BUFF,#DSP_BUFF      ; Display Buffer = DG-0
            SETB    TR0                     ; Strat Timer0
            SETB    EA                      ; Enable Global Interupt
            ;
            MOV     R2,#8
            MOV     R0,#DSP_BUFF
            MOV     A,#0AH
CLR_DSPY:   MOV     @R0,A
            INC     R0
            DJNZ    R2,CLR_DSPY
            MOV     MOVE_CNT,#39
            MOV     MOVE_INX,#00
            ;
MOVE_DSP:   MOV     DPTR,#TAB_MOVE
MOVEDSP1:   MOV     A,MOVE_INX
            MOVC    A,@A+DPTR
            LCALL   CONV_DSP
            INC     MOVE_INX                ; Move Index
            ;
            MOV     R2,#4
DSP_DLY1:   MOV     R3,#00
DSP_DLY2:   MOV     R4,#00
            DJNZ    R4,$
            DJNZ    R3,DSP_DLY2
            DJNZ    R2,DSP_DLY1
            ;
            DJNZ    MOVE_CNT,MOVEDSP1       ; Move Byte Count
            MOV     MOVE_CNT,#39
            MOV     MOVE_INX,#00
            SJMP    MOVE_DSP                ; Loop to Continue
            ;
TAB_MOVE:   DB      '01234567890 ABCDEFGHIJKLMNOPQRSTUVWXYZ'
            DB      '             '

;/*********************/;
;/** Reset Watch-Dog **/;
;/*********************/;
;
WATCHDOG:   CLR     RST_WATCH               ; Reset Watch-Dog
            SETB    RST_WATCH               ; Enable Watch-Dog
            RET

;************************************
;*  Print Data data to Serial Port  *
;*  to display buffer.              *
;*  Usage    : LCALL PRINT_SER      *
;*           : DB   'xxxx',00       *
;*  Register : ACC                  *
;*  Note     : last byte must be 00 *
;************************************
;
PRINT_SER:  POP     DPH
            POP     DPL
PRINT1:     CLR     A
            MOVC    A,@A+DPTR
            CJNE    A,#00H,PRINT2
            SJMP    PRINT3
PRINT2:     LCALL   TX_BYTE
            INC     DPTR
            SJMP    PRINT1
PRINT3:     PUSH    DPL
            PUSH    DPH
            RET

;/***********************************/;
;/* Subroutine for EEPROM 93AA46    */;
;/* 8-Bits Organization (128x8)     */;
;/* Reference to Microchips Data Ic */;
;/* EWEN  : Erase/Write Enable      */;
;/* EWDS  : Erase/Write Disable     */;
;/* WRITE : Write Data 8-Bit 1-Byte */;
;/* ERASE : Erase Data 8-Bit 1-Byte */;
;/* READ_WORD  : Read 8-Bit 1-Byte  */;
;/* WRITE_WORD : Write 8-Bit 1-Byte */;
;/***********************************/;
;
;/**********************/;
;/*    EWEN Command    */;
;/* Erase/Write Enable */;
;/**********************/;
;
EWEN:       CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Enable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS                  ; Standby
            RET

;/***********************/;
;/*     EWDS Command    */;
;/* Erase/Write Disable */;
;/***********************/;
;
EWDS:       CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase/Write Disable command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; 0
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            LCALL   CLOCK                   ; X(0)
            CLR     EEP_CS
            RET

;/*********************************/;
;/* WRITE DATA TO EEPROM (93AA46) */;
;/* Input    : ACC  = Address     */;
;/*          : B    = Data(8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
WRITE:      CLR     EEP_DIN
            SETB    EEP_CS                  ; Write command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
WREEP1:     RLC     A                       ; Write  address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP1               ; Repeat write address
            ;
            MOV     R2,#8                   ; Write data 8 bit in B
            MOV     A,B
WREEP2:     RLC     A                       ; Write data D7..D0
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,WREEP2
            ;
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCSL
            SETB    EEP_CS
WR_STAT:    LCALL   CLOCK                   ; wait ready state
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     WR_STAT                 ; Repeat if Busy
            CLR     EEP_CS                  ; OK ready So Standby
            RET

;/**************************/;
;/*        Erase byte      */;
;/* Input : ACC = Address  */;
;/* Reg.  : ACC,R2         */;
;/**************************/;
;
ERASE:      CLR     EEP_DIN
            SETB    EEP_CS                  ; Erase command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RAS1:       RLC     A                       ; Write Address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RAS1
            CLR     EEP_DIN
            CLR     EEP_CS
            LCALL   CLOCK                   ; TCLS
            SETB    EEP_CS
BUSY:       LCALL   CLOCK
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT
            JNC     BUSY                    ; Loop if Busy
            CLR     EEP_CS                  ; OK Ready So Standby
            RET

;/*********************************/;
;/* READ DATA FROM EEPROM(93AA46) */;
;/* Input    : ACC = Address      */;
;/* Output   : B   = Data (8-Bit) */;
;/* Reg.     : ACC,B,R2           */;
;/*********************************/;
;
READ_WORD:  CLR     EEP_DIN
            SETB    EEP_CS                  ; Read Command
            SETB    EEP_DIN
            LCALL   CLOCK                   ; 1
            LCALL   CLOCK                   ; 1
            CLR     EEP_DIN
            LCALL   CLOCK                   ; 0
            RL      A                       ; Shift A6 to A7
            MOV     R2,#7
            CLR     CY
RDEEP1:     RLC     A                       ; Write address 7 bit
            MOV     EEP_DIN,C
            LCALL   CLOCK
            DJNZ    R2,RDEEP1               ; Repeat write address
            CLR     EEP_DIN
            MOV     R1,#0
            MOV     R2,#8
RDEEP2:     LCALL   CLOCK                   ; Read data D0..D7
            SETB    EEP_DIN
            MOV     A,PORT_EEP
            MOV     C,EEP_DOUT              ; Read data out bit
            MOV     A,R1
            RLC     A
            MOV     R1,A
            DJNZ    R2,RDEEP2
            MOV     B,R1                    ; 1st Byte
            MOV     DPH,#0                  ; Dont Care 2nd Byte
            CLR     EEP_CS
            RET

;/***************************/;
;/* Write 'WORD' To EEPROM  */;
;/* Input : B    = Data     */;
;/*       : ACC  = Address  */;
;/* Reg.  : ACC,B,R2        */;
;/***************************/;
;
WRITE_WORD: PUSH    ACC
            LCALL   EWEN                    ; Erase/Write Enable
            POP     ACC                     ; Get address
            PUSH    ACC                     ; Save address
            LCALL   ERASE                   ; Erase before write
            POP     ACC                     ; Get address
            LCALL   WRITE                   ; Write
            LCALL   EWDS                    ; Erase/Write Disable
            RET

;/******************/;
;/*  Serial clock  */;
;/******************/;
;
CLOCK:      SETB    EEP_CLOCK               ; SK = hi
            NOP
            NOP
            NOP
            NOP
            CLR     EEP_CLOCK               ; SK = low
            NOP
            NOP
            NOP
            NOP
            RET

;*************************
;* Send 1-Byte to RS-232 *
;* Input   : ACC         *
;* Output  : Serial port *
;*************************
;
TX_BYTE:    PUSH   IE
            CLR    TI
            MOV    SBUF,A
            JNB    TI,$
            CLR    TI
            POP    IE
            RET

;******************************
;*    Send 2 byte of ASCII    *
;*  Input : ACC (HEX)         *
;*  Example : A = A3H         *
;*          : send 41H,33H    *
;******************************
;
TX_2ASC:    LCALL   HEX_2ASC
            LCALL   TX_BYTE
            MOV     A,B
            LCALL   TX_BYTE
            RET

;******************************
;*   Convert HEX to ASCII     *
;* Input    :  A              *
;* Output   :  A (high-byte)  *
;*          :  B (low-byte)   *
;******************************
;
HEX_2ASC:   PUSH    ACC
            LCALL   HEX_TO_ASC
            MOV     B,A              ; B = Lo byte
            POP     ACC
            SWAP    A
            LCALL   HEX_TO_ASC
            RET

;********************************
;*  Convert hex (0-F) to ASCII  *
;*  Input   : ACC (HEX code)    *
;*  Output  : ACC (ASCII code)  *
;********************************
;
HEX_TO_ASC: PUSH    PSW
            CLR     CY
            ANL     A,#00001111B
            CJNE    A,#09H,HTOA1
            SETB    CY
HTOA1:      JNB     CY,HTOA2       ;> 0AH
            ORL     A,#30H         ;< 0AH
            SJMP    END_HTOA
HTOA2:      SUBB    A,#09H         ;> 0AH
            ORL     A,#40H
END_HTOA:   POP     PSW
            RET

TEST_EEP:   MOV     A,#'T'
            LCALL   CONV_DSP
            MOV     A,#'E'
            LCALL   CONV_DSP
            MOV     A,#'S'
            LCALL   CONV_DSP
            MOV     A,#'T'
            LCALL   CONV_DSP
            MOV     A,#' '
            LCALL   CONV_DSP
            MOV     A,#'E'
            LCALL   CONV_DSP
            MOV     A,#'E'
            LCALL   CONV_DSP
            MOV     A,#'P'
            LCALL   CONV_DSP
            ;
            MOV     P1,#0FFH
            CLR     RTC_RESET
            CLR     EEP_CS                  ; CS Active = "1"
            CLR     EEP_CLOCK               ; 1st Clock = Rising Edge
            LCALL   READ_WORD
            ;
            LCALL   PRINT_SER
            DB      0CH,0DH,0AH,0DH,0AH
            DB      'Hardware  DSP14SEG',0DH,0AH
            DB      'ETT CO.,LTD',0DH,0AH
            DB      'Test Serial eeprom 93xx46 (128x8)',0DH,0AH
            DB      0DH,0AH,00H
            MOV     EEP_BUFF+0,SCAN_CNT
            MOV     EEP_BUFF+1,SCAN_BUF
            ;
            LCALL   PRINT_SER
            DB      'Write Address 00H = ',00H
            MOV     A,EEP_BUFF+0
            LCALL   TX_2ASC
            ;
            MOV     B,EEP_BUFF+0            ; Data To Write eeprom
            MOV     A,#00H                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read Address 00H = ',00H
            MOV     A,#00H                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            ;
            LCALL   PRINT_SER
            DB      0DH,0AH,0DH,0AH
            DB      'Write Address 7FH = ',00H
            MOV     A,EEP_BUFF+1
            LCALL   TX_2ASC
            ;
            MOV     B,EEP_BUFF+1            ; Data To Write eeprom
            MOV     A,#7FH                  ; Address EEprom
            LCALL   WRITE_WORD              ; Write Data to eeprom
            LCALL   PRINT_SER
            DB      0DH,0AH
            DB      'Read Address 7FH = ',00H
            MOV     A,#7FH                  ; Address to Read eeprom
            LCALL   READ_WORD               ; Read Data From eeprom
            MOV     A,B                     ; Data 1-Byte
            LCALL   TX_2ASC
            MOV     A,#0DH
            LCALL   TX_BYTE
            MOV     A,#0AH
            LCALL   TX_BYTE
            ;
            MOV     A,#0
            PUSH    ACC
            PUSH    ACC
            RET

            END


            

            
            
            
