' -----[ Title ]-----------------------------------------------------------
'
' File....... DS1620EZ.BAS
' Purpose.... Stamp <-> Dallas DS1620 Digital Thermometer Interface
' Author..... Jon Williams 
' Internet... jonwms@aol.com
' Started.... 1 April 1995 (no fooling!)
' Updated.... 4 April 1995


' -----[ Program Description ]---------------------------------------------
'
' This program provides an easy (EZ) interface between the Parallax BASIC
' Stamp and the Dallas DS1620 Digital Thermometer chip.
'
' The temperature input routine takes advantage of the Stamp's ability to
' overlay bit, byte and word variables making the access of temperature
' from the DS1620 very easy.  Subroutines handle sending commands and 
' getting data from the DS1620, as well as a detailed conversion routine
' that converts the 1/2 bit Celsius output of the DS1620 to Fahrenheit.
' 
' The DEBUG command is used to display the  temperature in this demo.  In
' actual practice, the SEROUT command would be more practical. Alternately,
' the temperature could be displayed directly on an LCD.  Note that nega-
' tive temperatures come from the DS1620 in a 2's compliment format and
' must be converted before display.
'
' Thanks to Richard Juengel for his DS1620 program.
' Comments and suggestions are welcomed.
'
'
' -------------------------------
' Notes on temperature conversion 
' -------------------------------
'
' Fahrenheit = Celsius * 1.8 + 32
'
' In the Stamp this equation can be evaluated as F = C * 9 / 5 + 32
'
' The problem is that this equation will not work directly with the 
' output of the DS1620 which is in 1/2C increments.  To make it work you
' must divide the DS1620 output by 2--which throws away 0.9F (nearly a
' whole degree!) of resolution.  Since each half degree change in Celsius
' is equal to 0.9 degrees Fahrenheit, we can keep the full resolution of
' the DS1620 by converting to Fahrenheit like this:
'
' F = C2 * 0.9 + 32 (Remember!... C2 is in 1/2 increments)
'
' In the Stamp we evaluate the expression like this: F = C2 * 9 / 10 + 32
'
' Now the problem is rounding... We can fix it (round up for values X.5
' or better) by modifying the equation like this:
'
' F = C2 * 9 + 325 / 10
'
' This is the same as F = Int(C2 * 0.9 + 32.5)
' 
'
' Connections for demo:
' --------------------------------
' Stamp.0 --> DS1620.3 (RST\)
' Stamp.1 --> DS1620.2 (CLK/CONV\)
' Stamp.2 <-> DS1620.1 (DQ)


' -----[ Revision History ]------------------------------------------------
'
' 4 April 1995 : Version 1 complete and working


' -----[ Constants ]-------------------------------------------------------
'

' DS1620 connections
'
SYMBOL  RST     = 0                     ' DS1620 Reset input
SYMBOL  CLK     = 1                     ' DS1620 Clock input
SYMBOL  DQPin   = 2                     ' DS1620 data I/O
SYMBOL  DQ      = Pin2  

' DS1620 commands
'
SYMBOL  TEMPR   = $AA                   ' Read temperature
SYMBOL  THW     = $01                   ' Write TH (high temp register)
SYMBOL  TLW     = $02                   ' Write TL (low temp register)
SYMBOL  THR     = $A1                   ' Read TH
SYMBOL  TLR     = $A2                   ' Read TL
SYMBOL  START   = $EE                   ' Start temperature conversion
SYMBOL  STOP    = $22                   ' Stop temperature conversion
SYMBOL  CFGW    = $0C                   ' Write configuration register
SYMBOL  CFGR    = $AC                   ' Read configuration register


' -----[ Variables ]-------------------------------------------------------
'
SYMBOL  cmdByte = B0                    ' command sent to DS1620
SYMBOL  loTmpC  = B0                    ' lo byte of temperature
SYMBOL  hiTmpC  = B1                    ' high byte (sign) of temperature
SYMBOL  tempC   = W0                    ' temperature (needs 9 bits)
SYMBOL  tempF   = W1                    ' temperature in Fahrenheit
SYMBOL  shift   = B4                    ' shift counter
SYMBOL  signC   = B5                    ' pos or neg indicator (C)
SYMBOL  signF   = B6                    ' pos or neg indicator (F)


' -----[ Initialization ]--------------------------------------------------
'
Init:   Dirs = %00000111                ' pins 0 - 2 are outputs
	Pins = %00000000                ' clear I/O pins

	HIGH RST                        ' alert the DS1620
	cmdByte = CFGW                  ' prepare to write config
	GOSUB CmdOut  
	cmdByte = %00000010             ' use with CPU; free run mode
	GOSUB CmdOut
	LOW RST
	PAUSE 10                        ' wait 10 ms for EEPROM write
	HIGH RST                        ' alert the DS1620
	cmdByte = START
	GOSUB CmdOut                    ' start continuous conversion
	LOW RST                         ' end DS1620 commo


' -----[ Main Code ]-------------------------------------------------------
'
Main:   HIGH RST                        ' alert the DS1620
	cmdByte = TEMPR                 ' prep for temperature read
	GOSUB CmdOut
	GOSUB TempIn                    ' get temp from the DS1620
	LOW RST                         ' end DS1620 commo

	' tempC = %111011110             ' negative test value (-17C/+1F)

	GOSUB ConvT                     ' do conversions
	IF signC = "-" THEN Show2
Show1:  DEBUG "Temperature: ", #tempC, "C / ",#tempF, "F", CR
	GOTO P2000
Show2:  IF signF <> "-" THEN Show3 
	DEBUG "Temperature: -", #tempC, "C / -",#tempF, "F", CR
	GOTO P2000
Show3:  DEBUG "Temperature: -", #tempC, "C / ",#tempF, "F", CR

P2000:  PAUSE 2000
	GOTO Main                       ' do it again


' -----[ Subroutines ]-----------------------------------------------------
'

' Send command to DS1620
' 8-bit command is sent LSB -> MSB
'
CmdOut: OUTPUT DQPin                    ' make DQ an output
	HIGH CLK
	FOR shift = 1 TO 8              ' shift eight bits
	  DQ = Bit0                     ' get LSB of cmdByte
	  PULSOUT CLK, 10               ' clock out the bit
	  cmdByte = cmdByte / 2         ' shift cmd for next bit
	NEXT shift
	RETURN


' Retrieve temperature from DS1620
' Data comes in LSB -> MSB in 9 bits
' - 1 bit for sign
' - 8 bits for temp (in 2's compliment format if negative)
'
TempIn: tempC = 0                       ' clear old reading
	INPUT DQPin                     ' make DQ an input
	HIGH CLK
	FOR shift = 1 TO 9
	  tempC = tempC / 2             ' shift temp bits
	  LOW CLK
	  Bit8 = DQ                     ' get data bit
	  HIGH CLK
	NEXT shift
	RETURN


' Convert temp reading to Fahrenheit
' - save sign
' - convert from 2's compliment if negative
' - round to nearest degree
' Round tempC to whole value
'
ConvT:  signC = " "                     ' assume positive
	IF hiTmpC = 0 THEN IsPos
	'
	' convert from negative
	'
	signC = "-"
	hiTmpC = 0                      ' clear negative flag
	loTmpC = loTmpC ^ $FF + 1       ' get 2's compliment
	IF loTmpC > 34 THEN Neg2        ' if < -17C then F will be negative
Neg1:   signF = " "
	tempF = tempC * 9               ' F value will be positive
	tempF = 325 - tempF / 10
	GOTO TFDone
Neg2:   signF = "-"
	tempF = tempC * 9 - 315 / 10    ' F value will be negative
	GOTO TFDone
IsPos:  tempF = tempC * 9 + 325 / 10    ' convert to F with rounding
TFDone: tempC = tempC / 2 + Bit0        ' make whole with rounding
	RETURN
                                                                                                                         