;/***************************************/;
;/* Example Form For : CP-JR08GP32 V2.0 */;
;/* Controller       : 68H(R)C908GP32   */;
;/* Assembler        : CASM08Z.EXE      */;
;/* File Name        : MAGCARD2.ASM     */;
;/* Write By         : Eakachai Makarn  */;
;/* File Update      : 15-Nov-2002      */;
;/***************************************/;
;/* Magnetic Card Reader  :  MCR-B02TTL */;
;/* Use Interrupt Read(Short Jumper INT)*/;
;/* -Display Data to RS232 (9600,N,8,1) */;
;/* -Display Data to Character LCD 40x2 */;
;/***************************************/;
;
;/***************************************/;
;/* Start of IO Register Address Equate */;
;/* For Processor : MC68H(R)C908GP32    */;
;/***************************************/;
PTA         EQU     $0000                   ; Port-A Data Register
PTB         EQU     $0001                   ; Port-B Data Register
PTC         EQU     $0002                   ; Port-C Data Register
PTD         EQU     $0003                   ; Port-D Data Regsiter
DDRA        EQU     $0004                   ; Port-A Direction Register
DDRB        EQU     $0005                   ; Port-B Direction Register
DDRC        EQU     $0006                   ; Port-C Direction Register
DDRD        EQU     $0007                   ; Port-D Direction Register
PTE         EQU     $0008                   ; Port-E Data Regsiter
DDRE        EQU     $000C                   ; Port-E Direction Register
PTAPUE      EQU     $000D                   ; Port-A Pull-Up Enable Register
PTCPUE      EQU     $000E                   ; Port-C Pull-Up Enable Register
PTDPUE      EQU     $000F                   ; Port-D Pull-Up Enable Register
;
;/* Bit of SPI Control Register
;/* SPCR Bit Address Equate
SPCR        EQU     $0010                   ; SPI Control Register
SPRIE       EQU     $07                     ; R/W
DMAS        EQU     $06                     ; R
SPMSTR      EQU     $05                     ; R/W
CPOL        EQU     $04                     ; R/W
CPHA        EQU     $03                     ; R/W
SPWOM       EQU     $02                     ; R/W
SPE         EQU     $01                     ; R/W
SPTIE       EQU     $00                     ; R/W
;
;/* Bit of SPI Status and Control
;/* SPSCR Bit Address Equate
SPSCR       EQU     $0011                   ; SPI Status and Control Register
SPRF        EQU     $07                     ; R
ERRIE       EQU     $06                     ; R/W
OVRF        EQU     $05                     ; R
MODF        EQU     $04                     ; R
SPTE        EQU     $03                     ; R
MODFEN      EQU     $02                     ; R/W
SPR1        EQU     $01                     ; R/W
SPR0        EQU     $00                     ; R/W
;
SPDR        EQU     $0012                   ; SPI Data Register
;
;/* Bit of SCI Control Register1
;/* SCC1 Bit Address Equate
SCC1        EQU     $0013                   ; SCI Control Register1
LOOPS       EQU     $07                     ; R/W
ENSCI       EQU     $06                     ; R/W
TXINV       EQU     $05                     ; R/W
M           EQU     $04                     ; R/W
WAKE        EQU     $03                     ; R/W
ILTY        EQU     $02                     ; R/W
PEN         EQU     $01                     ; R/W
PTY         EQU     $00                     ; R/W
;
;/* Bit of SCI Control Register2
;/* SCC2 Bit Address Equate
SCC2        EQU     $0014                   ; SCI Control Register2
SCTIE       EQU     $07                     ; R/W
TCIE        EQU     $06                     ; R/W
SCRIE       EQU     $05                     ; R/W
ILIE        EQU     $04                     ; R/W
TE          EQU     $03                     ; R/W
RE          EQU     $02                     ; R/W
RWU         EQU     $01                     ; R/W
SBK         EQU     $00                     ; R/W
;
;/* Bit of SCI Control Register3
;/* SCC3 Bit Address Equate
SCC3        EQU     $0015                   ; SCI Control Register3
R8          EQU     $07                     ; R
T8          EQU     $06                     ; R/W
DMARE       EQU     $05                     ; R/W
DMATE       EQU     $04                     ; R/W
ORIE        EQU     $03                     ; R/W
NEIE        EQU     $02                     ; R/W
FEIE        EQU     $01                     ; R/W
PEIE        EQU     $00                     ; R/W
;
;/* Bit Of SCI Status Register1
;/* SCS1 Bit Address Equate
SCS1        EQU     $0016                   ; SCI Status Register1
SCTE        EQU     $07                     ; R
TC          EQU     $06                     ; R
SCRF        EQU     $05                     ; R
IDLE        EQU     $04                     ; R
OR          EQU     $03                     ; R
NF          EQU     $02                     ; R
FE          EQU     $01                     ; R
PE          EQU     $00                     ; R
;
;/* Bit of SCI Status Register2
;/* SCS2 Bit Address Equate
SCS2        EQU     $0017                   ; SCI Status Register2
BKF         EQU     $01                     ; R
RPF         EQU     $00                     ; R
;
SCDR        EQU     $0018                   ; SCI Data Register
;
;/* Bit of SCI Baudrate Register
;/* SCBR Bit Address Equate
SCBR        EQU     $0019                   ; SCI Baudrate Register
SCP1        EQU     $05                     ; R/W
SCP0        EQU     $04                     ; R/W
SCR2        EQU     $02                     ; R/W
SCR1        EQU     $01                     ; R/W
SCR0        EQU     $00                     ; R/W
;
;/* Bit of Keyboard Status & Control
;/* INTKBSCR Bit Address Equate
INTKBSCR    EQU     $001A                   ; Keyboard Status and Control Register
KEYF        EQU     $03                     ; R
ACKK        EQU     $02                     ; W
IMASKK      EQU     $01                     ; R/W
MODEK       EQU     $00                     ; R/W
;
;/* Bit of Keyboard Interrupt Enable
;/* INTKBIER Bit Address Equate
INTKBIER    EQU     $001B                   ; Keyboard Interrupt Enable Register
KBIE7       EQU     $07                     ; R/W
KBIE6       EQU     $06                     ; R/W
KBIE5       EQU     $05                     ; R/W
KBIE4       EQU     $04                     ; R/W
KBIE3       EQU     $03                     ; R/W
KBIE2       EQU     $02                     ; R/W
KBIE1       EQU     $01                     ; R/W
KBIE0       EQU     $00                     ; R/W
;
;/* Bit of Time Base Module Control
;/* TBCR Bit Address Equate
TBCR        EQU     $001C                   ; Time Base Module Control Register
TBIF        EQU     $07                     ; R
TBR2        EQU     $06                     ; R/W
TBR1        EQU     $05                     ; R/W
TBR0        EQU     $04                     ; R/W
TACK        EQU     $03                     ; W
TBIE        EQU     $02                     ; R/W
TBON        EQU     $01                     ; R/W
;
;/* Bit of IRQ Status & Control
;/* INTSCR Bit Address Equate
INTSCR      EQU     $001D                   ; IRQ Status and Control Register
IRQF1       EQU     $03                     ; R
ACK1        EQU     $02                     ; W
IMASK1      EQU     $01                     ; R/W
MODE1       EQU     $00                     ; R/W
;
;/* Bit of Config Register2
;/* CONFIG2 Bit Address Equate
CONFIG2     EQU     $001E                   ; Config Register2
OSCSTOPENB  EQU     $01                     ; R/W
SCIBDSRC    EQU     $00                     ; R/W
;
;/* Bit of Config Register1
;/* CONFIG1 Bit Address Equate
CONFIG1     EQU     $001F                   ; Config Register1
COPRS       EQU     $07                     ; R/W
LVISTOP     EQU     $06                     ; R/W
LVIRSTD     EQU     $05                     ; R/W
LVIPWRD     EQU     $04                     ; R/W
LVI5OR3     EQU     $03                     ; R/W
SSREC       EQU     $02                     ; R/W
STOP        EQU     $01                     ; R/W
COPD        EQU     $00                     ; R/W
;
;/* Bit of Timer Status & Control
;/* TxSC Bit Address Equate
T1SC        EQU     $0020                   ; TIM1 Status & Control Register
T2SC        EQU     $002B                   ; TIM2 Status & Control Register
TOF         EQU     $07                     ; R
TOIE        EQU     $06                     ; R/W
TSTOP       EQU     $05                     ; R/W
TRST        EQU     $04                     ; W
PS2         EQU     $02                     ; R/W
PS1         EQU     $01                     ; R/W
PS0         EQU     $00                     ; R/W
;
;/* Bit of Timer CH0 Status & Control Register
;/* TxSC0 Bit Address Equate
T1SC0       EQU     $0025                   ; TIM1 CH0 Status & Control Register
T2SC0       EQU     $0030                   ; TIM2 CH0 Status & Control Register
CH0F        EQU     $07                     ; R/W
CH0IE       EQU     $06                     ; R/W
MS0B        EQU     $05                     ; R/W
MS0A        EQU     $04                     ; R/W
ELS0B       EQU     $03                     ; R/W
ELS0A       EQU     $02                     ; R/W
TOV0        EQU     $01                     ; R/W
CH0MAX      EQU     $00                     ; R/W
;
;/* Bit of Timer CH1 Status & Control Register
;/* TxSC1 Bit Address Equate
T1SC1       EQU     $0028                   ; TIM1 CH1 Status & Control Register
T2SC1       EQU     $0033                   ; TIM2 CH1 Status & Control Register
CH1F        EQU     $07                     ; R/W
CH1IE       EQU     $06                     ; R/W
MS1A        EQU     $04                     ; R/W
ELS1B       EQU     $03                     ; R/W
ELS1A       EQU     $02                     ; R/W
TOV1        EQU     $01                     ; R/W
CH1MAX      EQU     $00                     ; R/W
;
T1CNTH      EQU     $0021                   ; TIM1 Counter High Register
T1CNTL      EQU     $0022                   ; TIM1 Counter Low Register
T1MODH      EQU     $0023                   ; TIM1 Counter Modulo High Register
T1MODL      EQU     $0024                   ; TIM1 Counter Modulo Low Register
T1CH0H      EQU     $0026                   ; TIM1 CH0 High Register
T1CH0L      EQU     $0027                   ; TIM1 CH0 Low Register
T1CH1H      EQU     $0029                   ; TIM1 CH1 High Register
T1CH1L      EQU     $002A                   ; TIM1 CH1 Low Register
T2CNTH      EQU     $002C                   ; TIM2 Counter High Register
T2CNTL      EQU     $002D                   ; TIM2 Counter Low Register
T2MODH      EQU     $002E                   ; TIM2 Counter Modulo High Register
T2MODL      EQU     $002F                   ; TIM2 Counter Modulo Low Register
T2CH0H      EQU     $0031                   ; TIM2 CH0 High Register
T2CH0L      EQU     $0032                   ; TIM2 CH0 Low Register
T2CH1H      EQU     $0034                   ; TIM2 CH1 High Register
T2CH1L      EQU     $0035                   ; TIM2 CH1 Low Register
;
;/* Bit of PLL Control Register
;/* PCTL Bit Address Equate
PCTL        EQU     $0036                   ; PLL Control Register
PLLIE       EQU     $07                     ; R/W
PLLF        EQU     $06                     ; R
PLLON       EQU     $05                     ; R/W
BCS         EQU     $04                     ; R/W
PRE1        EQU     $03                     ; R/W
PRE0        EQU     $02                     ; R/W
VPR1        EQU     $01                     ; R/W
VPR0        EQU     $00                     ; R/W
;
;/* Bit of PLL Bandwidth Control Register
;/* PBWC Bit Address Equate
PBWC        EQU     $0037                   ; PLL Bandwidth Control Register
AUTO        EQU     $07                     ; R/W
LOCK        EQU     $06                     ; R
ACQ         EQU     $05                     ; R/W
;
;/* Bit of PLL Multiply Select High
;/* PMSH Bit Address Equate
PMSH        EQU     $0038                   ; PLL Multiply Select High Register
MUL11       EQU     $03                     ; R/W
MUL10       EQU     $02                     ; R/W
MUL9        EQU     $01                     ; R/W
MUL8        EQU     $00                     ; R/W
;
;/* Bit of Multiply Select Low
;/* PMSL Bit Address Equate
PMSL        EQU     $0039                   ; PLL Multiply Select Low Register
MUL7        EQU     $07                     ; R/W
MUL6        EQU     $06                     ; R/W
MUL5        EQU     $05                     ; R/W
MUL4        EQU     $04                     ; R/W
MUL3        EQU     $03                     ; R/W
MUL2        EQU     $02                     ; R/W
MUL1        EQU     $01                     ; R/W
MUL0        EQU     $00                     ; R/W
;
;/* Bit of PLL VCO Select Range
;/* PMRS Bit Address Equate
PMRS        EQU     $003A                   ; PLL VCO Select Range Register
VRS7        EQU     $07                     ; R/W
VRS6        EQU     $06                     ; R/W
VRS5        EQU     $05                     ; R/W
VRS4        EQU     $04                     ; R/W
VRS3        EQU     $03                     ; R/W
VRS2        EQU     $02                     ; R/W
VRS1        EQU     $01                     ; R/W
VRS0        EQU     $00                     ; R/W
;
;/* Bit of PLL Reference Divider Select
;/* PMDS Bit Address Equate
PMDS        EQU     $003B                   ; PLL Reference Divider Select Register
RDS3        EQU     $03                     ; R/W
RDS2        EQU     $02                     ; R/W
RDS1        EQU     $01                     ; R/W
RDS0        EQU     $00                     ; R/W
;
;/* Bit of ADC Status & Control
;/* ADSCR Bit Address Equate
ADSCR       EQU     $003C                   ; ADC Status & Control Register
COCO        EQU     $07                     ; R/W
AIEN        EQU     $06                     ; R/W
ADCO        EQU     $05                     ; R/W
ADCH4       EQU     $04                     ; R/W
ADCH3       EQU     $03                     ; R/W
ADCH2       EQU     $02                     ; R/W
ADCH1       EQU     $01                     ; R/W
ADCH0       EQU     $00                     ; R/W
;
ADR         EQU     $003D                   ; ADC Data Register
;
;/* Bit of ADC Clock Register
;/* ADCLK Bit Address Equate
ADCLK       EQU     $003E                   ; ADC Clock Register
ADIV2       EQU     $07                     ; R/W
ADIV1       EQU     $06                     ; R/W
ADIV0       EQU     $05                     ; R/W
ADICLK      EQU     $04                     ; R/W
;
;/* Bit of SIM Break Status Register
;/* SBSR Bit Address Equate
SBSR        EQU     $FE00                   ; SIM Break Status Register
SBSW        EQU     $01                     ; R
;
;/* Bit of SIM Reset Status Register
;/* SRSR Bit Address Equate
SRSR        EQU     $FE01                   ; SIM Reset Status Register
POR         EQU     $07                     ; R/W
PIN         EQU     $06                     ; R/W
COP         EQU     $05                     ; R/W
ILOP        EQU     $04                     ; R/W
ILAD        EQU     $03                     ; R/W
MODRST      EQU     $02                     ; R/W
LVI         EQU     $01                     ; R/W
;
SUBAR       EQU     $FE02                   ; SIM Upper Byte Address Register
;
;/* Bit of SIM Break Flag Control
;/* SBFCR Bit Address Equate
SBFCR       EQU     $FE03                   ; SIM Break Flag Control Register
BCFE        EQU     $07                     ; R/W
;
;/* Bit of Interrupt Status Register1
;/* INT1 Bit Address Equate
INT1        EQU     $FE04                   ; Interrupt Status1 Register
IF6         EQU     $07                     ; R
IF5         EQU     $06                     ; R
IF4         EQU     $05                     ; R
IF3         EQU     $04                     ; R
IF2         EQU     $03                     ; R
IF1         EQU     $02                     ; R
;
;/* Bit of Interrupt Status Register2
;/* INT2 Bit Address Equate
INT2        EQU     $FE05                   ; Interrupt Status2 Register
IF14        EQU     $07                     ; R
IF13        EQU     $06                     ; R
IF12        EQU     $05                     ; R
IF11        EQU     $04                     ; R
IF10        EQU     $03                     ; R
IF9         EQU     $02                     ; R
IF8         EQU     $01                     ; R
IF7         EQU     $00                     ; R
;
;/* Bit of Interrupt Status Register3
;/* INT3 Bit Address Equate Equate
INT3        EQU     $FE06                   ; Interrupt Status3 Register
IF16        EQU     $01                     ; R
IF15        EQU     $00                     ; R
;
;/* Bit of Flash Control Register
;/* FLCR Bit Address Equate
FLCR        EQU     $FE08                   ; Flash Control Register
HVEN        EQU     $03                     ; R/W
MASS        EQU     $02                     ; R/W
ERASE       EQU     $01                     ; R/W
PGM         EQU     $00                     ; R/W
;
BRKH        EQU     $FE09                   ; Break Address High Register
BRKL        EQU     $FE0A                   ; Break Address Low Register
;
;/* Bit of Break Status & Control Register
;/* BRKSCR Bit Address Equate
BRKSCR      EQU     $FE0B                   ; Break Status and Control Register
BRKE        EQU     $07                     ; R/W
BRKA        EQU     $06                     ; R/W
;
;/* Bit of LVI Status Register
;/* LVISR Bit Address Equate
LVISR       EQU     $FE0C                   ; LVI Status Register
LVIOUT      EQU     $07                     ; R/W
;
;/* Bit of Flash Block Protect
;/* FLBPR Bit Address Equate
FLBPR       EQU     $FF7E                   ; Flash Block Protect Register
BPR7        EQU     $07                     ; R/W
BPR6        EQU     $06                     ; R/W
BPR5        EQU     $05                     ; R/W
BPR4        EQU     $04                     ; R/W
BPR3        EQU     $03                     ; R/W
BPR2        EQU     $02                     ; R/W
BPR1        EQU     $01                     ; R/W
BPR0        EQU     $00                     ; R/W
COPCTL      EQU     $FFFF                   ; Reset COP Register
;/*************************************/;
;/* End of IO Register Address Equate */;
;/* For Processor : MC68H(R)C908GP32  */;
;/*************************************/;

;/*************************************/;
;/* Start of Equate Hardware I/O Bit  */;
;/* For Standard Board : CP-JR08GP32  */;
;/*************************************/;
;
;/****************************/;
;/* LCD Interface With 4-Bit */;
;/* EN-LCD = PTC2            */;
;/* RS-LCD = PTC4            */;
;/* D4-LCD = PTD0            */;
;/* D5-LCD = PTD1            */;
;/* D6-LCD = PTD2            */;
;/* D7-LCD = PTD3            */;
;/* RW,D0..D3 of LCD = GND   */;
;/****************************/;
DATA_LCD    EQU     PTD                     ; PORTD = Data LCD
DLCD_DIR    EQU     DDRD                    ; Direction For Data LCD
CTRL_LCD    EQU     PTC                     ; PORTC = Control LCD
CLCD_DIR    EQU     DDRC                    ; Direction For Control LCD
EN_LCD      EQU     02H                     ; PTC2
RS_LCD      EQU     04H                     ; PTC4
D4_LCD      EQU     00H                     ; PTD0
D5_LCD      EQU     01H                     ; PTD1
D6_LCD      EQU     02H                     ; PTD2
D7_LCD      EQU     03H                     ; PTD3
;
;/****************************/;
;/* I2C I/O Device Interface */;
;/* I2C Signal Select Jumper */;
;/*     - SDA = PTC0 or PTB6 */;
;/*     - SCL = PTC1 or PTB7 */;
;/* This Equate to PTC0,PTC1 */;
;/****************************/;
PORT_I2C    EQU     PTC                     ; PORTC = I2C Signal
PDIR_I2C    EQU     DDRC                    ; Direction Read/Write
PULL_I2C    EQU     PTCPUE                  ; Pull-Up Enable
I2C_SDA     EQU     $00                     ; PTC0 = SDA
I2C_SCL     EQU     $01                     ; PTC1 = SCL
;
;/****************************/;
;/* RS-485 Direction Control */;
;/****************************/;
PORT_485    EQU     PTC                     ; PORTC = RS485 Direction Signal Control
PDIR_485    EQU     DDRC                    ; Direction Bit Control (Setup = Output)
POUT_485    EQU     $03                     ; PTC3 = RS485 Bit Signal Direction
;
;/****************************/;
;/* Matrix Keyboard Interface*/;
;/****************************/;
PORT_KEY    EQU     PTA                     ; PORTA = Keyboard Signal
PDIR_KEY    EQU     DDRA                    ; Direction In/Out
PULL_KEY    EQU     PTAPUE                  ; Pull-Up Enable
ROW0_KEY    EQU     $00                     ; PTA0 = Row0
ROW1_KEY    EQU     $01                     ; PTA1 = Row1
ROW2_KEY    EQU     $02                     ; PTA2 = Row2
ROW3_KEY    EQU     $03                     ; PTA3 = Row3
COL0_KEY    EQU     $04                     ; PTA4 = Column0
COL1_KEY    EQU     $05                     ; PTA5 = Column1
COL2_KEY    EQU     $06                     ; PAT6 = Column2
COL3_KEY    EQU     $07                     ; PTA7 = Column3 (Jumper Select = 4x4)
;
;/****************************/;
;/* Speaker or Buzzer Control*/;
;/*   (Jumper Select = 4x3)  */;
;/****************************/;
PORT_SPK    EQU     PTA                     ; PORTA = Speaker Signal Control
PDIR_SPK    EQU     DDRA                    ; Direction Bit Control (Setup = Output)
POUT_SPK    EQU     $07                     ; PTA7 = Speaker Bit Signal
;
;/****************************/;
;/* Magnetic Card Interface  */;
;/* Data Magnetic  = PTD4    */;
;/* Clock Magnetic = PTD5    */;
;/****************************/;
PORT_MAG    EQU     PTD                     ; PORTD = Magnetic Card Reader Signal
PDIR_MAG    EQU     DDRD                    ; Direction Read/Write
PULL_MAG    EQU     PTDPUE                  ; Pull-Up Enable
MAG_DATA    EQU     $04                     ; PTD4 = Data
MAG_CLK     EQU     $05                     ; PTD5 = Clock
;
;/*************************************/;
;/*  End of Equate Hardware I/O Bit   */;
;/* For Standard Board : CP-JR08GP32  */;
;/*************************************/;


            ;/*************************/;
            ;/* RAM Buffer Start Here */;
            ;/* $0040-$023F = 512Byte */;
            ;/*************************/;
            ORG     $0040                   ; RAM Buffer : $0040-$023F
FLAG_BUF    DS      1
MAG_BUFF    DS      40
;
MAG_PASS    EQU     $00                     ; FLAG_BUF Bit-0
MAG_STRT    EQU     $01                     ; FLAG_BUF Bit-1
MAG_STOP    EQU     $01                     ; FLAG_BUF Bit-2


            ;/*******************************/;
            ;/* Usercode Program Start Here */;
            ;/* $8000-$FDFF = User Code     */;
            ;/* $FFDC-$FFFF = User Vector   */;
            ;/*******************************/;
            ORG     $8000                   ; Start Main (68HC908GP32 : $8000-$FDFF)
$BASE       10T                             ; Defualt System Number to Decimal
RESET:      BSET    0,CONFIG1               ; Disable COP Watch-Dog
            BSET    0,CONFIG2               ; SCI Baudrate Select From Internal Clock Bus
            LDHX    #$0240                  ; Last Address RAM = Stack (H:X=$0240)
            TXS                             ; [(H:X-1) -> SP] (SP = $023F)
            ;
            ;/*******************************/;
            ;/* Initial  PLL  For Clock Bus */;
            ;/* External X-TAL = 32.768 KHz */;
            ;/* Clock Bus Freq = 7.3728 MHz */;
            ;/*******************************/;
            ;
            BCLR    BCS,PCTL                ; Select External Clock
            BCLR    PLLON,PCTL              ; Turn-off PLL
            BCLR    PRE1,PCTL               ; P = 0
            BCLR    PRE0,PCTL
            BSET    VPR1,PCTL               ; E = 2
            BCLR    VPR0,PCTL
            MOV     #$01,PMDS               ; R = $01
            MOV     #$03,PMSH               ; N = $0384
            MOV     #$84,PMSL
            MOV     #$C0,PMRS               ; L = $C0
            LDA     PCTL                    ; Clear Status
            BSET    AUTO,PBWC               ; Auto Bandwidth Control
            BSET    PLLON,PCTL              ; Turn-on PLL
            BRCLR   LOCK,PBWC,*             ; Wait For PLL Lock Complete
            BSET    BCS,PCTL                ; Select PLL Output as Base Clock
            ;
            ;/*******************************/;
            ; Other User Program Start Here */;
            ;/*******************************/;
            ;
            ;/* Initial LCD Signal Direction */;
            BSET    RS_LCD,CLCD_DIR         ; Control LCD = Output
            BSET    EN_LCD,CLCD_DIR
            BSET    D4_LCD,DLCD_DIR         ; Data LCD = Output
            BSET    D5_LCD,DLCD_DIR
            BSET    D6_LCD,DLCD_DIR
            BSET    D7_LCD,DLCD_DIR
            ;
            ;/* Initial LCD Signal Logic */;
            BCLR    EN_LCD,CTRL_LCD         ; Initial Signal LCD
            BSET    RS_LCD,CTRL_LCD
            BSET    D4_LCD,DATA_LCD
            BSET    D5_LCD,DATA_LCD
            BSET    D6_LCD,DATA_LCD
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Initial I2C Signal Direction */;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output
            BSET    I2C_SDA,PULL_I2C        ; Enable Pull-up SDA
            BSET    I2C_SDA,PORT_I2C        ; SDA = "1"
            BSET    I2C_SCL,PORT_I2C        ; SCA = "1"
            ;
            ;/* Makesure I2C Stop Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Initial Bus For Stop I2C Device
            BCLR    I2C_SDA,PORT_I2C
            BSET    I2C_SCL,PORT_I2C
            BSET    I2C_SDA,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            ;
            ;/* Initial Magnetic Card Reader */;
            BCLR    MAG_DATA,PDIR_MAG       ; Data  = Input
            BCLR    MAG_CLK,PDIR_MAG        ; Clock = Input
            BSET    MAG_DATA,PULL_MAG       ; Enable Pull-Up Data
            BSET    MAG_CLK,PULL_MAG        ; Enable Pull-Up Clock
            ;
            ;/* Initial Keyboard & Speaker */;
            BCLR    ROW0_KEY,PDIR_KEY       ; Row0 = Input
            BCLR    ROW1_KEY,PDIR_KEY       ; Row1 = Input
            BCLR    ROW2_KEY,PDIR_KEY       ; Row2 = Input
            BCLR    ROW3_KEY,PDIR_KEY       ; Row3 = Input
            BSET    ROW0_KEY,PULL_KEY       ; Enable Row0 Pull-Up
            BSET    ROW1_KEY,PULL_KEY       ; Enable Row1 Pull-Up
            BSET    ROW2_KEY,PULL_KEY       ; Enable Row2 Pull-Up
            BSET    ROW3_KEY,PULL_KEY       ; Enable Row3 Pull-Up
            BSET    COL0_KEY,PDIR_KEY       ; Column0 = Output
            BSET    COL1_KEY,PDIR_KEY       ; Column1 = Output
            BSET    COL2_KEY,PDIR_KEY       ; Column2 = Output
           ;BSET    COL3_KEY,PDIR_KEY       ; Column3 = Output (Jumper Select = 4x4)
            BSET    POUT_SPK,PDIR_KEY       ; Speaker = Output (Jumper Select = 4x3)
            ;
            JSR     INIT_SCI                ; Initial RS-232 (9600,N,8,1,P)
            JSR     INIT_LCD                ; Initial 4-Bit LCD
            ;
            ;
            ;/*********************************/;
            ;/* Read Magnetic Card MCR-B02TTL */;
            ;/* - Display Data on SCI RS232   */;
            ;/* - Display Data on LCD 40x2    */;
            ;/*********************************/;
            ;
TEST_MAG:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A,$0A
            DB      'DEMO FOR MAGNETIC CARD READER MODEL : MCR-B02TTL'
            DB      $0D,$0A
            DB      'Reading By...Interrupt on IRQ (Short Jumper INT)'
            DB      $0D,$0A,$00
            ;
            LDA     #$00                    ; Start Cursor Line1
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'DEMO FOR MAGNETIC CARD READER MCR-B02TTL',$00
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      '                                        ',$00
            ;
            CLI                             ; Enable Global Interrupt
            JSR     BEEP_SPK                ; Start Beep
            ;
WAIT_MAG:   BRCLR   MAG_PASS,FLAG_BUF,*     ; Wait Interrupt Complete
            ;
            ;/* Start Display Magnetic Code */;
            LDHX    #MAG_BUFF               ; Start Buffer Display
DSP_MAG1:   LDA     0,X                     ; Get 1St Byte Data
            CBEQA   #$0B,DSP_MAG2           ; Check Start Code at 1St Byte
            BCLR    MAG_PASS,FLAG_BUF
            BRA     WAIT_MAG                ; Restart New Read Data
            ;
            ;/* Start Display Data to RS232 */;
DSP_MAG2:   AIX     #1                      ; Next Data
            LDA     0,X
            CBEQA   #$0F,DSP_MAG3           ; Not Display Stop Code
            ADD     #$30                    ; Convert to ASCII
            JSR     TX_BYTE
            BRA     DSP_MAG2                ; Next Byte Display
            ;
DSP_MAG3:   LDA     #$0D                    ; Carrier Return
            JSR     TX_BYTE
            LDA     #$0A                    ; Line Feed
            JSR     TX_BYTE
            ;
            ;/* Start Display Data to 4-Bit LCD */;
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD               ; Clear old Display
            DB      '                                        ',$00
            ;
            ;/* New Data Display */;
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            ;
            LDHX    #MAG_BUFF               ; Start Buffer Display
DSP_MAG4:   AIX     #1
            LDA     0,X
            CBEQA   #$0F,DSP_MAG5           ; Not Display Stop Code
            ADD     #$30                    ; Convert to ASCII
            JSR     WR_LCD
            BRA     DSP_MAG4                ; Next Byte Display
            ;
DSP_MAG5:   BCLR    MAG_PASS,FLAG_BUF
            JSR     BEEP_SPK                ; Beep on Exit
            BRA     WAIT_MAG                ; Wait New Data


;/*********************************/;
;/* Read Data From  Magnetic Card */;
;/* Get Number of Card (Modulo-5) */;
;/* Output : MAG_BUFF(BCD 40Byte) */;
;/* (This is IRQ Service Routine) */;
;/*********************************/;
;
READ_MAG:   AIS     #-2                         ; Reserve 2 Byte Stack
            ;
            LDHX    #MAG_BUFF                   ; Pointer Magnetic Buffer
            CLRA
READ_MG0:   STA     0,X                         ; Clear Buffer
            AIX     #1
            CPHX    #MAG_BUFF+40                ; 40 Byte Clear
            BNE     READ_MG0
            ;
            BCLR    MAG_PASS,FLAG_BUF           ; Clear Any Flag
            BCLR    MAG_STRT,FLAG_BUF
            BCLR    MAG_STOP,FLAG_BUF
            ;
            ;/*********************************/;
            ;/* Wait For Start Bit Data = "0" */;
            ;/* Start = "1" But Signal Invert */;
            ;/*********************************/;
READ_MG1:   JSR     GET_CLK                     ; Get Start Sentinel
            BRSET   MAG_DATA,PORT_MAG,READ_MG1  ; Wait Data = "0" (Invert = "1" = Start)
            ;
            LDA     #5                          ; Modulo-5 Format Counter
            STA     1,SP
            CLRA                                ; Clear Result
            STA     2,SP                        ; Clear Checksum
            LDHX    #MAG_BUFF                   ; Start Pointer Save Data
            BRA     READ_MG4                    ; Get Start Bit at 1St Byte
            ;
            ;/* Get Start Data Bit Ready  */;
            ;/* Check 1St Byte = XXX01011 */;
READ_MG2:   LDA     #5                          ; Modulo-5 Format Counter
            STA     1,SP
            CLRA                                ; Clear Result
            ;
READ_MG3:   JSR     GET_CLK                     ; Wait Clock Sync.
            BRCLR   MAG_DATA,PORT_MAG,READ_MG4  ; Invert Data Signal Logic
            CLC                                 ; C = "0"
            BRA     READ_MG5
READ_MG4:   SEC                                 ; C = "1"
READ_MG5:   RORA                                ; Shift D7..D3 (D3 First)
            DBNZ    1,SP,READ_MG3
            ;
            ;/* Now Result = A = XXXXX000
            LSRA                                ; Shift D3 to D2 (0 -> XXXXX000)
            LSRA                                ; Shift D3 to D1 (0 -> 0XXXXX00)
            LSRA                                ; Shift D3 to D0 (0 -> 00XXXXX0)
            ;
            ;/* Now Result = A = 000XXXXX
            ;
            PSHH                                ; Save Pointer
            PSHX
            ;
            LDHX    #TAB_MAG1
READ_MG6:   CBEQ    0,X+,READ_MG7               ; Data Parity OK
            CPHX    #TAB_MAG1+16
            BNE     READ_MG6
            BCLR    MAG_PASS,FLAG_BUF           ; Parity Error
            PULX                                ; Restore Pointer
            PULH
            BRA     EXIT_MAG                    ; Exit
            ;
READ_MG7:   PULX                                ; Restore Pointer
            PULH
            ;
            AND     #$0F                        ; Ignore Parity Bit
            BRSET   MAG_STOP,FLAG_BUF,READ_MG8  ; Goto Compare Checksum
            STA     0,X                         ; Save Result to Buffer
            EOR     2,SP                        ; Generate Checksum
            STA     2,SP
            LDA     0,X
            AIX     #1                          ; Next Pointer
            CMP     #$0F                        ; $0F = Stop Code
            BNE     READ_MG2
            BSET    MAG_STOP,FLAG_BUF           ; Stop Code Receive
            BRA     READ_MG2                    ; Goto Get Checksum
            ;
READ_MG8:   CBEQ    2,SP,READ_MG9               ; Compare Checksum
            BCLR    MAG_PASS,FLAG_BUF           ; Checksum Error
            BRA     EXIT_MAG
            ;
READ_MG9:   BSET    MAG_PASS,FLAG_BUF           ; Enable Display Result
            BIL     *                           ; Wait Present(IRQ) Return
EXIT_MAG:   AIS     #2                          ; Clean-Up 2 Byte Stack
            RTI

;/**********************/;
;/* Get Magnetic Clock */;
;/*    ---|___|---     */;
;/**********************/;
;
GET_CLK:    BRCLR   MAG_CLK,PORT_MAG,*  ; Wait Rising Clock
            BRSET   MAG_CLK,PORT_MAG,*  ; Wait Falling Clock
            RTS

;/**************************/;
;/* Table Data of Modulo-5 */;
;/* 4Bit BCD + Parity(odd) */;
;/**************************/;
;
TAB_MAG1:   DB      %00010000           ; 0(00H) = Data
            DB      %00000001           ; 1(01H) = Data
            DB      %00000010           ; 2(02H) = Data
            DB      %00010011           ; 3(03H) = Data
            DB      %00000100           ; 4(04H) = Data
            DB      %00010101           ; 5(05H) = Data
            DB      %00010110           ; 6(06H) = Data
            DB      %00000111           ; 7(07H) = Data
            DB      %00001000           ; 8(08H) = Data
            DB      %00011001           ; 9(09H) = Data
            DB      %00011010           ; A(0AH) = Control(:)
            DB      %00001011           ; B(0BH) = Start Sentinel(;)
            DB      %00011100           ; C(0CH) = Control(<)
            DB      %00001101           ; D(0DH) = Field Seperate(=)
            DB      %00001110           ; E(0EH) = Control(>)
            DB      %00011111           ; F(0FH) = End Sentinel(?)


;/***********************/;
;/* Generate Sound Beep */;
;/*   (Mini Speaker)    */;
;/***********************/;
;
BEEP_SPK:   AIS     #-2                 ; Reserve 2 Byte Stack
            PSHA                        ; Save ACC
            ;
            LDA     #$FF                ; Duration Beep
            STA     2,SP
BEEP_SP1:   LDA     #$FF                ; Freq. Beep
            STA     1,SP
            BSET    POUT_SPK,PORT_SPK   ; ON Time
            DBNZ    1,SP,*
            LDA     #$FF
            STA     1,SP
            BCLR    POUT_SPK,PORT_SPK   ; OFF Time
            DBNZ    1,SP,*
            DBNZ    2,SP,BEEP_SP1
            ;
            PULA                        ; Restore ACC
            AIS     #2                  ; Clean-Up 2 Byte Stack
            RTS

;/**********************************/;
;/* Start Subroutine For 4-Bit LCD */;
;/* -INIT_LCD = Initial 4-Bit LCD  */;
;/* -ENA_LCD  = Active Enable LCD  */;
;/* -BUSY     = Delay Wait LCD     */;
;/* -WR_LCD   = Write ASCII to LCD */;
;/* -WR_INS   = Write Command LCD  */;
;/* -GOTO_LCD = Set Cursor LCD     */;
;/* -PRINT_LCD= Print Display LCD  */;
;/**********************************/;
;
;/********************/;
;/* Initial Char LCD */;
;/* 4-Bit Interface  */;
;/* RS    = Bit-3    */;
;/* EN    = Bit-2    */;
;/* D4..7 = Bit-0..3 */;
;/********************/;
;
INIT_LCD:   LDA     #33H                    ; DL = 1 ,DL =1
            JSR     WR_INS
            LDA     #32H                    ; DL = 1 ,DL =0
            JSR     WR_INS
            LDA     #28H                    ; Function set
            JSR     WR_INS                  ; DL=0(4Bit),N=1(2Line),F=0(5X7)
            LDA     #0CH                    ; Display on/off Control
            JSR     WR_INS                  ; Entry Display,Cursor off,Not Blink
            LDA     #06H                    ; Entry Mode Set
            JSR     WR_INS                  ; I/D=1 Increment,S=0 Cursor Shift
CLR_LCD:    LDA     #01H                    ; Clear Display
            JSR     WR_INS                  ; Clear Display,Set DD RAM Addr=0
            RTS

;/**********************/;
;/* Enable Pin EN CLCD */;
;/* Pin EN LCD = Bit-2 */;
;/* Active Chip select */;
;/**********************/;
;
ENA_LCD:    JSR     BUSY                    ; Busy Delay
            BSET    EN_LCD,CTRL_LCD         ; Enable Chips Select
            ;
            ;/* Delay 10uS (76-Cycle)
            PSHX                            ; 2 Cycle
            LDX     #18                     ; 2 Cycle
ENA_LCD0:   NOP                             ; 1 Cycle
            DBNZX   ENA_LCD0                ; 3 Cycle (18x4 = 72)
            PULX                            ; 2 Cycle
            ;
            BCLR    EN_LCD,CTRL_LCD         ; Disable Chips Select
            RTS

;/*******************/;
;/* Busy CLCD Delay */;
;/* Default = 100uS */;
;/* Fbus = 7.3728Hz */;
;/* 100uS = 737Cycle*/;
;/*******************/;
;
BUSY:       PSHX                            ; 2 Cycle
            LDX     #182                    ; 2 Cycle
BUSY1:      NOP                             ; 1 Cycle
            DBNZX   BUSY1                   ; 3 Cycle (182x4 =728)
            PULX                            ; 2 Cycle
            RTS                             ; 4 Cycle

;/**********************/;
;/* Write ASCII to LCD */;
;/* Input : ACC(ASCII) */;
;/**********************/;
;
WR_LCD:     PSHA                            ; Save ACC(ASCII)
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            ;/* Send High Nibble */;
            LDA     2,SP                    ; Get ACC(ASCII)
            AND     #$F0                    ; High Nibble
            NSA                             ; Nibble Swap A (D7..D4 <-> D3..D0)
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD01
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_LCD01:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD02
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_LCD02:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD03
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_LCD03:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD04
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe High Nibble Data */;
WR_LCD04:   BSET    RS_LCD,CTRL_LCD         ; Set RS For Write Data Select
            JSR     ENA_LCD
            ;
            ;/* Send Low Nibble */;
            LDA     2,SP                    ; Get ACC (ASCII)
            AND     #$0F                    ; Low Nibble
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD05
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_LCD05:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD06
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_LCD06:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD07
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_LCD07:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD08
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe Low Nibble Data */;
WR_LCD08:   BSET    RS_LCD,CTRL_LCD         ; Set RS For Write Data Select
            JSR     ENA_LCD
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULA                            ; Restore ACC(ASCII)
            RTS

;/**************************/;
;/* Write Instruction LCD  */;
;/* Input  : ACC (Command) */;
;/* Output : Data bus LCD  */;
;/**************************/;
;
WR_INS:     PSHA                            ; Save ACC(Command)
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            ;/* Send High Nibble */;
            LDA     2,SP                    ; Get ACC(Command)
            AND     #$F0                    ; High Nibble
            NSA                             ; Nibble Swap A (D7..D4 <-> D3..D0)
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS01
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_INS01:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS02
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_INS02:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS03
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_INS03:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS04
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe High Nibble Command */;
WR_INS04:   BCLR    RS_LCD,CTRL_LCD         ; Clear RS For Write Instruction
            JSR     ENA_LCD
            ;
            ;/* Send Low Nibble */;
            LDA     2,SP                    ; Get ACC (Command)
            AND     #$0F                    ; Low Nibble
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS05
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_INS05:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS06
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_INS06:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS07
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_INS07:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS08
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe Low Nibble Command */;
WR_INS08:   BCLR    RS_LCD,CTRL_LCD         ; Clear RS For Write Instruction
            JSR     ENA_LCD
            ;
            LDA     #2                      ; Wait Command Ready = 2mS
            JSR     DLY_MSEC
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULA                            ; Restore ACC(Command)
            RTS

;/************************/;
;/* Goto position of LCD */;
;/* Input : ACC (addr.)  */;
;/************************/;
;
GOTO_LCD:   ORA     #$80                    ; Set ACC.7
            JSR     WR_INS
            RTS

;/********************************/;
;/* Print ASCII Data to LCD Port */;
;/* Usage    : JSR  PRINT_LCD    */;
;/*          : DB   'xxxxxxxx'   */;
;/*          : DB   $00          */;
;/* Note     : Last Byte = $00   */;
;/********************************/;
;
PRINT_LCD:  PULH                        ; Pop Stack PCH
            PULX                        ; Pop Stack PCL
            PSHA                        ; Save ACC
            ;
PRN_LCD1:   LDA     0,X                 ; Get Data For Print
            BEQ     PRN_LCD2            ; If Last Byte Jump
            JSR     WR_LCD              ; Write Data to LCD
            AIX     #1                  ; Inc (H:X)
            BRA     PRN_LCD1
            ;
PRN_LCD2:   AIX     #1                  ; Skip Last Byte(00)
            ;
            PULA                        ; Restore ACC
            PSHX                        ; Push PCL to Stack
            PSHH                        ; Push PCH to Stack
            RTS

;/***********************************/;
;/* End of Subroutine For 4-Bit LCD */;
;/***********************************/;


;/***********************************/;
;/* Start Subroutine For I2C Device */;
;/* -OUT_WRITE = Out Port PCF8574   */;
;/* -INP_READ  = Read Port PCF8574  */;
;/* -RTC_WRITE = Write RTC PCF8583  */;
;/* -RTC_READ1 = Read RTC PCF8583   */;
;/* -I2C_WRLF  = Shift Bit Out (LSB)*/;
;/* -I2C_RDLF  = Shift Bit In (LSB) */;
;/***********************************/;
;
;/*****************************/;
;/* I2C-BUS PCF8574 Interface */;
;/* Write Data to  PCF8574/A  */;
;/* Input : A (Data)          */;
;/*****************************/;
;
OUT_WRITE:  PSHA                            ; Save Data to Write
            PSHX                            ; Save X
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
OUT_WR0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* PCF8574  = 0100XXX0
            ;/* PCF8574A = 0111XXX0
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%01110000              ; 0111000,Write (PCF8574A)
            ;
OUT_WR1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   OUT_WR1
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,OUT_WR2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     OUT_WR0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
OUT_WR2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     3,SP                    ; Get ACC (Data)
            ;
OUT_WR3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   OUT_WR3
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Release Data Line
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            ;
            ;/* Stop Clock Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULX                            ; Restore X
            PULA                            ; Restore Data to Write
            RTS

;/*****************************/;
;/* I2C-BUS PCF8574 Interface */;
;/* Read Data From PCF8574/A  */;
;/* Output  : A (Data)        */;
;/*****************************/;
;
INP_READ:   AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save EEP Address
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
INP_RD0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* PCF8574  = 0100XXX1
            ;/* PCF8574A = 0111XXX1
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%01110001              ; 0111000,Read
            ;
INP_RD1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   INP_RD1                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,INP_RD2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     INP_RD0                 ; Resend Control Byte Until ACK
            ;
INP_RD2:    LDX     #8                      ; Read Data 8-Bit Count
            CLRA                            ; Clera Buffer For Read
            ;
INP_RD3:    BCLR    I2C_SCL,PORT_I2C        ;/* Wait to Receive SDA Signal */;
            JSR     I2C_RDLF                ; SDA Shift-IN D7..D0 (D0 First)
            DBNZX   INP_RD3                 ; Loop to Read Data 8-Bit
            STA     3,SP                    ; Save Result
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Not ACK At Stop Read
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Send ACK */;
            ;
            ;/* Stop Clock Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULX                            ; Restore EEP Address
            PULA                            ; Get Result
            RTS

;/*****************************/;
;/* I2C-RTC PCF8583 Interface */;
;/* Write Data to PCF8583 RTC */;
;/* Input : X (Address)       */;
;/*       : A (Data)          */;
;/*****************************/;
;
RTC_WRITE:  PSHA                            ; Save Data to Write
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
RTC_WR0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* PCF8583 = 101000X0
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100000              ; 1010000,Write
            ;
RTC_WR1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR1
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_WR2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_WR0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
RTC_WR2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get RTC Address
            ;
RTC_WR3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR3
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_WR4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_WR6                 ; Exit
            ;
RTC_WR4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Data 8-Bit Count
            LDA     3,SP                    ; Get Data For Write
            ;
RTC_WR5:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR5
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Release Data Line
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            ;
RTC_WR6:    BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore RTC Address
            PULA                            ; Restore RTC Data
            RTS

;/*****************************/;
;/* I2C-BUS PCF8583 Interface */;
;/* Read Data From PCF8583    */;
;/* -RTC_READ1 = Random Read  */;
;/*  Input  : X (Address)     */;
;/*  Output : A (Data)        */;
;/* -RTC_READ2 = Current Read */;
;/*  Output : ACC (Data)      */;
;/*****************************/;
;
RTC_READ1:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
RTC_RD0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* PCF8583 = 101000X0
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100000              ; 1010000,Write
            ;
RTC_RD1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD1                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD0                 ; Resend Control Byte Until ACK
            ;
RTC_RD2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get RTC Address
            ;
RTC_RD3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD3                 ; Loop to Write Addr High-Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD10                ; Exit
            ;
            ;/* New Start For Send Control Byte Read Command */;
RTC_RD4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BCLR    I2C_SCL,PORT_I2C        ; Falling New Start Clock
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* PCF8583 = 10100001
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100001              ; 1010000,Read
            BRA     RTC_RD7
            ;
            ;/* Current Address Read Operation */;
RTC_READ2:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* PCF8583 = 10100001
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100001              ; 1010000,Read
            ;
RTC_RD7:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD7                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD8
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD10                ; Exit
            ;
RTC_RD8:    LDX     #8                      ; Read Data 8-Bit Count
            CLRA                            ; Clera Buffer For Read
            ;
RTC_RD9:    BCLR    I2C_SCL,PORT_I2C        ;/* Wait to Receive SDA Signal */;
            JSR     I2C_RDLF                ; SDA Shift-IN D7..D0 (D0 First)
            DBNZX   RTC_RD9                 ; Loop to Read Data 8-Bit
            STA     3,SP                    ; Save Result
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Not ACK At Stop Read
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Send ACK */;
            ;
RTC_RD10:   BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore RTC Address
            PULA                            ; Get Result to ACC
            RTS

;/**************************/;
;/* Write SDA & Shift Left */;
;/* Input : ACC            */;
;/**************************/;
;
I2C_WRLF:   ROLA                            ; Shift D7..D0 (D7 First)
            BCS     I2C_WRL0                ; Jump if Data = "1"
            BCLR    I2C_SDA,PORT_I2C        ; SDA = "0"
            BRA     I2C_WRL1
I2C_WRL0:   BSET    I2C_SDA,PORT_I2C        ; SDA = "1"
I2C_WRL1:   NOP
            BSET    I2C_SCL,PORT_I2C        ; Shift Data to I2C Device
            RTS

;/*************************/;
;/* Read SDA & Shift Left */;
;/* Output : ACC          */;
;/*************************/;
;
I2C_RDLF:   BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read)
            BSET    I2C_SCL,PORT_I2C        ; Read Data From I2C-Device
            BRSET   I2C_SDA,PORT_I2C,I2C_RDL0
            CLC                             ; C = "0"
            BRA     I2C_RDL1
I2C_RDL0:   SEC                             ; C = "1"
I2C_RDL1:   ROLA                            ; Shift D7..D0 (D7 First)
            BSET    I2C_SCL,PORT_I2C        ; Shift Data to I2C Device
            RTS

;/************************************/;
;/* End of Subroutine For I2C Device */;
;/************************************/;


;/***********************************/;
;/* Start Subroutine For SCI(RS232) */;
;/* -OUT_WRITE = Out Port PCF8574   */;
;/* -INP_READ  = Read Port PCF8574  */;
;/* -RTC_WRITE = Write RTC PCF8583  */;
;/* -RTC_READ1 = Read RTC PCF8583   */;
;/* -I2C_WRLF  = Shift Bit Out (LSB)*/;
;/* -I2C_RDLF  = Shift Bit In (LSB) */;
;/***********************************/;
;
;/*************************************/;
;/* Initial SCI Port = Baudrate,N,8,1 */;
;/*  (SCI Clock = Fbus = 7.3728 MHz)  */;
;/* Baudrate = 7.3728MHz / (64xPDxBD) */;
;/* 64xPDxBD = 7.3728MHz / Baudrate   */;
;/*************************************/;
;/* Baudrate : PD x BD  %00PP0BBB     */;
;/*  1200    : 96(3x32) %00010101     */;
;/*  2400    : 48(3x16) %00010100     */;
;/*  4800    : 24(3x8)  %00010011     */;
;/*  9600    : 12(3x4)  %00010010     */;
;/*  19200   : 6(3x2)   %00010001     */;
;/*  28800   : 4(1x4)   %00000010     */;
;/*  38400   : 3(3x1)   %00010000     */;
;/*  57600   : 2(1x2)   %00000001     */;
;/*************************************/;
;
INIT_SCI:   ;MOV     #%00010101,SCBR         ; Baud Rate = 1200 / 7.3728MHz
            ;MOV     #%00010100,SCBR         ; Baud Rate = 2400 / 7.3728MHz
            ;MOV     #%00010011,SCBR         ; Baud Rate = 4800 / 7.3728MHz
            MOV     #%00010010,SCBR         ; Baud Rate = 9600 / 7.3728MHz
            ;MOV     #%00010001,SCBR         ; Baud Rate = 19200 / 7.3728MHz
            ;MOV     #%00000010,SCBR         ; Baud Rate = 28800 / 7.3728MHz
            ;MOV     #%00010000,SCBR         ; Baud Rate = 38400 / 7.3728MHz
            ;MOV     #%00000001,SCBR         ; Baud Rate = 57600 / 7.3728MHz
            ;
            BCLR    LOOPS,SCC1              ; Normal Operation(Not Loop RX & TX)
            BSET    ENSCI,SCC1              ; Enable the SCI (N,8,1)
            BCLR    TXINV,SCC1              ; TX Signal Normal (Not Invert)
            BCLR    M,SCC1                  ; Data 8-Bit SCI
            BCLR    WAKE,SCC1               ; Idle Line Wakeup(Not Address mark Wakeup)
            BCLR    ILTY,SCC1               ; Character Bit Count After Start Bit
            BCLR    PEN,SCC1                ; Not Parity Bit
            BCLR    PTY,SCC1                ; Defualt = Even Parity (If Parity Enable)
            ;
            BCLR    SCTIE,SCC2              ; Disable TX Interrupt
            BCLR    TCIE,SCC2               ; Disable TC Interrupt
            BCLR    SCRIE,SCC2              ; Disable RX Interrupt
            BCLR    ILIE,SCC2               ; Disable IDLE Interrupt
            BSET    TE,SCC2                 ; Enable TX SCI
            BSET    RE,SCC2                 ; Enable RX SCI
            BCLR    RWU,SCC2                ; Normal Operation(Not Standby Mode)
            BCLR    SBK,SCC2                ; Not Break Character
            ;
            BCLR    DMARE,SCC3              ; Disable SCI RX DMA
            BCLR    DMATE,SCC3              ; Disable SCI TX DMA
            BCLR    ORIE,SCC3               ; Disable RX Overrun Error Interrupt
            BCLR    NEIE,SCC3               ; Disable Noise Error Interrupt
            BCLR    FEIE,SCC3               ; Disable Framing Error Interrupt
            BCLR    PEIE,SCC3               ; Disable Parity Error Interrupt
            RTS

;/********************/;
;/* Send Byte to SCI */;
;/* Input : A        */;
;/********************/;
;
TX_BYTE:    BRCLR   SCTE,SCS1,*             ; Wait Buffer Empty
            STA     SCDR                    ; Send Byte Serial Port
            BRCLR   TC,SCS1,*               ; Wait Send Ready
            RTS

;/**********************/;
;/* Read Byte From SCI */;
;/* Output : A         */;
;/* (Polling Read Byte)*/;
;/**********************/;
;
RX_BYTE:    BRCLR   SCRF,SCS1,*             ; Wait Receive Ready.
            LDA     SCDR                    ; Read Byte Serial Port
            RTS

;/******************************/;
;/* Print ASCII Data to RS232  */;
;/* Usage    : JSR  PRINT_SER  */;
;/*          : DB   'xxxxxxxx' */;
;/*          : DB   $00        */;
;/* Note     : Last Byte = $00 */;
;/******************************/;
;
PRINT_SER:  PULH                        ; Pop Stack PCH
            PULX                        ; Pop Stack PCL
            PSHA                        ; Save ACC
            ;
PRN_SER1:   LDA     0,X                 ; Get Data For Print
            BEQ     PRN_SER2            ; If Last Byte Jump
            JSR     TX_BYTE             ; Send Byte to RS232
            AIX     #1                  ; Inc (H:X)
            BRA     PRN_SER1
            ;
PRN_SER2:   AIX     #1                  ; Skip Last Byte(00)
            ;
            PULA                        ; Restore ACC
            PSHX                        ; Push PCL to Stack
            PSHH                        ; Push PCH to Stack
            RTS

;/************************************/;
;/* End of Subroutine For SCI(RS232) */;
;/************************************/;

;/*******************/;
;/* Delay Time 1 mS */;
;/* Fbus = 7.372 MHz*/;
;/* 1mS = 7373 Cycle*/;
;/* Input : A (mSec)*/;
;/*******************/;
;
DLY_MSEC:   PSHH                            ; Save Register (2 Cycle)
            PSHX                            ; 2 Cycle
            ;
DLY_MS0:    LDHX    #816                    ; 3 Cycle
DLY_MS1:    NOP                             ; 1 Cycle
            AIX     #-1                     ; H:X - 1 (2 Cycle)
            CPHX    #0                      ; 3 Cycle
            BNE     DLY_MS1                 ; 3 Cycle ( 9 x 886 = 7344 Cycle)
            DBNZA   DLY_MS0                 ; 3 Cycle
            ;
            PULX                            ; 2 Cycle
            PULH                            ; 2 Cycle
            RTS                             ; 4 Cycle


            ;/**************************/;
            ;/* User Vector Start Here */;
            ;/*     2 Byte / Vector    */;
            ;/**************************/;
            ;/* ORG $Vector  Address   */;
            ;/* DW  $Routine Address   */;
            ;/**************************/;
            ;
            ORG     $FFDC                   ; TBM Vector
            DW      RESET
            ;
            ORG     $FFDE                   ; ADC Vector
            DW      RESET
            ;
            ORG     $FFE0                   ; KBI Vector
            DW      RESET
            ;
            ORG     $FFE2                   ; SCI TX Vector
            DW      RESET
            ;
            ORG     $FFE4                   ; SCI RX Vector
            DW      RESET
            ;
            ORG     $FFE6                   ; SCI Error Vector
            DW      RESET
            ;
            ORG     $FFE8                   ; SPI TX Vector
            DW      RESET
            ;
            ORG     $FFEA                   ; SPI RX Vector
            DW      RESET
            ;
            ORG     $FFEC                   ; TIM2 Overflow Vector
            DW      RESET
            ;
            ORG     $FFEE                   ; TIM2 CH1 Vector
            DW      RESET
            ;
            ORG     $FFF0                   ; TIM2 CH0 Vector
            DW      RESET
            ;
            ORG     $FFF2                   ; TIM1 Overflow Vector
            DW      RESET
            ;
            ORG     $FFF4                   ; TIM1 CH1 Vector
            DW      RESET
            ;
            ORG     $FFF6                   ; TIM1 CH0 Vector
            DW      RESET
            ;
            ORG     $FFF8                   ; PLL Vector
            DW      RESET
            ;
            ORG     $FFFA                   ; IRQ Vector
            DW      READ_MAG
            ;
            ORG     $FFFC                   ; SWI Vector
            DW      RESET
            ;
            ORG     $FFFE                   ; RST Vector
            DW      RESET

            END
