;/***************************************/;
;/* Example Form For : CP-JR08GP32 V2.0 */;
;/* Controller       : 68H(R)C908GP32   */;
;/* Assembler        : CASM08Z.EXE      */;
;/* File Name        : JR08GPV2.ASM     */;
;/* Write By         : Eakachai Makarn  */;
;/* File Update      : 15-Nov-2002      */;
;/***************************************/;
;/* System IO Self Test For CP-JR08GP32 */;
;/* -> Test PTA6 LED Output Selftest    */;
;/* -> Test PCF8574 Input/Output        */;
;/* -> Test PCF8583 Real Time Clock     */;
;/* -> Test 24C32/64/128/256/512 EEPROM */;
;/* -> Test 4-Bit LCD (16x1,16x2)       */;
;/* -> Test Magnetic Card : MCR-B02TTL  */;
;/* -> Test RS232/422/485               */;
;/* -> Test Matrix Keyboard (4x3,4x4)   */;
;/* -> Test Mini Speaker                */;
;/* -> Test Pulse Width Moduration(PWM) */;
;/***************************************/;
;
;/***************************************/;
;/* Start of IO Register Address Equate */;
;/* For Processor : MC68H(R)C908GP32    */;
;/***************************************/;
PTA         EQU     $0000                   ; Port-A Data Register
PTB         EQU     $0001                   ; Port-B Data Register
PTC         EQU     $0002                   ; Port-C Data Register
PTD         EQU     $0003                   ; Port-D Data Regsiter
DDRA        EQU     $0004                   ; Port-A Direction Register
DDRB        EQU     $0005                   ; Port-B Direction Register
DDRC        EQU     $0006                   ; Port-C Direction Register
DDRD        EQU     $0007                   ; Port-D Direction Register
PTE         EQU     $0008                   ; Port-E Data Regsiter
DDRE        EQU     $000C                   ; Port-E Direction Register
PTAPUE      EQU     $000D                   ; Port-A Pull-Up Enable Register
PTCPUE      EQU     $000E                   ; Port-C Pull-Up Enable Register
PTDPUE      EQU     $000F                   ; Port-D Pull-Up Enable Register
;
;/* Bit of SPI Control Register
;/* SPCR Bit Address Equate
SPCR        EQU     $0010                   ; SPI Control Register
SPRIE       EQU     $07                     ; R/W
DMAS        EQU     $06                     ; R
SPMSTR      EQU     $05                     ; R/W
CPOL        EQU     $04                     ; R/W
CPHA        EQU     $03                     ; R/W
SPWOM       EQU     $02                     ; R/W
SPE         EQU     $01                     ; R/W
SPTIE       EQU     $00                     ; R/W
;
;/* Bit of SPI Status and Control
;/* SPSCR Bit Address Equate
SPSCR       EQU     $0011                   ; SPI Status and Control Register
SPRF        EQU     $07                     ; R
ERRIE       EQU     $06                     ; R/W
OVRF        EQU     $05                     ; R
MODF        EQU     $04                     ; R
SPTE        EQU     $03                     ; R
MODFEN      EQU     $02                     ; R/W
SPR1        EQU     $01                     ; R/W
SPR0        EQU     $00                     ; R/W
;
SPDR        EQU     $0012                   ; SPI Data Register
;
;/* Bit of SCI Control Register1
;/* SCC1 Bit Address Equate
SCC1        EQU     $0013                   ; SCI Control Register1
LOOPS       EQU     $07                     ; R/W
ENSCI       EQU     $06                     ; R/W
TXINV       EQU     $05                     ; R/W
M           EQU     $04                     ; R/W
WAKE        EQU     $03                     ; R/W
ILTY        EQU     $02                     ; R/W
PEN         EQU     $01                     ; R/W
PTY         EQU     $00                     ; R/W
;
;/* Bit of SCI Control Register2
;/* SCC2 Bit Address Equate
SCC2        EQU     $0014                   ; SCI Control Register2
SCTIE       EQU     $07                     ; R/W
TCIE        EQU     $06                     ; R/W
SCRIE       EQU     $05                     ; R/W
ILIE        EQU     $04                     ; R/W
TE          EQU     $03                     ; R/W
RE          EQU     $02                     ; R/W
RWU         EQU     $01                     ; R/W
SBK         EQU     $00                     ; R/W
;
;/* Bit of SCI Control Register3
;/* SCC3 Bit Address Equate
SCC3        EQU     $0015                   ; SCI Control Register3
R8          EQU     $07                     ; R
T8          EQU     $06                     ; R/W
DMARE       EQU     $05                     ; R/W
DMATE       EQU     $04                     ; R/W
ORIE        EQU     $03                     ; R/W
NEIE        EQU     $02                     ; R/W
FEIE        EQU     $01                     ; R/W
PEIE        EQU     $00                     ; R/W
;
;/* Bit Of SCI Status Register1
;/* SCS1 Bit Address Equate
SCS1        EQU     $0016                   ; SCI Status Register1
SCTE        EQU     $07                     ; R
TC          EQU     $06                     ; R
SCRF        EQU     $05                     ; R
IDLE        EQU     $04                     ; R
OR          EQU     $03                     ; R
NF          EQU     $02                     ; R
FE          EQU     $01                     ; R
PE          EQU     $00                     ; R
;
;/* Bit of SCI Status Register2
;/* SCS2 Bit Address Equate
SCS2        EQU     $0017                   ; SCI Status Register2
BKF         EQU     $01                     ; R
RPF         EQU     $00                     ; R
;
SCDR        EQU     $0018                   ; SCI Data Register
;
;/* Bit of SCI Baudrate Register
;/* SCBR Bit Address Equate
SCBR        EQU     $0019                   ; SCI Baudrate Register
SCP1        EQU     $05                     ; R/W
SCP0        EQU     $04                     ; R/W
SCR2        EQU     $02                     ; R/W
SCR1        EQU     $01                     ; R/W
SCR0        EQU     $00                     ; R/W
;
;/* Bit of Keyboard Status & Control
;/* INTKBSCR Bit Address Equate
INTKBSCR    EQU     $001A                   ; Keyboard Status and Control Register
KEYF        EQU     $03                     ; R
ACKK        EQU     $02                     ; W
IMASKK      EQU     $01                     ; R/W
MODEK       EQU     $00                     ; R/W
;
;/* Bit of Keyboard Interrupt Enable
;/* INTKBIER Bit Address Equate
INTKBIER    EQU     $001B                   ; Keyboard Interrupt Enable Register
KBIE7       EQU     $07                     ; R/W
KBIE6       EQU     $06                     ; R/W
KBIE5       EQU     $05                     ; R/W
KBIE4       EQU     $04                     ; R/W
KBIE3       EQU     $03                     ; R/W
KBIE2       EQU     $02                     ; R/W
KBIE1       EQU     $01                     ; R/W
KBIE0       EQU     $00                     ; R/W
;
;/* Bit of Time Base Module Control
;/* TBCR Bit Address Equate
TBCR        EQU     $001C                   ; Time Base Module Control Register
TBIF        EQU     $07                     ; R
TBR2        EQU     $06                     ; R/W
TBR1        EQU     $05                     ; R/W
TBR0        EQU     $04                     ; R/W
TACK        EQU     $03                     ; W
TBIE        EQU     $02                     ; R/W
TBON        EQU     $01                     ; R/W
;
;/* Bit of IRQ Status & Control
;/* INTSCR Bit Address Equate
INTSCR      EQU     $001D                   ; IRQ Status and Control Register
IRQF1       EQU     $03                     ; R
ACK1        EQU     $02                     ; W
IMASK1      EQU     $01                     ; R/W
MODE1       EQU     $00                     ; R/W
;
;/* Bit of Config Register2
;/* CONFIG2 Bit Address Equate
CONFIG2     EQU     $001E                   ; Config Register2
OSCSTOPENB  EQU     $01                     ; R/W
SCIBDSRC    EQU     $00                     ; R/W
;
;/* Bit of Config Register1
;/* CONFIG1 Bit Address Equate
CONFIG1     EQU     $001F                   ; Config Register1
COPRS       EQU     $07                     ; R/W
LVISTOP     EQU     $06                     ; R/W
LVIRSTD     EQU     $05                     ; R/W
LVIPWRD     EQU     $04                     ; R/W
LVI5OR3     EQU     $03                     ; R/W
SSREC       EQU     $02                     ; R/W
STOP        EQU     $01                     ; R/W
COPD        EQU     $00                     ; R/W
;
;/* Bit of Timer Status & Control
;/* TxSC Bit Address Equate
T1SC        EQU     $0020                   ; TIM1 Status & Control Register
T2SC        EQU     $002B                   ; TIM2 Status & Control Register
TOF         EQU     $07                     ; R
TOIE        EQU     $06                     ; R/W
TSTOP       EQU     $05                     ; R/W
TRST        EQU     $04                     ; W
PS2         EQU     $02                     ; R/W
PS1         EQU     $01                     ; R/W
PS0         EQU     $00                     ; R/W
;
;/* Bit of Timer CH0 Status & Control Register
;/* TxSC0 Bit Address Equate
T1SC0       EQU     $0025                   ; TIM1 CH0 Status & Control Register
T2SC0       EQU     $0030                   ; TIM2 CH0 Status & Control Register
CH0F        EQU     $07                     ; R/W
CH0IE       EQU     $06                     ; R/W
MS0B        EQU     $05                     ; R/W
MS0A        EQU     $04                     ; R/W
ELS0B       EQU     $03                     ; R/W
ELS0A       EQU     $02                     ; R/W
TOV0        EQU     $01                     ; R/W
CH0MAX      EQU     $00                     ; R/W
;
;/* Bit of Timer CH1 Status & Control Register
;/* TxSC1 Bit Address Equate
T1SC1       EQU     $0028                   ; TIM1 CH1 Status & Control Register
T2SC1       EQU     $0033                   ; TIM2 CH1 Status & Control Register
CH1F        EQU     $07                     ; R/W
CH1IE       EQU     $06                     ; R/W
MS1A        EQU     $04                     ; R/W
ELS1B       EQU     $03                     ; R/W
ELS1A       EQU     $02                     ; R/W
TOV1        EQU     $01                     ; R/W
CH1MAX      EQU     $00                     ; R/W
;
T1CNTH      EQU     $0021                   ; TIM1 Counter High Register
T1CNTL      EQU     $0022                   ; TIM1 Counter Low Register
T1MODH      EQU     $0023                   ; TIM1 Counter Modulo High Register
T1MODL      EQU     $0024                   ; TIM1 Counter Modulo Low Register
T1CH0H      EQU     $0026                   ; TIM1 CH0 High Register
T1CH0L      EQU     $0027                   ; TIM1 CH0 Low Register
T1CH1H      EQU     $0029                   ; TIM1 CH1 High Register
T1CH1L      EQU     $002A                   ; TIM1 CH1 Low Register
T2CNTH      EQU     $002C                   ; TIM2 Counter High Register
T2CNTL      EQU     $002D                   ; TIM2 Counter Low Register
T2MODH      EQU     $002E                   ; TIM2 Counter Modulo High Register
T2MODL      EQU     $002F                   ; TIM2 Counter Modulo Low Register
T2CH0H      EQU     $0031                   ; TIM2 CH0 High Register
T2CH0L      EQU     $0032                   ; TIM2 CH0 Low Register
T2CH1H      EQU     $0034                   ; TIM2 CH1 High Register
T2CH1L      EQU     $0035                   ; TIM2 CH1 Low Register
;
;/* Bit of PLL Control Register
;/* PCTL Bit Address Equate
PCTL        EQU     $0036                   ; PLL Control Register
PLLIE       EQU     $07                     ; R/W
PLLF        EQU     $06                     ; R
PLLON       EQU     $05                     ; R/W
BCS         EQU     $04                     ; R/W
PRE1        EQU     $03                     ; R/W
PRE0        EQU     $02                     ; R/W
VPR1        EQU     $01                     ; R/W
VPR0        EQU     $00                     ; R/W
;
;/* Bit of PLL Bandwidth Control Register
;/* PBWC Bit Address Equate
PBWC        EQU     $0037                   ; PLL Bandwidth Control Register
AUTO        EQU     $07                     ; R/W
LOCK        EQU     $06                     ; R
ACQ         EQU     $05                     ; R/W
;
;/* Bit of PLL Multiply Select High
;/* PMSH Bit Address Equate
PMSH        EQU     $0038                   ; PLL Multiply Select High Register
MUL11       EQU     $03                     ; R/W
MUL10       EQU     $02                     ; R/W
MUL9        EQU     $01                     ; R/W
MUL8        EQU     $00                     ; R/W
;
;/* Bit of Multiply Select Low
;/* PMSL Bit Address Equate
PMSL        EQU     $0039                   ; PLL Multiply Select Low Register
MUL7        EQU     $07                     ; R/W
MUL6        EQU     $06                     ; R/W
MUL5        EQU     $05                     ; R/W
MUL4        EQU     $04                     ; R/W
MUL3        EQU     $03                     ; R/W
MUL2        EQU     $02                     ; R/W
MUL1        EQU     $01                     ; R/W
MUL0        EQU     $00                     ; R/W
;
;/* Bit of PLL VCO Select Range
;/* PMRS Bit Address Equate
PMRS        EQU     $003A                   ; PLL VCO Select Range Register
VRS7        EQU     $07                     ; R/W
VRS6        EQU     $06                     ; R/W
VRS5        EQU     $05                     ; R/W
VRS4        EQU     $04                     ; R/W
VRS3        EQU     $03                     ; R/W
VRS2        EQU     $02                     ; R/W
VRS1        EQU     $01                     ; R/W
VRS0        EQU     $00                     ; R/W
;
;/* Bit of PLL Reference Divider Select
;/* PMDS Bit Address Equate
PMDS        EQU     $003B                   ; PLL Reference Divider Select Register
RDS3        EQU     $03                     ; R/W
RDS2        EQU     $02                     ; R/W
RDS1        EQU     $01                     ; R/W
RDS0        EQU     $00                     ; R/W
;
;/* Bit of ADC Status & Control
;/* ADSCR Bit Address Equate
ADSCR       EQU     $003C                   ; ADC Status & Control Register
COCO        EQU     $07                     ; R/W
AIEN        EQU     $06                     ; R/W
ADCO        EQU     $05                     ; R/W
ADCH4       EQU     $04                     ; R/W
ADCH3       EQU     $03                     ; R/W
ADCH2       EQU     $02                     ; R/W
ADCH1       EQU     $01                     ; R/W
ADCH0       EQU     $00                     ; R/W
;
ADR         EQU     $003D                   ; ADC Data Register
;
;/* Bit of ADC Clock Register
;/* ADCLK Bit Address Equate
ADCLK       EQU     $003E                   ; ADC Clock Register
ADIV2       EQU     $07                     ; R/W
ADIV1       EQU     $06                     ; R/W
ADIV0       EQU     $05                     ; R/W
ADICLK      EQU     $04                     ; R/W
;
;/* Bit of SIM Break Status Register
;/* SBSR Bit Address Equate
SBSR        EQU     $FE00                   ; SIM Break Status Register
SBSW        EQU     $01                     ; R
;
;/* Bit of SIM Reset Status Register
;/* SRSR Bit Address Equate
SRSR        EQU     $FE01                   ; SIM Reset Status Register
POR         EQU     $07                     ; R/W
PIN         EQU     $06                     ; R/W
COP         EQU     $05                     ; R/W
ILOP        EQU     $04                     ; R/W
ILAD        EQU     $03                     ; R/W
MODRST      EQU     $02                     ; R/W
LVI         EQU     $01                     ; R/W
;
SUBAR       EQU     $FE02                   ; SIM Upper Byte Address Register
;
;/* Bit of SIM Break Flag Control
;/* SBFCR Bit Address Equate
SBFCR       EQU     $FE03                   ; SIM Break Flag Control Register
BCFE        EQU     $07                     ; R/W
;
;/* Bit of Interrupt Status Register1
;/* INT1 Bit Address Equate
INT1        EQU     $FE04                   ; Interrupt Status1 Register
IF6         EQU     $07                     ; R
IF5         EQU     $06                     ; R
IF4         EQU     $05                     ; R
IF3         EQU     $04                     ; R
IF2         EQU     $03                     ; R
IF1         EQU     $02                     ; R
;
;/* Bit of Interrupt Status Register2
;/* INT2 Bit Address Equate
INT2        EQU     $FE05                   ; Interrupt Status2 Register
IF14        EQU     $07                     ; R
IF13        EQU     $06                     ; R
IF12        EQU     $05                     ; R
IF11        EQU     $04                     ; R
IF10        EQU     $03                     ; R
IF9         EQU     $02                     ; R
IF8         EQU     $01                     ; R
IF7         EQU     $00                     ; R
;
;/* Bit of Interrupt Status Register3
;/* INT3 Bit Address Equate Equate
INT3        EQU     $FE06                   ; Interrupt Status3 Register
IF16        EQU     $01                     ; R
IF15        EQU     $00                     ; R
;
;/* Bit of Flash Control Register
;/* FLCR Bit Address Equate
FLCR        EQU     $FE08                   ; Flash Control Register
HVEN        EQU     $03                     ; R/W
MASS        EQU     $02                     ; R/W
ERASE       EQU     $01                     ; R/W
PGM         EQU     $00                     ; R/W
;
BRKH        EQU     $FE09                   ; Break Address High Register
BRKL        EQU     $FE0A                   ; Break Address Low Register
;
;/* Bit of Break Status & Control Register
;/* BRKSCR Bit Address Equate
BRKSCR      EQU     $FE0B                   ; Break Status and Control Register
BRKE        EQU     $07                     ; R/W
BRKA        EQU     $06                     ; R/W
;
;/* Bit of LVI Status Register
;/* LVISR Bit Address Equate
LVISR       EQU     $FE0C                   ; LVI Status Register
LVIOUT      EQU     $07                     ; R/W
;
;/* Bit of Flash Block Protect
;/* FLBPR Bit Address Equate
FLBPR       EQU     $FF7E                   ; Flash Block Protect Register
BPR7        EQU     $07                     ; R/W
BPR6        EQU     $06                     ; R/W
BPR5        EQU     $05                     ; R/W
BPR4        EQU     $04                     ; R/W
BPR3        EQU     $03                     ; R/W
BPR2        EQU     $02                     ; R/W
BPR1        EQU     $01                     ; R/W
BPR0        EQU     $00                     ; R/W
COPCTL      EQU     $FFFF                   ; Reset COP Register
;/*************************************/;
;/* End of IO Register Address Equate */;
;/* For Processor : MC68H(R)C908GP32  */;
;/*************************************/;

;/*************************************/;
;/* Start of Equate Hardware I/O Bit  */;
;/* For Standard Board : CP-JR08GP32  */;
;/*************************************/;
;
;/****************************/;
;/* LCD Interface With 4-Bit */;
;/* EN-LCD = PTC2            */;
;/* RS-LCD = PTC4            */;
;/* D4-LCD = PTD0            */;
;/* D5-LCD = PTD1            */;
;/* D6-LCD = PTD2            */;
;/* D7-LCD = PTD3            */;
;/* RW,D0..D3 of LCD = GND   */;
;/****************************/;
DATA_LCD    EQU     PTD                     ; PORTD = Data LCD
DLCD_DIR    EQU     DDRD                    ; Direction For Data LCD
CTRL_LCD    EQU     PTC                     ; PORTC = Control LCD
CLCD_DIR    EQU     DDRC                    ; Direction For Control LCD
EN_LCD      EQU     02H                     ; PTC2
RS_LCD      EQU     04H                     ; PTC4
D4_LCD      EQU     00H                     ; PTD0
D5_LCD      EQU     01H                     ; PTD1
D6_LCD      EQU     02H                     ; PTD2
D7_LCD      EQU     03H                     ; PTD3
;
;/****************************/;
;/* I2C I/O Device Interface */;
;/* I2C Signal Select Jumper */;
;/*     - SDA = PTC0 or PTB6 */;
;/*     - SCL = PTC1 or PTB7 */;
;/* This Equate to PTC0,PTC1 */;
;/****************************/;
PORT_I2C    EQU     PTC                     ; PORTC = I2C Signal
PDIR_I2C    EQU     DDRC                    ; Direction Read/Write
PULL_I2C    EQU     PTCPUE                  ; Pull-Up Enable
I2C_SDA     EQU     $00                     ; PTC0 = SDA
I2C_SCL     EQU     $01                     ; PTC1 = SCL
;
;/****************************/;
;/* RS-485 Direction Control */;
;/****************************/;
PORT_485    EQU     PTC                     ; PORTC = RS485 Direction Signal Control
PDIR_485    EQU     DDRC                    ; Direction Bit Control (Setup = Output)
POUT_485    EQU     $03                     ; PTC3 = RS485 Bit Signal Direction
;
;/****************************/;
;/* Matrix Keyboard Interface*/;
;/****************************/;
PORT_KEY    EQU     PTA                     ; PORTA = Keyboard Signal
PDIR_KEY    EQU     DDRA                    ; Direction In/Out
PULL_KEY    EQU     PTAPUE                  ; Pull-Up Enable
ROW0_KEY    EQU     $00                     ; PTA0 = Row0
ROW1_KEY    EQU     $01                     ; PTA1 = Row1
ROW2_KEY    EQU     $02                     ; PTA2 = Row2
ROW3_KEY    EQU     $03                     ; PTA3 = Row3
COL0_KEY    EQU     $04                     ; PTA4 = Column0
COL1_KEY    EQU     $05                     ; PTA5 = Column1
COL2_KEY    EQU     $06                     ; PAT6 = Column2
COL3_KEY    EQU     $07                     ; PTA7 = Column3 (Jumper Select = 4x4)
;
;/****************************/;
;/* Speaker or Buzzer Control*/;
;/*   (Jumper Select = 4x3)  */;
;/****************************/;
PORT_SPK    EQU     PTA                     ; PORTA = Speaker Signal Control
PDIR_SPK    EQU     DDRA                    ; Direction Bit Control (Setup = Output)
POUT_SPK    EQU     $07                     ; PTA7 = Speaker Bit Signal
;
;/*****************/;
;/* Relay Control */;
;/*****************/;
PORT_REL    EQU     PTB                     ; PORTB = Relay Signal Control
PDIR_REL    EQU     DDRB                    ; Direction Bit Control (Setup = Output)
POUT_REL    EQU     $07                     ; PTB7 = Speaker Bit Signal
;
;/****************************/;
;/* Magnetic Card Interface  */;
;/* Data Magnetic  = PTD4    */;
;/* Clock Magnetic = PTD5    */;
;/****************************/;
PORT_MAG    EQU     PTD                     ; PORTD = Magnetic Card Reader Signal
PDIR_MAG    EQU     DDRD                    ; Direction Read/Write
PULL_MAG    EQU     PTDPUE                  ; Pull-Up Enable
MAG_DATA    EQU     $04                     ; PTD4 = Data
MAG_CLK     EQU     $05                     ; PTD5 = Clock
;
;/************************/;
;/* RTC PCF8583 Register */;
;/************************/;
CTRL_RTC    EQU     $00                     ; Control/Status RTC
SEC_100     EQU     $01                     ; 1/100 of Second (00-99)
SECOND      EQU     $02                     ; Second (00-59)
MINUTE      EQU     $03                     ; Minute (00-59)
HOUR        EQU     $04                     ; Hour   (00-23) / 01-12(AM/PM)
YEAR_DATE   EQU     $05                     ; Year   (00-03) / Date(01-28,29,30,31)
WEEK_MONTH  EQU     $06                     ; Week   (00-06) / Month (01-12)
TIMER_RTC   EQU     $07                     ; Timer  (00-99)
;
ALARM_CTL   EQU     $08                     ; Alarm Control/Status
ALARM_SEC0  EQU     $09                     ; Alarm 1/100 Second
ALARM_SEC   EQU     $0A                     ; Alarm Second
ALARM_MIN   EQU     $0B                     ; Alarm Minute
ALARM_HOUR  EQU     $0C                     ; Alarm Hour
ALARM_DATE  EQU     $0D                     ; Alarm Date
ALARM_MONTH EQU     $0E                     ; Alarm Month
ALARM_TIMER EQU     $0F                     ; Alarm Timer
;
;/* Internal RTC RAM Address */;
RAM_START   EQU     $10                     ; $10..$FF = 240 Byte RAM
RAM_END     EQU     $FF
;
;/*************************************/;
;/*  End of Equate Hardware I/O Bit   */;
;/* For Standard Board : CP-JR08GP32  */;
;/*************************************/;


            ;/*************************/;
            ;/* RAM Buffer Start Here */;
            ;/* $0040-$023F = 512Byte */;
            ;/*************************/;
            ORG     $0040                   ; RAM Buffer : $0040-$023F
FLAG_BUF    DS      1
MAG_BUFF    DS      40
DUTY0_HI    DS      1                       ; PWM1 Duty Cycle
DUTY0_LO    DS      1
DUTY1_HI    DS      1                       ; PWM2 Duty Cycle
DUTY1_LO    DS      1
;
MAG_PASS    EQU     $00                     ; FLAG_BUF Bit-0
MAG_STRT    EQU     $01                     ; FLAG_BUF Bit-1
MAG_STOP    EQU     $01                     ; FLAG_BUF Bit-2
KEY_NEW     EQU     $03                     ; FLAG_BUF Bit-3

            ;/*******************************/;
            ;/* Usercode Program Start Here */;
            ;/* $8000-$FDFF = User Code     */;
            ;/* $FFDC-$FFFF = User Vector   */;
            ;/*******************************/;
            ORG     $8000                   ; Start Main (68HC908GP32 : $8000-$FDFF)
$BASE       10T                             ; Defualt System Number to Decimal
RESET:      BSET    0,CONFIG1               ; Disable COP Watch-Dog
            BSET    0,CONFIG2               ; SCI Baudrate Select From Internal Clock Bus
            LDHX    #$0240                  ; Last Address RAM = Stack (H:X=$0240)
            TXS                             ; [(H:X-1) -> SP] (SP = $023F)
            ;
            ;/*******************************/;
            ;/* Initial  PLL  For Clock Bus */;
            ;/* External X-TAL = 32.768 KHz */;
            ;/* Clock Bus Freq = 7.3728 MHz */;
            ;/*******************************/;
            ;
            BCLR    BCS,PCTL                ; Select External Clock
            BCLR    PLLON,PCTL              ; Turn-off PLL
            BCLR    PRE1,PCTL               ; P = 0
            BCLR    PRE0,PCTL
            BSET    VPR1,PCTL               ; E = 2
            BCLR    VPR0,PCTL
            MOV     #$01,PMDS               ; R = $01
            MOV     #$03,PMSH               ; N = $0384
            MOV     #$84,PMSL
            MOV     #$C0,PMRS               ; L = $C0
            LDA     PCTL                    ; Clear Status
            BSET    AUTO,PBWC               ; Auto Bandwidth Control
            BSET    PLLON,PCTL              ; Turn-on PLL
            BRCLR   LOCK,PBWC,*             ; Wait For PLL Lock Complete
            BSET    BCS,PCTL                ; Select PLL Output as Base Clock
            ;
            ;/*******************************/;
            ; Other User Program Start Here */;
            ;/*******************************/;
            ;
            ;/* Initial LCD Signal Direction */;
            BSET    RS_LCD,CLCD_DIR         ; Control LCD = Output
            BSET    EN_LCD,CLCD_DIR
            BSET    D4_LCD,DLCD_DIR         ; Data LCD = Output
            BSET    D5_LCD,DLCD_DIR
            BSET    D6_LCD,DLCD_DIR
            BSET    D7_LCD,DLCD_DIR
            ;
            ;/* Initial LCD Signal Logic */;
            BCLR    EN_LCD,CTRL_LCD         ; Initial Signal LCD
            BSET    RS_LCD,CTRL_LCD
            BSET    D4_LCD,DATA_LCD
            BSET    D5_LCD,DATA_LCD
            BSET    D6_LCD,DATA_LCD
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Initial I2C Signal Direction */;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output
            BSET    I2C_SDA,PULL_I2C        ; Enable Pull-up SDA
            BSET    I2C_SDA,PORT_I2C        ; SDA = "1"
            BSET    I2C_SCL,PORT_I2C        ; SCA = "1"
            ;
            ;/* Makesure I2C Stop Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Initial Bus For Stop I2C Device
            BCLR    I2C_SDA,PORT_I2C
            BSET    I2C_SCL,PORT_I2C
            BSET    I2C_SDA,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            ;
            ;/* Initial Magnetic Card Reader */;
            BCLR    MAG_DATA,PDIR_MAG       ; Data  = Input
            BCLR    MAG_CLK,PDIR_MAG        ; Clock = Input
            BSET    MAG_DATA,PULL_MAG       ; Enable Pull-Up Data
            BSET    MAG_CLK,PULL_MAG        ; Enable Pull-Up Clock
            ;
            ;/* Initial Keyboard & Speaker */;
            BCLR    ROW0_KEY,PDIR_KEY       ; Row0 = Input
            BCLR    ROW1_KEY,PDIR_KEY       ; Row1 = Input
            BCLR    ROW2_KEY,PDIR_KEY       ; Row2 = Input
            BCLR    ROW3_KEY,PDIR_KEY       ; Row3 = Input
            BSET    ROW0_KEY,PULL_KEY       ; Enable Row0 Pull-Up
            BSET    ROW1_KEY,PULL_KEY       ; Enable Row1 Pull-Up
            BSET    ROW2_KEY,PULL_KEY       ; Enable Row2 Pull-Up
            BSET    ROW3_KEY,PULL_KEY       ; Enable Row3 Pull-Up
            BSET    COL0_KEY,PDIR_KEY       ; Column0 = Output
            BSET    COL1_KEY,PDIR_KEY       ; Column1 = Output
            BSET    COL2_KEY,PDIR_KEY       ; Column2 = Output
            BSET    COL3_KEY,PDIR_KEY       ; Column3 = Output (Jumper Select = 4x4)
            BSET    POUT_SPK,PDIR_KEY       ; Speaker = Output (Jumper Select = 4x3)
            ;
            ;/* Initial RS485 Direction
            BSET    POUT_485,PDIR_485       ; Initial RS485 Signal = Output
            BCLR    POUT_485,PORT_485       ; Select Direction For Receive
            ;
            ;/*******************************/;
            ;/* Start Function Program Here */;
            ;/*******************************/;
            ;
            JSR     INIT_SCI                ; Initial RS232 (9600,N,8,1)
            JSR     PRINT_SER               ; Print Menu Function Test
            DB      $0D,$0A,$0A
            DB      'Demo Selftest For CP-JR08GP32 V2.0'
            DB      $0D,$0A
            DB      '0. Test PTA6 (LED Output Selftest)',$0D,$0A
            DB      '1. Test I2C Output (PCF8574A)',$0D,$0A
            DB      '2. Test I2C Input  (PCF8574A)',$0D,$0A
            DB      '3. Test I2C RTC    (PCF8583)',$0D,$0A
            DB      '4. Test Mini Speaker  (Beep)',$0D,$0A
            DB      '5. Test ADC 8Bit/8Ch',$0D,$0A
            DB      '6. Test I2C EEPROM (24C32/64/128/256/512)',$0D,$0A
            DB      '7. Test Character LCD 16x1',$0D,$0A
            DB      '8. Test Character LCD 16x2',$0D,$0A
            DB      '9. Test RS232/RS422 Receive/Transmit',$0D,$0A
            DB      'A. Test Magnetic Card Reader MCR-B02TTL & Relay',$0D,$0A
            DB      'B. Test Keyboard 4x3 + Speaker',$0D,$0A
            DB      'C. Test Keyboard 4x4',$0D,$0A
            DB      'D. Test Pulse Width Moduration PWM1',$0D,$0A
            DB      'E. Test Pulse Width Moduration PWM2',$0D,$0A
            DB      'F. Test RS485 Receive/Transmit',$0D,$0A,$0A
            DB      'Select (0..9,A..F) For Test...',$00
            ;
            ;/********************/;
            ;/* Wait Select Test */;
            ;/********************/;
GET_TEST:   JSR      RX_BYTE            ; Wait Command Test
            CBEQA    #'0',TEST_RUN0     ; PTA6 LED Output Selftest
            CBEQA    #'1',TEST_RUN1     ; PCF8574 Output
            CBEQA    #'2',TEST_RUN2     ; PCF8574 Input
            CBEQA    #'3',TEST_RUN3     ; PCF8583 RTC
            CBEQA    #'4',TEST_RUN4     ; Mini Speaker
            CBEQA    #'5',TEST_RUN5     ; ADC 8-Bit/8Ch
            CBEQA    #'6',TEST_RUN6     ; EEPROM 2432/64/128/512
            CBEQA    #'7',TEST_RUN7     ; LCD 16x1
            CBEQA    #'8',TEST_RUN8     ; LCD 16x2
            CBEQA    #'9',TEST_RUN9     ; RS232
            CBEQA    #'a',TEST_RUNA     ; Magnetic Card
            CBEQA    #'A',TEST_RUNA     ; Magnetic Card
            CBEQA    #'b',TEST_RUNB     ; Keyboard 4x3
            CBEQA    #'B',TEST_RUNB     ; Keyboard 4x3
            CBEQA    #'c',TEST_RUNC     ; Keyboard 4x4
            CBEQA    #'C',TEST_RUNC     ; Keyboard 4x4
            CBEQA    #'d',TEST_RUND     ; PWM1
            CBEQA    #'D',TEST_RUND     ; PWM1
            CBEQA    #'e',TEST_RUNE     ; PWM2
            CBEQA    #'E',TEST_RUNE     ; PWM2
            CBEQA    #'f',TEST_RUNF     ; RS485
            CBEQA    #'F',TEST_RUNF     ; RS485
            BRA      GET_TEST           ; Loop For Command Test

TEST_RUN0:  JMP      TEST_PA6           ; Test LED PTA6
TEST_RUN1:  JMP      TEST_OUT           ; Test PCF8574 Output
TEST_RUN2:  JMP      TEST_INP           ; Test PCF8574 Input
TEST_RUN3:  JMP      TEST_RTC           ; Test PCF8583 RTC
TEST_RUN4:  JMP      TEST_SPK           ; Test Mini Speaker
TEST_RUN5:  JMP      TEST_ADC           ; Test ADC 8 Bit / 8Ch
TEST_RUN6:  JMP      TEST_EEP           ; Test EEPROM
TEST_RUN7:  JMP      TEST_LCD1          ; Test LCD 16x1
TEST_RUN8:  JMP      TEST_LCD2          ; Test LCD 16x2
TEST_RUN9:  JMP      TEST_232           ; Test RS232
TEST_RUNA:  JMP      TEST_MAG           ; Test Magnetic Card Reader
TEST_RUNB:  JMP      TEST_KEY1          ; Test Keyboard 4x3 + Speaker
TEST_RUNC:  JMP      TEST_KEY2          ; Test Keyboard 4x4
TEST_RUND:  JMP      TEST_PWM1          ; Test PWM0
TEST_RUNE:  JMP      TEST_PWM2          ; Test PWM1
TEST_RUNF:  JMP      TEST_485           ; Test RS485

;/*****************/;
;/* PTA6 O/P Test */;
;/*****************/;
;
TEST_PA6:   JSR     PRINT_SER
            DB      $0D,$0A,$0A
            DB      'PTA6 (LED Output Selftest)'
            DB      $0D,$0A
            DB      'Short Jumper PTA6 For Test'
            DB      $0D,$0A,$00
            ;
            BSET    $06,DDRA            ; PTA6 = Output
            ;
LOOP_PA6:   BCLR    $06,PTA             ; ON LED
            LDA     #250                ; Delay 500mS
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            BSET    $06,PTA             ; OFF LED
            LDA     #250                ; Delay 500mS
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            BRA     LOOP_PA6            ; Loop Test PTA6

;/****************/;
;/* Test Speaker */;
;/****************/;
;
TEST_SPK:   JSR     PRINT_SER
            DB      $0D,$0A,$0A
            DB      'Mini Speaker Beep Output Test'
            DB      $0D,$0A
            DB      'Select Jumper 4X4/SPK = (SPK)'
            DB      $0D,$0A,$00
            ;
LOOP_SPK:   JSR     BEEP_SPK                ; Generate Sound Beep
            LDA     #250                    ; Delay 500mS
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            BRA     LOOP_SPK

;/*******************/;
;/* Test I2C Output */;
;/*******************/;
;
TEST_OUT:   JSR     PRINT_SER
            DB      $0D,$0A,$0A
            DB      'PCF8574A Output Port Sink Output Test'
            DB      $0D,$0A
            DB      'Connect 8-Bit Output to I2C(IN/OUT) Port'
            DB      $0D,$0A,$00
            ;
NEXT_OUT:   LDHX    #TAB_OUT8                   ; Table For Out Port
LOOP_OUT:   LDA     0,X
            JSR     OUT_WRITE
            LDA     #250                         ; Delay 500mS
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            AIX     #1                           ; Next Data
            CPHX    #TAB_OUT8+8                  ; 8 Byte Complete ?
            BNE     LOOP_OUT
            BRA     NEXT_OUT                     ; Restart Display

;/********************/;
;/* Table Output LED */;
;/********************/;
;
TAB_OUT8:   DB     %11111110                    ; Bit-0
            DB     %11111101                    ; Bit-1
            DB     %11111011                    ; Bit-2
            DB     %11110111                    ; Bit-3
            DB     %11101111                    ; Bit-4
            DB     %11011111                    ; Bit-5
            DB     %10111111                    ; Bit-6
            DB     %01111111                    ; Bit-7

;/******************/;
;/* Test I2C Input */;
;/******************/;
;
TEST_INP:   JSR     PRINT_SER
            DB      $0D,$0A,$0A
            DB      'PCF8574 Input Polling Read Mode Test'
            DB      $0D,$0A
            DB      'Connect 8-Bit Input to I2C(I/O) Port'
            DB      $0D,$0A,$00
            ;
LOOP_INP:   LDA     #$0D                    ; Carrier Return
            JSR     TX_BYTE
            ;
            JSR     INP_READ                ; Read PCF8574A Input
            ASLA                            ; C <- [D7...D0] <- 0
            ;
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-7
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-6
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-5
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-4
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-3
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-2
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-1
            PULA
            ;
            ASLA                             ; C <- [D7...D0] <- 0
            PSHA
            CLRA
            ADC     #$30                     ; Convert to ASCII
            JSR     TX_BYTE                  ; Display Bit-0
            PULA
            ;
            JMP     LOOP_INP                 ; New Test Input

;/*********************************/;
;/* Read Magnetic Card MCR-B02TTL */;
;/* - Display Data on SCI RS232   */;
;/*********************************/;
;
TEST_MAG:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A,$0A
            DB      'Magnetic Card Reader Model : MCR-B02TTL & Relay Test'
            DB      $0D,$0A
            DB      'Reading By Polling Data Read(Open Jumper INT)'
            DB      $0D,$0A,$00
            JSR     BEEP_SPK                ; Start Beep Test
            ;
            BSET    POUT_REL,PDIR_REL       ; Control Relay = Output
            BCLR    POUT_REL,PORT_REL       ; OFF Relay
            ;
WAIT_MAG:   JSR     READ_MAG                ; Polling Read Magnetic Card
            BRCLR   MAG_PASS,FLAG_BUF,WAIT_MAG
            ;
            ;/* Start Display Magnetic Code */;
            LDHX    #MAG_BUFF               ; Start Buffer Display
DSP_MAG1:   LDA     0,X                     ; Get 1St Byte Data
            CBEQA   #$0B,DSP_MAG2           ; Check Start Code at 1St Byte
            BCLR    MAG_PASS,FLAG_BUF
            BRA     WAIT_MAG                ; Restart New Read Data
            ;
            ;/* Start Display Data to RS232 */;
DSP_MAG2:   AIX     #1                      ; Next Data
            LDA     0,X
            CBEQA   #$0F,DSP_MAG3           ; Not Display Stop Code
            ADD     #$30                    ; Convert to ASCII
            JSR     TX_BYTE
            BRA     DSP_MAG2                ; Next Byte Display
            ;
DSP_MAG3:   LDA     #$0D                    ; Carrier Return
            JSR     TX_BYTE
            LDA     #$0A                    ; Line Feed
            JSR     TX_BYTE
            ;
            BCLR    MAG_PASS,FLAG_BUF       ; Clear Flag
            BSET    POUT_REL,PORT_REL       ; ON Relay
            JSR     BEEP_SPK                ; Beep on Exit
            BCLR    POUT_REL,PORT_REL       ; OFF Relay
            BRA     WAIT_MAG                ; Wait New Data

;/**********************/;
;/* Test RTC : PCF8583 */;
;/**********************/;
;
TEST_RTC:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Real Time Clock (RTC : PCF8583) Test'
            DB      $0D,$0A
            DB      'Reading By Polling (Open Jumper RTC)'
            DB      $0D,$0A,$00
            ;
LOOP_RTC:   LDA     #$0D                    ; Carrier Return
            JSR     TX_BYTE
            ;
            LDX     #HOUR                   ; Get Hour Time
            JSR     RTC_READ1
            PSHA
            NSA
            AND     #03H
            ADD     #30H
            JSR     TX_BYTE
            PULA
            AND     #0FH
            ADD     #30H
            JSR     TX_BYTE
            ;
            LDA     #':'                    ; Display Minute
            JSR     TX_BYTE
            ;
            LDX     #MINUTE                 ; Get Minute Time
            JSR     RTC_READ1
            PSHA
            NSA
            AND     #07H
            ADD     #30H
            JSR     TX_BYTE
            PULA
            AND     #0FH
            ADD     #30H
            JSR     TX_BYTE
            ;
            LDA     #':'                    ; Display Second
            JSR     TX_BYTE
            ;
            LDX     #SECOND                 ; Get Second Time
            JSR     RTC_READ1
            PSHA
            NSA
            AND     #07H
            ADD     #30H
            JSR     TX_BYTE
            PULA
            AND     #0FH
            ADD     #30H
            JSR     TX_BYTE
            ;
            BRA     LOOP_RTC                 ; Loop Display Time


;/*********************/;
;/* Test ADC 8Bit/8Ch */;
;/*********************/;
;
TEST_ADC:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Internal ADC 8-Bit/8Ch Test'
            DB      $0D,$0A
            DB      'Reading By.....Polling Read'
            DB      $0D,$0A,$00
            ;
            LDA     #%00000000              ; PortB = Input ADC
            STA     DDRB
            ;
            BCLR    ADIV2,ADCLK             ; 011 = ADC Clock Divide 8 (7.3728MHz/8)
            BSET    ADIV1,ADCLK
            BSET    ADIV0,ADCLK
            BSET    ADICLK,ADCLK            ; ADC Clock = Internal Bus Clock(7.3728 MHz)
            ;
LOOP_ADC:   LDA     #$0D                    ; Carrier Return
            JSR     TX_BYTE
            ;
            MOV     #%00000111,ADSCR        ; ADC-7 Single Convert & Disable Interrupt
WAIT_AD7:   BRCLR   COCO,ADSCR,WAIT_AD7     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-7 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000110,ADSCR        ; ADC-6 Single Convert & Disable Interrupt
WAIT_AD6:   BRCLR   COCO,ADSCR,WAIT_AD6     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-6 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000101,ADSCR        ; ADC-5 Single Convert & Disable Interrupt
WAIT_AD5:   BRCLR   COCO,ADSCR,WAIT_AD5     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-5 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000100,ADSCR        ; ADC-4 Single Convert & Disable Interrupt
WAIT_AD4:   BRCLR   COCO,ADSCR,WAIT_AD4     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-4 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000011,ADSCR        ; ADC-3 Single Convert & Disable Interrupt
WAIT_AD3:   BRCLR   COCO,ADSCR,WAIT_AD3     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-3 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000010,ADSCR        ; ADC-2 Single Convert & Disable Interrupt
WAIT_AD2:   BRCLR   COCO,ADSCR,WAIT_AD2     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-2 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000001,ADSCR        ; ADC-1 Single Convert & Disable Interrupt
WAIT_AD1:   BRCLR   COCO,ADSCR,WAIT_AD1     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-1 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            MOV     #%00000000,ADSCR        ; ADC-0 Single Convert & Disable Interrupt
WAIT_AD0:   BRCLR   COCO,ADSCR,WAIT_AD0     ; Wait COCO = "1"
            LDA     ADR                     ; Get ADC-0 Result
            JSR     TX_2ASC
            LDA     #$20                    ; Space
            JSR     TX_BYTE
            ;
            JMP     LOOP_ADC                ; New Test ADC

;/*****************/;
;/* Test LCD 16x1 */;
;/*****************/;
;
TEST_LCD1:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Character LCD (16x1) Display Test',$0D,$0A,$00
            JSR     INIT_LCD
            ;
LOOP_LCD1:  LDA     #$00                    ; Start Cursor Line1
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'Example ',$00
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      '4Bit LCD',$00
            ;
            LDA     #250                    ; Delay 2 Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            LDA     #$00                    ; Start Cursor Line1
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'CP-JR08G',$00
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'P32 V2.0',$00
            ;
            LDA     #250                    ; Delay 2 Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            JMP     LOOP_LCD1                ; Loop Test

;/*****************/;
;/* Test LCD 16x2 */;
;/*****************/;
;
TEST_LCD2:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Character LCD (16x2) Display Test',$0D,$0A,$00
            JSR     INIT_LCD
            ;
LOOP_LCD2:  LDA     #$00                    ; Start Cursor Line1
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'Example 4Bit LCD',$00
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'CP-JR08GP32 V2.0',$00
            ;
            LDA     #250                    ; Delay 2 Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            LDA     #$00                    ; Start Cursor Line1
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'CP-JR08GP32 V2.0',$00
            LDA     #$40                    ; Start Cursor Line2
            JSR     GOTO_LCD
            JSR     PRINT_LCD
            DB      'Example 4Bit LCD',$00
            ;
            LDA     #250                    ; Delay 2 Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            JMP     LOOP_LCD2                ; Loop Test

;/***************/;
;/* Test EEPROM */;
;/***************/;
;
TEST_EEP:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'I2C EEPROM (24C32/64/128/256/512) Test'
            DB      $0D,$0A
            DB      'Test EEPROM Read/Write...',$00
            ;
            LDHX    #0000H                  ; Test Write 55H to 0000H
            LDA     #055H
            JSR     EEP_WRITE
            LDHX    #0000H                  ; Read Address 0000H
            JSR     EEP_READ1
            CBEQA   #055H,TEST_EE1          ; Verify OK
            BRA     EEP_FAIL
            ;
TEST_EE1:   LDHX    #0000H                  ; Test Write AAH to 0000H
            LDA     #0AAH
            JSR     EEP_WRITE
            LDHX    #0000H                  ; Read Address 00FFH
            JSR     EEP_READ1
            CBEQA   #0AAH,TEST_EE2          ; Verify OK
            BRA     EEP_FAIL
            ;
TEST_EE2:   LDHX    #0FFFH                 ; Test Write 55H to 0FFFH
            LDA     #055H
            JSR     EEP_WRITE
            LDHX    #0FFFH                 ; Read Address 0FFFH
            JSR     EEP_READ1
            CBEQA   #055H,TEST_EE3         ; Verify OK
            BRA     EEP_FAIL
            ;
TEST_EE3:   LDHX    #0FFFH                 ; Test Write AAH to 0FFFH
            LDA     #0AAH
            JSR     EEP_WRITE
            LDHX    #0FFFH                 ; Read Address 0FFFH
            JSR     EEP_READ1
            CBEQA   #0AAH,EEP_PASS         ; Verify OK
            ;
EEP_FAIL:   JSR     PRINT_SER
            DB      'FAIL',00H
            BRA     $
            ;
EEP_PASS:   JSR     PRINT_SER
            DB      'PASS',00H
            BRA     $


;/*************************/;
;/* Test RS232/RS422 Echo */;
;/*************************/;
;
TEST_232:   JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'RS232/RS422 Receive & Transmit Test'
            DB      $0D,$0A
            DB      'Press Any Key For Test.......'
            DB      $0D,$0A,$00
            ;
LOOP_SCI:   JSR     RX_BYTE                 ; Receive RS232
            CBEQA   #$0D,TEST_232           ; If Enter Restart
            JSR     TX_BYTE                 ; Transmit RS232
            BRA     LOOP_SCI


;/***************************/;
;/* Test Keyboard 4x3 + SPK */;
;/***************************/;
;
TEST_KEY1:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Matrix Keyboard 4x3 + Speaker Test'
            DB      $0D,$0A,$00
            ;
GET_K4X3:   JSR     SCANK4X3                ; Scankey 4x3
            CBEQA   #$FF,GET_K4X3
            JSR     TX_2ASC
            LDA     #$0D
            JSR     TX_BYTE                 ; Carrier Return
            JSR     BEEP_SPK                ; Beep on Keypress
            BRA     GET_K4X3



;/*********************/;
;/* Test Keyboard 4x4 */;
;/*********************/;
;
TEST_KEY2:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Matrix Keyboard 4x4 Test'
            DB      $0D,$0A,$00
            ;
GET_K4X4:   JSR     SCANK4X4                ; Scankey 4x4
            CBEQA   #$FF,GET_K4X4
            JSR     TX_2ASC
            LDA     #$0D
            JSR     TX_BYTE                 ; Carrier Return
            BRA     GET_K4X4

;/*************/;
;/* Test PWM1 */;
;/*************/;
;
TEST_PWM1:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Pulse Width Modulation (PWM1) Test'
            DB      $0D,$0A,$00
            ;
            LDA     #%00000000              ; X0XXXXXX (Clear TOIE)
            ORA     #%00110000              ; XX11XXXX (Stop & Reset Timer Count)
            ORA     #%00000000              ; XXXXX000 (Prescale = 1)
            STA     T1SC                    ; Clear TOIE + Stop & Reset + Prescale:1
            ;
            LDA     #%01000000              ; X1XXXXXX (CH0IE = 1: Enable Interrupt)
            ORA     #%00010000              ; XX01XXXX (Mode Select = Unbuffer PWM)
            ORA     #%00001000              ; XXXX10XX (Clear Output on Compare)
            ORA     #%00000010              ; XXXXXX1X (Toggle on Overflow)
            STA     T1SC0
            ;
            ;/* Define Period = 2.0mS */;
            MOV     #$39,T1MODH             ; Define Period of PWM ($399A = 2.0mS)
            MOV     #$9A,T1MODL
            ;
            ;/* Start Duty Cycle 1ms of 2.0mS */;
            MOV     #$1C,DUTY0_HI           ; 1.0ms Duty Cycle
            MOV     #$CD,DUTY0_LO
            MOV     #$1C,T1CH0H             ; Define Duty Cycle of PWM
            MOV     #$CD,T1CH0L
            ;
            CLI                             ; Enable CPU Interrupt
            BCLR    TSTOP,T1SC              ; Enable & Strat Timer(TSTOP)
            ;
            ;/* Start Generate PWM */;
LOOP_PWM1:  MOV     #$1C,DUTY0_HI           ; 1.0ms Duty Cycle
            MOV     #$CD,DUTY0_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            MOV     #$2B,DUTY0_HI           ; 1.5mS Duty Cycle
            MOV     #$33,DUTY0_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            MOV     #$39,DUTY0_HI           ; 2.0mS Duty Cycle
            MOV     #$9A,DUTY0_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            BRA     LOOP_PWM1               ; Loop Adjust Duty Cycle


;/*************/;
;/* Test PWM2 */;
;/*************/;
;
TEST_PWM2:  JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'Pulse Width Modulation (PWM2) Test'
            DB      $0D,$0A,$00
            ;
            LDA     #%00000000              ; X0XXXXXX (Clear TOIE)
            ORA     #%00110000              ; XX11XXXX (Stop & Reset Timer Count)
            ORA     #%00000000              ; XXXXX000 (Prescale = 1)
            STA     T1SC                    ; Clear TOIE + Stop & Reset + Prescale:1
            ;
            LDA     #%01000000              ; X1XXXXXX (CH1IE = 1: Enable Interrupt)
            ORA     #%00010000              ; XX01XXXX (Mode Select = Unbuffer PWM)
            ORA     #%00001000              ; XXXX10XX (Clear Output on Compare)
            ORA     #%00000010              ; XXXXXX1X (Toggle on Overflow)
            STA     T1SC1
            ;
            ;/* Define Period = 2.0mS */;
            MOV     #$39,T1MODH             ; Define Period of PWM ($399A = 2.0mS)
            MOV     #$9A,T1MODL
            ;
            ;/* Start Duty Cycle 1ms of 2.0mS */;
            MOV     #$1C,DUTY1_HI           ; 1.0ms Duty Cycle
            MOV     #$CD,DUTY1_LO
            MOV     #$1C,T1CH1H             ; Define Duty Cycle of PWM
            MOV     #$CD,T1CH1L
            ;
            CLI                             ; Enable CPU Interrupt
            BCLR    TSTOP,T1SC              ; Enable & Strat Timer(TSTOP)
            ;
            ;/* Start Generate PWM */;
LOOP_PWM2:  MOV     #$1C,DUTY1_HI           ; 1.0ms Duty Cycle
            MOV     #$CD,DUTY1_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            MOV     #$2B,DUTY1_HI           ; 1.5mS Duty Cycle
            MOV     #$33,DUTY1_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            ;
            MOV     #$39,DUTY1_HI           ; 2.0mS Duty Cycle
            MOV     #$9A,DUTY1_LO
            LDA     #250                    ; Delay 1-Second
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            LDA     #250
            JSR     DLY_MSEC
            BRA     LOOP_PWM2               ; Loop Adjust Duty Cycle

;/*******************************/;
;/* Test RS485 Receive/Transmit */;
;/*******************************/;
;
TEST_485:   BSET    POUT_485,PDIR_485       ; Initial RS485 Signal = Output Direction
            BSET    POUT_485,PORT_485       ; Transmit Direction
            ;
            JSR     PRINT_SER               ; Display Sign-on Logo
            DB      $0D,$0A,$0A
            DB      'RS485 Receive & Transmit Test'
            DB      $0D,$0A
            DB      'Press [*][0][1][Ent] For Test'
            DB      $0D,$0A,$00
            LDA     #2                      ; Delay 2mS For Last Byte Send Complete
            JSR     DLY_MSEC
            BCLR    POUT_485,PORT_485       ; Select Direction For Receive
            ;
            ;/***************************/;
            ;/* Wait Receive Frame Data */;
            ;/* Defualt ID = 01         */;
            ;/* Start,ID0,ID1,Stop      */;
            ;/* Start = '*'   ($2A)     */;
            ;/* ID    = '01'  ($30,$31) */;
            ;/* End   = Enter ($0D)     */;
            ;/* Frame =  *   0   1  Ent */;
            ;/*       = $2A,$30,$31,$0D */;
            ;/***************************/;
            ;
WAIT_485:   JSR     RX_BYTE                 ; Start Frame Data = *
            CMP     #'*'
            BNE     WAIT_485
            JSR     RX_BYTE                 ; ID Code High Byte = 0
            CMP     #'0'
            BNE     WAIT_485
            JSR     RX_BYTE                 ; ID Code Low Byte = 1
            CMP     #'1'
            BNE     WAIT_485
            JSR     RX_BYTE                 ; End of Frame = Enter ($0D)
            CMP     #$0D
            BNE     WAIT_485
            ;
            ;/* Receive Frame Data OK */;
            ;/* Echo Data to Master   */;
ECHO_485:   BSET    POUT_485,PORT_485       ; Select Direction For Transmit
            JSR     PRINT_SER               ; Display Sign-on Logo
            DB      'Station : 01 Receive Data OK'
            DB      $0D,$0A,$00
            LDA     #2                      ; Delay 2mS For Send Last Byte Complete
            JSR     DLY_MSEC
            BCLR    POUT_485,PORT_485       ; Select Direction For Receive
            BRA     WAIT_485


;/**********************************/;
;/* Start Subroutine For 4-Bit LCD */;
;/* -INIT_LCD = Initial 4-Bit LCD  */;
;/* -ENA_LCD  = Active Enable LCD  */;
;/* -BUSY     = Delay Wait LCD     */;
;/* -WR_LCD   = Write ASCII to LCD */;
;/* -WR_INS   = Write Command LCD  */;
;/* -GOTO_LCD = Set Cursor LCD     */;
;/* -PRINT_LCD= Print Display LCD  */;
;/**********************************/;
;
;/********************/;
;/* Initial Char LCD */;
;/* 4-Bit Interface  */;
;/* RS    = Bit-3    */;
;/* EN    = Bit-2    */;
;/* D4..7 = Bit-0..3 */;
;/********************/;
;
INIT_LCD:   LDA     #33H                    ; DL = 1 ,DL =1
            JSR     WR_INS
            LDA     #32H                    ; DL = 1 ,DL =0
            JSR     WR_INS
            LDA     #28H                    ; Function set
            JSR     WR_INS                  ; DL=0(4Bit),N=1(2Line),F=0(5X7)
            LDA     #0CH                    ; Display on/off Control
            JSR     WR_INS                  ; Entry Display,Cursor off,Not Blink
            LDA     #06H                    ; Entry Mode Set
            JSR     WR_INS                  ; I/D=1 Increment,S=0 Cursor Shift
CLR_LCD:    LDA     #01H                    ; Clear Display
            JSR     WR_INS                  ; Clear Display,Set DD RAM Addr=0
            RTS

;/**********************/;
;/* Enable Pin EN CLCD */;
;/* Pin EN LCD = Bit-2 */;
;/* Active Chip select */;
;/**********************/;
;
ENA_LCD:    JSR     BUSY                    ; Busy Delay
            BSET    EN_LCD,CTRL_LCD         ; Enable Chips Select
            ;
            ;/* Delay 10uS (76-Cycle)
            PSHX                            ; 2 Cycle
            LDX     #18                     ; 2 Cycle
ENA_LCD0:   NOP                             ; 1 Cycle
            DBNZX   ENA_LCD0                ; 3 Cycle (18x4 = 72)
            PULX                            ; 2 Cycle
            ;
            BCLR    EN_LCD,CTRL_LCD         ; Disable Chips Select
            RTS

;/*******************/;
;/* Busy CLCD Delay */;
;/* Default = 100uS */;
;/* Fbus = 7.3728Hz */;
;/* 100uS = 737Cycle*/;
;/*******************/;
;
BUSY:       PSHX                            ; 2 Cycle
            LDX     #182                    ; 2 Cycle
BUSY1:      NOP                             ; 1 Cycle
            DBNZX   BUSY1                   ; 3 Cycle (182x4 =728)
            PULX                            ; 2 Cycle
            RTS                             ; 4 Cycle

;/**********************/;
;/* Write ASCII to LCD */;
;/* Input : ACC(ASCII) */;
;/**********************/;
;
WR_LCD:     PSHA                            ; Save ACC(ASCII)
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            ;/* Send High Nibble */;
            LDA     2,SP                    ; Get ACC(ASCII)
            AND     #$F0                    ; High Nibble
            NSA                             ; Nibble Swap A (D7..D4 <-> D3..D0)
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD01
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_LCD01:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD02
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_LCD02:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD03
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_LCD03:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD04
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe High Nibble Data */;
WR_LCD04:   BSET    RS_LCD,CTRL_LCD         ; Set RS For Write Data Select
            JSR     ENA_LCD
            ;
            ;/* Send Low Nibble */;
            LDA     2,SP                    ; Get ACC (ASCII)
            AND     #$0F                    ; Low Nibble
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD05
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_LCD05:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD06
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_LCD06:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD07
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_LCD07:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_LCD08
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe Low Nibble Data */;
WR_LCD08:   BSET    RS_LCD,CTRL_LCD         ; Set RS For Write Data Select
            JSR     ENA_LCD
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULA                            ; Restore ACC(ASCII)
            RTS

;/**************************/;
;/* Write Instruction LCD  */;
;/* Input  : ACC (Command) */;
;/* Output : Data bus LCD  */;
;/**************************/;
;
WR_INS:     PSHA                            ; Save ACC(Command)
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            ;/* Send High Nibble */;
            LDA     2,SP                    ; Get ACC(Command)
            AND     #$F0                    ; High Nibble
            NSA                             ; Nibble Swap A (D7..D4 <-> D3..D0)
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS01
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_INS01:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS02
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_INS02:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS03
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_INS03:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS04
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe High Nibble Command */;
WR_INS04:   BCLR    RS_LCD,CTRL_LCD         ; Clear RS For Write Instruction
            JSR     ENA_LCD
            ;
            ;/* Send Low Nibble */;
            LDA     2,SP                    ; Get ACC (Command)
            AND     #$0F                    ; Low Nibble
            BCLR    D4_LCD,DATA_LCD         ; Clear old Signal
            BCLR    D5_LCD,DATA_LCD
            BCLR    D6_LCD,DATA_LCD
            BCLR    D7_LCD,DATA_LCD
            ;
            ;/* Update D4-LCD */;
            LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS05
            BSET    D4_LCD,DATA_LCD
            ;
            ;/* Update D5-LCD */;
WR_INS05:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS06
            BSET    D5_LCD,DATA_LCD
            ;
            ;/* Update D6-LCD */;
WR_INS06:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS07
            BSET    D6_LCD,DATA_LCD
            ;
            ;/* Update D7-LCD */;
WR_INS07:   LSRA                            ; [0] -> [ACC] -> [CY]
            BCC     WR_INS08
            BSET    D7_LCD,DATA_LCD
            ;
            ;/* Strobe Low Nibble Command */;
WR_INS08:   BCLR    RS_LCD,CTRL_LCD         ; Clear RS For Write Instruction
            JSR     ENA_LCD
            ;
            LDA     #2                      ; Wait Command Ready = 2mS
            JSR     DLY_MSEC
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULA                            ; Restore ACC(Command)
            RTS

;/************************/;
;/* Goto position of LCD */;
;/* Input : ACC (addr.)  */;
;/************************/;
;
GOTO_LCD:   ORA     #$80                    ; Set ACC.7
            JSR     WR_INS
            RTS

;/********************************/;
;/* Print ASCII Data to LCD Port */;
;/* Usage    : JSR  PRINT_LCD    */;
;/*          : DB   'xxxxxxxx'   */;
;/*          : DB   $00          */;
;/* Note     : Last Byte = $00   */;
;/********************************/;
;
PRINT_LCD:  PULH                        ; Pop Stack PCH
            PULX                        ; Pop Stack PCL
            PSHA                        ; Save ACC
            ;
PRN_LCD1:   LDA     0,X                 ; Get Data For Print
            BEQ     PRN_LCD2            ; If Last Byte Jump
            JSR     WR_LCD              ; Write Data to LCD
            AIX     #1                  ; Inc (H:X)
            BRA     PRN_LCD1
            ;
PRN_LCD2:   AIX     #1                  ; Skip Last Byte(00)
            ;
            PULA                        ; Restore ACC
            PSHX                        ; Push PCL to Stack
            PSHH                        ; Push PCH to Stack
            RTS

;/***********************************/;
;/* End of Subroutine For 4-Bit LCD */;
;/***********************************/;


;/***********************************/;
;/* Start Subroutine For I2C Device */;
;/* -OUT_WRITE = Out Port PCF8574   */;
;/* -INP_READ  = Read Port PCF8574  */;
;/* -RTC_WRITE = Write RTC PCF8583  */;
;/* -RTC_READ1 = Read RTC PCF8583   */;
;/* -EEP_WRITE = Write EEP 24C32... */;
;/* -EEP_READ1 = Read EEP 24C32...  */;
;/* -I2C_WRLF  = Shift Bit Out (LSB)*/;
;/* -I2C_RDLF  = Shift Bit In (LSB) */;
;/***********************************/;
;
;/*****************************/;
;/* I2C-BUS PCF8574 Interface */;
;/* Write Data to  PCF8574/A  */;
;/* Input : A (Data)          */;
;/*    (Pin A2,A1,A0 = "0")   */;
;/*****************************/;
;
OUT_WRITE:  PSHA                            ; Save Data to Write
            PSHX                            ; Save X
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
OUT_WR0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte (Write) */;
            ;/* PCF8574  = 0100XXX0 (XXX = Pin A2,A1,A0)
            ;/* PCF8574A = 0111XXX0 (XXX = Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%01110000              ; 0111000,Write (PCF8574A : Pin A2,A1,A0 = 000)
            ;
OUT_WR1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   OUT_WR1
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,OUT_WR2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     OUT_WR0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
            ;/* Data For Output Port (A) */;
OUT_WR2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     3,SP                    ; Get ACC (Data)
            ;
OUT_WR3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   OUT_WR3
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Release Data Line
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            ;
            ;/* Stop Clock Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULX                            ; Restore X
            PULA                            ; Restore Data to Write
            RTS

;/*****************************/;
;/* I2C-BUS PCF8574 Interface */;
;/* Read Data From PCF8574/A  */;
;/* Output  : A (Data)        */;
;/*    (Pin A2,A1,A0 = "0")   */;
;/*****************************/;
;
INP_READ:   AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save EEP Address
            TPA                             ; CCR -> ACC
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
INP_RD0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte (Write) */;
            ;/* PCF8574  = 0100XXX1 (XXX = Pin A2,A1,A0)
            ;/* PCF8574A = 0111XXX1 (XXX = Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%01110001              ; 0111000,Read (PCF8574A : Pin A2,A1,A0 = 000)
            ;
INP_RD1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   INP_RD1                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,INP_RD2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     INP_RD0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Read Input Port */;
INP_RD2:    LDX     #8                      ; Read Data 8-Bit Count
            CLRA                            ; Clera Buffer For Read
            ;
INP_RD3:    BCLR    I2C_SCL,PORT_I2C        ;/* Wait to Receive SDA Signal */;
            JSR     I2C_RDLF                ; SDA Shift-IN D7..D0 (D0 First)
            DBNZX   INP_RD3                 ; Loop to Read Data 8-Bit
            STA     3,SP                    ; Save Result
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Not ACK At Stop Read
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Send ACK */;
            ;
            ;/* Stop Clock Condition */;
            BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; ACC -> CCR
            PULX                            ; Restore EEP Address
            PULA                            ; Get Result
            RTS

;/*****************************/;
;/* I2C-RTC PCF8583 Interface */;
;/* Write Data to PCF8583 RTC */;
;/* Input : X (Address)       */;
;/*       : A (Data)          */;
;/*   (Pin A0 = Logic "1")    */;
;/*****************************/;
;
RTC_WRITE:  PSHA                            ; Save Data to Write
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
RTC_WR0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte (Write) */;
            ;/* PCF8583 = 101000X0 (X = Pin A0)
            ;/* Pin A0 = "0" = 10100000
            ;/* Pin A0 = "1" = 10100010
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100010              ; 1010001,Write (Pin A0 = "1")
            ;
RTC_WR1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR1
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_WR2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_WR0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
            ;/* Byte Address (X) */;
RTC_WR2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get RTC Address
            ;
RTC_WR3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR3
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_WR4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_WR6                 ; Exit
            ;
            ;/* Ready to Start 3rd Byte Send */;
            ;/* Data for Write (A) */;
RTC_WR4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Data 8-Bit Count
            LDA     3,SP                    ; Get Data For Write
            ;
RTC_WR5:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_WR5
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Release Data Line
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            ;
RTC_WR6:    BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore RTC Address
            PULA                            ; Restore RTC Data
            RTS

;/*****************************/;
;/* I2C-BUS PCF8583 Interface */;
;/* Read Data From PCF8583    */;
;/* -RTC_READ1 = Random Read  */;
;/*  Input  : X (Address)     */;
;/*  Output : A (Data)        */;
;/* -RTC_READ2 = Current Read */;
;/*  Output : ACC (Data)      */;
;/*   (Pin A0 = Logic "1")    */;
;/*****************************/;
;
RTC_READ1:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
RTC_RD0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte (Write) */;
            ;/* PCF8583 = 101000X0 (X = Pin A0)
            ;/* Pin A0 = "0" = 10100000
            ;/* Pin A0 = "1" = 10100010
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100010              ; 1010001,Write (Pin A0 = "1")
            ;
RTC_RD1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD1                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
            ;/* Address Byte (X) */;
RTC_RD2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get RTC Address
            ;
RTC_RD3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD3                 ; Loop to Write Addr High-Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD10                ; Exit
            ;
            ;/* New Start For Send Control Byte Read Command */;
            ;/* Control Byte (Read) */;
RTC_RD4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BCLR    I2C_SCL,PORT_I2C        ; Falling New Start Clock
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Control Byte (Read) */;
            ;/* PCF8583 = 101000X1 (X = Pin A0)
            ;/* Pin A0 = "0" = 10100001
            ;/* Pin A0 = "1" = 10100011
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100011              ; 1010001,Read (Pin A0 = "1")
            BRA     RTC_RD7
            ;
            ;/* Current Address Read Operation */;
RTC_READ2:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHX                            ; Save RTC Address
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Control Byte (Read) */;
            ;/* PCF8583 = 101000X1 (X = Pin A0)
            ;/* Pin A0 = "0" = 10100001
            ;/* Pin A0 = "1" = 10100011
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10100011              ; 1010001,Read (Pin A0 = "1")
            ;
RTC_RD7:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   RTC_RD7                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,RTC_RD8
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     RTC_RD10                ; Exit
            ;
            ;/* Ready to Read Data From RTC */;
RTC_RD8:    LDX     #8                      ; Read Data 8-Bit Count
            CLRA                            ; Clera Buffer For Read
            ;
RTC_RD9:    BCLR    I2C_SCL,PORT_I2C        ;/* Wait to Receive SDA Signal */;
            JSR     I2C_RDLF                ; SDA Shift-IN D7..D0 (D0 First)
            DBNZX   RTC_RD9                 ; Loop to Read Data 8-Bit
            STA     3,SP                    ; Save Result
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Not ACK At Stop Read
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Send ACK */;
            ;
RTC_RD10:   BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore RTC Address
            PULA                            ; Get Result to ACC
            RTS

;/*****************************/;
;/* I2C-EEPROM(24C32..24C515) */;
;/* Write Data to  I2C EEPROM */;
;/* Input : H:X (Address)     */;
;/*       : A (Data)          */;
;/*****************************/;
;
EEP_WRITE:  PSHA                            ; Save Data to Write
            PSHH                            ; Save EEP Address
            PSHX
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
EEP_WR0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte Code (Write) */;
            ;/* 24C32/64/128/515 = 1010XXX0  (XXX = Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10101000              ; 1010100,Write (Pin A2 = "1",A1 & A0 = 00)
            ;
EEP_WR1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_WR1
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_WR2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_WR0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
            ;/* High Byte Address Word (H) */;
EEP_WR2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     3,SP                    ; Get EEP Address High(H)
            ;
EEP_WR3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_WR3
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_WR4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_WR8                 ; Exit
            ;
            ;/* Ready to Start 3rd Byte Send */;
            ;/* Low Byte Address Word (X) */;
EEP_WR4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get EEP Address Low(X)
            ;
EEP_WR5:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_WR5
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_WR6
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_WR8                 ; Exit
            ;
            ;/* Ready to Start 4th Byte Send */;
            ;/* Data For Write (A) */;
EEP_WR6:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Data 8-Bit Count
            LDA     4,SP                    ; Get Data For Write
            ;
EEP_WR7:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_WR7
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Release Data Line
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            ;
EEP_WR8:    BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore EEP Address
            PULH
            PULA                            ; Restore EEP Data
            RTS

;/*****************************/;
;/* I2C-EEPROM(24C32..24C515) */;
;/* Read Data From I2C EEPROM */;
;/* -EEP_READ1 = Random Read  */;
;/*  Input  : H:X (Address)   */;
;/*  Output : A (Data)        */;
;/* -EEP_READ2 = Current Read */;
;/*  Output : ACC (Data)      */;
;/*****************************/;
;
EEP_READ1:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHH                            ; Save EEP Address
            PSHX
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            ;
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
EEP_RD0:    BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send */;
            ;/* Control Byte (Write) */;
            ;/* 24C32/64/128/256/515 = 1010XXX0 (XXX = Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10101000              ; 1010100,Write (Pin A2 = "1",A1 & A0 = 00)
            ;
EEP_RD1:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_RD1                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_RD2
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_RD0                 ; Resend Control Byte Until ACK
            ;
            ;/* Ready to Start 2nd Byte Send */;
            ;/* Address High Byte(H) */;
EEP_RD2:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     3,SP                    ; Get EEP Address High(H)
            ;
EEP_RD3:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_RD3                 ; Loop to Write Addr High-Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_RD4
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_RD10                ; Exit
            ;
            ;/* Ready to Start 3nd Byte Send */;
            ;/* Address Low Byte(X) */;
EEP_RD4:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            LDX     #8                      ; Write Address Count
            LDA     2,SP                    ; Get EEP Address Low(X)
            ;
EEP_RD5:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_RD5                 ; Loop to Write Addr Low-Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_RD6
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_RD10                ; Exit
            ;
            ;
            ;/* New Start For Send Control Byte Read of Random Read */;
            ;/* Control Byte (Read) */;
EEP_RD6:    BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BCLR    I2C_SCL,PORT_I2C        ; Falling New Start Clock
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 4th Byte Send */;
            ;/* Control Byte Read */;
            ;/* 24C32/64/128/256/515 = 1010XXX1 (XXX = Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10101001              ; 1010100,Read (Pin A2 = "1", A1 & A0 = 00)
            BRA     EEP_RD7
            ;
            ;/**********************************/;
            ;/* Current Address Read Operation */;
            ;/**********************************/;
EEP_READ2:  AIS     #-1                     ; Reserve Stack 1 Byte
            PSHH                            ; Save EEP Address
            PSHX
            TPA                             ; CCR -> A
            PSHA                            ; Save CCR
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BSET    I2C_SCL,PDIR_I2C        ; SCL = Output Direction
            ;
            BSET    I2C_SCL,PORT_I2C        ; Initial Bus For Start
            BSET    I2C_SDA,PORT_I2C
            BCLR    I2C_SDA,PORT_I2C        ; Start Condition THD:STA(600nS)
            BCLR    I2C_SCL,PORT_I2C
            ;
            ;/* Ready to Start 1st Byte Send of Current Read */;
            ;/* Control Byte (Read) */;
            ;/* 24C32/64/128/256/515 = 1010XXX1 (Pin A2,A1,A0)
            LDX     #8                      ; Write Control Byte 8-Bit Count
            LDA     #%10101001              ; 1010100,Read (Pin A2 = "1", A1 & A0 = 00)
            ;
EEP_RD7:    BCLR    I2C_SCL,PORT_I2C        ;/* Change Data to SDA Signal */;
            JSR     I2C_WRLF                ; SDA Shift-Out D7..D0 (D7 First)
            DBNZX   EEP_RD7                 ; Loop to Write Control Byte
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read ACK)
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Get ACK */;
            BRCLR   I2C_SDA,PORT_I2C,EEP_RD8
            BSET    I2C_SDA,PDIR_I2C        ; SDA = Output Direction
            BRA     EEP_RD10                ; Exit
            ;
            ;/* Ready to Receive Data */;
            ;/* Data Read From EEPROM */;
EEP_RD8:    LDX     #8                      ; Read Data 8-Bit Count
            CLRA                            ; Clera Buffer For Read
            ;
EEP_RD9:    BCLR    I2C_SCL,PORT_I2C        ;/* Wait to Receive SDA Signal */;
            JSR     I2C_RDLF                ; SDA Shift-IN D7..D0 (D0 First)
            DBNZX   EEP_RD9                 ; Loop to Read Data 8-Bit
            STA     4,SP                    ; Save Result
            ;
            BCLR    I2C_SCL,PORT_I2C        ;*/ Falling 9'th Clock For ACK */;
            BSET    I2C_SDA,PORT_I2C        ; Not ACK At Stop Read
            BSET    I2C_SCL,PORT_I2C        ;*/ Rising 9'th Clock for Send ACK */;
            ;
EEP_RD10:   BCLR    I2C_SCL,PORT_I2C        ; Falling Stop Clock
            BCLR    I2C_SDA,PORT_I2C        ; Wait For Stop
            BSET    I2C_SCL,PORT_I2C        ; Stop Condition (SCL=1,SDA=1)
            BSET    I2C_SDA,PORT_I2C
            ;
            PULA                            ; Restore CCR
            TAP                             ; A -> CCR
            PULX                            ; Restore EEP Address
            PULH
            PULA                            ; Get Result to ACC
            RTS

;/**************************/;
;/* Write SDA & Shift Left */;
;/* Input : ACC            */;
;/**************************/;
;
I2C_WRLF:   ROLA                            ; Shift D7..D0 (D7 First)
            BCS     I2C_WRL0                ; Jump if Data = "1"
            BCLR    I2C_SDA,PORT_I2C        ; SDA = "0"
            BRA     I2C_WRL1
I2C_WRL0:   BSET    I2C_SDA,PORT_I2C        ; SDA = "1"
I2C_WRL1:   NOP
            BSET    I2C_SCL,PORT_I2C        ; Shift Data to I2C Device
            RTS

;/*************************/;
;/* Read SDA & Shift Left */;
;/* Output : ACC          */;
;/*************************/;
;
I2C_RDLF:   BCLR    I2C_SDA,PDIR_I2C        ; SDA = Input (Read)
            BSET    I2C_SCL,PORT_I2C        ; Read Data From I2C-Device
            BRSET   I2C_SDA,PORT_I2C,I2C_RDL0
            CLC                             ; C = "0"
            BRA     I2C_RDL1
I2C_RDL0:   SEC                             ; C = "1"
I2C_RDL1:   ROLA                            ; Shift D7..D0 (D7 First)
            BSET    I2C_SCL,PORT_I2C        ; Shift Data to I2C Device
            RTS

;/************************************/;
;/* End of Subroutine For I2C Device */;
;/************************************/;


;/***********************************/;
;/* Start Subroutine For SCI(RS232) */;
;/* -INIT_SCI = Initial 9600,N,8,1  */;
;/* -TX_BYTE  = Transmit 1 Byte     */;
;/* -RX_BYTE  = Receive 1 Byte      */;
;/***********************************/;
;
;/*************************************/;
;/* Initial SCI Port = Baudrate,N,8,1 */;
;/*  (SCI Clock = Fbus = 7.3728 MHz)  */;
;/* Baudrate = 7.3728MHz / (64xPDxBD) */;
;/* 64xPDxBD = 7.3728MHz / Baudrate   */;
;/*************************************/;
;/* Baudrate : PD x BD  %00PP0BBB     */;
;/*  1200    : 96(3x32) %00010101     */;
;/*  2400    : 48(3x16) %00010100     */;
;/*  4800    : 24(3x8)  %00010011     */;
;/*  9600    : 12(3x4)  %00010010     */;
;/*  19200   : 6(3x2)   %00010001     */;
;/*  28800   : 4(1x4)   %00000010     */;
;/*  38400   : 3(3x1)   %00010000     */;
;/*  57600   : 2(1x2)   %00000001     */;
;/*  115200  : 1(1x1)   %00000000     */;
;/*************************************/;
;
INIT_SCI:   ;MOV     #%00010101,SCBR         ; Baud Rate = 1200 / 7.3728MHz
            ;MOV     #%00010100,SCBR         ; Baud Rate = 2400 / 7.3728MHz
            ;MOV     #%00010011,SCBR         ; Baud Rate = 4800 / 7.3728MHz
            MOV     #%00010010,SCBR         ; Baud Rate = 9600 / 7.3728MHz
            ;MOV     #%00010001,SCBR         ; Baud Rate = 19200 / 7.3728MHz
            ;MOV     #%00000010,SCBR         ; Baud Rate = 28800 / 7.3728MHz
            ;MOV     #%00010000,SCBR         ; Baud Rate = 38400 / 7.3728MHz
            ;MOV     #%00000001,SCBR         ; Baud Rate = 57600 / 7.3728MHz
            ;MOV     #%00000000,SCBR         ; Baud Rate = 115200 / 7.3728MHz
            ;
            BCLR    LOOPS,SCC1              ; Normal Operation(Not Loop RX & TX)
            BSET    ENSCI,SCC1              ; Enable the SCI (N,8,1)
            BCLR    TXINV,SCC1              ; TX Signal Normal (Not Invert)
            BCLR    M,SCC1                  ; Data 8-Bit SCI
            BCLR    WAKE,SCC1               ; Idle Line Wakeup(Not Address mark Wakeup)
            BCLR    ILTY,SCC1               ; Character Bit Count After Start Bit
            BCLR    PEN,SCC1                ; Not Parity Bit
            BCLR    PTY,SCC1                ; Defualt = Even Parity (If Parity Enable)
            ;
            BCLR    SCTIE,SCC2              ; Disable TX Interrupt
            BCLR    TCIE,SCC2               ; Disable TC Interrupt
            BCLR    SCRIE,SCC2              ; Disable RX Interrupt
            BCLR    ILIE,SCC2               ; Disable IDLE Interrupt
            BSET    TE,SCC2                 ; Enable TX SCI
            BSET    RE,SCC2                 ; Enable RX SCI
            BCLR    RWU,SCC2                ; Normal Operation(Not Standby Mode)
            BCLR    SBK,SCC2                ; Not Break Character
            ;
            BCLR    DMARE,SCC3              ; Disable SCI RX DMA
            BCLR    DMATE,SCC3              ; Disable SCI TX DMA
            BCLR    ORIE,SCC3               ; Disable RX Overrun Error Interrupt
            BCLR    NEIE,SCC3               ; Disable Noise Error Interrupt
            BCLR    FEIE,SCC3               ; Disable Framing Error Interrupt
            BCLR    PEIE,SCC3               ; Disable Parity Error Interrupt
            RTS

;/********************/;
;/* Send Byte to SCI */;
;/* Input : A        */;
;/********************/;
;
TX_BYTE:    BRCLR   SCTE,SCS1,*             ; Wait Buffer Empty
            STA     SCDR                    ; Send Byte Serial Port
            BRCLR   TC,SCS1,*               ; Wait Send Ready
            RTS

;/**********************/;
;/* Read Byte From SCI */;
;/* Output : A         */;
;/* (Polling Read Byte)*/;
;/**********************/;
;
RX_BYTE:    BRCLR   SCRF,SCS1,*             ; Wait Receive Ready.
            LDA     SCDR                    ; Read Byte Serial Port
            RTS

;/******************************/;
;/* Print ASCII Data to RS232  */;
;/* Usage    : JSR  PRINT_SER  */;
;/*          : DB   'xxxxxxxx' */;
;/*          : DB   $00        */;
;/* Note     : Last Byte = $00 */;
;/******************************/;
;
PRINT_SER:  PULH                        ; Pop Stack PCH
            PULX                        ; Pop Stack PCL
            PSHA                        ; Save ACC
            ;
PRN_SER1:   LDA     0,X                 ; Get Data For Print
            BEQ     PRN_SER2            ; If Last Byte Jump
            JSR     TX_BYTE             ; Send Byte to RS232
            AIX     #1                  ; Inc (H:X)
            BRA     PRN_SER1
            ;
PRN_SER2:   AIX     #1                  ; Skip Last Byte(00)
            ;
            PULA                        ; Restore ACC
            PSHX                        ; Push PCL to Stack
            PSHH                        ; Push PCH to Stack
            RTS

;/***********************/;
;/* Send Hex to 2 ASCII */;
;/* Input : ACC (HEX)   */;
;/***********************/;
;
TX_2ASC:    PSHA                            ; Save HEX Byte
            NSA                             ; High Byte
            JSR     HEX_ASCII
            JSR     TX_BYTE
            PULA
            JSR     HEX_ASCII
            JSR     TX_BYTE
            RTS

;/************************************/;
;/* End of Subroutine For SCI(RS232) */;
;/************************************/;


;/*******************************************/;
;/* Start of Subroutine For Magnetic Reader */;
;/*******************************************/;
;
;/*********************************/;
;/* Read Data From  Magnetic Card */;
;/* Get Number of Card (Modulo-5) */;
;/* Output : MAG_BUFF(BCD 40Byte) */;
;/* (This is IRQ Service Routine) */;
;/*********************************/;
;
READ_MAG:   AIS     #-2                         ; Reserve 2 Byte Stack
            ;
            LDHX    #MAG_BUFF                   ; Pointer Magnetic Buffer
            CLRA
READ_MG0:   STA     0,X                         ; Clear Buffer
            AIX     #1
            CPHX    #MAG_BUFF+40                ; 40 Byte Clear
            BNE     READ_MG0
            ;
            BCLR    MAG_PASS,FLAG_BUF           ; Clear Any Flag
            BCLR    MAG_STRT,FLAG_BUF
            BCLR    MAG_STOP,FLAG_BUF
            ;
            ;/*********************************/;
            ;/* Wait For Start Bit Data = "0" */;
            ;/* Start = "1" But Signal Invert */;
            ;/*********************************/;
READ_MG1:   JSR     GET_CLK                     ; Get Start Sentinel
            BRSET   MAG_DATA,PORT_MAG,READ_MG1  ; Wait Data = "0" (Invert = "1" = Start)
            ;
            LDA     #5                          ; Modulo-5 Format Counter
            STA     1,SP
            CLRA                                ; Clear Result
            STA     2,SP                        ; Clear Checksum
            LDHX    #MAG_BUFF                   ; Start Pointer Save Data
            BRA     READ_MG4                    ; Get Start Bit at 1St Byte
            ;
            ;/* Get Start Data Bit Ready  */;
            ;/* Check 1St Byte = XXX01011 */;
READ_MG2:   LDA     #5                          ; Modulo-5 Format Counter
            STA     1,SP
            CLRA                                ; Clear Result
            ;
READ_MG3:   JSR     GET_CLK                     ; Wait Clock Sync.
            BRCLR   MAG_DATA,PORT_MAG,READ_MG4  ; Invert Data Signal Logic
            CLC                                 ; C = "0"
            BRA     READ_MG5
READ_MG4:   SEC                                 ; C = "1"
READ_MG5:   RORA                                ; Shift D7..D3 (D3 First)
            DBNZ    1,SP,READ_MG3
            ;
            ;/* Now Result = A = XXXXX000
            LSRA                                ; Shift D3 to D2 (0 -> XXXXX000)
            LSRA                                ; Shift D3 to D1 (0 -> 0XXXXX00)
            LSRA                                ; Shift D3 to D0 (0 -> 00XXXXX0)
            ;
            ;/* Now Result = A = 000XXXXX
            ;
            PSHH                                ; Save Pointer
            PSHX
            ;
            LDHX    #TAB_MAG1
READ_MG6:   CBEQ    0,X+,READ_MG7               ; Data Parity OK
            CPHX    #TAB_MAG1+16
            BNE     READ_MG6
            BCLR    MAG_PASS,FLAG_BUF           ; Parity Error
            PULX                                ; Restore Pointer
            PULH
            BRA     EXIT_MAG                    ; Exit
            ;
READ_MG7:   PULX                                ; Restore Pointer
            PULH
            ;
            AND     #$0F                        ; Ignore Parity Bit
            BRSET   MAG_STOP,FLAG_BUF,READ_MG8  ; Goto Compare Checksum
            STA     0,X                         ; Save Result to Buffer
            EOR     2,SP                        ; Generate Checksum
            STA     2,SP
            LDA     0,X
            AIX     #1                          ; Next Pointer
            CMP     #$0F                        ; $0F = Stop Code
            BNE     READ_MG2
            BSET    MAG_STOP,FLAG_BUF           ; Stop Code Receive
            BRA     READ_MG2                    ; Goto Get Checksum
            ;
READ_MG8:   CBEQ    2,SP,READ_MG9               ; Compare Checksum
            BCLR    MAG_PASS,FLAG_BUF           ; Checksum Error
            BRA     EXIT_MAG
            ;
READ_MG9:   BSET    MAG_PASS,FLAG_BUF           ; Enable Display Result
EXIT_MAG:   AIS     #2                          ; Clean-Up 2 Byte Stack
            RTS

;/**********************/;
;/* Get Magnetic Clock */;
;/*    ---|___|---     */;
;/**********************/;
;
GET_CLK:    BRCLR   MAG_CLK,PORT_MAG,*  ; Wait Rising Clock
            BRSET   MAG_CLK,PORT_MAG,*  ; Wait Falling Clock
            RTS

;/**************************/;
;/* Table Data of Modulo-5 */;
;/* 4Bit BCD + Parity(odd) */;
;/**************************/;
;
TAB_MAG1:   DB      %00010000           ; 0(00H) = Data
            DB      %00000001           ; 1(01H) = Data
            DB      %00000010           ; 2(02H) = Data
            DB      %00010011           ; 3(03H) = Data
            DB      %00000100           ; 4(04H) = Data
            DB      %00010101           ; 5(05H) = Data
            DB      %00010110           ; 6(06H) = Data
            DB      %00000111           ; 7(07H) = Data
            DB      %00001000           ; 8(08H) = Data
            DB      %00011001           ; 9(09H) = Data
            DB      %00011010           ; A(0AH) = Control(:)
            DB      %00001011           ; B(0BH) = Start Sentinel(;)
            DB      %00011100           ; C(0CH) = Control(<)
            DB      %00001101           ; D(0DH) = Field Seperate(=)
            DB      %00001110           ; E(0EH) = Control(>)
            DB      %00011111           ; F(0FH) = End Sentinel(?)

;/*****************************************/;
;/* End of Subroutine For Magnetic Reader */;
;/*****************************************/;


;/*****************************************/;
;/* Start of Subroutine For Scan Keyboard */;
;/*****************************************/;
;
;/**********************/;
;/* Scankey Matrix 4x3 */;
;/* Output : A         */;
;/* Row-0 = Pin 1,2    */;
;/* Row-1 = Pin 3      */;
;/* Row-2 = Pin 5      */;
;/* Row-3 = Pin 6      */;
;/* Col-0 = Pin 7      */;
;/* Col-1 = Pin 4,8    */;
;/* Col-2 = Pin 9      */;
;/**********************/;
;
SCANK4X3:   AIS     #-2                     ; Reserve 2 Byte Stack
            PSHH                            ; Save Register
            PSHX
            ;
            BCLR    COL0_KEY,PORT_KEY       ; Column-0
            BSET    COL1_KEY,PORT_KEY
            BSET    COL2_KEY,PORT_KEY
            BCLR    POUT_SPK,PORT_SPK
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K01
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K07
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K01
            BRA     SCAN_K04
            ;
SCAN_K01:   BSET    COL0_KEY,PORT_KEY       ; Column-1
            BCLR    COL1_KEY,PORT_KEY
            BSET    COL2_KEY,PORT_KEY
            BCLR    POUT_SPK,PORT_SPK
            ;
            LDA     PORT_KEY
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K02
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K07
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K02
            ORA     #%00010000
            BRA     SCAN_K04
            ;
SCAN_K02:   BSET    COL0_KEY,PORT_KEY       ; Column-2
            BSET    COL1_KEY,PORT_KEY
            BCLR    COL2_KEY,PORT_KEY
            BCLR    POUT_SPK,PORT_SPK
            ;
            LDA     PORT_KEY
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K03
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K07
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K03
            ORA     #%00100000
            BRA     SCAN_K04
            ;
SCAN_K03:   BCLR    KEY_NEW,FLAG_BUF        ; Key Release
            BRA     SCAN_K07
            ;
SCAN_K04:   BSET    KEY_NEW,FLAG_BUF        ; New Press
            STA     3,SP                    ; Save Matrix Keycode
            ;
            LDHX    #TAB_K4X3
            CLR     4,SP                    ; Position Count
SCAN_K05:   LDA     0,X                     ; Get Code From Table
            CBEQ    3,SP,SCAN_K06           ; Verify Keycode
            INC     4,SP                    ; Increment Position Count
            AIX     #1                      ; Next Code in Table
            CPHX    #TAB_K4X3+12            ; Check Last Key
            BNE     SCAN_K05
            BRA     SCAN_K07                ; Verify Not Match
            ;
SCAN_K06:   LDA     4,SP                    ; Get Key Position
            BRA     SCAN_K08
            ;
SCAN_K07:   LDA     #$FF                    ; Not Keypress
            ;
SCAN_K08:   PULX                            ; Restore Register
            PULH
            AIS     #2                      ; Clean-Up 2 Byte Stack
            RTS

;/**********************/;
;/* Matrix Keycode 4x3 */;
;/*   [1]  [2]  [3]    */;
;/*   [4]  [5]  [6]    */;
;/*   [7]  [8]  [9]    */;
;/*   [*]  [0]  [#]    */;
;/*   [12345--6789]    */;
;/**********************/;
;
TAB_K4X3:   DB      1EH,27H,17H,07H         ; 0 1 2 3
            DB      2BH,1BH,0BH,2DH         ; 4 5 6 7
            DB      1DH,0DH,2EH,0EH         ; 8 9 * #

;/**********************/;
;/* Scankey Matrix 4x4 */;
;/* Output : A         */;
;/* Row-0 = Pin 1      */;
;/* Row-1 = Pin 2      */;
;/* Row-2 = Pin 3      */;
;/* Row-3 = Pin 4      */;
;/* Col-0 = Pin 5      */;
;/* Col-1 = Pin 6      */;
;/* Col-2 = Pin 7      */;
;/* Col-3 = Pin 8      */;
;/**********************/;
;
SCANK4X4:   AIS     #-2                     ; Reserve 2 Byte Stack
            PSHH                            ; Save Register
            PSHX
            ;
            BCLR    COL0_KEY,PORT_KEY       ; Column-0
            BSET    COL1_KEY,PORT_KEY
            BSET    COL2_KEY,PORT_KEY
            BSET    POUT_SPK,PORT_SPK
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K11
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K11
            BRCLR   KEY_NEW,FLAG_BUF,SCAN_K15
            JMP     SCAN_K18
            ;
SCAN_K11:   BSET    COL0_KEY,PORT_KEY       ; Column-1
            BCLR    COL1_KEY,PORT_KEY
            BSET    COL2_KEY,PORT_KEY
            BSET    POUT_SPK,PORT_SPK
            ;
            LDA     PORT_KEY
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K12
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K18
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K12
            ORA     #%00010000
            BRA     SCAN_K15
            ;
SCAN_K12:   BSET    COL0_KEY,PORT_KEY       ; Column-2
            BSET    COL1_KEY,PORT_KEY
            BCLR    COL2_KEY,PORT_KEY
            BSET    POUT_SPK,PORT_SPK
            ;
            LDA     PORT_KEY
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K13
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K18
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K13
            ORA     #%00100000
            BRA     SCAN_K15
            ;
SCAN_K13:   BSET    COL0_KEY,PORT_KEY       ; Column-3
            BSET    COL1_KEY,PORT_KEY
            BSET    COL2_KEY,PORT_KEY
            BCLR    POUT_SPK,PORT_SPK
            ;
            LDA     PORT_KEY
            AND     #%00001111
            CBEQA   #%00001111,SCAN_K14
            BRSET   KEY_NEW,FLAG_BUF,SCAN_K18
            LDA     #20                     ; Debounce 20mS
            JSR     DLY_MSEC
            LDA     PORT_KEY                ; Read Row
            AND     #%00001111              ; Ignore Column
            CBEQA   #%00001111,SCAN_K14
            ORA     #%01000000
            BRA     SCAN_K15
            ;
SCAN_K14:   BCLR    KEY_NEW,FLAG_BUF        ; Key Release
            BRA     SCAN_K18
            ;
SCAN_K15:   BSET    KEY_NEW,FLAG_BUF        ; New Press
            STA     3,SP                    ; Save Matrix Keycode
            ;
            LDHX    #TAB_K4X4
            CLR     4,SP                    ; Position Count
SCAN_K16:   LDA     0,X                     ; Get Code From Table
            CBEQ    3,SP,SCAN_K17           ; Verify Keycode
            INC     4,SP                    ; Increment Position Count
            AIX     #1                      ; Next Code in Table
            CPHX    #TAB_K4X4+16            ; Check Last Key
            BNE     SCAN_K16
            BRA     SCAN_K18                ; Verify Not Match
            ;
SCAN_K17:   LDA     4,SP                    ; Get Key Position
            BRA     SCAN_K19
            ;
SCAN_K18:   LDA     #$FF                    ; Not Keypress
            ;
SCAN_K19:   PULX                            ; Restore Register
            PULH
            AIS     #2                      ; Clean-Up 2 Byte Stack
            RTS

;/**********************/;
;/* Matrix Keycode 4x4 */;
;/*     [12345678]     */;
;/* [1]  [2]  [3]  [U] */;
;/* [4]  [5]  [6]  [D] */;
;/* [7]  [8]  [9]  [S] */;
;/* [C]  [0]  [H]  [E] */;
;/**********************/;
;
TAB_K4X4:   DB      4DH,0EH,0DH,0BH         ; 0   1   2   3
            DB      1EH,1DH,1BH,2EH         ; 4   5   6   7
            DB      2DH,2BH,4EH,4BH         ; 8   9   Clr Hlp
            DB      07H,17H,27H,47H         ; Up  Dn  2nd Ent

;/***************************************/;
;/* End of Subroutine For Scan Keyboard */;
;/***************************************/;

;/***********************/;
;/* Generate Sound Beep */;
;/*   (Mini Speaker)    */;
;/***********************/;
;
BEEP_SPK:   AIS     #-2                 ; Reserve 2 Byte Stack
            PSHA                        ; Save ACC
            ;
            LDA     #$FF                ; Duration Beep
            STA     2,SP
BEEP_SP1:   LDA     #$FF                ; Freq. Beep
            STA     1,SP
            BSET    POUT_SPK,PORT_SPK   ; ON Time
            DBNZ    1,SP,*
            LDA     #$FF
            STA     1,SP
            BCLR    POUT_SPK,PORT_SPK   ; OFF Time
            DBNZ    1,SP,*
            DBNZ    2,SP,BEEP_SP1
            ;
            PULA                        ; Restore ACC
            AIS     #2                  ; Clean-Up 2 Byte Stack
            RTS

;/*******************/;
;/* Delay Time 1 mS */;
;/* Fbus = 7.372 MHz*/;
;/* 1mS = 7373 Cycle*/;
;/* Input : A (mSec)*/;
;/*******************/;
;
DLY_MSEC:   PSHH                            ; Save Register (2 Cycle)
            PSHX                            ; 2 Cycle
            ;
DLY_MS0:    LDHX    #816                    ; 3 Cycle
DLY_MS1:    NOP                             ; 1 Cycle
            AIX     #-1                     ; H:X - 1 (2 Cycle)
            CPHX    #0                      ; 3 Cycle
            BNE     DLY_MS1                 ; 3 Cycle ( 9 x 886 = 7344 Cycle)
            DBNZA   DLY_MS0                 ; 3 Cycle
            ;
            PULX                            ; 2 Cycle
            PULH                            ; 2 Cycle
            RTS                             ; 4 Cycle

;/************************/;
;/* Convert HEX to ASCII */;
;/* Input  : A (HEX)     */;
;/* Output : A (ASCII)   */;
;/************************/;
;
HEX_ASCII:  AND     #0FH                ; Clear High Nibble
            CMP     #9
            BHI     HEX_ASC1            ; A..F Jump
            ADD     #$30                ; Convert 0..9 to '0..9'
            BRA     HEX_ASC2
            ;
HEX_ASC1:   SUB     #9                  ; Convert A..F to 1..6
            ADD     #$40                ; Convert 1..6 to 'A..F'
            ;
HEX_ASC2:   RTS


;/*******************/;
;/* T1CH0 Interrupt */;
;/*   (TIM1 CH0)    */;
;/*******************/;
;
T1CH0_SRV:  PSHH                            ; Save Register
            LDA     T1SC0                   ; Read CH0 Status
            BCLR    CH0F,T1SC0              ; Clear CH0F
            ;
            ;/* Change Duty Cycle Here */;
            MOV     DUTY0_HI,T1CH0H         ; Define Duty Cycle of PWM
            MOV     DUTY0_LO,T1CH0L
            ;
            PULH                            ; Restore Register
            RTI

;/*******************/;
;/* T1CH1 Interrupt */;
;/*    (TIM1 CH1)   */;
;/*******************/;
;
T1CH1_SRV:  PSHH                            ; Save Register
            LDA     T1SC1                   ; Read CH1 Status
            BCLR    CH1F,T1SC1              ; Clear CH1F
            ;
            ;/* Change Duty Cycle Here */;
            MOV     DUTY1_HI,T1CH1H         ; Define Duty Cycle of PWM
            MOV     DUTY1_LO,T1CH1L
            ;
            PULH                            ; Restore Register
            RTI

            ;/**************************/;
            ;/* User Vector Start Here */;
            ;/*     2 Byte / Vector    */;
            ;/**************************/;
            ;/* ORG $Vector  Address   */;
            ;/* DW  $Routine Address   */;
            ;/**************************/;
            ;
            ORG     $FFDC                   ; TBM Vector
            DW      RESET
            ;
            ORG     $FFDE                   ; ADC Vector
            DW      RESET
            ;
            ORG     $FFE0                   ; KBI Vector
            DW      RESET
            ;
            ORG     $FFE2                   ; SCI TX Vector
            DW      RESET
            ;
            ORG     $FFE4                   ; SCI RX Vector
            DW      RESET
            ;
            ORG     $FFE6                   ; SCI Error Vector
            DW      RESET
            ;
            ORG     $FFE8                   ; SPI TX Vector
            DW      RESET
            ;
            ORG     $FFEA                   ; SPI RX Vector
            DW      RESET
            ;
            ORG     $FFEC                   ; TIM2 Overflow Vector
            DW      RESET
            ;
            ORG     $FFEE                   ; TIM2 CH1 Vector
            DW      RESET
            ;
            ORG     $FFF0                   ; TIM2 CH0 Vector
            DW      RESET
            ;
            ORG     $FFF2                   ; TIM1 Overflow Vector
            DW      RESET
            ;
            ORG     $FFF4                   ; TIM1 CH1 Vector
            DW      T1CH1_SRV
            ;
            ORG     $FFF6                   ; TIM1 CH0 Vector
            DW      T1CH0_SRV
            ;
            ORG     $FFF8                   ; PLL Vector
            DW      RESET
            ;
            ORG     $FFFA                   ; IRQ Vector
            DW      RESET
            ;
            ORG     $FFFC                   ; SWI Vector
            DW      RESET
            ;
            ORG     $FFFE                   ; RST Vector
            DW      RESET

            END
