/*******************************************/
/* Example Program For ET-BASE51 AC3 V1.0  */
/* MCU      : AT89C51AC3(XTAL=29.4912 MHz) */
/*          : Frequency Bus = 58.9824 MHz  */
/* Compiler : Keil C51 (V7.50)             */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)*/
/*******************************************/
/* Demo Character LCD(16x1) 4Bit Interface */

/* Include  Section */
#include <reg52.h>						// Standard 8052 SFR : File
#include <stdio.h>						// sprintf Function

/* AT89C51AC3 SFR */
sfr CKCON = 0x8F;						// Clock Control
sfr P4 = 0xC0;							// Port P4

/* LCD Interface */
sfr PORT_LCD = 0xA0;					// LCD Interface = Port P2
sbit RS = PORT_LCD^1;					// RS LCD (0=Instruction,1=Data)
sbit RW = PORT_LCD^2;					// RW LCD (0=Write,1=Read)
sbit E  = PORT_LCD^3;					// Enable LCD(Active = "1")

char lcdbuf[16+1];						// LCD Display Buffer

/* pototype  section */
void init_lcd(void);					// Initial Character LCD(4-Bit Interface)
void gotolcd(unsigned char);			// Set Cursor LCD
void write_ins(unsigned char);      	// Write Instruction LCD
void write_data(unsigned char);			// Write Data LCD
void enable_lcd(void);	 				// Enable Pulse
char busy_lcd(void);					// Read Busy LCD Status
void printlcd(void);					// Display Message LCD
void delay(unsigned long);				// Delay Time Function(1..4294967295)

/*--------------------------------------------
The main C function.  Program execution Here 
---------------------------------------------*/
void main(void)
{
  CKCON = 0x01;							// Initial X2 Mode (BUS Clock = 58.9824 MHz) 
  init_lcd();							// Initial LCD

  while(1)
  {
    gotolcd(0);							// Set Cursor Line-1
    sprintf(lcdbuf,"ET-BASE51 AC3 V1"); // Display Line-1
    printlcd();
    gotolcd(0x40);						// Set Cursor Line-2
    sprintf(lcdbuf,"BY..ETT CO.,LTD."); // Display Line-2
    printlcd();	
    delay(150000);						// Display Delay

    gotolcd(0);							// Set Cursor Line-1
    sprintf(lcdbuf,"MCS51 High Speed"); // Display Line-1  
    printlcd();
    gotolcd(0x40);						// Set Cursor Line-2
    sprintf(lcdbuf,"58.98MHz Execute"); // Display Line-2
    printlcd();	
    delay(150000);						// Display Delay
  }  
}

/*******************************/
/* Initial LCD 4-Bit Interface */
/*******************************/
void init_lcd(void)
{
  unsigned int i;						// Delay Count

  E = 0;								// Start LCD Control (Disable)
  RS = 0;								// Default Instruction
  RW = 0;								// Default = Write Direction
  for (i=0;i<10000;i++);				// Power-On Delay (15 mS)  
  	  
  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  for (i=0;i<2500;i++);					// Delay 4.1mS

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  for (i=0;i<100;i++);					// delay 100uS

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  while(busy_lcd());      				// Wait LCD Execute Complete

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x20;						// DB5:DB4 = 1:0
  enable_lcd();							// Enable Pulse
  while(busy_lcd());      				// Wait LCD Execute Complete
  
  write_ins(0x28);  					// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)
  write_ins(0x0C);  					// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  write_ins(0x06);  					// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)
  write_ins(0x01);  					// Clear Display  (Clear Display,Set DD RAM Address=0)
}

/******************/
/* Set LCD Cursor */
/******************/
void gotolcd(unsigned char i)
{
  i |= 0x80;							// Set DD-RAM Address Command
  write_ins(i);  
}

/****************************/
/* Write Instruction to LCD */
/****************************/
void write_ins(unsigned char i)
{
  RS  = 0;			 					// Instruction Select
  RW  = 0;								// Write Select

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= i & 0xF0;   				// Strobe High Nibble Command
  enable_lcd();							// Enable Pulse
       
  PORT_LCD &= 0x0F;        				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= (i << 4) & 0xF0;			// Strobe Low Nibble Command
  enable_lcd();							// Enable Pulse
    
  while(busy_lcd());      				// Wait LCD Execute Complete
}

/****************************/
/* Write Data(ASCII) to LCD */
/****************************/
void write_data(unsigned char i)
{
  RS  = 1;				  				// Data Select
  RW  = 0;								// Write Select

  PORT_LCD &= 0x0F;    					// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= i & 0xF0;  				// Strobe High Nibble Data
  enable_lcd();							// Enable Pulse  
    			
  PORT_LCD &= 0x0F;    					// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= (i << 4) & 0xF0; 			// Strobe Low Nibble Data
  enable_lcd();							// Enable Pulse
 
  while(busy_lcd());      				// Wait LCD Execute Complete
}

/***********************/
/* Enable Pulse to LCD */
/***********************/
void enable_lcd(void)	 				// Enable Pulse
{
  unsigned int i;						// Delay Count
  E = 1;   								// Enable ON
  for (i=0;i<500;i++);
  E = 0;   								// Enable OFF 
}

/******************/
/* Wait LCD Ready */
/******************/
char busy_lcd(void)
{
  unsigned char busy_status;			// Busy Status Read

  RS  = 0;			 					// Instruction Select
  RW  = 1; 								// Read Direction
  E = 1;								// Start Read Busy
  busy_status = PORT_LCD;				// Read LCD Data

  if(busy_status & 0x80)   				// Read & Check Busy Flag
  {
    E = 0;       						// Disable Read
	RW = 0;								// Default = Write Direction
    return 1;							// LCD Busy Status
  }
  else
  {
    E = 0;       						// Disable Read
	RW = 0;								// Default = Write Direction
    return 0;							// LCD Ready Status
  }
}


/****************************/
/* Print Data(ASCII) to LCD */
/****************************/
void printlcd(void)
{
  char *p;
 
  p = lcdbuf;
 
  do 									// Get ASCII & Write to LCD Until null
  {
    write_data(*p); 					// Write ASCII to LCD
    p++;								// Next ASCII
  }
  while(*p != '\0');					// End of ASCII (null)
 
  return;
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long i)
{
  while(i > 0) {i--;}					// Loop Decrease Counter	
  return;
}




