/*******************************************/
/* Example Program For ET-BASE51 AC3 V1.0  */
/* MCU      : AT89C51AC3(XTAL=29.4912 MHz) */
/*          : Frequency Bus = 58.9824 MHz  */
/* Compiler : Keil C51 (V7.50)             */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)*/
/*******************************************/
/* Demo Character LCD(16x1) 4Bit Interface */

/* Include  Section */
#include <reg52.h>						// Standard 8052 SFR : File
#include <stdio.h>						// sprintf Function

// Global Variable
unsigned int val;		   				// ADC Result (HEX)
float volt;								// ADC Result Volt

/* AT89C51AC3 SFR */
sfr CKCON = 0x8F;						// Clock Control
sfr P4 = 0xC0;							// Port P4
sfr ADCLK = 0xF2;                		// ADC Clock Control Register
sfr ADCON = 0xF3;                		// ADC Control Register
sfr ADDL  = 0xF4;                		// ADC Data Low Byte
sfr ADDH  = 0xF5;                		// ADC Data High Byte
sfr ADCF  = 0xF6;                		// ADC Config Register
sbit led   = P1^0;						// P1.0 = lED ON/OFF

/* LCD Interface */
sfr PORT_LCD = 0xA0;					// LCD Interface = Port P2
sbit RS = PORT_LCD^1;					// RS LCD (0=Instruction,1=Data)
sbit RW = PORT_LCD^2;					// RW LCD (0=Write,1=Read)
sbit E  = PORT_LCD^3;					// Enable LCD(Active = "1")

char lcdbuf[16+1];						// LCD Display Buffer

/* pototype  section */
void init_lcd(void);					// Initial Character LCD(4-Bit Interface)
void gotolcd(unsigned char);			// Set Cursor LCD
void write_ins(unsigned char);      	// Write Instruction LCD
void write_data(unsigned char);			// Write Data LCD
void enable_lcd(void);	 				// Enable Pulse
char busy_lcd(void);					// Read Busy LCD Status
void printlcd(void);					// Display Message LCD
void delay(unsigned long);				// Delay Time Function(1..4294967295)

void test_led(void);
void test_adc8(void);
void test_lcd(void);
void test_manual(void);

/*--------------------------------------------
The main C function.  Program execution Here 
---------------------------------------------*/
void main(void)
{
  unsigned char rs232_data;				// RS232 Get Byte

  CKCON = 0x01;							// Initial X2 Mode (BUS Clock = 58.9824 MHz) 
  CKCON = 0x01;							// Initial X2 Mode (58.9824 MHz)
	                  
  /* Initial MCS51 Serial Port */
  TMOD &=  0x0F;  						// Reset old Timer1 Mode Config
  TMOD |=  0x20;  						// Update Timer1 = 8 Bit Auto Reload             
  SCON  =  0x50;  						// Serial Port Mode 1 (N,8,1)
  ES    =  0;     						// Disable Serial Interupt
  ET1   =  0;     						// Disable Timer1 Interrupt
  PCON &=  0x7F;  						// SMOD1 = 0 (Disable Double Baudrate)                  
  TH1   =  0xF0;  						// Setup Timer1 Baudrate 9600BPS / 58.9824 MHz
  TL1   =  0xF0;
  TR1   =  1;     						// Start Timer1 Generate Baudrate                      
  TI    =  1;     						// Set TI to send First char of UART
  
  printf("\n\n\n");
  printf("****************************\n");
  printf("***  ET-BASE51 AC3 V1.0  ***\n"); 
  printf(">>>>> SYSTEM SELF TEST <<<<<\n");
  printf("1. Test LED(P1.0)\n");  
  printf("2. Test ADC 10Bit 8Ch\n");
  printf("3. Test LCD(Char16x2)\n");
  printf("4. Test PORT I/O(Manual)\n");
  printf("****************************\n");
  printf("\n");
  printf("Select Choice(1...4)......");

  // Loop Print Out Message to UART0 //
  while(1)													// Loop Continue
  {
    rs232_data = getchar(); 								// Get Choice
	switch (rs232_data)
	{
	  case '1' : printf("\n\n");
	             printf("Test LED(P1.0)");
				 test_led();
	             break;

	  case '2' : printf("\n\n");
	             printf("Test ADC 10Bit 8Ch(PORT-P1)\n");
				 test_adc8();
	             break;

	  case '3' : printf("\n\n");
	             printf("Test LCD 16x2");	 
				 test_lcd();
	             break;	  

	  case '4' : printf("\n\n");
	             printf("Test All Port I/O Manual Output\n");
				 printf("Press Any Key For Test Output\n");
				 test_manual();
				 break;
	}
  }  
}

/*******************************/
/* Initial LCD 4-Bit Interface */
/*******************************/
void init_lcd(void)
{
  unsigned int i;						// Delay Count

  E = 0;								// Start LCD Control (Disable)
  RS = 0;								// Default Instruction
  RW = 0;								// Default = Write Direction
  for (i=0;i<10000;i++);				// Power-On Delay (15 mS)  
  	  
  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  for (i=0;i<2500;i++);					// Delay 4.1mS

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  for (i=0;i<100;i++);					// delay 100uS

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x30;						// DB5:DB4 = 1:1
  enable_lcd();							// Enable Pulse
  while(busy_lcd());      				// Wait LCD Execute Complete

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= 0x20;						// DB5:DB4 = 1:0
  enable_lcd();							// Enable Pulse
  while(busy_lcd());      				// Wait LCD Execute Complete
  
  write_ins(0x28);  					// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)
  write_ins(0x0C);  					// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  write_ins(0x06);  					// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)
  write_ins(0x01);  					// Clear Display  (Clear Display,Set DD RAM Address=0)
}

/******************/
/* Set LCD Cursor */
/******************/
void gotolcd(unsigned char i)
{
  i |= 0x80;							// Set DD-RAM Address Command
  write_ins(i);  
}

/****************************/
/* Write Instruction to LCD */
/****************************/
void write_ins(unsigned char i)
{
  RS  = 0;			 					// Instruction Select
  RW  = 0;								// Write Select

  PORT_LCD &= 0x0F;       				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= i & 0xF0;   				// Strobe High Nibble Command
  enable_lcd();							// Enable Pulse
       
  PORT_LCD &= 0x0F;        				// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= (i << 4) & 0xF0;			// Strobe Low Nibble Command
  enable_lcd();							// Enable Pulse
    
  while(busy_lcd());      				// Wait LCD Execute Complete
}

/****************************/
/* Write Data(ASCII) to LCD */
/****************************/
void write_data(unsigned char i)
{
  RS  = 1;				  				// Data Select
  RW  = 0;								// Write Select

  PORT_LCD &= 0x0F;    					// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= i & 0xF0;  				// Strobe High Nibble Data
  enable_lcd();							// Enable Pulse  
    			
  PORT_LCD &= 0x0F;    					// Clear old LCD Data (Bit[7..4])
  PORT_LCD |= (i << 4) & 0xF0; 			// Strobe Low Nibble Data
  enable_lcd();							// Enable Pulse
 
  while(busy_lcd());      				// Wait LCD Execute Complete
}

/***********************/
/* Enable Pulse to LCD */
/***********************/
void enable_lcd(void)	 				// Enable Pulse
{
  unsigned int i;						// Delay Count
  E = 1;   								// Enable ON
  for (i=0;i<500;i++);
  E = 0;   								// Enable OFF 
}

/******************/
/* Wait LCD Ready */
/******************/
char busy_lcd(void)
{
  unsigned char busy_status;			// Busy Status Read

  RS  = 0;			 					// Instruction Select
  RW  = 1; 								// Read Direction
  E = 1;								// Start Read Busy
  busy_status = PORT_LCD;				// Read LCD Data

  if(busy_status & 0x80)   				// Read & Check Busy Flag
  {
    E = 0;       						// Disable Read
	RW = 0;								// Default = Write Direction
    return 1;							// LCD Busy Status
  }
  else
  {
    E = 0;       						// Disable Read
	RW = 0;								// Default = Write Direction
    return 0;							// LCD Ready Status
  }
}

/****************************/
/* Print Data(ASCII) to LCD */
/****************************/
void printlcd(void)
{
  char *p;
 
  p = lcdbuf;
 
  do 									// Get ASCII & Write to LCD Until null
  {
    write_data(*p); 					// Write ASCII to LCD
    p++;								// Next ASCII
  }
  while(*p != '\0');					// End of ASCII (null)
 
  return;
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long i)
{
  while(i > 0) {i--;}					// Loop Decrease Counter	
  return;
}


void test_led(void)
{
  while (1) 								// Loop Continue 
  {   	     
	 led = ~led;							// Tiggle LED
	 delay(50000);							// Delay Blink
  }	
}

//
void test_lcd(void)
{
  init_lcd();							// Initial LCD

  while(1)
  {
    gotolcd(0);							// Set Cursor Line-1
    sprintf(lcdbuf,"ET-BASE51 AC3 V1"); // Display Line-1
    printlcd();
    gotolcd(0x40);						// Set Cursor Line-2
    sprintf(lcdbuf,"BY..ETT CO.,LTD."); // Display Line-2
    printlcd();	
    delay(150000);						// Display Delay

    gotolcd(0);							// Set Cursor Line-1
    sprintf(lcdbuf,"MCS51 High Speed"); // Display Line-1  
    printlcd();
    gotolcd(0x40);						// Set Cursor Line-2
    sprintf(lcdbuf,"58.98MHz Execute"); // Display Line-2
    printlcd();	
    delay(150000);						// Display Delay
  }    
}

//
void test_adc8(void)
{
  /* Print Message to RS232 */
  printf ("ADC0[P1.0] 10Bit[0-3V] Read Demo\n"); 
  ADCF  = 0xFF;         						// P1[7..0] = ADC[7..0]
  ADCON = 0x20;       							// Enable ADC Function

  // Start Test Read ADC0 and Display on UART0 //
  while(1)										// Loop Continue
  {
    // Read AD0
    ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x00;       						// Select ADC = Ch0            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf("\rADC0 Result = %1.1f Volt.",volt);	// Display 3-Digit Result(0-3.3V)

    /*
	// Read AD1
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x01;       						// Select ADC = Ch1            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD2
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x02;       						// Select ADC = Ch2            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD3
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x03;       						// Select ADC = Ch3            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD4
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x04;       						// Select ADC = Ch4            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD5
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x05;       						// Select ADC = Ch5            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD6
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x06;       						// Select ADC = Ch6            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f",volt);					// Display 3-Digit Result(0-3.3V)

	// Read AD7
	ADCON &= 0xF8;       						// Reset ADC Channel Select
    ADCON |= 0x07;       						// Select ADC = Ch7            
    ADCON |= 0x08;        						// Start ADC Convert (ADSST=1)      
	while((ADCON & 0x10)!= 0x10)				// Wait ADEOC = 1 (ADC Conversion Complete)
	val = ADDH<<2;          					// Read 8 MSB Bits 
	val |= (ADDL & 0x03);   					// Read 2 LSB Bits 
    ADCON &= 0xEF;       						// Clear ADEOC = 0
	volt = val * (3.0 / 1023.0);	  			// Volt = ADC Result x [3.0V / 1024]	 
	printf(" : %1.1f Volt.",volt);				// Display 3-Digit Result(0-3.3V)
	*/
  }      
}

void test_manual(void)
{
  unsigned char rs232_data;					// RS232 Get Byte
  unsigned int j;				   			// LED Output Loop Count
  P0    |= 0xFF;							// OFF P0[7..0] Output
  P1    |= 0xFF;							// OFF P1[7..0] Output
  P2    |= 0xFF;							// OFF P2[7..0] Output
  P3    |= 0xFC;							// OFF P3[7..2] Output
  P4    |= 0x1F;							// OFF P4[4..0] Output

  while (1) 								// Loop Continue 
  {   
	/* Forward Direction */
    // Shift Left P0[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  			// Blink LED P0.7 <- P0.0
    {   
      P0 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0) 			 
      rs232_data = getchar(); 				// Wait Continue
    }
	P0 |= 0xFF;	 							// OFF P0[7..0] LED Output

	// Shift Left P1[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  			// Blink LED P1.7 <- P1.0
    {   
      P1 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 				// Wait Continue
    }
	P1 |= 0xFF;								// OFF P1[7..0] LED Output

	// Shift Left P2[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  			// Blink LED P2.7 <- P2.0
    {   
      P2 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 				// Wait Continue
    }
	P2 |= 0xFF;								// OFF P2[7..0] LED Output

	// Shift Left P3[7..2]                 							
    for (j=0x04; j<= 0x80; j<<=1)  			// Blink LED P3.7 <- P3.2
    {   
      P3 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 				// Wait Continue
    }
	P3 |= 0xFC;								// OFF P3[7..2] LED Output

	// Shift Left P4[4..0]                 							
    for (j=0x01; j<= 0x10; j<<=1)  			// Blink LED P4.4 <- P4.0
    {   
      P4 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0) 
      rs232_data = getchar(); 				// Wait Continue
    }
	P4 |= 0x1F;				 				// OFF P4[4..0] LED Output


	/* Reverse Direction */
	// Shift Right P4[4..0]
    for (j=0x10; j>= 0x01; j>>=1)  			// Blink LED P4.4 -> P4.0
    {   
      P4 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 				// Wait Continue
    }
	P4 |= 0x1F;								// OFF P4[4..0] LED Output

	// Shift Right P3[7..2]
    for (j=0x80; j>= 0x04; j>>=1)  			// Blink LED P3.7 -> P3.2
    {   
      P3 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 				// Wait Continue
    }
	P3 |= 0xFC;								// OFF P3[7..2] LED Output

	// Shift Right P2[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  			// Blink LED P2.7 -> P2.0
    {   
      P2 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 				// Wait Continue
    }
	P2 |= 0xFF;								// OFF P2[7..0] LED Output
	
	// Shift Right P1[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  			// Blink LED P1.7 -> P1.0
    {   
      P1 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 				// Wait Continue
    }
	P1 |= 0xFF;								// OFF P1[7..0] LED Output

	// Shift Right P0[7..0]
    for (j=0x80; j>= 0x01; j>>=1)  			// Blink LED P0.7 -> P0.0
    {   
      P0 = 0xFF-j;                         	// Output to LED Port(Toggle Active=0)  
      rs232_data = getchar(); 				// Wait Continue
    }
	P0 |= 0xFF;								// OFF P0[7..0] LED Output
  }
}



