

/***********************************************************************************************
 *                                                                                             *
 * Ex1-TFT43_Clock : GLCD_Touch Screen 480x272 (ET-TFT43-EVE) "ET-BASE PIC8722 (ICD2)"         *
 *                                                                                             *
 *                  Interface GLCD Data SPI Software                                           *
 *                                                                                             *
 ***********************************************************************************************
 *                                                                                             *
 * Target MCU        : PIC18F8722                                                              *
 *                   : X-TAL : 10 MHz                                                          *
 *                   : CPU Clock = 40 MHz (x4)                                                 *                                                                  *
 * Editor-Compiler   : CCS Version 4.124                                                       *
 * Create By         : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                    *
 * Last Update       : 9/March/2014                                                            *
 *                                                                                             *
 * Port Interface :  ---- MCU-PIC18F8722 ----                ---- ET-TFT43-EVE ----            *
 *                                                                                             *
 *      SPI       :                                                                            *
 *                     RJ0 (MISO-In)       Connect          MISO              [Pin0]           *
 *                     RJ1 (MOSI-Out)      Connect          MOSI              [Pin1]           *
 *                     RJ2 (SCK-Out)       Connect          SCK  (Ideal=Low)  [Pin2]           *
 *                     RJ3 (CS-Out)        Connect          CS#  (Active Low) [Pin3]           *
 *                     RJ4 (OUT))          Connect          PD#  (Active=Low) [Pin4]           *
 *                     RJ5 (IN)            Connect          INT# (Active=Low) [Pin5]           *
 *                     RJ6 (OUT)           Connect          SD_CS(Active=Low) [Pin6]           *
 *                                                                                             *
 *                                                                                             *
 ***********************************************************************************************/


#include <18F8722.h>
#include <stdio.h>

//#define TX1   PIN_C6
//#define RX1   PIN_C7

#fuses H4,NOLVP,NOWDT,NOPROTECT,NOSTVREN           //Setup MCU
#use delay (clock = 40000000)                      //delay Clock = (ms)

//#use rs232(baud = 9600, xmit = TX1, rcv = RX1)   //Set Rs232

#use fast_io(ALL)                                  //Stop Set direction Auto for All Port  by Compiler  


//********************* 1.Define-FT800 Memory Map Addresses *************************

#define RAM_CMD                     0x108000       //Address Start Graphic Engine Command Buffer
#define RAM_DL                      0x100000       //Address Start Display list ram


//**************** 2.Define-FT800 Command Memory Read/Write ********************

#define MEM_WRITE                    0x80          //FT800 Host Memory Write Mark bit7..8 = 10
#define MEM_READ                     0x3F          //FT800 Host Memory Read Mark bit7..8 =00


//*************** 3.Define-FT800 Register Address (0x102400-0x10257F) ******************

#define      REG_ID                  0x102400
#define      REG_FRAMES              0x102404
#define      REG_CLOCK               0x102408
#define      REG_FREQUENCY           0x10240C
#define      REG_RENDERMODE          0x102410
#define      REG_SNAPY               0x102414
#define      REG_SNAPSHOT            0x102418
#define      REG_CPURESET            0x10241C
#define      REG_TAP_CRC             0x102420
#define      REG_TAP_MASK            0x102424
#define      REG_HCYCLE              0x102428
#define      REG_HOFFSET             0x10242C
#define      REG_HSIZE               0x102430
#define      REG_HSYNC0              0x102434
#define      REG_HSYNC1              0x102438
#define      REG_VCYCLE              0x10243C
#define      REG_VOFFSET             0x102440
#define      REG_VSIZE               0x102444
#define      REG_VSYNC0              0x102448
#define      REG_VSYNC1              0x10244C
#define      REG_DLSWAP              0x102450
#define      REG_ROTATE              0x102454
#define      REG_OUTBITS             0x102458

#define      REG_DITHER              0x10245C
#define      REG_SWIZZLE             0x102460
#define      REG_CSPREAD             0x102464
#define      REG_PCLK_POL            0x102468
#define      REG_PCLK                0x10246C
#define      REG_TAG_X               0x102470
#define      REG_TAG_Y               0x102474
#define      REG_TAG                 0x102478
#define      REG_VOL_PB              0x10247C
#define      REG_VOL_SOUND           0x102480
#define      REG_SOUND               0x102484
#define      REG_PLAY                0x102488
#define      REG_GPIO_DIR            0x10248C
#define      REG_GPIO                0x102490
#define      REG_INT_FLAGS           0x102498
#define      REG_INT_EN              0x10249C
#define      REG_INT_MASK            0x1024A0
#define      REG_PLAYBACK_START      0x1024A4
#define      REG_PLAYBACK_LENGTH     0x1024A8
#define      REG_PLAYBACK_READPTR    0x1024AC
#define      REG_PLAYBACK_FREQ       0x1024B0
#define      REG_PLAYBACK_FORMAT     0x1024B4
#define      REG_PLAYBACK_LOOP       0x1024B8

#define      REG_PLAYBACK_PLAY       0x1024BC
#define      REG_PWM_HZ              0x1024C0
#define      REG_PWM_DUTY            0x1024C4
#define      REG_MACRO_0             0x1024C8
#define      REG_MACRO_1             0x1024CC
#define      REG_CMD_READ            0x1024E4
#define      REG_CMD_WRITE           0x1024E8
#define      REG_CMD_DL              0x1024EC
#define      REG_TOUCH_MODE          0x1024F0
#define      REG_TOUCH_ADC_MODE      0x1024F4
#define      REG_TOUCH_CHARGE        0x1024F8
#define      REG_TOUCH_SETTLE        0x1024FC
#define      REG_TOUCH_OVERSAMPLE    0x102500
#define      REG_TOUCH_RZTHRESH      0x102504
#define      REG_TOUCH_RAW_XY        0x102508
#define      REG_TOUCH_RZ            0x10250C
#define      REG_TOUCH_SCREEN_XY     0x102510
#define      REG_TOUCH_TAG_XY        0x102514
#define      REG_TOUCH_TAG           0x102518
#define      REG_TOUCH_TRANSFORM_A   0x10251C

#define      REG_TOUCH_TRANSFORM_B   0x102520
#define      REG_TOUCH_TRANSFORM_C   0x102524
#define      REG_TOUCH_TRANSFORM_D   0x102528
#define      REG_TOUCH_TRANSFORM_E   0x10252C
#define      REG_TOUCH_TRANSFORM_F   0x102530
#define      REG_TOUCH_DIRECT_XY     0x102574
#define      REG_TOUCH_DIRECT_Z1Z2   0x102578
#define      REG_TRACKER             0x109000


/************************************************************************ 
 **                                                                    **
 **                4.Define-FT800 Host Commands                        **
 **                                                                    **
 **  -Use by FT800_HostCmdWrite(). Write Control to FT800              **
 **                                                                    **
 ************************************************************************/

//Group Power Mode

#define FT800_ACTIVE      0x00         //Initializes FT800
#define FT800_STANDBY     0x41         //Place FT800 in Standby (clk running)
#define FT800_SLEEP       0x42         //Place FT800 in Sleep (clk off)
#define FT800_PWRDOWN     0x50         //Place FT800 in Power Down (core off)

//Group Clock Switch

#define FT800_CLKEXT      0x44         //Select external clock source
#define FT800_CLK48M      0x62         //Select 48MHz PLL
#define FT800_CLK36M      0x61         //Select 36MHz PLL

//Group Miscellaneous

#define FT800_CORERST     0x68         //Reset core - all registers default


/*************************************************************************** 
 **                                                                       **
 **           5.Define-FT800 Display list commands(DL-CMD)                **
 **                                                                       **
 **  -Use by FT800memWritexx()/FT800memReadxx(). Write to Ram-DL          **
 **                                                                       **
 ***************************************************************************/

#define DL_ALPHA_FUNC       0x09000000 // requires OR'd arguments
#define DL_BITMAP_HANDLE    0x05000000 // requires OR'd arguments
#define DL_BITMAP_LAYOUT    0x07000000 // requires OR'd arguments
#define DL_BITMAP_SIZE      0x08000000 // requires OR'd arguments
#define DL_BITMAP_SOURCE    0x01000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_A   0x15000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_B   0x16000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_C   0x17000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_D   0x18000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_E   0x19000000 // requires OR'd arguments
#define DL_BITMAP_TFORM_F   0x1A000000 // requires OR'd arguments
#define DL_BLEND_FUNC       0x0B000000 // requires OR'd arguments
#define DL_BEGIN            0x1F000000 // requires OR'd arguments
#define DL_CALL             0x1D000000 // requires OR'd arguments
#define DL_CLEAR            0x26000000 // requires OR'd arguments
#define DL_CELL             0x06000000 // requires OR'd arguments
#define DL_CLEAR_RGB        0x02000000 // requires OR'd arguments
#define DL_CLEAR_STENCIL    0x11000000 // requires OR'd arguments
#define DL_CLEAR_TAG        0x12000000 // requires OR'd arguments
#define DL_COLOR_A          0x0F000000 // requires OR'd arguments
#define DL_COLOR_MASK       0x20000000 // requires OR'd arguments
#define DL_COLOR_RGB        0x04000000 // requires OR'd arguments
#define DL_DISPLAY          0x00000000
#define DL_END              0x21000000
#define DL_JUMP             0x1E000000 // requires OR'd arguments
#define DL_LINE_WIDTH       0x0E000000 // requires OR'd arguments
#define DL_MACRO            0x25000000 // requires OR'd arguments
#define DL_POINT_SIZE       0x0D000000 // requires OR'd arguments
#define DL_RESTORE_CONTEXT  0x23000000
#define DL_RETURN           0x24000000
#define DL_SAVE_CONTEXT     0x22000000
#define DL_SCISSOR_SIZE     0x1C000000 // requires OR'd arguments
#define DL_SCISSOR_XY       0x1B000000 // requires OR'd arguments
#define DL_STENCIL_FUNC     0x0A000000 // requires OR'd arguments
#define DL_STENCIL_MASK     0x13000000 // requires OR'd arguments
#define DL_STENCIL_OP       0x0C000000 // requires OR'd arguments
#define DL_TAG              0x03000000 // requires OR'd arguments
#define DL_TAG_MASK         0x14000000 // requires OR'd arguments
#define DL_VERTEX2F         0x40000000 // requires OR'd arguments
#define DL_VERTEX2II        0x80000000 // requires OR'd arguments


/******************************************************************************
 **                                                                          **
 **           6.Define-FT800 Co-Processor Engine Commands(CO-CMD)            **
 **                                                                          **
 **         -Use by FT800memWritexx()/FT800memReadxx().Write to Ram-CMD      **
 **                                                                          **
 ******************************************************************************/

#define    CMD_APPEND            0xFFFFFF1E    //Append memory to display list
#define    CMD_BGCOLOR           0xFFFFFF09    //Set the background color
#define    CMD_BUTTON            0xFFFFFF0D    //Draw a button
#define    CMD_CALIBRATE         0xFFFFFF15    //Execute the touch screen calibration routine
#define    CMD_CLOCK             0xFFFFFF14    //Draw an analog clock
#define    CMD_COLDSTART         0xFFFFFF32    //Set co-processor engine state to default values
#define    CMD_DIAL              0xFFFFFF2D    //Draw a rotary dial control
#define    CMD_DLSTART           0xFFFFFF00    //Start a new display list
#define    CMD_FGCOLOR           0xFFFFFF0A    //Set the foreground color
#define    CMD_GAUGE             0xFFFFFF13    //Draw a gauge
#define    CMD_GETMATRIX         0xFFFFFF33    //Retrieves the current matrix coefficients
#define    CMD_GETPTR            0xFFFFFF23    //Get the end memory address of inflated data
#define    CMD_GRADCOLOR         0xFFFFFF34    //Set the 3D Button highlight color
#define    CMD_GRADIENT          0xFFFFFF0B    //Draw a smooth color gradient
#define    CMD_INFLATE           0xFFFFFF22    //Decompress data into memory
#define    CMD_INTERRUPT_TRIG    0xFFFFFF02    //Trigger interrupt INT_CMDFLAG
#define    CMD_KEYS              0xFFFFFF0E    //Draw a row of keys
#define    CMD_LOADIDENTITY      0xFFFFFF26    //Set the current matrix to the identity matrix
#define    CMD_LOADIMAGE         0xFFFFFF24    //Load a JPEG image
#define    CMD_LOGO              0xFFFFFF31    //Play FTDI logo animation
#define    CMD_MEMCPY            0xFFFFFF1D    //Copy a block of memory
#define    CMD_MEMCRC            0xFFFFFF18    //Compute a CRC-32 for memory
#define    CMD_MEMSET            0xFFFFFF1B    //Fill memory with a byte value
#define    CMD_MEMWRITE          0xFFFFFF1A    //Write bytes into memory
#define    CMD_MEMZERO           0xFFFFFF1C    //Write zero to a block of memory
#define    CMD_NUMBER            0xFFFFFF2E    //Draw a decimal number 
#define    CMD_PROGRESS          0xFFFFFF0F    //Draw a process bar
#define    CMD_REGREAD           0xFFFFFF19    //Read a register value
#define    CMD_ROTATE            0xFFFFFF29    //Apply a rotation to the current matrix
#define    CMD_SCALE             0xFFFFFF28    //Apply a scale to the current matrix
#define    CMD_SCREENSAVER       0xFFFFFF2F    //Start an animated screensaver
#define    CMD_SCROLLBAR         0xFFFFFF11    //Draw a scroll bar
#define    CMD_SETFONT           0xFFFFFF2B    //Set up a custom font 
#define    CMD_SETMATRIX         0xFFFFFF2A    //Write the current matrix to the display list
#define    CMD_SKETCH            0xFFFFFF30    //Start a continuous sketch update
#define    CMD_SLIDER            0xFFFFFF10    //Draw a slider
#define    CMD_SNAPSHOT          0xFFFFFF1F    //Take a snapshot of the current screen
#define    CMD_SPINNER           0xFFFFFF16    //Start an animated spinner
#define    CMD_STOP              0xFFFFFF17    //Stop an of spinner, screensaver or sketch
#define    CMD_SWAP              0xFFFFFF01    //Swap the current display list
#define    CMD_TEXT              0xFFFFFF0C    //Draw text
#define    CMD_TOGGLE            0xFFFFFF12    //Draw a toggle switch
#define    CMD_TRACK             0xFFFFFF2C    //Track touches for a graphics object
#define    CMD_TRANSLATE         0xFFFFFF27    //Apply a translation to the current matrix

#define    CMD_IDCT              0xFFFFFF06
#define    CMD_MARCH             0xFFFFFF05
#define    CMD_TOUCH_TRANSFORM   0xFFFFFF20 
#define    CMD_BITMAP_TRANSFORM  0xFFFFFF21
#define    CMD_CRC               0xFFFFFF03
#define    CMD_EXECUTE           0xFFFFFF07 
#define    CMD_GETPOINT          0xFFFFFF08
#define    CMD_GETPROPS          0xFFFFFF25
#define    CMD_HAMMERAUX         0xFFFFFF04 



//*********** 7.Data Parameter Option of DL-CMD,Co-CMD and Register *************

#define CLR_COL              0x4
#define CLR_STN              0x2
#define CLR_TAG              0x1
#define DECR                 4
#define DECR_WRAP            7
#define DLSWAP_DONE          0
#define DLSWAP_FRAME         2
#define DLSWAP_LINE          1
#define DST_ALPHA            3
#define EDGE_STRIP_A         7
#define EDGE_STRIP_B         8
#define EDGE_STRIP_L         6
#define EDGE_STRIP_R         5
#define EQUAL                5
#define GEQUAL               4
#define GREATER              3
#define INCR                 3
#define INCR_WRAP            6
#define INT_CMDEMPTY         32
#define INT_CMDFLAG          64
#define INT_CONVCOMPLETE     128
#define INT_PLAYBACK         16
#define INT_SOUND            8
#define INT_SWAP             1
#define INT_TAG              4
#define INT_TOUCH            2
#define INVERT               5
#define KEEP                 1
#define L1                   1
#define L4                   2
#define L8                   3
#define LEQUAL               2
#define LESS                 1
#define LINEAR_SAMPLES       0
#define LINES                3
#define LINE_STRIP           4
#define NEAREST              0
#define NEVER                0
#define NOTEQUAL             6
#define ONE                  1
#define ONE_MINUS_DST_ALPHA  5
#define ONE_MINUS_SRC_ALPHA  4
#define OPT_CENTER           1536    // 0x6000
#define OPT_CENTERX          512     // 0x0200
#define OPT_CENTERY          1024    // 0x0400
#define OPT_FLAT             256     // 0x0100
#define OPT_MONO             1
#define OPT_NOBACK           4096    // 0x1000
#define OPT_NODL             2
#define OPT_NOHANDS          49152   // 0xC168
#define OPT_NOHM             16384   // 0x4000
#define OPT_NOPOINTER        16384   // 0x4000
#define OPT_NOSECS           32768   // 0x8000
#define OPT_NOTICKS          8192    // 0x2000
#define OPT_RIGHTX           2048    // 0x0800
#define OPT_SIGNED           256     // 0x0100
#define PALETTED             8
#define PLAYCOLOR            0x00A0A080
#define FTPOINTS             2       // "POINTS" is a reserved word
#define RECTS                9
#define REPEAT               1
#define REPLACE              2
#define RGB332               4
#define RGB565               7
#define SRC_ALPHA            2
#define TEXT8X8              9
#define TEXTVGA              10
#define TOUCHMODE_CONTINUOUS 3
#define TOUCHMODE_FRAME      2
#define TOUCHMODE_OFF        0
#define TOUCHMODE_ONESHOT    1
#define ULAW_SAMPLES         1
#define ZERO                 0
#define BITMAPS              1



//***************** 8.Define Color RGB 24 bit *******************


#define      AliceBlue            0xF0F8FF
#define      AntiqueWhite         0xFAEBD7
#define      Aquamarine           0x7FFFD4
#define      Azure                0xF0FFFF
#define      Beige                0xF5F5DC
#define      Bisque               0xFFFAF0
#define      Black                0x000000
#define      BlanchedAlmond       0xFFEBCD
#define      Blue                 0x0000FF
#define      BlueViolet           0x8A2BE2
#define      Brown                0xA52A2A
#define      Burlywood            0xDEB887
#define      CadetBlue            0x5F9EA0
#define      Chartreuse           0x7FFF00
#define      Chocolate            0xD2691E
#define      Coral                0xFF7F50
#define      CornflowerBlue       0x6495ED
#define      Cornsilk             0xFFF8DC
#define      Cyan                 0x00FFFF
#define      DarkGoldenrod        0xB8860B
#define      DarkGreen            0x006400
#define      DarkKaki             0xBDB76B
#define      DarkOliveGreen       0x556B2F
#define      DarkOrange           0xFF8C00
#define      DarkOrchid           0x9932CC
#define      DarkSalmon           0xE9967A
#define      DarkSeaGreen         0x8FBC8F
#define      DarkSlateBlue        0x483D8B
#define      DarkSlateGray        0x2F4F4F
#define      DarkTurquoise        0x00CED1
#define      DarkViolet           0x9400D3
#define      DeepPink             0xFF1493
#define      DeepSkyBlue          0x00BFFF
#define      DimGrey              0x696969
#define      DodgeBlue            0x1E90FF
#define      Firebrick            0xB22222
#define      FloralWhite          0xFFFAF0
#define      ForestGreen          0x228B22
#define      Gainsboro            0xDCDCDC
#define      GhostWhite           0xF8F8FF
#define      Gold                 0xFFD700
#define      Goldenrod            0xDAA520
#define      Gray                 0xBEBEBE
#define      Green                0x00FF00
#define      GreenYellow          0xADFF2F
#define      Honeydew             0xF0FFF0
#define      HotPink              0xFF69B4
#define      IndianRed            0xCD5C5C
#define      LtGoldenrodYellow    0xFAFAD2
#define      Ivory                0xFFFFF0
#define      lavender             0xE6E6FA
#define      Lavender_Blush       0xFFF0F5
#define      LawnGreen            0x7CFC00
#define      LemonChiffon         0xFFFACD
#define      LightBlue            0xADD8E6
#define      LightCoral           0xF08080
#define      LightCyan            0xE0FFFF
#define      LightGoldenrod       0xEEDD82
#define      LightGray            0xD3D3D3
#define      LightPink            0xFFB6C1
#define      LightSeaGreen        0x20B2AA
#define      LightSkyBlue         0x87CEFA
#define      LightSalmon          0xFFA07A
#define      LightSlateBlue       0x8470FF
#define      LightSlateGray       0x778899
#define      LightSteelBlue       0xB0C4DE
#define      LightYellow          0xFFFFE0
#define      LimeGreen            0x32CD32
#define      Linen                0xFAF0E6
#define      Magenta              0xFF00FF
#define      Maroon               0xB03060
#define      MediumAquamarine     0x66CDAA
#define      MediumBlue           0x0000CD
#define      MediumOrchid         0xBA55D3
#define      MediumPurple         0x9370DB
#define      MediumSeaGreen       0x3CB371
#define      MediumSlateBlue      0x7B68EE
#define      MediumTurquoise      0x48D1CC
#define      MediumVioletRed      0xC71585
#define      MedSpringGreen       0x00FA9A
#define      MidnightBlue         0x191970
#define      MintCream            0xF5FFFA
#define      MistyRose            0xFFE4E1
#define      Moccasin             0xFFE4B5
#define      NavyBlue             0x000080
#define      NavajoWhite          0xFFDEAD
#define      OldLace              0xFDF5E6
#define      OliveDrab            0x6B8E23
#define      Orange               0xFFA500
#define      OrangeRed            0xFF4500
#define      Orchid               0xDA70D6
#define      PaleGoldenrod        0xEEE8AA
#define      PaleGreen            0x98FB98
#define      PaleTurquoise        0xAFEEEE
#define      PaleVioletRed        0xDB7093
#define      PapayaWhip           0xFFEFD5
#define      PeachPuff            0xFFDAB9
#define      Peru                 0xCD853F
#define      Pink                 0xFFc0CB
#define      Plum                 0xDDA0DD
#define      PowderBlue           0xB0E0E6
#define      Purple               0xA020F0
#define      Red                  0xFF0000
#define      RosyBrown            0xBC8F8F
#define      RoyalBlue            0x4169E1
#define      Salmon               0xFA8072
#define      SaddleBrown          0x8B4513
#define      SandyBrown           0xF4A460
#define      SeaGreen             0x2E8B57
#define      Seashell             0xFFF5EE
#define      Sienna               0xA0522D
#define      SkyBlue              0x87CEEB
#define      SlateBlue            0x6A5ACD
#define      SlateGrey            0x708090
#define      Snow                 0xFFFAFA
#define      SpringGreen          0x00FF7F
#define      SteelBlue            0x4682B4
#define      Tan                  0xD2B48C
#define      Tomato               0xFF6347   
#define      Thistle              0xD8BFD8
#define      Turquoise            0x40E0D0
#define      Violet               0xEE82EE
#define      Wheat                0xF5DEB3
#define      White                0xFFFFFF
#define      WhiteSmoke           0xF5F5F5
#define      Yellow               0xFFFF00
#define      YellowGreen          0x9ACD32
#define      VioletRed            0xD02090




//***************** 9.Define Sound Effect ,Use with REG_SOUND 8 bit Low  *******************
                                     
                                      /* Continuous / Pitch adjus midi */    
                              
#define      Silence         0x00    //       yes    /        no          :No-sound
#define      SquareWave      0x01    //       yes    /        yes        
#define      SineWave        0x02    //       yes    /        yes       
#define      SawtoothWave    0x03    //       yes    /        yes        
#define      TriangleWave    0x04    //       yes    /        yes        
#define      Beeping         0x05    //       yes    /        yes 
#define      Alarm           0x06    //       yes    /        yes 
#define      Warble          0x07    //       yes    /        yes 
#define      Carousel        0x08    //       yes    /        yes 
#define      Short1Pip       0x10    //       no     /        yes   
#define      Short2Pips      0x11    //       no     /        yes  
#define      Short3Pips      0x12    //       no     /        yes  
#define      Short4Pips      0x13    //       no     /        yes  
#define      Short5Pips      0x14    //       no     /        yes  
#define      Short6Pips      0x15    //       no     /        yes  
#define      Short7Pips      0x16    //       no     /        yes  
#define      Short8Pips      0x17    //       no     /        yes  
#define      Short9Pips      0x18    //       no     /        yes  
#define      Short10Pips     0x19    //       no     /        yes  
#define      Short11Pips     0x1A    //       no     /        yes  
#define      Short12Pips     0x1B    //       no     /        yes  
#define      Short13Pips     0x1C    //       no     /        yes  
#define      Short14Pips     0x1D    //       no     /        yes  
#define      Short15Pips     0x1E    //       no     /        yes  
#define      Short16Pips     0x1F    //       no     /        yes  
#define      DTMF_shap       0x23    //       yes    /        no  
#define      DTMF_star       0x2A    //       yes    /        no  
#define      DTMF0           0x30    //       yes    /        no  
#define      DTMF1           0x31    //       yes    /        no  
#define      DTMF2           0x32    //       yes    /        no  
#define      DTMF3           0x33    //       yes    /        no  
#define      DTMF4           0x34    //       yes    /        no  
#define      DTMF5           0x35    //       yes    /        no  
#define      DTMF6           0x36    //       yes    /        no  
#define      DTMF7           0x37    //       yes    /        no  
#define      DTMF8           0x38    //       yes    /        no  
#define      DTMF9           0x39    //       yes    /        no  
#define      Harp            0x40    //       no     /        yes  
#define      Xylophone       0x41    //       no     /        yes  
#define      Tuba            0x42    //       no     /        yes  
#define      Glockenspiel    0x43    //       no     /        yes  
#define      Organ           0x44    //       no     /        yes  
#define      Trumpet         0x45    //       no     /        yes  
#define      Piano           0x46    //       no     /        yes  
#define      Chimes          0x47    //       no     /        yes  
#define      MusicBox        0x48    //       no     /        yes  
#define      Bell            0x49    //       no     /        yes  
#define      Click           0x50    //       no     /        yes  
#define      Swit            0x51    //       no     /        yes  
#define      Cowbell         0x52    //       no     /        yes  
#define      Notch           0x53    //       no     /        yes  
#define      Hihat           0x54    //       no     /        yes  
#define      Kickdrum        0x55    //       no     /        yes  
#define      Pop             0x56    //       no     /        yes  
#define      Clack           0x57    //       no     /        yes  
#define      Chack           0x58    //       no     /        yes  
#define      Mute            0x60    //       no     /        yes  
#define      Unmute          0x61    //       no     /        yes  



//***************** 10.Define MIDI Note Effect ,Use with REG_SOUND 8 bit High  *******************

                              /* Freq (Hz)*/

#define      midC       0        //Level MiddelC
#define      A0        21        //27.5   
#define      A_0       22        //29.1
#define      B0        23        //30.9
#define      C1        24        //32.7
#define      C_1       25        //34.6
#define      D1        26        //36.7
#define      D_1       27        //38.9
#define      E1        28        //41.2
#define      F1        29        //43.7
#define      F_1       30        //46.2
#define      G1        31        //49.0
#define      G_1       32        //51.9
#define      A1        33        //55.0
#define      A_1       34        //58.3
#define      B1        35        //61.7
#define      C2        36        //65.4
#define      C_2       37        //69.3
#define      D2        38        //73.4
#define      D_2       39        //77.8
#define      E2        40        //82.4
#define      F2        41        //87.3
#define      F_2       42        //92.5
#define      G2        43        //98.0
#define      G_2       44        //103.8
#define      A2        45        //110.0
#define      A_2       46        //116.5
#define      B2        47        //123.5
#define      C3        48        //130.8
#define      C_3       49        //138.6
#define      D3        50        //146.8
#define      D_3       51        //155.6
#define      E3        52        //164.8
#define      F3        53        //174.6
#define      F_3       54        //185.0
#define      G3        55        //196.0
#define      G_3       56        //207.7
#define      A3        57        //220.0
#define      A_3       58        //233.1
#define      B3        59        //246.9
#define      C4        60        //261.6
#define      C_4       61        //277.2
#define      D4        62        //293.7
#define      D_4       63        //311.1
#define      E4        64        //329.6
#define      F4        65        //349.2
#define      F_4       66        //370.0
#define      G4        67        //392.0
#define      G_4       68        //415.3
#define      A4        69        //440.0
#define      A_4       70        //466.2
#define      B4        71        //493.9
#define      C5        72        //523.3
#define      C_5       73        //554.4
#define      D5        74        //587.3
#define      D_5       75        //622.3
#define      E5        76        //659.3
#define      F5        77        //698.5
#define      F_5       78        //740.0
#define      G5        79        //784.0
#define      G_5       80        //830.6
#define      A5        81        //880.0
#define      A_5       82        //932.3
#define      B5        83        //987.8
#define      C6        84        //1046.5
#define      C_6       85        //1108.7
#define      D6        86        //1174.7
#define      D_6       87        //1244.5
#define      E6        88        //1318.5
#define      F6        89        //1396.9
#define      F_6       90        //1480.0
#define      G6        91        //1568.0
#define      G_6       92        //1661.2
#define      A6        93        //1760.0 
#define      A_6       94        //1864.7
#define      B6        95        //1975.5
#define      C7        96        //2093.0
#define      C_7       97        //2217.5
#define      D7        98        //2349.3
#define      D_7       99        //2489.0
#define      E7       100        //2637.0
#define      F7       101        //2793.8
#define      F_7      102        //2960.0
#define      G7       103        //3136.0
#define      G_7      104        //3322.4
#define      A7       105        //3520.0
#define      A_7      106        //3729.3
#define      B7       107        //3951.1
#define      C8       108        //4186.0




/*#################################################################
 ##                                                              ##
 ##                        Define Variable                       ##
 ##                                                              ##
 #################################################################*/


unsigned int16 DL_Offset   = 0x0000  ;          //Offset Address Pointer of display list memory 
unsigned int16 cmdOffset   = 0x0000  ;          //Offset Address Pointer of Command Buffer memmory

unsigned int16 cmdBufferRd = 0x0000  ;          //Used to navigate command ring buffer
unsigned int16 cmdBufferWr = 0x0000  ;          //Used to navigate command ring buffer

unsigned int8 ft800Gpio              ;          //Used for FT800 GPIO register

unsigned int8 num_cha                ;          //keep Number Print Text 
unsigned int16 tc_x,tc_y             ;          //keeop Positin Touch Screen X,Y 


//----------- Valiable for Application -----------

 signed int16 hh=12,mm=0,ss=0        ;
 unsigned int8 mode=0                ; 


/*#################################################################
 ##                                                              ##
 ##              Define Function Sub-program                     ##
 ##                                                              ##
 ##################################################################*/

                               

/****************************************************************
 **                                                            **
 **    1.Function-About Manage Hardware/Software of MCU        **
 **                                                            **     
 ****************************************************************/

void init_port(void)                 ;



/***********************************************************************************
 **                                                                               **
 **         2. Function-About Software SPI Read/Write 8 bit (RJ0-RJ6)             **
 **                                                                               **
 ***********************************************************************************/

void spi_sck_hi(void)             ;
void spi_sck_lo(void)             ;
void spi_cs_hi(void)              ;
void spi_cs_lo(void)              ;
void spi_sdo_hi(void)             ;
void spi_sdo_lo(void)             ;
unsigned int8 spi_rw(unsigned int8 Db)   ;



/*******************************************************************************
 **                                                                           **
 **                       3.Function- About Write FT800                       **
 **                                                                           **
 *******************************************************************************/

//-------------- Function-FT800 Power-Down[RJ4] ------------------ 

void FT800_PD_hi(void)            ;
void FT800_PD_lo(void)            ;   //Set Power Down FT800

//------------Function-Write Register,Ram-DL,Ram-CMD (Address[22bit] and Data Parameter 8,16,32 bit)---------

void FT800memWrite8(unsigned int32 addr,unsigned int8 data8)       ;
void FT800memWrite16(unsigned int32 addr,unsigned int16 data16)    ;
void FT800memWrite32(unsigned int32 addr,unsigned int32 data32)    ;

//-------------Function-Write Address Ram-CMD and Data Text-String -----------------

void FT800memWriteStr(unsigned int32 addr,int8 *str)               ;  //Command for Write Address + Text-String 


//------------ Function-Write Host Command  (Command[6-bit]+0x00+0x00)-------------

void FT800_HostCmdWrite(unsigned int8 Command)                     ;




/*******************************************************************************
 **                                                                           **
 **                         4.Function-About Read FT800                       **
 **                                                                           **
 *******************************************************************************/


//------------Function-Read Register,Ram-DL,RAm-CMD (Address[22bit] + Data Parameter 8,16,32 bit)---------

unsigned int8 FT800memRead8(unsigned int32 addr)                   ;
unsigned int16 FT800memRead16(unsigned int32 addr)                 ;
unsigned int32 FT800memRead32(unsigned int32 addr)                 ; 




/***********************************************************************************
 **                                                                               **
 **           5.Function-About Manage memory address FT800 Ring Buffer            **              
 **                                                                               **
 ***********************************************************************************/


//----------- Function-Increment Address Offset Pointer for Ram-CMD --------------

unsigned int16 incCMDOffset(unsigned int16 currentOffset,unsigned int8 commandSize)     ; 
unsigned int16 End_incCMDOffset(unsigned int16 currentOffset,unsigned int8 commandSize) ; //Use when end Co-Processor CMD 

//-------- Function-Start Offset pointer & Waite Co-Processor CMD Finish executing -------------

void Start_CMDOffset(void) ;



/*************************************************************************************
 **                                                                                 **
 **                         6.Function-About Touch Screen                           **
 **                                                                                 **
 *************************************************************************************/

int8 Touch_Detect(void)    ;
void Touch_Read_XY(void)   ;
void Touch_Calibrate(void) ;



/*************************************************************************************
 **                                                                                 **
 **                               7.Function-About Audio                            **
 **                                                                                 **
 *************************************************************************************/

 void Sound_Effect(unsigned int8 vol,int16 midi,int16 effect) ;


/*************************************************************************************
 **                                                                                 **
 **                         8.Function-About FT800 DL-CMD,Co-CMD                    **
 **                                                                                 **
 *************************************************************************************/



void Start_CMDList(int32 screen_color);
void End_CMDList(void)                ;
void Background_DL(int32 bg_color)    ;
void Button_CMD(int8 *str,int16 x,int16 y,int16 wide,int16 high,int16 font,int16 opt,int32 tx_color,int32 bt_color) ;
void Clock_CMD(int16 x,int16 y,int16 radius,int16 opt,int16 hour,int16 minute,int16 second ,int16 ms,int32 pin_color,int32 clock_color) ;
void Text_CMD(int8 *str,int16 x,int16 y,int16 font,int16 opt,int32 tx_color) ;


/*************************************************************************************
 **                                                                                 **
 **                       9.Function-About User Application                         **
 **                                                                                 **
 *************************************************************************************/

void FT800_Initial(void)   ;
void Update_Screen(int32 btset_color,int32 tx_color,int32 bt_color,int32 btup_color,int32 btdown_color);
void User_App(void)        ;





/*##########################################################################
 ##                                                                       ##
 ##                            MAIN PROGRAM                               ##
 ##                                                                       ##
 ###########################################################################*/

 void main(void)
 {
  
   init_port()        ;
   FT800_Initial()    ;
   Touch_Calibrate()  ;
   User_App()         ;

}
    


/*################################################################################
  ##                                                                            ##
  ##                          Function SUB-PROGRAM                              ##
  ##                                                                            ##
  ################################################################################ */

 

/*****************************************************************************
 **                                                                         **
 **            1.Function-About Manage Hardware/Software of MCU             **
 **                                                                         **     
 *****************************************************************************/


 /*+++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                ++
  ++         1.1) Function-Initial Port             ++
  ++                                                ++ 
  +++++++++++++++++++++++++++++++++++++++++++++++++++*/

 void init_port(void)
  {
   set_tris_J(0x21)          ;   //Set RJ1[MOSI],RJ2[SCK],RJ3[CS],RJ4[PD#],RJ6[SD_CS] = Output:0  ; RJ0[MISO],RJ5(INT)=Input:1
 
   //------ Set Value Default Port ------ 
         
   output_J(0x5A)            ; //RJ4(PD)=1(Disable Power down),RJ3(CS)=1,RJ2(SCK)=0,RJ1(MOSI)=1 ,RJ6(SD_CS) =1  

  }



/***********************************************************************************
 **                                                                               **
 **         2. Function-About Software SPI Read/Write 8 bit (RJ0-RJ6)             **
 **                                                                               **
 ***********************************************************************************/


//++++++++++++++++ Data MOSI [RJ1] ++++++++++++++++++

void spi_sdo_hi(void)
 {
   output_high(PIN_J1) ;   //RJ1=1
 }

void spi_sdo_lo(void)
 {
   output_low(PIN_J1) ;   //RJ1=0
 }


//+++++++++++++++++ SCK [RJ2] ++++++++++++++++++++++

void spi_sck_hi(void)
 {
   output_high(PIN_J2);   //RJ2=1
 }

void spi_sck_lo(void) 
 {
  output_low(PIN_J2)  ;   //RJ2=1(Ideal)
 }


//+++++++++++++++++ CS [RJ3] ++++++++++++++++++++++

 void spi_cs_hi(void)
 {
   output_high(PIN_J3);   //RJ3=1
 }

void spi_cs_lo(void)
 {
   output_low(PIN_J3) ;   //RJ3=0(Active)
 }



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 ++                                                                                   ++
 ++                  2.1) Function-SPI Read/Write Data 1 Byte                         ++
 ++                                                                                   ++
 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 ++   Input :                                                                         ++
 ++           Db = Data or Cmmand  1 Byte (Send data most significant bit first)      ++
 ++                                                                                   ++
 ++   Output:                                                                         ++
 ++           result = Return data 1 byte Pass Function                               ++
 ++                                                                                   ++
 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


unsigned int8 spi_rw(unsigned int8 Db)              
{
  unsigned int8 bt,result=0;                   
  
     for(bt=0;bt<8;bt++)                      //Loop read/write 8 bit
      {       
      //--------- Write Data(RJ1) ---------
 
        if((Db & 0x80)== 0x80)
          spi_sdo_hi()                     ;    //Set bit data(RJ1) = 1
        else
          spi_sdo_lo()                     ;    //Clear bit data(RJ1) = 0
 
        spi_sck_hi()                       ;    //Start Clock  High (active rise edge) 
     
        Db  <<= 1                          ;    //Shift Next Bit Data
         
        //---------- Read Data(RJ0) -------
        
        result <<= 1                       ;    //Shift data read to MSB bit     
     
        if(input(PIN_J0)== 1)                   //If SDI = "1"                  
         {
           result |= 0x01                  ;    //Keep Result of Read-data        
         } 
        spi_sck_lo()                       ;      
     }
 
     return result                         ;
}



/*******************************************************************************
 **                                                                           **
 **                       3.Function- About Write FT800                       **
 **                                                                           **
 *******************************************************************************/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                         ++
  ++                   3.1) Function-FT800 Power-Down[RJ4]                   ++
  ++                                                                         ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void FT800_PD_hi(void)
{
   output_high(PIN_J4)     ;   //RJ4=1:Disable Power-Down FT800
}


void FT800_PD_lo(void)
{
   output_low(PIN_J4)      ;   //RJ4=0:Enable Power-Down FT800
}


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                         ++
  ++          3.2) Function-Write Register,Ram-DL,Ram-CMD +Data              ++
  ++                                                                         ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++ Input:                                                                  ++
  ++         addr    = FT800 memory space address Ram-DL,Ram-CMD,Reg.        ++
  ++                                                                         ++
  ++         dataxx  = Data for Co-CMD,DL-CMD or Reg.                        ++
  ++                                                                         ++
  ++ Overview:         Writes FT800 internal address space                   ++
  ++ Note:             "xx" is data of 8, 16 or 32 bit                       ++
  ++                                                                         ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


//++++++++++++++ Function-Write Address Register,Ram-DL,Ram-CMD + Data 8 bit ++++++++++++++++


void FT800memWrite8(unsigned int32 addr, unsigned int8 data8)
{
  spi_cs_lo()                     ; // Set CS# low
  
  //sent address Ram
  spi_rw(((addr>>16)|MEM_WRITE))  ; // Send Memory Write plus high address byte
  spi_rw(addr>>8)                 ; // Send middle address byte
  spi_rw(addr)                    ; // Send low address byte
 
 //sent data 1 byte
  spi_rw(data8)                   ; // Send data byte

  spi_cs_hi()                     ; // Set CS# high
}


//++++++++++++++ Function-Write Address Register,Ram-DL,Ram-CMD + Data 16 bit++++++++++++++++

void FT800memWrite16(unsigned int32 addr, unsigned int16 data16)
{

  spi_cs_lo()                     ;   // Set CS# low

  //sent address Ram
  spi_rw(((addr>>16)|MEM_WRITE))  ;   // Send Memory Write plus high address byte
  spi_rw(addr>>8)                 ;   // Send middle address byte
  spi_rw(addr)                    ;   // Send low address byte

 //sent data 2 byte
  spi_rw(data16)                  ;   // Send data low byte
  spi_rw(data16>>8)               ;   // Send data high byte
  
  spi_cs_hi()                     ;   // Set CS# high

}

//++++++++++++ Function-Write Address Register,Ram-DL,Ram-CMD + Data 32 bit +++++++++++++++

void FT800memWrite32(unsigned int32 addr, unsigned int32 data32)
{
  spi_cs_lo()                     ;   // Set CS# low

//sent address Ram
  spi_rw(((addr>>16)|MEM_WRITE))  ;   // Send Memory Write plus high address byte
  spi_rw(addr>>8)                 ;   // Send middle address byte
  spi_rw(addr)                    ;   // Send low address byte

//Sent data 4 byte
  spi_rw(data32)                  ;   // Send data low byte
  spi_rw(data32>>8)               ;   
  spi_rw(data32>>16)              ;   
  spi_rw(data32>>24)              ;   // Send data high byte
  
  spi_cs_hi()                     ;   // Set CS# high

}



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                        ++
  ++        3.3) Function-Write Address Ram-CMD and Data Text-String        ++
  ++                                                                        ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++ Input  :                                                               ++
  ++           addr = FT800 memory space address Ram-CMD                    ++
  ++           str  = Data Text String                                      ++
  ++                                                                        ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

                                                          
void FT800memWriteStr(unsigned int32 addr,int8 *str)
{
  unsigned int16 i ;
  
  num_cha = 0                     ;
  spi_cs_lo()                     ;   // Set CS# low
  
  //sent address ram
  spi_rw(((addr>>16)|MEM_WRITE))  ;   // Send Memory Write plus high address byte
  spi_rw(addr>>8)                 ;   // Send middle address byte
  spi_rw(addr)                    ;   // Send low address byte

 // sent data string 
 for (i=0; str[i] != '\0'; i++)      //Check End String In C
  {                 
     spi_rw(str[i])              ;
     num_cha +=  1               ;
  }

  spi_rw(0)                      ;   //Sent End Write String Sent 0

  num_cha = num_cha+1            ;

  spi_cs_hi()                    ;   // Set CS# high

}



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                        ++
  ++      3.4) Function-Write Host Command  (Command[6-bit]+0x00+0x00)      ++
  ++                                                                        ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++  Input :                                                               ++
  ++           Command = FT800 Command                                      ++
  ++                                                                        ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void FT800_HostCmdWrite(unsigned int8 Command)
{
  spi_cs_lo()            ;  // Set CS# low

  spi_rw(Command)        ;  // Send command
  spi_rw(0x00)           ;  // Commands consist of two more zero bytes
  spi_rw(0x00)           ;  // Send last zero byte

  spi_cs_hi()            ;  // Set CS# high
}



/*******************************************************************************
 **                                                                           **
 **                         4.Function-About Read FT800                       **
 **                                                                           **
 *******************************************************************************/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                         ++
  ++            4.1) Function-Read data Register,Ram-DL,Ram-CMD              ++
  ++                                                                         ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++ Input    :                                                              ++
  ++            Address = FT800 memory space address Ram-DL,CMD,Reg          ++
  ++ Output   :                                                              ++
  ++            Dataxx  = data Read for Co-CMD,DL-CMD or Reg.                ++
  ++                                                                         ++
  ++ Note     : "xx" is data of 8, 16 or 32 bit                              ++
  ++                                                                         ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


//++++++++++++++++++ Function-Read data Memory 8 bit +++++++++++++++++++++++

unsigned int8 FT800memRead8(unsigned int32 addr)
{
  unsigned int8 data8 ;

  spi_cs_lo()                     ; // Set CS# low

  //Sent Address Ram for read
  spi_rw(((addr>>16)&MEM_READ))   ; // Send Memory Write plus high address byte
  spi_rw(addr>>8)                 ; // Send middle address byte
  spi_rw(addr)                    ; // Send low address byte
  
  spi_rw(ZERO)                    ; // Send dummy byte
  
  //Read data 1 byte
  data8 = spi_rw(ZERO)            ; // Read data byte

  spi_cs_hi()                     ; // Set CS# high
  
  return data8                    ; // Return byte read


}


//++++++++++++++++++++ Function-Read data Memory 16 bit ++++++++++++++++++++

unsigned int16 FT800memRead16(unsigned int32 addr)
{
  unsigned int16 data16,dat;

  spi_cs_lo()                                  ; // Set CS# low
  
   //Sent Address Ram for read
  spi_rw(((addr>>16)&MEM_READ))                ; // Send Memory Write plus high address byte
  spi_rw(addr>>8)                              ; // Send middle address byte
  spi_rw(addr)                                 ; // Send low address byte
  
  spi_rw(ZERO)                                 ; // Send dummy byte

  //Read data 2 byte 
  data16 = spi_rw(ZERO)                        ; // Read low byte
  dat    = spi_rw(ZERO)                        ; // Read high byte
  data16 = (dat << 8)| data16                  ; 
 

  spi_cs_hi()                                  ; // Set CS# high
    
  return data16;                                 // Return integer read
}


//+++++++++++++++++ Function-Read data Memory 32 bit +++++++++++++++++++++

unsigned int32 FT800memRead32(unsigned int32 addr)
{
  unsigned int32 data32,dat;

  spi_cs_lo()                                  ; // Set CS# low

   //Sent Address Ram for read
  spi_rw(((addr>>16)&MEM_READ))                ; // Send Memory Write plus high address byte
  spi_rw(addr>>8)                              ; // Send middle address byte
  spi_rw(addr)                                 ; // Send low address byte
 
  spi_rw(ZERO)                                 ; // Send dummy byte

  //Read data 4 byte 
  data32 = spi_rw(ZERO)                        ; // Read low byte
  dat    = spi_rw(ZERO)                        ; 
  data32 = (dat<<8)| data32                    ;
  dat    = spi_rw(ZERO)                        ; 
  data32 = (dat<<16)| data32                   ;
  dat    = spi_rw(ZERO)                        ; // Read high byte 
  data32 = (dat<<24)| data32                   ;

  spi_cs_hi()                                  ; // Set CS# high
    
  return data32                                ; // Return integer read
}


/***********************************************************************************
 **                                                                               **
 **           5.Function-About Manage memory address FT800 Ring Buffer            **              
 **                                                                               **
 ***********************************************************************************/

 
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                        ++
  ++     5.1) Function-Waite Co-Processor Finish executing and              ++
  ++          Start Offset pointer                                          ++
  ++                                                                        ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void Start_CMDOffset(void) 

 {
  //------------Determine Pointer Offset for Ram_CMD Address---------------
    do
     {
       cmdBufferRd = FT800memRead16(REG_CMD_READ)      ;   //***Read Reg.the graphics processor read pointer
       cmdBufferWr = FT800memRead16(REG_CMD_WRITE)     ;   //***Read Reg.the graphics processor write pointer
  
     }while(cmdBufferWr != cmdBufferRd)                ;   //Wait until the two registers match
  
      cmdOffset = cmdBufferWr                          ;   //The new starting point the first location after the last command

  }


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                                 ++
  ++         5.2) Function-Increment Offset Address Pointer for Ram-CMD              ++
  ++                                                                                 ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++ Input    :                                                                      ++
  ++            currentOffset = graphics processor command list pointer              ++
  ++            commandSize   = number of bytes data or CMD to increment the offset  ++
  ++                                                                                 ++
  ++ Output   : newOffset = the new ring buffer pointer after adding the command     ++ 
  ++                                                                                 ++
  ++ Overview : Adds commandSize to the currentOffset.                               ++
  ++            Checks for 4K ring-buffer offset roll-over                           ++
  ++                                                                                 ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


  //++++++++++++++++++ Function-Increment Offset Address +++++++++++++++++++++++

unsigned int16 incCMDOffset(unsigned int16 currentOffset, unsigned int8 commandSize)
{
    unsigned int16 newOffset                  ;   //used to hold new offset

    newOffset = currentOffset + commandSize   ;   //Calculate new offset

    if(newOffset > 4095)                          //If new offset past boundary...
    {
        newOffset = (newOffset - 4096)        ;   //... roll over pointer
    }

    return newOffset;                             //Return new offset
}



 //++++++++++++ Function-Increment Offset Address Use when end Co-Processor CMD (offset Address/4 = integer)+++++++++++++

unsigned int16 End_incCMDOffset(unsigned int16 currentOffset, unsigned int8 commandSize)
{
  unsigned int16 newOffset                    ;  //used to hold new offset

   newOffset = currentOffset + commandSize    ;  //Calculate new offset

   if(newOffset > 4095)                          //If new offset past boundary...
    {
        newOffset = (newOffset - 4096)        ;  //... roll over pointer
    }
   

  //increment the length and align it by 4 bytes//

   newOffset = (newOffset+3) & 4092           ;

   return newOffset ;                            //Return new offset
}



/*************************************************************************************
 **                                                                                 **
 **                         6.Function-About Touch Screen                           **
 **                                                                                 **
 *************************************************************************************/

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                             ++
  ++               6.1) Function-Read Statust Toucu  (Not Toch = 0xFFFF)         ++
  ++                                                                             ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

 int8 Touch_Detect(void)
  {
    unsigned int16 td  ;

    td = FT800memRead16(REG_TOUCH_RAW_XY)  ;   //***Read Reg.Toucu Detect
    
   if(td != 0xFFFF)   
     return 1       ;       //Touch return 1
   else
     return 0       ;       //No Touch return 0
      
  }

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                                ++
  ++          6.2) Function-Read Position Touch Screen XY [X=0-479,Y=0-271]         ++
  ++                                                                                ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void Touch_Read_XY(void)
 {
    unsigned int32 xy ;  

   xy = FT800memRead32(REG_TOUCH_SCREEN_XY)   ; //***Read Reg : Read Value Position XY Touch Screen

   tc_x = (xy >> 16) & 0x000001FF             ; //Mark X-Position 0-479 in Pixel

   tc_y = xy & 0x1FF                          ; //Mark Y-Position 0-271 in Pixel 

 }


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                      ++
  ++                6.3) Function-Calibrate Touch Screen                  ++
  ++                                                                      ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void Touch_Calibrate(void)
 {
   unsigned int8 strbuf[50];
    // Calibrate Touch Screen //

    //--- 1.Start Display List ---
   Start_CMDList(Black)    ;
    
    //--- 2.1 Co-Processor Draw Text-CMD --- 
   
   sprintf (strbuf,"Please tap on a dot")          ;
   Text_CMD(strbuf,240,139,26,OPT_CENTER,White)    ;

   //--- 2.2 Co-Processor Calibrate CMD --- 

    FT800memWrite32(RAM_CMD + cmdOffset,CMD_CALIBRATE)             ; //***Sent Co-CMD for Calibrate Touch Screen
    cmdOffset = incCMDOffset(cmdOffset,4)                          ; //Update Offset pointer Ram-CMD 4 byte for Next CMD


   //--- 3.End Display List and Update Execute ----

    End_CMDList();

    Start_CMDOffset()                                              ; //Wait Calibrate Execute  


}


/*************************************************************************************
 **                                                                                 **
 **                               7.Function-About Audio                            **
 **                                                                                 **
 *************************************************************************************/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                      ++
  ++                    7.1) Function-Sound Effect                        ++
  ++                                                                      ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++     
  ++   Input : vol    = Value Volume(0-255) by min=0,max=255              ++
  ++           midi   = MIDI note for the sound effect (value 22-108)     ++                                          
  ++           effect = Sound Effect (0x00-0x08,0x10-0x1F,23,2C...)       ++
  ++                    00 : No Sound                                     ++                                                     
  ++                                                                      ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

 void Sound_Effect(unsigned int8 vol,int16 midi,int16 effect)
  {

    midi = midi << 8                                ;  //Set data Midi 8 bit High
    FT800memWrite8(REG_VOL_SOUND,vol)               ;  //***Write data to Reg : Turn synthesizer volume 0-0xFF   
    FT800memWrite16(REG_SOUND,midi | effect)        ;  //***Write data to Reg : Select Sound Effect = Silence(No Sound) 
    FT800memWrite8(REG_PLAY,0x01)                   ;  //***Write data to Reg : Play Sound Effect

  }


/*************************************************************************************
 **                                                                                 **
 **                     8.Function-About FT800 DL-CMD,Co-CMD                        **
 **                                                                                 **
 *************************************************************************************/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++     8.1) Function:Start Co-Processor Command List (Write Program to Ram_CMD) ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++  Input :                                                                     ++
  ++           screen_color  : Color Background Screen                            ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
  
  void Start_CMDList(int32 screen_color)
   {

    //-------------------- Start Display List ------------------------

     Start_CMDOffset()                                                  ; //Start Offset pointer for Ram-CMD

     FT800memWrite32(RAM_CMD + cmdOffset,CMD_DLSTART)                   ; //*** Sent Co-CMD Start the display list                                                     
     cmdOffset = incCMDOffset(cmdOffset,4)                              ; //Update Offset pointer Ram-CMD 4 byte for Next CMD     
    
     FT800memWrite32(RAM_CMD + cmdOffset,(DL_CLEAR_RGB | screen_color)) ; //*** Sent DL-CMD Clear color BackGround to Black (Set BackGround)                  
     cmdOffset = incCMDOffset(cmdOffset,4)                              ; //Update Offset pointer Ram-CMD 4 byte for Next CMD

     FT800memWrite32(RAM_CMD + cmdOffset,(DL_CLEAR | CLR_COL | CLR_STN | CLR_TAG));   //*** Sent DL-CMD Clear the screen - this and the previous prevent artifacts between lists                                                                               // Attributes are the color, stencil and tag buffers
     cmdOffset = incCMDOffset(cmdOffset,4)                              ;             //Update Offset pointer Ram-CMD 4 byte for Next CMD

   }


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++     8.2) Function:End Co-Processor Command List (Write Program to Ram_CMD)   ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

  void End_CMDList(void)               
  {
    //----------------- End Display List and Update Execute -----------------

    FT800memWrite32(RAM_CMD + cmdOffset,DL_DISPLAY)          ;//*** Sent DL-CMD End Display List                  
    cmdOffset = incCMDOffset(cmdOffset,4)                    ;//Update Offset pointer Ram-CMD 4 byte for Next CMD
 
    FT800memWrite32(RAM_CMD + cmdOffset,CMD_SWAP)            ;//*** Sent Co-CMD Requse Display List Swap
    cmdOffset = incCMDOffset(cmdOffset,4)                    ;//Update Offset pointer Ram-CMD 4 byte for Next CMD

    FT800memWrite16(REG_CMD_WRITE,cmdOffset)                 ;//*** Sent Data cmdOffset to reg. CMD_Write for
                                                              //Update the ring buffer pointer so the graphics processor starts executing
  } 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++     8.3) Function:Display-List Set Background  (Write Program to Ram_DL)     ++
  ++                                                                              ++                                                                       
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

  void Background_DL(int32 bg_color)               
  {

   FT800memWrite32(RAM_DL,(DL_CLEAR_RGB | bg_color))        ; //*** Sent DL-CMD Set Background Color  
     
  } 
   


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++     8.4) Function:Co-Processor Draw Button (Write Program to Ram_CMD)        ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++   Input :     *str =  Text Button label                                      ++
  ++               x,y  =  x,y codinate button top-left ,in pixel                 ++
  ++               wide =  Wide Button in pixel                                   ++
  ++               high =  High Button in pixel                                   ++
  ++               font =  Format font for Button label                           ++
  ++                opt =  0=3D Button ,256 = Flat Button                         ++
  ++           tx_color =  Color Text Button label                                ++
  ++           bt_color =  Color Button                                           ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void Button_CMD(int8 *str,int16 x,int16 y,int16 wide,int16 high,int16 font,int16 opt,int32 tx_color,int32 bt_color)
 {

      // Set Attribute Button //

    FT800memWrite32(RAM_CMD + cmdOffset,(DL_COLOR_RGB | tx_color))      ; //*** Sent DL-CMD Set Color Text              
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD


    FT800memWrite32(RAM_CMD + cmdOffset,CMD_FGCOLOR)                    ; //*** Sent Co-CMD Set Color Button                                                  
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD
   
    FT800memWrite32(RAM_CMD + cmdOffset,bt_color)                       ; //---Sent Data Color Button
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD

    // Start Draw Button1 //   

    FT800memWrite32(RAM_CMD + cmdOffset,CMD_BUTTON)                     ; //***Sent Co-CMD Build Button
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,x)                              ; //---Sent data X-Point top-left (Pixel) 
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,y)                              ; //---Sent data Y-Point  top-left (Pixel)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,wide)                           ; //---Sent data wide-Button (pixel)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,high)                           ; //---Sent data high-Button (Pixel)                                                     
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,font)                           ; //---Sent data format font in Button label (16,18,20,21,22...)                                                     
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,opt)                            ; //---Sent data Option Button Drawn 3D                                                     
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWriteStr(RAM_CMD + cmdOffset,str)                           ; //---Sent data Text Button Label (String)
    cmdOffset = End_incCMDOffset(cmdOffset,num_cha)                     ; //Update Offset pointer Ram-CMD for End CMD-Button  
 
 }


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                                       ++
  ++     8.5) Function:Co-Processor Draw Analog Clock  (Write Program to Ram_CMD)          ++
  ++                                                                                       ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++   Input :                                                                             ++
  ++               x,y  =  x,y codinate Clock Center ,in pixel                             ++
  ++            radius  =  Radius Clock in pixel                                           ++
  ++               opt  =  Set OPT_3D,OPT_FLAT,OPT_NOBACK,OPT_NOTICKS,OPT_NOSECS           ++
  ++                           OPT_NOHANDS,OPT_NOHM                                        ++
  ++              hour  =  Set Hour 1-12                                                   ++
  ++            minute  =  Set Minute 0-59                                                 ++
  ++            second  =  Set Second 0-59                                                 ++
  ++                ms  =  Set millisec 0-999                                              ++                    
  ++          pin_color =  Color pin Clock                                                 ++
  ++        clock_color =  Color Clock                                                     ++
  ++                                                                                       ++
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

void Clock_CMD(int16 x,int16 y,int16 radius,int16 opt,int16 hour,int16 minute,int16 second ,int16 ms,int32 pin_color,int32 clock_color)
 {

    // Set Attribute Clock //
 
    FT800memWrite32(RAM_CMD + cmdOffset,(DL_COLOR_RGB | pin_color))     ; //*** Sent DL-CMD Set Color Pin-Clock              
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD

    FT800memWrite32(RAM_CMD + cmdOffset,CMD_BGCOLOR)                    ; //*** Sent Co-CMD Set Background color of CLOCK
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD

    FT800memWrite32(RAM_CMD + cmdOffset,clock_color)                    ; //--- Sent data color bacground clock                                                      
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next CMD
   
   //Start Draw Analog Clock//    

    FT800memWrite32(RAM_CMD + cmdOffset,CMD_CLOCK)                      ; //***Sent Command Draw Clock
    cmdOffset = incCMDOffset(cmdOffset,4)                               ; //Update Offset pointer Ram-CMD 4 byte for Next data

    FT800memWrite16(RAM_CMD + cmdOffset,x)                              ; //---Sent data Clock Center Point-X (Pixel) 
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,y)                              ; //---Sent data Clock Center Point Y (Pixel) 
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,radius)                         ; //---Sent data Radius for Clock (pixel)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,opt)                            ; //---Sent datat Option Clock Drawn 3D   
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,hour)                           ; //---Sent data Time Houre(hh:0-23)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data 

    FT800memWrite16(RAM_CMD + cmdOffset,minute)                         ; //---Sent data Time Minute(mm:0-59)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,second)                         ; //---Sent data Time Secound (ss:0-59)
    cmdOffset = incCMDOffset(cmdOffset,2)                               ; //Update Offset pointer Ram-CMD 2 byte for Next Data

    FT800memWrite16(RAM_CMD + cmdOffset,ms)                             ; //---Sent Time ms (ms:0-1000)
    cmdOffset = End_incCMDOffset(cmdOffset,2)                           ; //Update Offset pointer Ram-CMD for End CMD-Clock

  }



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++       8.6) Function:Co-Processor Draw Text (Write Program to Ram_CMD)        ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++   Input :                                                                    ++
  ++                str =  Text String                                            ++
  ++                x,y =  x,y codinate Text base ,in pixel                       ++
  ++               font =  format font text(0-31)                                 ++
  ++               opt  =  Set Position Text default 0 top-left  or               ++
  ++                       OPT_CENTERX,OPT_CENTERY,OPT_CENTER,OPT_RIGHTX          ++
  ++                                                                              ++
  ++           tx_color =  Color Text                                             ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


void Text_CMD(int8 *str,int16 x,int16 y,int16 font,int16 opt,int32 tx_color)
 {

    // Set Attribute Clock //

    FT800memWrite32(RAM_CMD + cmdOffset,(DL_COLOR_RGB | tx_color))    ; //***Sent DL-CMD Set Color Text-White  
    cmdOffset = incCMDOffset(cmdOffset,4)                             ; //Update Offset pointer Ram-CMD 4 byte for Next CMD

   
   //Start Draw Text//    

    FT800memWrite32(RAM_CMD + cmdOffset,CMD_TEXT)                     ; //***Sent Co-CMD Draw TEXT
    cmdOffset = incCMDOffset(cmdOffset,4)                             ; //Update Offset pointer Ram-CMD 4 byte for Next data

    FT800memWrite16(RAM_CMD + cmdOffset,x)                            ; //---Sent data X-Point top-left (Pixel) 
    cmdOffset = incCMDOffset(cmdOffset,2)                             ; //Update Offset pointer Ram-CMD 2 byte for Next data

    FT800memWrite16(RAM_CMD + cmdOffset,y)                            ; //---Sent data Y-Point top-left (Pixel) 
    cmdOffset = incCMDOffset(cmdOffset,2)                             ; //Update Offset pointer Ram-CMD 2 byte for Next data
  
    FT800memWrite16(RAM_CMD + cmdOffset,font)                         ; //---Sent data Set format font(16,18,20,21,22,23,24...)                                                     
    cmdOffset = incCMDOffset(cmdOffset,2)                             ; //Update Offset pointer Ram-CMD 2 byte for Next data

    FT800memWrite16(RAM_CMD + cmdOffset,opt)                          ; //---Sent data Option : Position Draw  Text                                                     
    cmdOffset = incCMDOffset(cmdOffset,2)                             ; //Update Offset pointer Ram-CMD 2 byte for Next data

    FT800memWriteStr(RAM_CMD + cmdOffset,str)                         ; //---Sent data Text   String      
    cmdOffset = End_incCMDOffset(cmdOffset,num_cha)                   ; //Update Offset pointer Ram-CMD for End CMD-Text
  
  }



/*************************************************************************************
 **                                                                                 **
 **                       9.Function-About User Application                         **
 **                                                                                 **
 *************************************************************************************/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                   ++
  ++                     9.1)Function:FT800 Initial                    ++
  ++                                                                   ++                                       
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


void FT800_Initial(void)         
 {
 //---------------------- 1.Walk-Up FT800 -----------------------

     
   FT800_PD_lo()                            ;  //Power-Down FT800
   delay_ms(20)                             ;  //delay 20 ms
    
   FT800_PD_hi()                            ;  //Power-Up FT800  
   delay_ms(20)                             ;  //delay 20 ms


   FT800_HostCmdWrite(FT800_ACTIVE)         ;  //***Sent Host-CMD Start FT800
   delay_ms(5);                                //Give some time to process

   FT800_HostCmdWrite(FT800_CLKEXT)         ;  //***Sent Host-CMD Set FT800 for external clock
   delay_ms(5);                                //Give some time to process

   FT800_HostCmdWrite(FT800_CLK48M)         ;  //***Sent Host-CMD Set FT800 for 48MHz PLL
   delay_ms(5)                              ;  //Give some time to process
                                               //Now FT800 can accept commands at up to 30MHz clock on SPI bus
                                               //This application leaves the SPI bus at 4MHz
 
  if(FT800memRead8(REG_ID) != 0x7C)            //***Read ID register - is it 0x7C?
   {
     while(true);                                 //If we don't get 0x7C, the ineface isn't working - halt with infinite loop
   }
  
   FT800memWrite8(REG_PCLK,ZERO)           ;   //***Write data zero to Reg.PCLK  for don't clock the LCD until later
   FT800memWrite8(REG_PWM_DUTY,ZERO)       ;   //***Write data zero to Reg.PWM_Duty for Turn off backlight
  

 //-------------------- 2.Configure Display ------------------------

  FT800memWrite16(REG_HSIZE,480)           ;  //***Write data to Reg : Active display width =480 Pixel
  FT800memWrite16(REG_HCYCLE,548)          ;  //***Write data to Reg : Total number of clocks per line, incl front/back porch
  FT800memWrite16(REG_HOFFSET,43)          ;  //***Write data to Reg : start of active line
  FT800memWrite16(REG_HSYNC0,0)            ;  //***Write data to Reg : Start of horizontal sync pulse
  FT800memWrite16(REG_HSYNC1,41)           ;  //***Write data to Reg : End of horizontal sync pulse
  
  FT800memWrite16(REG_VSIZE,272)           ;  //***Write data to Reg : Active display height =272 Pixel
  FT800memWrite16(REG_VCYCLE,292)          ;  //***Write data to Reg : Total number of lines per screen, incl pre/post
  FT800memWrite16(REG_VOFFSET,12)          ;  //***Write data to Reg : Start of active screen
  FT800memWrite16(REG_VSYNC0,0)            ;  //***Write data to Reg : Start of vertical sync pulse
  FT800memWrite16(REG_VSYNC1,10)           ;  //***Write data to Reg : End of vertical sync pulse
  
  FT800memWrite8(REG_SWIZZLE,0)            ;  //***Write data to Reg : FT800 output to LCD - pin order
  FT800memWrite8(REG_PCLK_POL,1)           ;  //***Write data to Reg : LCD data is clocked in on this PCLK edge
                                              //***Write data to Reg : Don't set PCLK yet - wait for just after the first display list


 //------------------- 3.Configure Touch -----------------
  
  FT800memWrite8(REG_TOUCH_MODE,0x03)      ;  //***Write data to Reg : Touch Mode Continuous
  FT800memWrite16(REG_TOUCH_RZTHRESH,1200) ;  //***Write data to Reg : Eliminate any false touches
  


//-------------------- 4. Configure Audio ----------------


  FT800memWrite8(REG_GPIO_DIR,0x82)        ;   //***Write Data to Reg :  Set Direction  GPIO 7,1 = Output

  ft800Gpio = FT800memRead8(REG_GPIO)      ;   //***Read the FT800 GPIO register for a read/modify/write operation
  ft800Gpio = ft800Gpio | 0x02             ;   //set bit 1 of FT800 GPIO register (Audio) - others are inputs

  FT800memWrite8(REG_GPIO,ft800Gpio)       ;   //***Write Data to Reg : Enable(1) the Audio Amp
  
  FT800memWrite8(REG_VOL_PB,ZERO)          ;  //***Write data to Reg : Turn recorded audio volume 
  FT800memWrite8(REG_VOL_SOUND,0x80)       ;  //***Write data to Reg : Turn synthesizer volume 50%(100%=0xFF)  
 
  FT800memWrite16(REG_SOUND,0x0000)        ;  //***Write data to Reg : Select Sound Effect = Silence(No Sound) 
  FT800memWrite8(REG_PLAY,0x01)            ;  //***Write data to Reg : Play Sound Effect

 //------------------ 5.Write Initial Display List & Enable Display -----------------

  FT800memWrite32(RAM_DL+DL_Offset,DL_CLEAR_RGB)  ;     //***Sent DL-CMD Clear color BackGround to Black (default)
  DL_Offset += 4                                  ;     //Update Offset pointer Ram-DL 4 byte for Next DL-CMD

  FT800memWrite32(RAM_DL+DL_Offset,(DL_CLEAR | CLR_COL | CLR_STN | CLR_TAG));   //***Sent DL-CMD Clear C/S/T define which parameters to clear
  DL_Offset += 4                                  ;     //Update Offset pointer Ram-DL 4 byte for Next DL-CMD

  FT800memWrite32(RAM_DL+DL_Offset,DL_DISPLAY)    ;     //***Sent DL-CMD for end of display list

  FT800memWrite32(REG_DLSWAP,DLSWAP_FRAME )       ;     //***Write data(0x02) to Reg : Display List Execute
                                                
  DL_Offset = 0                                   ;     //Reset Display List offset pointer=0 for new Display list
  
  ft800Gpio = FT800memRead8(REG_GPIO)             ;     //***Read the FT800 GPIO register for a read/modify/write operation
  ft800Gpio = ft800Gpio | 0x80                    ;     //set bit 7 of FT800 GPIO register (DISP) - others are inputs

  FT800memWrite8(REG_GPIO,ft800Gpio)              ;     //***Write Data to Reg : Enable the DISP signal to the LCD panel
  FT800memWrite8(REG_PCLK,5)                      ;     //***Write Data to Reg : Now start clocking data to the LCD panel
 

  FT800memWrite8(REG_PWM_DUTY,128)                ;    //***Write Data to Reg : Turn on backlight by set PWM_duty Max for full brighness
  delay_ms(10)                                    ;

 }


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++                      9.2)Function: Update_Screen Application                 ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++  Input :                                                                     ++
  ++          btset_color  = Set Color Button "SET--->HH--->MM"                   ++ 
  ++          tx_color     = Set Color text for Button "Color" and Pin Clock      ++
  ++          bt_color     = Set Color Button "Color" and back ground Clock       ++
  ++          btup_color   = Set Color Button "+"                                 ++
  ++          btdown_color = Set Color Button "-"                                 ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

  void Update_Screen(int32 btset_color,int32 tx_color,int32 bt_color,int32 btup_color,int32 btdown_color)
   {
     unsigned int8 strbuf[50];
     
    //----------------- 1.Start Display List ------------------------
      
     Start_CMDList(Black)   ;

  
   //------------------ 2.User Program ------------------------- 
  
    sprintf(strbuf,"ET-DEMO CLOCK")                                     ;
    Text_CMD(strbuf,250,30,24,0,Yellow)                                 ;  //Display Text "ET-DEMO CLOCK"
   
    sprintf(strbuf,"Color")                                             ;
    Button_CMD(strbuf,330,100,50,25,27,0,tx_color,bt_color)             ;  //Draw Button"Color"

    Clock_CMD(120,136,100,0,hh,mm,ss,0,tx_color,bt_color)               ;  //Draw Clock   

    sprintf(strbuf,"+")                                                 ;
    Button_CMD(strbuf,405,130,25,25,23,0,White,btup_color)              ;  //Draw Button"+"
    sprintf(strbuf,"-")                                                 ;
    Button_CMD(strbuf,280,130,25,25,23,0,White,btdown_color)            ;  //Draw Button"-"
   
    switch(mode)
    {
    
      case 0 :   
                sprintf(strbuf,"SET")                                                    ;
                Button_CMD(strbuf,330,160,50,25,27,0,White,btset_color)                  ;  //Draw Button"Set"       
               
                break;

       case 1 :
                sprintf(strbuf,"HH")                                                     ;
                Button_CMD(strbuf,330,160,50,25,27,0,White,btset_color)                  ;  //Draw Button"HH"         
                
                break;

       case 2 :   
                sprintf(strbuf,"MM")                                                     ;
                Button_CMD(strbuf,330,160,50,25,27,0,White,btset_color)                  ;  //Draw Button"MM" 
           
                break;

    }

    //----------------- 3.End Display List and Update Execute -------------

   End_CMDList();

 }

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ++                                                                              ++
  ++                      9.3)Function: User Application                          ++
  ++                                                                              ++
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


 void User_App(void)
  {
   signed int16  tm = 0      ;
   unsigned int8 cb = 0        ;
   int8 fag_bt = 0             ;

   int32 bt_color[5] = {Blue,Red,Green,Magenta,Yellow}           ;   //Color Background Clock
   int32 tx_color[5] = {White,Black,NavyBlue,LimeGreen,Brown}    ;   //Color  Pin Clock
   
   
   Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red)   ;

   while(true)
    { 

     //-------- Check Statust Touch Screen ---------
         
     if(Touch_Detect())                          //if Touch =1 
      {
        Touch_Read_XY()                     ;    //Read Position Touch Screen XY

        //------Check Button"Color,Set"------//

           //X-Touch Button "Color,Set"

       if((tc_x>330) & (tc_x <380))             //Check Position Touch X-axis for Button"Color,Set"
         { 
          //Y-Touch Button"Color"//
           if((tc_y>100)&(tc_y<125))            //Check Position Touch Y-axis for  Button"Color"
           {
            if(fag_bt == 0)                  
             { 
               cb = cb+1                    ;   //Increment Pointer array for color

               if(cb == 5)
                 cb = 0                     ;   
              
               Sound_Effect(100,C7,Swit)    ;   //Set Sound Press Button
         
               Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red)  ;
               
               fag_bt = 1                   ;   //Set fag_bt protect Press Button double (for give Release Button) 
             }
           } 

             //Y-Touch Button"Set"//

          if((tc_y>160)&(tc_y<185))             //Check Position Touch Y-axis for Button"Color"
           {
              mode = mode+1                ;    //Chang Text on Button Set
              if(mode ==3)
                 mode=0                    ;
           
              Sound_Effect(100,C7,Swit)    ;   //Set Sound Press Button
            
              Update_Screen(LightBlue,tx_color[cb],bt_color[cb],Red,Red)  ; 

              while(Touch_Detect()){;}     ;   // Wait Release Button "+"

              Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red) ;  //Return Color Button "SET-->HH-->MM"
                 
           }  //if tc_y
        }   //if tc_x

         //-------- Check Button "-,+"---------//
         
         //Y-Touch Button"-,+"


       if(mode >0) //Enable button "+,-"
        {

         if((tc_y>130) & (tc_y <155))         //Check Position Touch Y-axis for Button"-,+"
           { 
           //X-Touch Button"-"//

             if((tc_x>280)&(tc_x<305))       //Check Position Touch X-axis for Button"-"
              {
                if(mode==1)                  //Set Down-- Hour :12-1
                 {   
                   hh = hh-1         ;       //Set Decrease Hour step 1 hour 

                   if(hh==0)                 //Check Hour if=0 give hh=12
                     hh=12           ;           
                 }

                if(mode==2)                  //Set Down-- Minute :59-0
                 {
                   mm = mm-1         ;       //Set Decrease minute step 1 minute 

                   if(mm == -1)              //Check minute if= -1 give mm= 59
                    {
                      mm = 59        ;
                   
                      hh = hh-1      ;
                      if(hh==0)               //Check Hour if=0 give hh=12
                        hh=12        ;       
                    }
                 }

                Sound_Effect(100,C7,Swit)                                          ;  //Set Sound Press Button
                Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,MistyRose)  ;

                while(Touch_Detect()){;}                                           ;  // Wait Release Button "+"

                Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red)        ;  //Return Color Button "-"
             
              } //if tc_x

               //X-Touch Button"+"//

              if((tc_x>405)&(tc_x<430))           //Check Position Touch X-axis for Button"+"
               {
                if(mode==1)                       //Set Up++ Hour :1-12
                 {
                   hh = hh+1                ;     //Set Increase Hour step 1 hour 
 
                   if(hh==13)                     //Check Hour if=13 give hh=1
                     hh = 1                 ;
            
                 }
                if(mode==2)                       //Set Up++ Minute :0-59
                 {
                  mm = mm+1                 ;     //Set Increase Minute step 1 minute 

                  if(mm==60)                      //Check Minute if=60 give minute=0
                   {
                     mm = 0                 ;
                     hh = hh+1              ;
                     if(hh==13)                   //Check Hour if=13 give hh=1
                       hh = 1               ;      
                   }  
                 }

                 Sound_Effect(100,C7,Swit)  ;  //Set Sound Press Button
                 Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],MistyRose,Red)  ;

                 while(Touch_Detect()){;}   ;  // Wait Release Button "+"
         
                 Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red)        ;   //Return Color Button "+"
                          

              } //if tc_x
           
          }  // if tc_y

        } //if mode >0

      }  //if Touch_Detect
     else
       fag_bt = 0  ;  //Clear fag_bt for Check Release Touch button "color"


     
    //-----------------  Update Time  --------------

    if(mode==0)                            //Update Time When Button = Set
     {
       delay_ms(10)                   ;

       tm += 10                       ;

       if(tm >1000)                        //check  1000 ms = 1s
        {
          tm =0                       ;
          ss = ss+1                   ;    //Update ss
          Sound_Effect(100,G_5,Click) ;    //Sound effect Pin-Second
            
          if(ss == 60)                     //Check SS=60
          {
            ss =0                     ;
            mm = mm+1                 ;   //Update mm

            if(mm==60)                    //Check mm=60
             {
              mm=0                    ;
              hh= hh+1                ;   //Update hh
            
              if(hh==24)
               hh = 0                 ;   //Clear hh
             }

          } // if ss

          Update_Screen(MediumBlue,tx_color[cb],bt_color[cb],Red,Red)  ;

       }    //if tm      
     }      // if mode
  }         //while
}           //end User App





