/*********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0 */
/* MCU      : Zilog MCS51 / Z51F6412         */
/*          : Run Internal RC (8.00 MHz)     */
/* Compiler : Keil C51 (V9.50a)              */
/* Editor   : uVision V4.53.0.6              */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)  */
/* Function : Demo Timer Generate PWM        */
/*          : PWM1 Pin = P5[3] = 10Bit PWM   */
/*          : PWM2 Pin = P5[4] = 16Bit PWM   */
/*          : PWM3 Pin = P5[5] = 16Bit PWM   */
/*          : PWM4 Pin = P5[6] = 16Bit PWM   */
/*          : PWM5 Pin = P5[7] = 16Bit PWM   */
/*********************************************/
// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>															// Z8051 : Z51F6412

//Timer5 PWM SFR 
#define PWM5HDR     T5H														// PWM5HDR Same Address SFR T5H
#define PWM5LDR 		T5L														// PWM5LDR Same Address SFR T5L
#define PWM5HPR			T5DRH													// PWM5HPR Same Address SFR T5DRH
#define PWM5LPR    	T5DRL													// PWM5LPR Same Address SFR T5DRL

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN				(1 << 1)										// P8[1] = LED
#define LED_PxIO			P8IO
#define LED_Px				P8
#define LED_INIT()    LED_PxIO |=  (LED_PIN)
#define LED_OFF()			LED_Px   &= ~(LED_PIN)
#define LED_ON()  		LED_Px   |=  (LED_PIN)
#define LED_TOGGLE()  LED_Px   ^=  (LED_PIN)

/***********************
 * Function Prototypes *
 ***********************/
void init_pwm1_timer1(void);
void init_pwm2_timer2(void);
void init_pwm3_timer3(void);
void init_pwm4_timer4(void);
void init_pwm5_timer5(void);

/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																		// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
	
	//Initial PWM1,2,3,4,5
  init_pwm1_timer1();															// P5[3] = 10Bit PWM (7.1825KHz)
	init_pwm2_timer2();   													// P5[4] = 16Bit PWM (122.07Hz)
	init_pwm3_timer3();   													// P5[5] = 16Bit PWM (122.07Hz)
	init_pwm4_timer4();   													// P5[6] = 16Bit PWM (122.07Hz)
	init_pwm5_timer5();   													// P5[7] = 16Bit PWM (122.07Hz)
	
	// 1Cycle = 125nS
	// Duty Cycle = nTime/125nS
	// 10Bit PWM Duty Cycle Max = 1023 Cycle
  // 16Bit PWM Duty Cycle Max = 65535 Cycle
	
	// PWM1 = 10Bit PWM
	// Set PWM1(P5.3) Duty Cycle = 10uS(Max=127.875uS)
	// Example 10uS Duty = 80 Cycle of 65535
	// 80 = 0x0050
	T1PDR	   =  0x50;																// LSB Duty[7..0]
	T1PWHR	&= ~0x03;																// MSB Duty[9:8]
	
	// PWM2 = 16Bit PWM
	// Set PWM2(P5.4) Duty Cycle = 2mS(Max=8.191mS)
	// Example 2mS Duty = 16000 Cycle of 65535
	// 16000 = 0x3E80
	PWM2HDR	   =  0x3E;															// MSB Duty[15..8]
	PWM2LDR	   =  0x80;															// LSB Duty[7..0]
	
	// PWM3 = 16Bit PWM
	// Set PWM3(P5.5) Duty Cycle = 3mS(Max=8.191mS)
	// Example 3mS Duty = 24000 Cycle of 65535
	// 24000 = 0x5DC0
	PWM3HDR	   =  0x5D;															// MSB Duty[15..8]
	PWM3LDR	   =  0xC0;															// LSB Duty[7..0]
	
	// PWM4 = 16Bit PWM
	// Set PWM4(P5.6) Duty Cycle = 4mS(Max=8.191mS)
	// Example 4mS Duty = 32000 Cycle of 65535
	// 32000 = 0x7D00
	PWM4HDR	   =  0x7D;															// MSB Duty[15..8]
	PWM4LDR	   =  0x00;															// LSB Duty[7..0]
	
	// PWM5 = 16Bit PWM
	// Set PWM5(P5.7) Duty Cycle = 5mS(Max=8.191mS)
	// Example 5mS Duty = 40000 Cycle of 65535
	// 40000 = 0x9D40
	PWM5HDR	   =  0x9C;															// MSB Duty[15..8]
	PWM5LDR	   =  0x40;															// LSB Duty[7..0]
	
	// Indicator MCU Running By LED ON 
	LED_INIT();					
	LED_ON();
	
	//Main Loop 
	while(1);
}

/*
 * Function    : init_pwm1_timer1
 * Description : initialize timer[1] = PWM 10Bit Mode
 */
void init_pwm1_timer1(void)
{
	//Timer1 = 10Bit PWM Mode
	// fx = 8MHz
	// 1Cycle = 125nS
	// PWM Freq = 7.8KHz
	// Period = 1024x125nS
	//        = 128uS
	// Freq   = 1/128uS
	//        = 7.8125 KHz
	// Duty Cycle = 1...1023
	
  // Initial T1CR = 1,0,1,0,00,1,1
  // 1xxxxxxx : PWM Polarity High
  // x0xxxxxx : 8 Bit Timer Mode
  // xx1xxxxx : PWM Enable
	// xxx0xxxx : Timer/Counter Mode
  // xxxx00xx : Clock Source = Fx
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Counter1 Start
	T1CR = 0xA3;
	
	// Initial T1PWHR = 1,0,0,0,0,0,0,0
  // 1xxxxxxx : Enable Timer1 Output(P5.3=PWM1)
  // x000xxxx : Default For Reserve Bit
  // xxxx00xx : Period[9:8]
	// xxxxxx00 : Duty[9:8]
	T1PWHR |= 0x80;
	
	//Period = [PW1H3:PW1H2]+T1PPR
	//T1PWHR = xxxx11xx
	T1PPR   = 0xFF;																	// LSB Period[7..0]
	T1PWHR |= 0x0C;																	// MSB Period[9:8]
	
	//Duty = [PW1H1:PW1H0]+T1PDR
	//P1PWHR = xxxxxx??
	//50% Duty  = (1024/2)-1
	//          = 511
	//          = 0x01FF
	T1PDR	   =  0xFF;																// LSB Duty[7..0]
	T1PWHR	|=  0x01;																// MSB Duty[9:8]
	
  // xxxxxx0x :	Disable Timer1 Interrupt									
	IE2 &= ~0x02;																		// Disable Timer1 Interrupt
}

/*
 * Function    : init_pwm2_timer2
 * Description : initialize timer[2] = PWM 16Bit Mode
 */
void init_pwm2_timer2(void)
{
	// Timer2,3,4,5 = 16Bit PWM Mode
	// fx = 8MHz
	// 1Cycle = 125nS
	// PWM Freq = 122.07 Hz
	// Period = 65536x125nS
	//        = 8.192mS
	// Freq   = 1/8.192mS
	//        = 122.07 Hz
	// Duty Cycle = 1...65535
	
	// Initial TxCR = 1,1,,0,000,1,1 
  // 1xxxxxxx : Timer2 Enable
  // x1xxxxxx : PWM Enable
  // xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Clock Source = Fx/1
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Clear Counter & Start
	T2CR = 0xC3;
	
	// Initial TxCR1 = 00000,0,1,1 
  // 00000xxx : Default for Reserve Bit
	// xxxxx0xx : External Clock Disable
  // xxxxxx1x : Timer Output Enable(P5.4=PWM2)
  // xxxxxxx1	: PWM Polarity High
	T2CR1 = 0x03;
	
  //Period = PWMxHPR+PWMxLPR
	PWM2HPR = 0xFF;																	// MSB Period[15..8]
  PWM2LPR = 0xFF;																	// LSB Period[7..0]
	
  //Duty = PWMxHDR+PWMxLDR
	//50% Dyty = (65536/2)-1
	//         = 32768-1
	//         = 32767
	//         = 0x7FFF
	PWM2HDR	   =  0x7F;															// MSB Duty[15..8]
	PWM2LDR	   =  0xFF;															// LSB Duty[7..0]
	
  // xxxxx0xx :	Disable Timer2 Interrupt									
	IE2 &= ~0x04;																		// Disable Timer2 Interrupt
}

/*
 * Function    : init_pwm3_timer3
 * Description : initialize timer[3] = PWM 16Bit Mode
 */
void init_pwm3_timer3(void)
{
	// Timer2,3,4,5 = 16Bit PWM Mode
	// fx = 8MHz
	// 1Cycle = 125nS
	// PWM Freq = 122.07 Hz
	// Period = 65536x125nS
	//        = 8.192mS
	// Freq   = 1/8.192mS
	//        = 122.07 Hz
	// Duty Cycle = 1...65535
	
	// Initial TxCR = 1,1,,0,000,1,1 
  // 1xxxxxxx : Timer2 Enable
  // x1xxxxxx : PWM Enable
  // xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Clock Source = Fx/1
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Clear Counter & Start
	T3CR = 0xC3;
	
	// Initial TxCR1 = 00000,0,1,1 
  // 00000xxx : Default for Reserve Bit
	// xxxxx0xx : External Clock Disable
  // xxxxxx1x : Timer Output Enable(P5.5=PWM3)
  // xxxxxxx1	: PWM Polarity High
	T3CR1 = 0x03;
	
  //Period = PWMxHPR+PWMxLPR
	PWM3HPR = 0xFF;																	// MSB Period[15..8]
  PWM3LPR = 0xFF;																	// LSB Period[7..0]
	
  //Duty = PWMxHDR+PWMxLDR
	//50% Dyty = (65536/2)-1
	//         = 32768-1
	//         = 32767
	//         = 0x7FFF
	PWM3HDR	   =  0x7F;															// MSB Duty[15..8]
	PWM3LDR	   =  0xFF;															// LSB Duty[7..0]
	
	// xxxx0xxx :	Disable Timer3 Interrupt									
	IE2 &= ~0x08;																		// Disable Timer3 Interrupt
}

/*
 * Function    : init_pwm4_timer4
 * Description : initialize timer[4] = PWM 16Bit Mode
 */
void init_pwm4_timer4(void)
{
	// Timer2,3,4,5 = 16Bit PWM Mode
	// fx = 8MHz
	// 1Cycle = 125nS
	// PWM Freq = 122.07 Hz
	// Period = 65536x125nS
	//        = 8.192mS
	// Freq   = 1/8.192mS
	//        = 122.07 Hz
	// Duty Cycle = 1...65535
	
	// Initial TxCR = 1,1,,0,000,1,1 
  // 1xxxxxxx : Timer2 Enable
  // x1xxxxxx : PWM Enable
  // xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Clock Source = Fx/1
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Clear Counter & Start
	T4CR = 0xC3;
	
	// Initial TxCR1 = 00000,0,1,1 
  // 00000xxx : Default for Reserve Bit
	// xxxxx0xx : External Clock Disable
  // xxxxxx1x : Timer Output Enable(P5.6=PWM4)
  // xxxxxxx1	: PWM Polarity High
	T4CR1 = 0x03;
	
  //Period = PWMxHPR+PWMxLPR
	PWM4HPR = 0xFF;																	// MSB Period[15..8]
  PWM4LPR = 0xFF;																	// LSB Period[7..0]
	
  //Duty = PWMxHDR+PWMxLDR
	//50% Dyty = (65536/2)-1
	//         = 32768-1
	//         = 32767
	//         = 0x7FFF
	PWM4HDR	   =  0x7F;															// MSB Duty[15..8]
	PWM4LDR	   =  0xFF;															// LSB Duty[7..0]
	
	// xxx0xxxx :	Disable Timer4 Interrupt									
	IE2 &= ~0x10;																		// Disable Timer4 Interrupt
}

/*
 * Function    : init_pwm5_timer5
 * Description : initialize timer[5] = PWM 16Bit Mode
 */
void init_pwm5_timer5(void)
{
	// Timer2,3,4,5 = 16Bit PWM Mode
	// fx = 8MHz
	// 1Cycle = 125nS
	// PWM Freq = 122.07 Hz
	// Period = 65536x125nS
	//        = 8.192mS
	// Freq   = 1/8.192mS
	//        = 122.07 Hz
	// Duty Cycle = 1...65535
	
	// Initial TxCR = 1,1,,0,000,1,1 
  // 1xxxxxxx : Timer2 Enable
  // x1xxxxxx : PWM Enable
  // xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Clock Source = Fx/1
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Clear Counter & Start
	T5CR = 0xC3;
	
	// Initial TxCR1 = 00000,0,1,1 
  // 00000xxx : Default for Reserve Bit
	// xxxxx0xx : External Clock Disable
  // xxxxxx1x : Timer Output Enable(P5.7=PWM5)
  // xxxxxxx1	: PWM Polarity High
	T5CR1 = 0x03;
	
  //Period = PWMxHPR+PWMxLPR
	PWM5HPR = 0xFF;																	// MSB Period[15..8]
  PWM5LPR = 0xFF;																	// LSB Period[7..0]
	
  //Duty = PWMxHDR+PWMxLDR
	//50% Dyty = (65536/2)-1
	//         = 32768-1
	//         = 32767
	//         = 0x7FFF
	PWM5HDR	   =  0x7F;															// MSB Duty[15..8]
	PWM5LDR	   =  0xFF;															// LSB Duty[7..0]
	
	// xx0xxxxx :	Disable Timer5 Interrupt									
	IE2 &= ~0x20;																		// Disable Timer5 Interrupt
}

