/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run Internal RC (8.00 MHz)      */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Hardware QC Test                */
/* Update   : 18 August 2012                  */
/**********************************************/
// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>																							// Z8051 : Z51F6412
	
/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)			

/*******************
 * Global Variable *
 *******************/
#define LED_SPEED						250																		// 250 LED Blink
static volatile unsigned int ledTicks = 0;	
static volatile unsigned int ledPTR = 0;
static volatile unsigned int BuzzerTick = 0;	
static bit this_sw;															
static bit last_sw;	
sbit SW_READ = P8^0;    

// Port LED Display
code  unsigned char PORT_OUT[8] = { ~0x01,
                                    ~0x02,
                                    ~0x04,
                                    ~0x08,
                                    ~0x10,
                                    ~0x20,
                                    ~0x40,
                                    ~0x80};	

/**********************************
 * Start of UART FIFO Buffer Area *
 **********************************/
// FIFO USART Buffer
// Ring buffers for transfering data to and from the USART ISR:
//  - (Save pointer == Read pointer) is defined as an empty FIFO
//  - (Save pointer == Read pointer - 1), accounting for wraparound,
//    is defined as a completely full FIFO.  As a result, the max data 
//    in a FIFO is the buffer size - 1.
static volatile char UART_RX0_FIFO[32];														// 31 Byte Max. Receive
static volatile char UART_RX1_FIFO[32];														// 31 Byte Max. Receive
static volatile char *RX0SavePtr;
static volatile char *RX0ReadPtr;
static volatile char *RX1SavePtr;
static volatile char *RX1ReadPtr;

/***********************
 * Function Prototypes *
 ***********************/
void init_uart0(unsigned long freq, unsigned long baud);
void init_uart1(unsigned long freq, unsigned long baud);
void my_putchar0(char ch);
void my_putchar1(char ch);
void print_uart0(char *uart_buf);
void print_uart1(char *uart_buf);
char getchar0_fifo(void);
char getchar1_fifo(void);
void init_timer2(void);

/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	char ch;
	
	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
  
	/* Start of Config All GPIO = Output */
  LED_PORT_INIT();  																							// Configure LED = Output 
	P0IO |=   0xFF; 																								// 11111111 	
	P1IO |=   0xFF; 																								// 11111111
	P2IO |=   0xFF; 																								// 11111111	
	P3IO |=   0xFF; 																								// 11111111 
	P4IO |=   0xFF; 																								// 11111111
	P5IO |=   0xFF; 																								// 11111111
	P6IO |=   0xFF; 																								// 11111111
	P7IO |=   0xFF; 																								// 11111111
	
	// Reserve P0[2],P0[3] For UART0
	P0IO &= ~(0x0C); 																								// xxxx00xx : Reserve P0[2]=TX0,P0[3]=RX0	
	
	// If Install Sub X-TAL(32.768KHz)
	//P0IO &= ~(0x30); 																								// xx00xxxx : Reserve P0[4]=SUBXIN,P0[5]=SUBXOUT	
	
	// Reserve P3[2],P3[3] For UART1
	P3IO &= ~(0x0C); 																								// xxxx00xx : Reserve P3[2]=RX1,P3[3]=TX1
	
	// Reserve P5[0] For Buzzer 
	P5IO &= ~(0x01); 																								// xxxxxxx0 : Reserve P5[0] = Buzzer
	
	// If Install Main X-TAL(10MHz) 
	//P6IO &= ~(0x0C); 																							// xxxx00xx : Reserve P6[2]=XIN,P6[3]=XOUT
	
	// Default OFF LED
  LED_OFF();
  P0 = 0xFF;
	P1 = 0xFF;
	P2 = 0xFF;
	P3 = 0xFF;
	P4 = 0xFF;
	P5 = 0xFF;
	P6 = 0xFF;
	P7 = 0xFF;
	/*End of Config All GPIO */
	
	//Config SW
	SW_PORT_INIT();																									// Config SW = Input 
	SW_DEBOUNCE_INIT();																							// Enable Debounce For SW
  SW_PULLUP_INIT();                           										// Enable Pull-Up For SW 
	SW_LOGIC_INIT();																								// Default Input Logic = "1"
														
  //Default SW Status = Release 
  last_sw = 0;		
	
	// Config Buzzer
	// BUZCR = 00000,11,1
	// 00000xxx : Default Reserve Bit
	// xxxxx11x : Buzzer Clock Source = Fx/256
	// xxxxxxx0 : Buzzer Disable
	BUZCR |= 0x06;																	
	BUZDR =  0x1F;
	BUZCR &= ~0x01;																									// OFF Buzzer
	BuzzerTick = 0;
	
	// Initial UART0,UART1
	init_uart0(MY_FREQ, MY_BAUD);
  init_uart1(MY_FREQ, MY_BAUD);
	RX0SavePtr = &UART_RX0_FIFO[0];																	// UART0 Interrupt Buffer Service
  RX0ReadPtr = &UART_RX0_FIFO[0];
  RX1SavePtr = &UART_RX1_FIFO[0];																	// UART1 Interrupt Buffer Service
  RX1ReadPtr = &UART_RX1_FIFO[0];
	
	init_timer2();
	ledTicks = 0;
	
	EA = 1;																													// Enable Glocbal Interrupt
	
	print_uart0("\n\n\r");
	print_uart0("\n\r=============================================");
	print_uart0("\n\rET-BASE Z51F6412 V1.0 Run Internal RC at 8MHZ");
	print_uart0("\n\r>>>>>>>>>>>>>>QC Hardware Test<<<<<<<<<<<<<<<");
	print_uart0("\n\r=============================================");
	print_uart0("\n\r ->Press SW For Test Buzzer Beep");
	print_uart0("\n\r ->Connect LED to Port Fot Test Output Port");
	print_uart0("\n\r   P0[7..4,1..0] : Reerve P0[2]=TX0,P0[3]=RX0");
	print_uart0("\n\r   P1[7...0]     : 8 Bit Output");
	print_uart0("\n\r   P2[7...0]     : 8 Bit Output");
	print_uart0("\n\r   P3[7..4,1..0] : Reserve P3[2]=TX1,P3[3]=RX1");
	print_uart0("\n\r   P4[7...0]     : 8 Bit Output");
	print_uart0("\n\r   P5[7...1]     : Reserve P5[0] = Buzzer");
	print_uart0("\n\r   P6[7...0]     : 8 Bit Output");
	print_uart0("\n\r   P7[7...0]     : 8 Bit Output");
	print_uart0("\n\r ->Press Any Key For Test UART0");
	print_uart0("\n\r=============================================");
	print_uart0("\n\rUART0>");
	
	print_uart1("\n\n\r");
	print_uart1("\n\r=============================================");
	print_uart1("\n\rET-BASE Z51F6412 V1.0 Run Internal RC at 8MHZ");
	print_uart1("\n\r>>>>>>>>>>>>>>QC Hardware Test<<<<<<<<<<<<<<<");
	print_uart1("\n\r=============================================");
	print_uart1("\n\r ->Press SW For Test Buzzer Beep");
	print_uart1("\n\r ->Connect LED to Port Fot Test Output Port");
	print_uart1("\n\r   P0[7..4,1..0] : Reerve P0[2]=TX0,P0[3]=RX0");
	print_uart1("\n\r   P1[7...0]     : 8 Bit Output");
	print_uart1("\n\r   P2[7...0]     : 8 Bit Output");
	print_uart1("\n\r   P3[7..4,1..0] : Reserve P3[2]=TX1,P3[3]=RX1");
	print_uart1("\n\r   P4[7...0]     : 8 Bit Output");
	print_uart1("\n\r   P5[7...1]     : Reserve P5[0] = Buzzer");
	print_uart1("\n\r   P6[7...0]     : 8 Bit Output");
	print_uart1("\n\r   P7[7...0]     : 8 Bit Output");
	print_uart1("\n\r ->Press Any Key For Test UART1");
	print_uart1("\n\r=============================================");
	print_uart1("\n\rUART1>");
  
	//Main Loop
	while(1)
	{
		// Read New SW Status
		this_sw = SW_READ;																						// Read Bit SW
		
		if(this_sw != last_sw)																				// If SW Status Change
    {
			//Verify SW Press & Service
			if((last_sw==1)&&(this_sw==0))
			{
        BuzzerTick = 250;  				
				BUZCR |= 0x01;																						// Buzzer ON
      }	
			
			//Verify SW Release & Service
			if((last_sw==0)&&(this_sw==1))
			{
      }	
			
			//Update SW Reference Status
			last_sw = this_sw;		
    }			

		// Verify uart0 receive
		if(RX0ReadPtr != RX0SavePtr)																	// If FIFO Buffer Data Receive
    {	
			ch = getchar0_fifo();																				// get char from fofo uart0
		  my_putchar0(ch);																						// echo char to uart0
			
		  if(ch==0x0D)																								// if Enter print prompt
		  {
				print_uart0("\n\rUART0>");
      }
	  }	
		
		// Verify uart1 receive
    if(RX1ReadPtr != RX1SavePtr)																	// If FIFO Buffer Data Receive
    {	
			ch = getchar1_fifo();																				// get char from fofo uart1
		  my_putchar1(ch);																						// echo char to uart1
			
		  if(ch==0x0D)																								// if Enter print prompt
		  {			  
				print_uart1("\n\rUART1>");
      }
	  }				
	} 
}

/*
 * Function    : init_uart0()
 * Description : initialize UART0 resources
 *             : baudrates, parirty, stop bit
 *             : Enable RX0 Interrupt
 */
void init_uart0(unsigned long freq, unsigned long baud)
{
	// UCTRL01 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL01 = 0x06;

	// UCTRL02 = 0,0,1,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx1xxxxx : Enable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL02 = 0x2F;
	
	// UCTRL03 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL03 = 0x00;
	
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:8MHz,Baudrate:9600
	// UBAUDx   = (8MHz/(9600*8))-1
	//          = (8MHz/76800)-1
	//          = 104.166-1
  //          = 103.166
  //          = 103 = 0x67  
	UBAUD0 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxxxxxx1 :	Enable UART0 RX Interrupt									
	IE1 |= 0x01;	
}

/*
 * Function    : init_uart1()
 * Description : initialize UART1 resources
 *             : baudrates, parirty, stop bit
 *             : Enable RX1 Interrupt
 */
void init_uart1(unsigned long freq, unsigned long baud)
{
	// UCTRL11 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL11 = 0x06;

	// UCTRL12 = 0,0,1,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx1xxxxx : Enable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL12 = 0x2F;
	
	// UCTRL13 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL13 = 0x00;
		
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:8MHz,Baudrate:9600
	// UBAUDx   = (8MHz/(9600*8))-1
	//          = (8MHz/76800)-1
	//          = 104.166-1
  //          = 103.166
  //          = 103 = 0x67  
	UBAUD1 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxx1xxxx :	Enable UART1 RX Interrupt									
	IE1 |= 0x10;	
}

/*
 * Function    : my_putchar0
 * Description : Output a character to uart0
 */
void my_putchar0(char ch)
{
  while(!(USTAT0 & 0x80));																				// Wait until Trammit Buffer is Empty
	UDATA0 = ch;
}

/*
 * Function    : my_putchar1
 * Description : Output a character to uart1
 */
void my_putchar1(char ch)
{
  while(!(USTAT1 & 0x80));																				// Wait until Trammit Buffer is Empty
	UDATA1 = ch;
}

/*
 * Function    : print_uart0
 * Description : print a string to uart0
 */
void print_uart0(char *uart_buf)
{
	while(*uart_buf != '\0')
	{	
	  my_putchar0(*uart_buf++);	
	}	
}

/*
 * Function    : print_uart1
 * Description : print a string to uart1
 */
void print_uart1(char *uart_buf)
{
	while(*uart_buf != '\0')
  {
	  my_putchar1(*uart_buf++);	
	}	
}

/*
 * Function			: getchar0_fifo()
 * Description  : read a character from uart0 fifo
 */
char getchar0_fifo(void)
{
	char rx0_chr;

  rx0_chr = *RX0ReadPtr++;														
  if(RX0ReadPtr >= &UART_RX0_FIFO[0] + sizeof(UART_RX0_FIFO)) 
	{	
    RX0ReadPtr = &UART_RX0_FIFO[0];
  }
	
  return (rx0_chr);
}

/*
 * Function			: getchar1_fifo()
 * Description  : read a character from uart1 fifo
 */
char getchar1_fifo(void)
{
	char rx1_chr;

  rx1_chr = *RX1ReadPtr++;														
  if(RX1ReadPtr >= &UART_RX1_FIFO[0] + sizeof(UART_RX1_FIFO)) 
	{	
    RX1ReadPtr = &UART_RX1_FIFO[0];
  }
	
  return (rx1_chr);
}

/*
 *  Function    : RX0_ISR()
 *  Description : UART0 Receive Interrupt.
 *  If Used Keil Compiler Set Vector = 6(Address Vector:0x0033)
 */
void RX0_ISR(void) interrupt 6 	using 1
{
	char ch;
	
	ch = UDATA0;																										// Get UART0 Data
	
	// Ring buffers for transfering data to and from the USART ISR:
  //  - (Save pointer == Read pointer) is defined as an empty FIFO
  //  - (Save pointer == Read pointer - 1), accounting for wraparound,
  //    is defined as a completely full FIFO.  As a result, the max data 
  //    in a FIFO is the buffer size - 1.
  // Copy the byte into the local FIFO, if it won't cause an overflow
  if(RX0SavePtr != RX0ReadPtr - 1)
	{
	  if((RX0SavePtr != &UART_RX0_FIFO[0] + sizeof(UART_RX0_FIFO)) || (RX0ReadPtr != &UART_RX0_FIFO[0]))
	  {
	    *RX0SavePtr++ = ch;
		  if(RX0SavePtr >= &UART_RX0_FIFO[0] + sizeof(UART_RX0_FIFO)) 
		  RX0SavePtr = &UART_RX0_FIFO[0];		  
	  }
	}
}

/*
 *  Function    : RX1_ISR()
 *  Description : UART1 Receive Interrupt.
 *  If Used Keil Compiler Set Vector = 10(Address Vector:0x0053)
 */
void RX1_ISR(void) interrupt 10 using 2
{
	char ch;
	
	ch = UDATA1;																										// Get UART1 Data
	
	// Ring buffers for transfering data to and from the USART ISR:
  //  - (Save pointer == Read pointer) is defined as an empty FIFO
  //  - (Save pointer == Read pointer - 1), accounting for wraparound,
  //    is defined as a completely full FIFO.  As a result, the max data 
  //    in a FIFO is the buffer size - 1.
  // Copy the byte into the local FIFO, if it won't cause an overflow
  if(RX1SavePtr != RX1ReadPtr - 1)
	{
	  if((RX1SavePtr != &UART_RX1_FIFO[0] + sizeof(UART_RX1_FIFO)) || (RX1ReadPtr != &UART_RX1_FIFO[0]))
	  {
	    *RX1SavePtr++ = ch;
		  if(RX1SavePtr >= &UART_RX1_FIFO[0] + sizeof(UART_RX1_FIFO)) 
		  RX1SavePtr = &UART_RX1_FIFO[0];		  
	  }
	}
}

/*
 * Function    : init_timer2
 * Description : initialize timer[2] = 16Bit Mode
 *               Trigger Interrupt 1mS(fx 8.00 Mhz)
 */
void init_timer2(void)
{
	//Timer2 = 16Bit Mode
	//Fx = 8.00 MHz
	//Timer Clock Source = 8.00 MHz
	//1 Cycle = 1/8.00MHz
	//        = 125nS
	// 1mS Period = 1mS/125nS
	//            = 8000 Cycle
	//            = 0x1F40
	T2DRH = 0x1F;		 																								// MSB : 1mS period ((8MHZ/1)*1ms)
	T2DRL = 0x40;																										// LSB : 1mS
	
  // Initial Timer2 = 1,0,0,000,1,1
  // 1xxxxxxx : Timer2 Enable
  // x0xxxxxx : PWM Disable
  // xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Clock Source = Fx/1
  // xxxxxx1x : Continue Count
  // xxxxxxx1	: Clear Counter & Start
	T2CR = 0x83;
	                  
  // xxxxx1xx :	Enable Timer2 Interrupt									
	IE2 |= 0x04;																										// Enable Timer2 Interrupt
}

/*
 * Function    : timer2_isr 
 * Description : timer 2 service interrupt 
 *               occurs on every 1ms. 
 *  If Used Keil Compiler Set Vector = 14(Address Vector:0x0073)
 */
void TIMER2_isr(void) interrupt 14 using 1
{	
  ledTicks++;

	//Verify Toggle LED 
	if(ledTicks > LED_SPEED)
 	{
	  LED_TOGGLE();																									// Toggle LED
		P0 = PORT_OUT[ledPTR];                     													
		P1 = PORT_OUT[ledPTR];
		P2 = PORT_OUT[ledPTR];
		P3 = PORT_OUT[ledPTR];
		P4 = PORT_OUT[ledPTR];
		P5 = PORT_OUT[ledPTR];
		P6 = PORT_OUT[ledPTR];
		P7 = PORT_OUT[ledPTR];

		ledTicks = 0;																									// Restart Blink Time
		ledPTR++;
		if(ledPTR>7) ledPTR=0x00;
	}
	
	if(BuzzerTick > 0)
	{  
		BuzzerTick--;
		if(BuzzerTick <= 0x00) BUZCR &= ~0x01;												// OFF Buzzer
  }
}




