/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run Internal RC (8.00 MHz)      */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Demo I2C Bus Function           */
/* Update   : 18 August 2012                  */
/**********************************************/
//
// Test Demo I2C with "ET-MINI PCF8574" Board
//
// I2C Interface
// ET-BASE Z51F66412 <----> ET-MINI PCF8574
// P0.6(SCL)        ------>  SCL
// P0.7(SDA)        <----->  SDA
// Setup Jumper A2:A1:A0 of PCF8574/A = 0:0:0

// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>																							// Z8051 : Z51F6412

/**********************************
 * ET-MINI PCF8574 Device ID Code *
 **********************************/
#define PCF8574_ID      		0x40																	// PCF8574 ID Code
#define PCF8574A_ID     		0x70																	// PCF8574A ID Code

/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)

/***********************
 * Function Prototypes *
 ***********************/
void init_i2c(void);																							// Initial I2C
void pcf8574_write_byte(unsigned char device_id,
                        unsigned char i2c_data);									
unsigned char pcf8574_read_byte(unsigned char device_id);		
void delay(unsigned long);																				// Delay Time Function(1..4294967295)

void init_uart0(unsigned long freq, unsigned long baud);
void my_putchar0(char ch);
void print_uart0(char *uart_buf);
void print_hex_byte(unsigned char hexval);
                  
/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	unsigned int j;				   																				// LED Output Loop Count
	unsigned char last_input,this_input;

	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
	
	LED_PORT_INIT();  																							// Configure LED = Output 
	LED_OFF();
	
	// Initial i2c
	init_i2c();
	
	//Main Loop Test PCF8574 Output Mode
	while(1)
	{
		// Shift Left Bit[7..0]                 							
    for (j=0x01; j<= 0x80; j<<=1)  																// Blink LED P0.7 <- P0.0
    {   
			LED_TOGGLE();
      pcf8574_write_byte(PCF8574A_ID,~j);
      delay(25000);																								// Delay Blink
    }
	}

	// Initial UART
	//init_uart0(MY_FREQ, MY_BAUD);
	//print_uart0("\n\rET-BASE Z51F6412 V1.0 Run Internal RC 8.00 MHz");
	//print_uart0("\n\rDemo Test I2C PCF8574/A Input Mode\n\r");
	//pcf8574_write_byte(PCF8574A_ID,0xFF);															// Makesure PCF8574/A = Input
	//last_input = 0x55;
	
  //Main Loop Test PCF8574 Input Mode
	//while(1)
	//{
	//	this_input=pcf8574_read_byte(PCF8574A_ID);
	//	
	//	//Verify Second Time Change
	//	if(this_input!=last_input)
	//	{
  //    last_input=this_input;
  //    print_uart0("\n\rInput = ");			
	//		print_hex_byte(this_input);
	//	}	
	//}
	//End of Test Read PCF8574/A
}

/*
 * Function    : init_i2c()
 * Description : initialize I2C resources
 */
void init_i2c(void)
{
	// I2CMR = 0,1,0,0,0,0,0,0
  // 0xxxxxxx : Disable Generate Interrupt
  // x1xxxxxx : I2C Enable
  // xx0xxxxx : Disable Auto Initial I2C on Reset
	// xxx0xxxx : Disable I2C Interrupt
	// xxxx0xxx : Disable ACK
	// xxxxx0xx : Default Read Only Bit
	// xxxxxx0x : Disable Stop
	// xxxxxxx0 : Disable Start
	I2CMR = 0x40;

	// I2C Speed(100KHz-400KHz)
	// I2CSCLLR = tsclk x (4 x SCLL + 1)
	// I2CSCLHR = tsclk x (4 x SCLL + 3)
	// FI2C     = 1 / tsclk x (4x(SCLL+SCLH)+4)
	// Fx       = 8.00 MHz
	// tsclk    = 1/8.00MHz
	//          = 125nS
  // FI2C     = 1 / 125nS x (4x(SCLL+SCLH)+4)
	//          = 1 / 125nS x (4x(10+10)+4)
	//          = 95.23KHz
	I2CSCLLR = 10; 																						// Low Period pf SCL
	I2CSCLHR = 10;																						// High Periof of SCL
	I2CSDAHR = 5;																							// 1/2 of SCLL
	
	// xxxx0xxx :	Disable I2C Interrupt									
	IE1 &= ~0x08;	
}

/*
 * Function    : pcf8574_write_byte()
 * Description : Write Data to DS1307
 */
void pcf8574_write_byte(unsigned char device_id,unsigned char i2c_data)
{  
	I2CDR = device_id;																					// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																						// Start Conditiontion + Send Byte Address	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 															// Wait Status Complete  
		
	I2CDR = i2c_data;																						// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
  I2CMR |= (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100010
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx1x : I2C is Transmitt
	while((I2CSR & 0x22) != 0x22); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
}

/*
 * Function    : pcf8574_read_byte()
 * Description : Read Data From PCF8574
 */
unsigned char pcf8574_read_byte(unsigned char device_id)
{  
	unsigned char i2c_data;																			// Dummy Read Byte
	
	/* Start Write ID+Read */
	I2CDR = (device_id|0x01);																		// Send ID Code + Read(xxxxxxx+1)   
	I2CMR |= (1<<0);																						// Re-Start Conditiontion + Send Byte ID Read	
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 10000101
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x85) != 0x85); 															// Wait Status Complete  

  // Enable Read Byte */
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	/* Read Byte & Stop Read */
	i2c_data = I2CDR;																						// Read Byte Data	
	I2CMR &= ~(0x08);																						// Disable ACK(Stop Read)
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx0 : Not Receive ACK on 9th SCL Cycle
  while((I2CSR & 0x44) != 0x44); 															// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |=  (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100000
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx0x : I2C is Receive
	while((I2CSR & 0x20) != 0x20); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	
	return i2c_data;
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long i)
{
  while(i > 0) {i--;}																					// Loop Decrease Counter	
  return;
}

/*
 * Function    : init_uart0()
 * Description : initialize UART0 resources
 *             : baudrates, parirty, stop bit
 */
void init_uart0(unsigned long freq, unsigned long baud)
{
	// UCTRL01 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL01 = 0x06;

	// UCTRL02 = 0,0,0,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx0xxxxx : Disable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL02 = 0x0F;
	
	// UCTRL03 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL03 = 0x00;
	
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:14.7456MHz,Baudrate:9600
	// UBAUDx   = (14.7456MHz/(9600*8))-1
	//          = (14.7456MHz/76800)-1
	//          = 192-1
  //          = 191 = 0xBF  
	UBAUD0 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxxxxxx0 :	Disable UART0 RX Interrupt									
	IE1 &= ~0x01;	
}

/*
 * Function    : my_putchar0
 * Description : Output a character to uart0
 */
void my_putchar0(char ch)
{
  while(!(USTAT0 & 0x80));																				// Wait until Trammit Buffer is Empty
	UDATA0 = ch;
}

/*
 * Function    : print_uart0
 * Description : print a string to uart0
 */
void print_uart0(char *uart_buf)
{
	while(*uart_buf != '\0')
	{	
	  my_putchar0(*uart_buf++);	
	}	
}

/*
 * Function    : print_hex_byte()
 * Description : print a character in hexadecimal format
 */
void print_hex_byte(unsigned char hexval)
{
  unsigned int temp;
	char i;

	for (i = 1; i>= 0; i--)
	{
		temp = hexval;
		temp >>= 4*i;
		temp &= 0x0F;

		if (temp >= 10) 
		{
			temp -= 10;
			temp += 'A';
		}
		else 
		{	
			temp +='0';
		}	
		my_putchar0(temp);
	}
}



