/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run Internal RC (8.00 MHz)      */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Demo I2C Bus Function           */
/* Update   : 18 August 2012                  */
/**********************************************/
//
// Test Demo I2C with "ET-MINI DS1307" Board
//
// I2C Interface
// ET-BASE Z51F66412 <----> ET-MINI DS1307
// P0.6(SCL)        ------>  SCL
// P0.7(SDA)        <----->  SDA

// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>																							// Z8051 : Z51F6412

/*********************************
 * ET-MINI DS1307 Device ID Code *
 *********************************/
#define DS1307_ID			  		0xD0																	// DS1307 ID Code
#define RTC_SECOND      		0
#define RTC_MINUTE      		1
#define RTC_HOUR        		2
#define RTC_WKDAY       		3
#define RTC_DATE        		4
#define RTC_MONTH      	 		5
#define RTC_YEAR        		6
#define RTC_CTRL        		7

/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)

/*******************
 * Global Variable *
 *******************/
unsigned char second;
unsigned char minute;
unsigned char hour;
unsigned char wkday;
unsigned char date;
unsigned char month;
unsigned char year;
unsigned char ctrl;

/***********************
 * Function Prototypes *
 ***********************/
void init_uart0(unsigned long freq, unsigned long baud);
void my_putchar0(char ch);
void print_uart0(char *uart_buf);
unsigned char BCDtoDecimal(unsigned char bcdval);
void PrintBCD(unsigned char bcdval);
void PrintDayName(unsigned char name);
void PrintMonthName(unsigned char name);

void init_i2c(void);																							// Initial I2C
void ds1307_write_byte(unsigned char i2c_address,									// Write DS1307
                      unsigned char i2c_data);
unsigned char ds1307_read_byte(unsigned char i2c_address);				// Read DS1307
void ds1307_set_clock(void);						  												// Save RTC Clock Time to DS1307                     
void ds1307_get_clock(void);																			// Read RTC Clock Time From DS1307											


/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	unsigned char last_second;
	
	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
	
	LED_PORT_INIT();  																							// Configure LED = Output 
	LED_OFF();
	
	// Initial UART
	init_uart0(MY_FREQ, MY_BAUD);
  print_uart0("\n\rET-BASE Z51F6412 V1.0 Run Internal RC at 8MHZ");
	print_uart0("\n\rDemo I2C Interface DS1307");
	
	// Initial i2c
	init_i2c();
	second = 0x55;
	minute=0x59;
	hour=0x23;
	wkday=0x07;
	date=0x31;
	month=0x12;
	year=0x08;
	ctrl=0x00;
	ds1307_set_clock();																							// Setup RTC Clock Time
	LED_ON();
	
	//Main Loop
	while(1)
	{
		ds1307_get_clock();																						// Read RTC Clock Time
		
		//Verify Second Time Change
		if(second!=last_second)
		{	
		  last_second=second;
			LED_TOGGLE();
			print_uart0("\r\nTime : ");
			PrintBCD(hour);
			my_putchar0(':');
			PrintBCD(minute);
			my_putchar0(':');
			PrintBCD(second);
			my_putchar0(':');
			PrintDayName(BCDtoDecimal(wkday));
			my_putchar0('-');
			PrintBCD(date);
			my_putchar0('-');
			PrintMonthName(BCDtoDecimal(month));
			print_uart0("-20");
			PrintBCD(year);
		}
	} 
}

/*
 * Function    : init_uart0()
 * Description : initialize UART0 resources
 *             : baudrates, parirty, stop bit
 */
void init_uart0(unsigned long freq, unsigned long baud)
{
	// UCTRL01 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL01 = 0x06;

	// UCTRL02 = 0,0,0,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx0xxxxx : Disable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL02 = 0x0F;
	
	// UCTRL03 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL03 = 0x00;
	
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:8MHz,Baudrate:9600
	// UBAUDx   = (8MHz/(9600*8))-1
	//          = (8MHz/76800)-1
	//          = 104.166-1
  //          = 103.166
  //          = 103 = 0x67  
	UBAUD0 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxxxxxx0 :	Disable UART0 RX Interrupt									
	IE1 &= ~0x01;	
}

/*
 * Function    : my_putchar0
 * Description : Output a character to uart0
 */
void my_putchar0(char ch)
{
  while(!(USTAT0 & 0x80));																		// Wait until Trammit Buffer is Empty
	UDATA0 = ch;
}

/*
 * Function    : print_uart0
 * Description : print a string to uart0
 */
void print_uart0(char *uart_buf)
{
	while(*uart_buf != '\0')
	{	
	  my_putchar0(*uart_buf++);	
	}	
}

/*
 * Function    : init_i2c()
 * Description : initialize I2C resources
 */
void init_i2c(void)
{
	// I2CMR = 0,1,0,0,0,0,0,0
  // 0xxxxxxx : Disable Generate Interrupt
  // x1xxxxxx : I2C Enable
  // xx0xxxxx : Disable Auto Initial I2C on Reset
	// xxx0xxxx : Disable I2C Interrupt
	// xxxx0xxx : Disable ACK
	// xxxxx0xx : Default Read Only Bit
	// xxxxxx0x : Disable Stop
	// xxxxxxx0 : Disable Start
	I2CMR = 0x40;

	// I2C Speed(100KHz-400KHz)
	// I2CSCLLR = tsclk x (4 x SCLL + 1)
	// I2CSCLHR = tsclk x (4 x SCLL + 3)
	// FI2C     = 1 / tsclk x (4x(SCLL+SCLH)+4)
	// Fx       = 8.00 MHz
	// tsclk    = 1/8.00MHz
	//          = 125nS
  // FI2C     = 1 / 125nS x (4x(SCLL+SCLH)+4)
	//          = 1 / 125nS x (4x(10+10)+4)
	//          = 95.23KHz
	I2CSCLLR = 10; 																						// Low Period pf SCL
	I2CSCLHR = 10;																						// High Periof of SCL
	I2CSDAHR = 5;																							// 1/2 of SCLL
	
	// xxxx0xxx :	Disable I2C Interrupt									
	IE1 &= ~0x08;	
}

/*
 * Function    : ds1307_write_byte()
 * Description : Write Data to DS1307
 */
void ds1307_write_byte(unsigned char i2c_address,unsigned char i2c_data)
{  
	I2CDR = DS1307_ID;																					// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																						// Start Conditiontion + Send Byte Address	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 															// Wait Status Complete  
	
  I2CDR = i2c_address;																				// Send Address   
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
		
	I2CDR = i2c_data;																						// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
  I2CMR |= (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100010
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx1x : I2C is Transmitt
	while((I2CSR & 0x22) != 0x22); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
}

/*
 * Function    : ds1307_read_byte()
 * Description : Read Data From DS1307
 */
unsigned char ds1307_read_byte(unsigned char i2c_address)
{  
	unsigned char i2c_data;																			// Dummy Read Byte
	
	/* Start Write ID+Write */
	I2CDR = DS1307_ID;																					// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																						// Start Conditiontion + Send Byte ID Write	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 															// Wait Status Complete  
	
	/* Write Address RTC For Read */
  I2CDR = i2c_address;																				// Send Address   
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
	/* Re-Start Write ID+Read */
	I2CDR = (DS1307_ID|0x01);																		// Send ID Code + Read(xxxxxxx+1)   
	I2CMR |= (1<<0);																						// Re-Start Conditiontion + Send Byte ID Read	
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 10000101
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x85) != 0x85); 															// Wait Status Complete  

  // Enable Read Byte */
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	/* Read Byte & Stop Read */
	i2c_data = I2CDR;																						// Read Byte Data	
	I2CMR &= ~(0x08);																						// Disable ACK(Stop Read)
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx0 : Not Receive ACK on 9th SCL Cycle
  while((I2CSR & 0x44) != 0x44); 															// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |=  (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100000
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx0x : I2C is Receive
	while((I2CSR & 0x20) != 0x20); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	
	return i2c_data;
}

/*
 * Function    : ds1307_set_clock()
 * Input       : second,minute,hour,wkday,date,month,year,ctrl            
 * Description : setup RTC Clock Time to DS1307
 */
void ds1307_set_clock(void)
{ 
  /* Start Condition + IDCode+Write	*/
	I2CDR = DS1307_ID;																					// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																						// Start Conditiontion + Send Byte Address	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 															// Wait Status Complete  
	
	/* Address = (Second) */
  I2CDR = RTC_SECOND;																			  	// Send RTC Address = Second   
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
	/* 1st Data = second */
	I2CDR = second;					  																	// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
	/* 2nd Data = minute */
	I2CDR = minute;					  																	// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 3rd data = hour */
	I2CDR = hour;					  																		// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 4th data = week day */
	I2CDR = wkday;					  																	// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 5th data = date */
	I2CDR = date;					  																		// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 6th data = month */
	I2CDR = month;					  																	// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 7th data = year */
	I2CDR = year;					  																		// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* 8th data = control */
	I2CDR = ctrl;					  																		// Send Data
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |= (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100010
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx1x : I2C is Transmitt
	while((I2CSR & 0x22) != 0x22); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
}

/*
 * Function    : ds1307_get_clock()
 * Output      : second,minute,hour,wkday,date,month,year,ctrl    
 * Description : Read RTC Clock Time From DS1307
 */
void ds1307_get_clock(void)
{  
	/* Start Write ID+Write */
	I2CDR = DS1307_ID;																					// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																						// Start Conditiontion + Send Byte ID Write	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 															// Wait Status Complete  
	
	/* Write Address RTC For Read */
  I2CDR = RTC_SECOND;																					// Address = second   
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 															// Wait Status Complete	
	
	/* Re-Start Write ID+Read */
	I2CDR = (DS1307_ID|0x01);																		// Send ID Code + Read(xxxxxxx+1)   
	I2CMR |= (1<<0);																						// Re-Start Conditiontion + Send Byte ID Read	
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 10000101
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x85) != 0x85); 															// Wait Status Complete  

  // Enable Read Byte */
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 1st Byte */
	second = I2CDR;																				    	// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 2nd Byte */
	minute = I2CDR;																				    	// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 3rd Byte */
	hour = I2CDR;																				    		// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 4th Byte */
	wkday = I2CDR;																				    	// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 5th Byte */
	date = I2CDR;																				    		// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 6th Byte */
	month = I2CDR;																				    	// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	// Read 7th Byte */
	year = I2CDR;																				    		// Read Byte Data	
	I2CMR |= (0x08);																						// Enable ACK
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 															// Wait Status Complete
	
	/* Read 8th Byte & Stop Read */
	ctrl = I2CDR;																					    	// Read Byte Data	
	I2CMR &= ~(0x08);																						// Disable ACK(Stop Read)
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx0 : Not Receive ACK on 9th SCL Cycle
  while((I2CSR & 0x44) != 0x44); 															// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |=  (0x02);																						// Enable Generate Stop Condition
	I2CSR = 0x00;																								// Write Dummy For Reset Status
	//Wait I2CSR = 00100000
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx0x : I2C is Receive
	while((I2CSR & 0x20) != 0x20); 															// Wait Status Complete
	I2CSR = 0x00;																								// Write Dummy For Reset Status
}

unsigned char BCDtoDecimal(unsigned char bcdval)
{
	return bcdval / 16 * 10 + bcdval % 16;
}

void PrintBCD(unsigned char bcdval)
{
	my_putchar0((bcdval >> 4)   | 0x30);
	my_putchar0((bcdval & 0x0F) | 0x30);
}

void PrintDayName(unsigned char name)
{
  switch(name)
	{
	  case 1: print_uart0("Sunday");
			      break;
		
		case 2: print_uart0("Monday");
			      break;
		
		case 3: print_uart0("Tuesday");
			      break;
		
		case 4: print_uart0("Wednesday");
			      break;
		
		case 5: print_uart0("Thursday");
			      break;
		
		case 6: print_uart0("Friday");
			      break;
		
		case 7: print_uart0("Saturday");
			      break;
		
		default : print_uart0("???");
  }
}

void PrintMonthName(unsigned char name)
{
	switch(name)
	{
		case 1:  print_uart0("January");
			       break;
		
		case 2:  print_uart0("February");
			       break;
		
		case 3:  print_uart0("March");
			       break;
		
		case 4:  print_uart0("April");
			       break;
		
		case 5:  print_uart0("May");
			       break;
		
		case 6:  print_uart0("June");
			       break;
		
		case 7:  print_uart0("July");
			       break;
		
		case 8:  print_uart0("August");
			       break;
		
		case 9:  print_uart0("September");
			       break;
		
		case 10: print_uart0("October");
			       break;
		
		case 11: print_uart0("November");
			       break;
		
		case 12: print_uart0("December");
			       break;
		
		default : print_uart0("???");
  }
}
