/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run Internal RC (8.00 MHz)      */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Demo I2C Bus Function           */
/* Update   : 18 August 2012                  */
/**********************************************/
//
// Test Demo I2C with "ET-MINI 24XX" Board
//
// I2C Interface
// ET-BASE Z51F66412 <----> ET-MINI 24XX(24LC32)
// P0.6(SCL)        ------>  SCL
// P0.7(SDA)        <----->  SDA

#include <z51f6412.h>																							// Z8051 : Z51F6412

/***************************************/
/* ET-MINI 24XX(24LC32) Device ID Code */
/***************************************/
#define EEP_DEVICE0_ID  		0xA8																	// 1010100+0
#define EEP_DEVICE1_ID  		0xAA																	// 1010101+0
#define EEP_DEVICE2_ID  		0xAC																	// 1010110+0
#define EEP_DEVICE3_ID  		0xAE																	// 1010111+0

/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)

/*******************
 * Global Variable *
 *******************/
unsigned char eep_buff[32];
code unsigned char msg[4][32] = { "This is Data Test Device[0]",
	                                "This is Data Test Device[1]",
	                                "This is Data Test Device[2]",
	                                "This is Data Test Device[3]"};
/***********************
 * Function Prototypes *
 ***********************/
void init_uart0(unsigned long freq, unsigned long baud);
void my_putchar0(char ch);
void print_uart0(char *uart_buf);
void print_hex_byte(unsigned char hexval);

void init_i2c(void);																							// Initial I2C
void eep_write_byte(unsigned char i2c_device,
                    unsigned int i2c_address,						
                    unsigned char i2c_data);										
void eep_write_page(unsigned char i2c_device,
                    unsigned int i2c_address,						
                    unsigned char *buf_data,
										unsigned char length);																				
unsigned char eep_read_byte(unsigned char i2c_device,
                            unsigned int i2c_address);															
void eep_read_page(unsigned char device,
                   unsigned int i2c_address,						
                   unsigned char *buf_data,
									 unsigned char length);	
										
/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	unsigned char byte_data;
	unsigned char byte_count;
	
	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
	
	LED_PORT_INIT();  																							// Configure GPIO Port = Output Connect LED
	LED_OFF();
	
	// Initial UART
	init_uart0(MY_FREQ, MY_BAUD);
  print_uart0("\n\n\rET-BASE Z51F6412 V1.0 Run Internal RC at 8MHZ");
	print_uart0("\n\rDemo I2C Interface EEPROM 24LC32\n\r");
	
	// Initial i2c
	init_i2c();
	LED_ON();
	
	print_uart0("\n\rTest Write Byte Mode EEPROM 24LC32");
	eep_write_byte(EEP_DEVICE0_ID,0x0000,0x30);											// Write Device0 Address:0x0000	= 0x30
	eep_write_byte(EEP_DEVICE1_ID,0x0000,0x31);											// Write Device1 Address:0x0000	= 0x31
	eep_write_byte(EEP_DEVICE2_ID,0x0000,0x32);											// Write Device2 Address:0x0000	= 0x32
	eep_write_byte(EEP_DEVICE3_ID,0x0000,0x33);											// Write Device3 Address:0x0000	= 0x33
	
	byte_data = eep_read_byte(EEP_DEVICE0_ID,0x0000);								// Read Device0 Address:0x0000
	print_uart0("\n\rTest Write Device0 Address:0x0000 = 0x30 & Read = ");
	print_hex_byte(byte_data);
	byte_data = eep_read_byte(EEP_DEVICE1_ID,0x0000);								// Read Device1 Address:0x0000
	print_uart0("\n\rTest Write Device1 Address:0x0000 = 0x31 & Read = ");
	print_hex_byte(byte_data);
	byte_data = eep_read_byte(EEP_DEVICE2_ID,0x0000);								// Read Device2 Address:0x0000
	print_uart0("\n\rTest Write Device2 Address:0x0000 = 0x32 & Read = ");
	print_hex_byte(byte_data);
	byte_data = eep_read_byte(EEP_DEVICE3_ID,0x0000);								// Read Device3 Address:0x0000
	print_uart0("\n\rTest Write Device3 Address:0x0000 = 0x33 & Read = ");
	print_hex_byte(byte_data);
	
	print_uart0("\n\n\rTest Write Page Mode EEPROM 24LC32");
	eep_write_page(EEP_DEVICE0_ID,0x0040,msg[0],sizeof(msg[0]));
	eep_write_page(EEP_DEVICE1_ID,0x0040,msg[1],sizeof(msg[1]));
	eep_write_page(EEP_DEVICE2_ID,0x0040,msg[2],sizeof(msg[2]));
	eep_write_page(EEP_DEVICE3_ID,0x0040,msg[3],sizeof(msg[3]));
	
	eep_read_page(EEP_DEVICE0_ID,0x0040,eep_buff,sizeof(msg[0]));
	print_uart0("\n\rTest Read Page Device0 Address:0x0040 = ");
	for(byte_count=0;byte_count<sizeof(msg[0]);byte_count++)
	{
		my_putchar0(eep_buff[byte_count]);
	}
	
	eep_read_page(EEP_DEVICE1_ID,0x0040,eep_buff,sizeof(msg[1]));
	print_uart0("\n\rTest Read Page Device1 Address:0x0040 = ");
	for(byte_count=0;byte_count<sizeof(msg[1]);byte_count++)
	{
		my_putchar0(eep_buff[byte_count]);
	}
	
	eep_read_page(EEP_DEVICE2_ID,0x0040,eep_buff,sizeof(msg[2]));
	print_uart0("\n\rTest Read Page Device2 Address:0x0040 = ");
	for(byte_count=0;byte_count<sizeof(msg[2]);byte_count++)
	{
		my_putchar0(eep_buff[byte_count]);
	}
	
	eep_read_page(EEP_DEVICE3_ID,0x0040,eep_buff,sizeof(msg[3]));
	print_uart0("\n\rTest Read Page Device3 Address:0x0040 = ");
	for(byte_count=0;byte_count<sizeof(msg[3]);byte_count++)
	{
		my_putchar0(eep_buff[byte_count]);
	}
	
	//Main Loop
	while(1)
	{
	} 
}

/*
 * Function    : init_uart0()
 * Description : initialize UART0 resources
 *             : baudrates, parirty, stop bit
 */
void init_uart0(unsigned long freq, unsigned long baud)
{
	// UCTRL01 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL01 = 0x06;

	// UCTRL02 = 0,0,0,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx0xxxxx : Disable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL02 = 0x0F;
	
	// UCTRL03 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL03 = 0x00;
	
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:8MHz,Baudrate:9600
	// UBAUDx   = (8MHz/(9600*8))-1
	//          = (8MHz/76800)-1
	//          = 104.166-1
  //          = 103.166
  //          = 103 = 0x67  
	UBAUD0 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxxxxxx0 :	Disable UART0 RX Interrupt									
	IE1 &= ~0x01;	
}

/*
 * Function    : my_putchar0
 * Description : Output a character to uart0
 */
void my_putchar0(char ch)
{
  while(!(USTAT0 & 0x80));																				// Wait until Trammit Buffer is Empty
	UDATA0 = ch;
}

/*
 * Function    : print_uart0
 * Description : print a string to uart0
 */
void print_uart0(char *uart_buf)
{
	while(*uart_buf != '\0')
	{	
	  my_putchar0(*uart_buf++);	
	}	
}

/*
 * Function    : print_hex_byte()
 * Description : print a character in hexadecimal format
 */
void print_hex_byte(unsigned char hexval)
{
  unsigned int temp;
	char i;

	for (i = 1; i>= 0; i--)
	{
		temp = hexval;
		temp >>= 4*i;
		temp &= 0x0F;

		if (temp >= 10) 
		{
			temp -= 10;
			temp += 'A';
		}
		else 
		{	
			temp +='0';
		}	
		my_putchar0(temp);
	}
}

/*
 * Function    : init_i2c()
 * Description : initialize I2C resources
 */
void init_i2c(void)
{
	// I2CMR = 0,1,0,0,0,0,0,0
  // 0xxxxxxx : Disable Generate Interrupt
  // x1xxxxxx : I2C Enable
  // xx0xxxxx : Disable Auto Initial I2C on Reset
	// xxx0xxxx : Disable I2C Interrupt
	// xxxx0xxx : Disable ACK
	// xxxxx0xx : Default Read Only Bit
	// xxxxxx0x : Disable Stop
	// xxxxxxx0 : Disable Start
	I2CMR = 0x40;

	// I2C Speed(100KHz-400KHz)
	// I2CSCLLR = tsclk x (4 x SCLL + 1)
	// I2CSCLHR = tsclk x (4 x SCLL + 3)
	// FI2C     = 1 / tsclk x (4x(SCLL+SCLH)+4)
	// Fx       = 8.00 MHz
	// tsclk    = 1/8.00MHz
	//          = 125nS
  // FI2C     = 1 / 125nS x (4x(SCLL+SCLH)+4)
	//          = 1 / 125nS x (4x(10+10)+4)
	//          = 95.23KHz
	I2CSCLLR = 10; 																						// Low Period pf SCL
	I2CSCLHR = 10;																						// High Periof of SCL
	I2CSDAHR = 5;																							// 1/2 of SCLL
	
	// xxxx0xxx :	Disable I2C Interrupt									
	IE1 &= ~0x08;	
}

/*
 * Function    : eep_write_byte()
 * Description : Write Byte Data to 24LC32
 */
void eep_write_byte(unsigned char i2c_device,											// Device ID Code
                    unsigned int i2c_address,											// Address for Write
                    unsigned char i2c_data)												// Data for Write
{
	I2CDR = i2c_device;																							// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																								// Start Conditiontion + Send Byte Address	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 																	// Wait Status Complete  
	
  I2CDR = (i2c_address>>8)&0x0F;																	// Send Address MSB:A[11...8]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	I2CDR = i2c_address & 0xFF;																			// Send Address LSB:A[7...0]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	I2CDR = i2c_data;																								// Send Data
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
  I2CMR |= (0x02);																								// Enable Generate Stop Condition
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 00100010
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx1x : I2C is Transmitt
	while((I2CSR & 0x22) != 0x22); 																	// Wait Status Complete
	I2CSR = 0x00;																										// Write Dummy For Reset Status
}

/*
 * Function    : eep_write_page()
 * Description : Write Page Data to 24LC32(Max 32Byte)
 */
void eep_write_page(unsigned char i2c_device,											// Device ID Code
                    unsigned int i2c_address,											// Start Address of Page
                    unsigned char *buf_data,											// Buffer Data fo Write
										unsigned char length)													// Length of Data For Write
{
	unsigned char byte_cnt;																					// Counter For Write Data
	
	I2CDR = i2c_device;																							// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																								// Start Conditiontion + Send Byte Address	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 																	// Wait Status Complete  
	
  I2CDR = (i2c_address>>8)&0x0F;																	// Send Address MSB:A[11...8]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	I2CDR = i2c_address & 0xFF;																			// Send Address LSB:A[7...0]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	for(byte_cnt=0;byte_cnt<length;byte_cnt++)
	{
	  I2CDR = buf_data[byte_cnt]; 																	// Send Data
	  I2CSR = 0x00;																									// Write Dummy For Reset Status
	  //Wait I2CSR = 01000111
	  //x1xxxxxx : Byte Transfer Complete
	  //xxxxx1xx : I2C is Busy
	  //xxxxxx1x : I2C is Transmitt
	  //xxxxxxx1 : ACK Receive on 9th SCL Cycle
    while((I2CSR & 0x47) != 0x47); 																// Wait Status Complete	
	}	
	
  I2CMR |= (0x02);																								// Enable Generate Stop Condition
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 00100010
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx1x : I2C is Transmitt
	while((I2CSR & 0x22) != 0x22); 																	// Wait Status Complete
	I2CSR = 0x00;																										// Write Dummy For Reset Status
}
										
/*
 * Function    : eep_read_byte()
 * Description : Read Byte Data From 24LC32
 */
unsigned char eep_read_byte(unsigned char i2c_device,							// Device ID Code
                            unsigned int i2c_address)							// Address For Read
{
	unsigned char i2c_data;																					// Dummy Read Byte
	
	/* Start Write ID+Write */
	I2CDR = i2c_device;																							// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																								// Start Conditiontion + Send Byte ID Write	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 																	// Wait Status Complete  
	
	/* Write Address RTC For Read */
  I2CDR = (i2c_address>>8)&0x0F;																	// Send Address MSB:A[11...8]  
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	/* Write Address RTC For Read */
  I2CDR = i2c_address & 0xFF;																			// Send Address LSB:A[7...0]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	/* Re-Start Write ID+Read */
	I2CDR = (i2c_device|0x01);																			// Send ID Code + Read(xxxxxxx+1)   
	I2CMR |= (1<<0);																								// Re-Start Conditiontion + Send Byte ID Read	
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 10000101
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x85) != 0x85); 																	// Wait Status Complete  

  // Enable Read Byte */
	I2CMR |= (0x08);																								// Enable ACK
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x45) != 0x45); 																	// Wait Status Complete
	i2c_data = I2CDR;																								// Read Byte Data	
	
	/* Stop Read */
	I2CMR &= ~(0x08);																								// Disable ACK(Stop Read)
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx0 : Not Receive ACK on 9th SCL Cycle
  while((I2CSR & 0x44) != 0x44); 																	// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |=  (0x02);																								// Enable Generate Stop Condition
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 00100000
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx0x : I2C is Receive
	while((I2CSR & 0x20) != 0x20); 																	// Wait Status Complete
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	
	return i2c_data;
}	

/*
 * Function    : eep_read_page()
 * Description : Read Page Data From 24LC32(Max 32Byte)
 */
void eep_read_page(unsigned char i2c_device,											// Device ID
                    unsigned int i2c_address,						    			// Start Address of Page
                    unsigned char *buf_data,                			// Buffer Save Data
										unsigned char length)													// Length of Data Read
{
	unsigned char byte_cnt;																					// Counter for Read Data
	
	/* Start Write ID+Write */
	I2CDR = i2c_device;																							// Send ID Code + Write(xxxxxxx+0)   
	I2CMR |= (1<<0);																								// Start Conditiontion + Send Byte ID Write	
	//Wait I2CSR = 10000111
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x87) != 0x87); 																	// Wait Status Complete  
	
	/* Write Address RTC For Read */
  I2CDR = (i2c_address>>8)&0x0F;																	// Send Address MSB:A[11...8]  
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	/* Write Address RTC For Read */
  I2CDR = i2c_address & 0xFF;																			// Send Address LSB:A[7...0]
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000111
	//x1xxxxxx : Byte Transfer Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx1x : I2C is Transmitt
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x47) != 0x47); 																	// Wait Status Complete	
	
	/* Re-Start Write ID+Read */
	I2CDR = (i2c_device|0x01);																			// Send ID Code + Read(xxxxxxx+1)   
	I2CMR |= (1<<0);																								// Re-Start Conditiontion + Send Byte ID Read	
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 10000101
	//1xxxxxxx : Receive ACK
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx1 : ACK Receive on 9th SCL Cycle
  while((I2CSR & 0x85) != 0x85); 																	// Wait Status Complete  

  //Read Page Data
	for(byte_cnt=0;byte_cnt<length;byte_cnt++)
	{
		// Enable Read Byte */
	  I2CMR |= (0x08);																							// Enable ACK
	  I2CSR = 0x00;																									// Write Dummy For Reset Status
	  //Wait I2CSR = 01000101
	  //x1xxxxxx : Byte Read Transmit Complete
	  //xxxxx1xx : I2C is Busy
	  //xxxxxx0x : I2C is Receive
	  //xxxxxxx1 : ACK Receive on 9th SCL Cycle
    while((I2CSR & 0x45) != 0x45); 																// Wait Status Complete
		buf_data[byte_cnt] = I2CDR;	
	}
	
	/* Read Byte & Stop Read */
	I2CMR &= ~(0x08);																								// Disable ACK(Stop Read)
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 01000101
	//x1xxxxxx : Byte Read Transmit Complete
	//xxxxx1xx : I2C is Busy
	//xxxxxx0x : I2C is Receive
	//xxxxxxx0 : Not Receive ACK on 9th SCL Cycle
  while((I2CSR & 0x44) != 0x44); 																	// Wait Status Complete
	
	/* Stop Condition */
  I2CMR |=  (0x02);																								// Enable Generate Stop Condition
	I2CSR = 0x00;																										// Write Dummy For Reset Status
	//Wait I2CSR = 00100000
	//xx1xxxxx : Stop Condition Detect
	//xxxxxx0x : I2C is Receive
	while((I2CSR & 0x20) != 0x20); 																	// Wait Status Complete
	I2CSR = 0x00;																										// Write Dummy For Reset Status
}

