/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run XTAL-10MHz                  */
/* Note     : Program Fuse Config = 0x08 for  */
/*          : Enable External Main Oscillator */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Demo Buzzer Drive Sound         */
/* Update   : 18 August 2012                  */
/**********************************************/
// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>																							// Z8051 : Z51F6412

/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)

/****************************************/
/* ET-BASE Z51F6412 Hardware Buzzer Pin */
/****************************************/
#define BUZZER_ENABLE()  		BUZCR |=  0x01 												// Enable Buzzer
#define BUZZER_DISABLE() 		BUZCR &= ~0x01 												// Disable Buzzer
#define BUZZER_DELAY			  100																		// 100mS Buzzer Delay

/*******************
 * Global Variable *
 *******************/
// Bit Variable
static bit this_sw;															
static bit last_sw;	
sbit SW_READ = P8^0;    																					// Bit Buffer Read P8[0]

static volatile unsigned int BuzzerTick = 0;											// LED Blinking Count(1..65536mS)
code  unsigned char buzzer_data[4]={0x1F, 0x17, 0x10, 0x07};			//Buzzer Frequency
static unsigned char sound_index = 0;

/***********************
 * Function Prototypes *
 ***********************/
void init_timer0(void);

/********************
 * Function: main() *
 ********************/
void main(void)
{	 
  /* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
  
	//Config LED
  LED_PORT_INIT();																								// Config LED = Output 
	LED_ON();
	
	//Config SW
	SW_PORT_INIT();																									// Config SW = Input 
	SW_DEBOUNCE_INIT();																							// Enable Debounce For SW
  SW_PULLUP_INIT();                           										// Enable Pull-Up For SW 
	SW_LOGIC_INIT();																								// Default Input Logic = "1"
														
  //Default SW Status = Release 
  last_sw = 0;																										// Default SW Status = Release
	
	// Config Buzzer
	// BUZCR = 00000,11,1
	// 00000xxx : Default Reserve Bit
	// xxxxx11x : Buzzer Clock Source = Fx/256
	// xxxxxxx0 : Buzzer Disable
	BUZCR |= 0x06;																

	init_timer0();
	BuzzerTick = 0;
	EA = 1;	
	
	sound_index = 0;
	BUZDR =  buzzer_data[sound_index];
	BUZCR |= 0x01;																									// Enable Buzzer
	
	while(1)
	{
		// Read New SW Status
		this_sw = SW_READ;																						// Read Bit SW
		
		if(this_sw != last_sw)																				// If SW Status Change
    {
			//Verify SW Press & Service
			if((last_sw==1)&&(this_sw==0))
			{
				sound_index++;
				if(sound_index > 3)
				{	
				 sound_index = 0;	
				}	
				BUZDR =  buzzer_data[sound_index];				
      }	
			
			//Verify SW Release & Service
			if((last_sw==0)&&(this_sw==1))
			{
      }	
			
			//Update SW Reference Status
			last_sw = this_sw;		
    }			
	} 
}

/*
 * Function    : init_timer0
 * Description : initialize timer[1:0] = 16Bit Mode
 *               Trigger Interrupt 1mS(fx 8.00 Mhz)
 */
void init_timer0(void)
{
	//Timer1:Timer0 = 16Bit Mode
	//Fx = 8.00 MHz
	//Timer Clock Source = 4.00 MHz
	//1 Cycle = 1/4.00MHz
	//        = 250nS
	// 1mS Period = 1mS/250nS
	//            = 4000 Cycle
	//            = 0x0FA0
	T1DR = 0x0F;		  																							// MSB : 1mS period ((8MHZ/2)*1ms)
	T0DR = 0xA0;																										// LSB : 1mS
	
  // Initial Timer1 = 0,1,0,0,11,1,0
  // 0xxxxxxx : Don't care(Used in PWM)
  // x1xxxxxx : 16 Bit Timer Mode
  // xx0xxxxx : PWM Disable
	// xxx0xxxx : Timer/Counter Mode
  // xxxx11xx : Clock Source = Timer0 Clock
  // xxxxxx1x : Continue Count
  // xxxxxxx0	: Counter1 Stop
	T1CR = 0x4E;
	
	// Initial Timer0 = 1,0,0,000,1,1
	// 1xxxxxxx : Enable Timer0
	// x0xxxxxx : Timer Output Disable
	// xx0xxxxx : Timer/Counter Mode
	// xxx000xx : Timer Clock Source = Fx/2
	// xxxxxx1x : Continue Count
	// xxxxxxx1 : Clear & Start Counter0
	T0CR = 0x83;
	                  
  // xxxxxxx1 :	Enable Timer0 Interrupt									
	IE2 |= 0x01;																										// Enable Timer0 Interrupt
}

/*
 * Function    : timer0_isr 
 * Description : timer 0 service interrupt 
 *               occurs on every 1ms. 
 *  If Used Keil Compiler Set Vector = 12(Address Vector:0x0063)
 */
void TIMER0_isr(void) interrupt 12 using 1
{
  if(BuzzerTick++ > BUZZER_DELAY)
	{
	  BuzzerTick = 0;  																							// Reset Buzzer Ticks
		BUZCR ^= 0x01;																								// ON/OFF Buzer
	}
}
