/**********************************************/
/* Example Program For ET-BASE Z51F6412 V1.0  */
/* MCU      : Zilog MCS51 / Z51F6412          */
/*          : Run Internal RC (8.00 MHz)      */
/* Compiler : Keil C51 (V9.50a)               */
/* Editor   : uVision V4.53.0.6               */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)   */
/* Function : Demo Analog(P2.0:ADC0) & UART0  */
/* Update   : 18 August 2012                  */
/**********************************************/
// ET-BASE Z51F6412 Hardware Statdard I/O
// SW1    = P8[0] GPIO Input
// LED    = P8[1] GPIO Output
// BUZZER = P5[0] Buzzer Drive
// UART0  = P0[2]:TX0,P0[3]:RX0
// UART1  = P3[2]:TX1,P3[3]:RX1
//
#include <z51f6412.h>																							// Z8051 : Z51F6412
#include <stdio.h>

/******************************************/
/* ET-BASE Z51F6412 Hardware System Clock */
/******************************************/
#define FREQ_32KHZ    			(32768ul)															// 32.768 KHz
#define FREQ_125KHZ					(125000ul)														// 125 KHz
#define FREQ_1MHZ						(1000000ul)														// 1.00 MHZ
#define FREQ_2MHZ						(2000000ul)														// 2.00 MHz
#define FREQ_4MHZ						(4000000ul)   												// 4.00 MHz
#define FREQ_8MHZ						(8000000ul)   												// 8.00 MHz
#define FREQ_10MHZ					(10000000ul)   												// 10.00 MHz
#define FREQ_12MHZ					(12000000ul)  												// 12.00 MHz
#define FREQ_16MHZ					(16000000ul)  												// 16.00 MHz
#define FREQ_PLL	    			(14745600ul)  												// 14.7456MHz
#define MY_FREQ							FREQ_8MHZ

/*******************************************/
/* ET-BASE Z51F6412 Hardware UART Baudrate */
/*******************************************/
#define BAUD_1200     			(1200ul)															// 1200
#define BAUD_2400     			(2400ul)															// 2400
#define BAUD_4800     			(4800ul)															// 4800
#define BAUD_9600     			(9600ul)															// 9600
#define BAUD_19200    			(19200ul)															// 19200
#define BAUD_38400    			(38400ul)															// 38400
#define BAUD_57600    			(57600ul)															// 57600
#define BAUD_115200   			(115200ul)														// 115200
#define MY_BAUD							BAUD_9600					
	
/************************************/
/* ET-BASE Z51F6412 Hardware SW Pin */
/************************************/
#define SW_PIN							(1 << 0)															// P8[0] = SW
#define SW_PORT_DIR					P8IO																	// Port Direction
#define SW_PORT_DATA				P8																		// Port Data
#define SW_PORT_PULLUP			P8PU																	// Port Pull-Up
#define SW_PORT_DEBOUNCE		P8DB																	// Port Debounce
#define SW_PORT_INIT() 			SW_PORT_DIR      	&= ~(SW_PIN)				// SW = Input 
#define SW_LOGIC_INIT()     SW_PORT_DATA     	|=  (SW_PIN)				// SW Defualt Logic = "1"
#define SW_PULLUP_INIT()		SW_PORT_PULLUP   	|=  (SW_PIN)				// Enable Pull-Up SW Pin
#define SW_DEBOUNCE_INIT()  SW_PORT_DEBOUNCE 	|=  (SW_PIN)				// Enable Debounce SW Pin

/*************************************/
/* ET-BASE Z51F6412 Hardware LED Pin */
/*************************************/
#define LED_PIN							(1 << 1)															// P8[1] = LED
#define LED_PORT_DIR				P8IO
#define LED_PORT_DATA 			P8
#define LED_PORT_INIT()    	LED_PORT_DIR 			|=  (LED_PIN)				// LED = Output
#define LED_OFF()						LED_PORT_DATA			&= ~(LED_PIN) 			// LED Pin = 0
#define LED_ON()  					LED_PORT_DATA			|=  (LED_PIN) 			// LED Pin = 1
#define LED_TOGGLE()  			LED_PORT_DATA			^=  (LED_PIN) 			// LED Pin = Toggle

/*************************************/
/* ET-BASE Z51F6412 Hardware SPI Pin */
/*************************************/
#define SPI0_CS_PIN	  			(1 << 4)															// P3[4] = CS0#
#define SPI0_PORT_DIR				P3IO
#define SPI0_PORT_DATA			P3
#define SPI0_CS_INIT()  		SPI0_PORT_DIR 		|=  (SPI0_CS_PIN)
#define SPI0_CS_LOW()				SPI0_PORT_DATA  	&= ~(SPI0_CS_PIN)
#define SPI0_CS_HIGH()  		SPI0_PORT_DATA  	|=  (SPI0_CS_PIN)

#define SPI1_CS_PIN	  			(1 << 4)															// P4[4] = CS1#
#define SPI1_PORT_DIR				P4IO
#define SPI1_PORT_DATA			P4
#define SPI1_CS_INIT()  		SPI1_PORT_DIR 		|=  (SPI1_CS_PIN)
#define SPI1_CS_LOW()				SPI1_PORT_DATA  	&= ~(SPI1_CS_PIN)
#define SPI1_CS_HIGH()  		SPI1_PORT_DATA  	|=  (SPI1_CS_PIN)

/*******************
 * Global Variable *
 *******************/
char xdata uart_buf[256]; 																				// "sprint" UART[] Buffer

/***********************
 * Function Prototypes *
 ***********************/
void init_adc0(void); 
void init_uart0(unsigned long freq, unsigned long baud);
void my_putchar0(char ch);
void print_uart0(char *uart_buf);
void delay(unsigned long);																				// Delay Time Function(1..4294967295)

/********************
 * Function: main() *
 ********************/
void main(void)
{	  
	unsigned int val;		   																					// ADC Result (HEX)	
	float volt;	
	
	/* Config System Clock = Internal RC 8.00 MHz */
	//PLLCR = 0,0,0,00,00,0
	// 0xxxxxxx : PLL Output Status
	// x0xxxxxx : PLL Output Bypass
	// xx0xxxxx : Power PLL = Default
	// xxx00xxx : FBDiv = Default
	// xxxxx00x : PLL M = Default
	// xxxxxxx0 : PLL Disable
	PLLCR = 0x00;																										// Disable PLL
	
	// SCCR = 0,01,0,0,1,00
	// 0xxxxxxx : Stop Mode = Mode 2
	// x01xxxxx : INTRC Clock Divide = INTRC(16MHz)/2 = 8MHz
	// xxx0xxxx : Clock Change By Hardware
	// xxxx0xxx : RC Oscillator Enable
	// xxxxx1xx : XTAL Oscillator Disable
	// xxxxxx00 : System Clock Source = INTRC(16MHz)
	SCCR = 0x24;  																									// INT-RC 8MHz	
	/* Now System Clock = 8.00MHz */
	
	// Initial LED */ 
	LED_PORT_INIT();					
	LED_OFF();
	
  // Initial UART
	init_uart0(MY_FREQ, MY_BAUD);
	init_adc0();
  
	print_uart0("\n\rET-BASE Z51F6412 V1.0 Run Internal RC at 8MHZ");
	print_uart0("\n\rDemo Test ADC0(P2.0) Polling Read\n\r");
	
	//Main Loop
	while(1)
	{ 
		LED_TOGGLE();
		
    // x1xxxxxx : ADC Start
	  ADCM = 0x40;																									// ADC Start
		
		//Wait Convert Complete
		while(!(ADCM & 0x10));																				// Wait AFLAG=1(Conversion Complete)
		
		// ADCRH = xxxx,11,10,9,8
	  // ADCRL = 7,6,5,4,3,2,1,0
	  val  = ADCRH<<8;    							 														// Read 4 MSB Bits 
	  val |= ADCRL;   												 											// Read 8 LSB Bits         
		val &= 0x0FFF;
	  ADCM = 0x00;																									// ADC Stop
		
		//ADC 12Bit Mode
	  volt = val * (5.0 / 4096.0);	  				 											// Volt = ADC Result x [5.0V / 4096]		
	  sprintf(uart_buf,"\rADC0 : %1.1fV.",volt);	  								// Display Result(0.0-5.0V)	    
		print_uart0(uart_buf);
		delay(5000);
	} 
}

/*
 * Function    : init_adc0()
 * Description : initialize adc0 resources
 */
void init_adc0(void)
{
	// PSR0 = 0,0,0,0,0,0,0,1
	// 0xxxxxxx : Disable ADC7(P2.7=Digital)
	// x0xxxxxx : Disable ADC6(P2.6=Digital)
	// xx0xxxxx : Disable ADC5(P2.5=Digital)
	// xxx0xxxx : Disable ADC4(P2.4=Digital)
	// xxxx0xxx : Disable ADC3(P2.3=Digital)
	// xxxxx0xx : Disable ADC2(P2.2=Digital)
	// xxxxxx0x : Disable ADC1(P2.1=Digital)
	// xxxxxxx1 : Enable  ADC0(P2.0=Analog)
	PSR0 = 0x01;
	
	// ADCM2 = 0,000,1,1,01
  // 0xxxxxxx : Disable External Trigger
  // x000xxxx : Don't Care Trigger Source
	// xxxx1xxx : ADC Clock Source = fx/2
	// xxxxx1xx : LSB Align(ADCRH[3..0],ADCRL[7..0])
	// xxxxxx01 : ADC Convert Clock = fx/4
	ADCM2 = 0x0D;
	
	// ADCM = 0,1,0,0,0000
  // 0xxxxxxx : Enable ADC
  // x1xxxxxx : ADC Start
  // xx0xxxxx : Internal Reference(+VDD)
	// xxx0xxxx : Wait Conversion
	// xxxx0000 : ADC Channel0
	ADCM = 0x40;
}

/*
 * Function    : init_uart0()
 * Description : initialize UART0 resources
 *             : baudrates, parirty, stop bit
 */
void init_uart0(unsigned long freq, unsigned long baud)
{
	// UCTRL01 = 00,00,011,0
  // 00xxxxxx : UART Mode
  // xx00xxxx : No-Parity 
  // xxxx011x : 8 Bit Data
	// xxxxxxx0 : Don't Care in UART Mode
	UCTRL01 = 0x06;

	// UCTRL02 = 0,0,0,0,1,1,1,1
  // 0xxxxxxx : Disable Data Empty Interrupt
  // x0xxxxxx : Disable TX Complete Interrupt
	// xx0xxxxx : Disable RX Complete Interrupt
	// xxx0xxxx : Disable Wakeup Interrupt
	// xxxx1xxx : Enable TX UART
	// xxxxx1xx : Enable RX UART
	// xxxxxx1x : Enable UART Module Function
	// xxxxxxx1 : Enable Double Baudrate
	UCTRL02 = 0x0F;
	
	// UCTRL03 = 0,0,0,0,0,0,0,0
	// 0xxxxxxx : Don't Care in UART Mode
	// x0xxxxxx : Normal Mode
	// xx0xxxxx : Don't Care in UART Mode
	// xxx0xxxx : Don't Care in UARt Mode
	// xxxx0xxx : Reserve Bit
	// xxxxx0xx : 1 Stop Bit
	// xxxxxx0x : Don't Care TX8 in 8 Bit Mode
	// xxxxxxx0 : Don't Care RX8 in 8 Bit Mode	
	UCTRL03 = 0x00;
	
	// Config UBARDx = Double Speed Mode Baudrate 
	// baudrate = (Fx / (8 * (UBARDx + 1)))
	// UBAUDx   = (Fx / (baudrate * 8))-1
	// Example Fx:8MHz,Baudrate:9600
	// UBAUDx   = (8MHz/(9600*8))-1
	//          = (8MHz/76800)-1
	//          = 104.166-1
  //          = 103.166
  //          = 103 = 0x67  
	UBAUD0 = (freq  / ((unsigned long)baud * 8ul)) - 1ul;
	
	// xxxxxxx0 :	Disable UART0 RX Interrupt									
	IE1 &= ~0x01;	
}

/*
 * Function    : my_putchar0
 * Description : Output a character to uart0
 */
void my_putchar0(char ch)
{
  while(!(USTAT0 & 0x80));																				// Wait until Trammit Buffer is Empty
	UDATA0 = ch;
}

/*
 * Function    : print_uart0
 * Description : print a string to uart0
 */
void print_uart0(char *uart_buf)
{
	while(*uart_buf != '\0')
	{	
	  my_putchar0(*uart_buf++);	
	}	
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long i)
{
  while(i > 0) {i--;}																							// Loop Decrease Counter	
  return;
}



