/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example Interface Character LCD */
/*             : 4 Bit Interface on RD0..RD6     */
/*************************************************/
/* Character 16x2 4-Bit Mode Interface           */
/* D4 = RD0						  				 */ 
/* D5 = RD1						   				 */
/* D6 = RD2						   				 */
/* D7 = RD3						   				 */
/* RS = RD4						   				 */
/* RW = RD5						   				 */
/* EN = RD6										 */
/*************************************************/

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register
#include "stdio.h"													// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

// Character LCD Interface Pins 
#define TRIS_DATA_PIN_4     TRISDbits.TRISD0						// Direction D4
#define TRIS_DATA_PIN_5     TRISDbits.TRISD1						// Direction D5
#define TRIS_DATA_PIN_6     TRISDbits.TRISD2						// Direction D6
#define TRIS_DATA_PIN_7     TRISDbits.TRISD3						// Direction D7
#define TRIS_RS             TRISDbits.TRISD4   						// Direction RS
#define TRIS_RW             TRISDbits.TRISD5   						// Direction RW
#define TRIS_E              TRISDbits.TRISD6 						// Direction E

#define DATA_PIN_4          LATDbits.LATD0							// RD0 = D4 LCD
#define DATA_PIN_5          LATDbits.LATD1							// RD1 = D5 LCD
#define DATA_PIN_6          LATDbits.LATD2							// RD2 = D6 LCD
#define DATA_PIN_7          LATDbits.LATD3							// RD3 = D7 LCD
#define RS_PIN              LATDbits.LATD4     						// RD4 = RS LCD
#define RW_PIN              LATDbits.LATD5     						// RD5 = RW LCD
#define E_PIN               LATDbits.LATD6     						// RD6 = E  LCD

#define READ_PIN_BUSY	    PORTDbits.RD3							// Read LCD Busy Pin

/* Display ON/OFF Control */
#define DON                 0x0F  									// Display on      
#define DOFF                0x0B  									// Display off     
#define CURSOR_ON           0x0F  									// Cursor on       
#define CURSOR_OFF          0x0D  									// Cursor off      
#define BLINK_ON            0x0F  									// Cursor Blink    
#define BLINK_OFF           0x0E  									// Cursor No Blink 

/* Cursor or Display Shift */
#define SHIFT_CUR_LEFT      0x13  									// Cursor shifts to the left   
#define SHIFT_CUR_RIGHT     0x17  									// Cursor shifts to the right  
#define SHIFT_DISP_LEFT     0x1B  									// Display shifts to the left  
#define SHIFT_DISP_RIGHT    0x1F  									// Display shifts to the right 

char lcd_buf[16]; 													// "sprint" LCD Buffer

/* Function Prototypes */
void Initial_4bitLCD(void);											// Initial LCD Interface
void SetCursor(unsigned char);										// Set Cursor Address
void WriteCmdLCD(unsigned char);									// Write Command
void WriteDataLCD(unsigned char);									// Write Data
void Delay_tW_LCD(void);											// Enable Pulse Delay	
char Read_BusyLCD(void);											// Read LCD Busy
void print_LCD(void);												// Print String to LCD
void Delay(unsigned long int);										// Delay Time Function

int main(void)
{  
  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */

  Initial_4bitLCD();												// Initial LCD 4 Bit Interface
 
  // Loop Print Message to LCD16 x 2 //
  while(1)	  														// Loop Continue
  {
    SetCursor(0x00);												// Start Cursor Line-1   
    sprintf(lcd_buf,"ET-dsPIC33WEB-V1");
    print_LCD();
    SetCursor(0x40);												// Start Cursor Line-2   
    sprintf(lcd_buf,"Run Speed 40MIPS");
    print_LCD();
    Delay(10000000);       											// Delay Display

    SetCursor(0x00);												// Start Cursor Line-1    
    sprintf(lcd_buf,"24 Bit dsPIC MCU");
    print_LCD();
    SetCursor(0x40);												// Start Cursor Line-2   
    sprintf(lcd_buf,"BY...ETT CO.,LTD");
    print_LCD();
    Delay(10000000);												// Delay Display      
   }  

}	  

/********************/
/* Initial 4Bit LCD */
/********************/
void Initial_4bitLCD(void)
{
  TRIS_DATA_PIN_4 = 0;												// Set Port Direction = Output
  TRIS_DATA_PIN_5 = 0;
  TRIS_DATA_PIN_6 = 0;
  TRIS_DATA_PIN_7 = 0;
  TRIS_RS = 0;
  TRIS_RW = 0;
  TRIS_E = 0;
  DATA_PIN_4 = 1;													// Start 4 Bit Initial 
  DATA_PIN_5 = 1;
  DATA_PIN_6 = 0;
  DATA_PIN_7 = 0;
  RS_PIN = 0;														// Instruction Select
  RW_PIN = 0;														// Write Select
  E_PIN = 0;														// Disable LCD Read/Write
  // Wait 15mS
  Delay(600000);													// Power-ON Delay (minimum of 15ms) 

  DATA_PIN_4 = 1;													// Start 4 Bit Initial 
  DATA_PIN_5 = 1;
  DATA_PIN_6 = 0;
  DATA_PIN_7 = 0;
  RS_PIN = 0;														// Instruction Select
  RW_PIN = 0;														// Write Select
  E_PIN = 1;														// Strobe High Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;
  // Wait 4.1mS
  Delay(164000);													// Wait 4.1 mS 
  DATA_PIN_4 = 1;													// Start 4 Bit Initial 
  DATA_PIN_5 = 1;
  DATA_PIN_6 = 0;
  DATA_PIN_7 = 0;
  RS_PIN = 0;														// Instruction Select
  RW_PIN = 0;														// Write Select
  E_PIN = 1;														// Strobe High Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;
  // Wait 100uS
  Delay(4000);														// Wait 100uS 
  
  WriteCmdLCD(0x32);												// Start Function Set 4Bit LCD Interface
  WriteCmdLCD(0x28);												// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)
  WriteCmdLCD(0x0C);												// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  WriteCmdLCD(0x06);  												// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)	  
  WriteCmdLCD(0x01);    											// Clear Display  (Clear Display,Set DD RAM Address=0) 
}

/**************************/
/* Set Cursor Address LCD */
/**************************/
void SetCursor(unsigned char address)
{  
  RS_PIN = 0;   													// Select Instruction 
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = 1;													// DB7 Must be 1 For Setting DDRam ADDR 
  DATA_PIN_6 = ((address & 0x40)>>6); 
  DATA_PIN_5 = ((address & 0x20)>>5);
  DATA_PIN_4 = ((address & 0x10)>>4);
  E_PIN = 1;														// Strobe High Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;
  Delay_tW_LCD();													// Enable Pulse Delay
  RS_PIN = 0;   													// Select Instruction 
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = ((address & 0x08)>>3);
  DATA_PIN_6 = ((address & 0x04)>>2);
  DATA_PIN_5 = ((address & 0x02)>>1);
  DATA_PIN_4 = ((address & 0x01)>>0);
  E_PIN = 1;    													// Strobe Low Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN =0;

  while(Read_BusyLCD());											// Wait LCD Ready
} 

/************************/
/* Write Command to LCD */
/************************/
void WriteCmdLCD(unsigned char cmd)
{
  RS_PIN = 0;   													// Select Instruction 
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = ((cmd & 0x80)>>7);
  DATA_PIN_6 = ((cmd & 0x40)>>6);
  DATA_PIN_5 = ((cmd & 0x20)>>5);
  DATA_PIN_4 = ((cmd & 0x10)>>4);
  E_PIN = 1;														// Strobe High Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;
  Delay_tW_LCD();													// Enable Pulse Delay
  RS_PIN = 0;   													// Select Instruction 
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = ((cmd & 0x08)>>3);
  DATA_PIN_6 = ((cmd & 0x04)>>2);
  DATA_PIN_5 = ((cmd & 0x02)>>1);
  DATA_PIN_4 = (cmd & 0x01);
  E_PIN = 1;														// Strobe Low Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;

  while(Read_BusyLCD());											// Wait LCD Ready
}  

/*********************/
/* Write Data to LCD */
/*********************/
void WriteDataLCD(unsigned char data)
{
  RS_PIN = 1;  														// Select Data                    
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = ((data & 0x80)>>7);
  DATA_PIN_6 = ((data & 0x40)>>6);
  DATA_PIN_5 = ((data & 0x20)>>5);
  DATA_PIN_4 = ((data & 0x10)>>4);
  E_PIN = 1;														// Strobe High Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;
  Delay_tW_LCD();													// Enable Pulse Delay
  RS_PIN = 1;  														// Select Data   
  RW_PIN = 0;														// Write Select
  DATA_PIN_7 = ((data & 0x08)>>3);
  DATA_PIN_6 = ((data & 0x04)>>2);
  DATA_PIN_5 = ((data & 0x02)>>1);
  DATA_PIN_4 = (data & 0x01);
  E_PIN = 1;														// Strobe Low Nibble
  Delay_tW_LCD();													// Enable Pulse Delay
  E_PIN = 0;

  while(Read_BusyLCD());											// Wait LCD Ready
}  

/*************************/
/* Pulse Delay Write LCD */
/*************************/
void Delay_tW_LCD(void)   											// Enable Pulse Delay
{
  int i;
  for(i=0;i<500;i++);       										// Wait 10uS
}

/******************/
/* Wait LCD Ready */
/******************/
char Read_BusyLCD(void)
{
  TRIS_DATA_PIN_7 = 1;												// Set Port Direction = Input

  RW_PIN = 1;          												// Read Function Select
  RS_PIN = 0;           											// Instruction Select
  Delay_tW_LCD();													// Enable Pulse Delay

  E_PIN = 1;            											// Enable Read
  Delay_tW_LCD();													// Enable Pulse Delay

  if(READ_PIN_BUSY)        											// Verify Busy Flag
  {
    TRIS_DATA_PIN_7 = 0;											// Set Port Direction = Output
    E_PIN = 0;       												// Disable Read
    return 1;
  }
  else
  {
    TRIS_DATA_PIN_7 = 0;											// Set Port Direction = Output
    E_PIN = 0;       												// Disable Read
    return 0;
  }
}

/***********************/
/* Print String to LCD */
/***********************/
void print_LCD(void)
{
  char *p;															// Pointer Buffer
  p = lcd_buf;														// LCD Buffer
 
  do 																// Get char & Print Until null
  {
    WriteDataLCD(*p); 												// Write char to UART1
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void Delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}

