/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example ADC on UART1 = 9600 BPS */
/*************************************************/
/* Used RF2 = RXD1                               */
/* Used RF3 = TXD1                               */
/* Used RB5 = ADC[0] 12 Bit Mode                 */
/*************************************************/

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register
#include "stdio.h"													// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

char uart1_buf[40]; 												// "sprint" UART1 Buffer
unsigned int  adc_buff;												// 1 Channel ADC Result

/* pototype  section */
void putchar1(char ch);  											// Put Char To UART-1
char getchar1(void);  												// Get Char From UART-1
void print_uart1(void);												// Print String to UART1 
void delay(unsigned long int);										// Delay Time Function

int main(void)
{  
  double adc_volt;													// ADC Result in Volt

  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */

  /* Confog UART[1] */	
  U1MODEbits.UARTEN   = 1;											// Enable RXD UART
  U1MODEbits.USIDL    = 0;											// UART Continue in IDLE
  U1MODEbits.IREN     = 0;											// IRDA Disable
  U1MODEbits.RTSMD    = 0;											// RTS = Flow Control		
  U1MODEbits.UEN1     = 0;											// Enable RX,TX Pin
  U1MODEbits.UEN0     = 0;											
  U1MODEbits.WAKE     = 0;											// Disable Wakeup
  U1MODEbits.LPBACK   = 0;											// Disable Loop Back
  U1MODEbits.ABAUD    = 0;											// Disable Auto Baudrate
  U1MODEbits.URXINV   = 0;											// RXD Pin in IDLE = "1"
  U1MODEbits.BRGH     = 0;											// Baudrate = 16 Cycle Clock
  U1MODEbits.PDSEL1   = 0;											// 8 Bit Data,No Parity
  U1MODEbits.PDSEL0   = 0;											
  U1MODEbits.STSEL    = 0;											// 1 Stop Bit
  U1STAbits.UTXISEL1  = 0;											// Interrupt By TX 1-Char
  U1STAbits.UTXISEL0  = 0;										
  U1STAbits.UTXINV	  = 0;											// IRDA Encode in IDLE = 0
  U1STAbits.UTXBRK	  = 0;											// Disable Sync Break
  U1STAbits.UTXEN     = 1;  										// Enable TXD UART 
  U1STAbits.URXISEL1  = 0;											// Interrupt BY RX 1-Char
  U1STAbits.URXISEL0  = 0;										
  U1STAbits.ADDEN	  = 0;											// Disable Address Detect

  // UART[1] Baudrate 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // UxBRG = [Fcy/(16xBaud)]-1
  //       = [ 40MHz / (16 x 9600) ] - 1
  //       = 259 
  U1BRG  = 259;														// UART1 Baudrate 9600/40MHz

  // UART[1] Interrupt Control 
  IEC0bits.U1RXIE     = 0;											// Disable RX Interrupt
  IEC0bits.U1TXIE     = 0;											// Disable TX Interrupt

  /* Start of Config ADC */
  // Config[1]
  AD1CON1bits.ADON = 0;												// Turn-OFF ADC Before Change Config
  AD1CON1bits.ADSIDL = 1;											// ADC Stop in Idle Mode
  AD1CON1bits.ADDMABM = 1;											// DMA Write on Conversion
  AD1CON1bits.AD12B = 1;											// ADC = 12Bit : 1 Channel
  AD1CON1bits.FORM1 = 0;											// ADC Result =Integer Format(0000 dddd dddd dddd)
  AD1CON1bits.FORM0 = 0; 
  AD1CON1bits.SSRC2 = 1;											// Sampling Clock = Auto Convert Continue
  AD1CON1bits.SSRC1 = 1;
  AD1CON1bits.SSRC0 = 1;
  AD1CON1bits.SIMSAM = 0;											// ADC Multiple Channel
  AD1CON1bits.ASAM = 1;												// ADC Sampling Auto Start
  AD1CON1bits.SAMP = 1;												// Enable Sampling

  // Config[2]
  AD1CON2bits.VCFG2 = 0;											// ADC Range = A[VDD]..A[VSS]
  AD1CON2bits.VCFG1 = 0;
  AD1CON2bits.VCFG0 = 0;
  AD1CON2bits.CSCNA = 1;											// Enable Scan Input
  AD1CON2bits.CHPS1 = 0;											// ADC Channel = 0
  AD1CON2bits.CHPS0 = 0;
  AD1CON2bits.SMPI3 = 0;											// Incresae DMA Address Every Sampling
  AD1CON2bits.SMPI2 = 0;
  AD1CON2bits.SMPI1 = 0;
  AD1CON2bits.SMPI0 = 0;
  AD1CON2bits.BUFM = 0;												// Allway Start Buffer From Beginning
  AD1CON2bits.ALTS = 0;												// Allway Channel Select

  // Config[3]
  AD1CON3bits.ADRC = 0;												// ADC Clock = System Clock
  AD1CON3bits.SAMC4 = 1;											// Max Sampling Time
  AD1CON3bits.SAMC3 = 1;
  AD1CON3bits.SAMC2 = 1;
  AD1CON3bits.SAMC1 = 1;
  AD1CON3bits.SAMC0 = 1;
  AD1CON3bits.ADCS5 = 1;											// Conversion Clock = 64 * Tcy
  AD1CON3bits.ADCS4 = 1;
  AD1CON3bits.ADCS3 = 1;
  AD1CON3bits.ADCS2 = 1;
  AD1CON3bits.ADCS1 = 1;
  AD1CON3bits.ADCS0 = 1;

  // Config[4]
  AD1CON4bits.DMABL2 = 0;											// Allocate DMA Buffer = 1-Word / Channel
  AD1CON4bits.DMABL1 = 0;
  AD1CON4bits.DMABL0 = 0;

  // ADC1 Input Channel[1,2,3] Select(Ignore in 12Bit Mode ADC)
  AD1CHS123bits.CH123NB1 = 0;										// Input[-] = Vref[-]
  AD1CHS123bits.CH123NB0 = 0;
  AD1CHS123bits.CH123SB = 1;										// Input[+] = AIN[0..2]
  AD1CHS123bits.CH123NA1 = 0;										// Input[-] = Vref[-]
  AD1CHS123bits.CH123NA0 = 0;
  AD1CHS123bits.CH123SA = 1;										// Input[+] = AIN[0..2]

  // ADC1 Input Channel[0] Select
  // For Sample-B
  AD1CHS0bits.CH0NB = 0;											// Input[-] = Vref[-]
  AD1CHS0bits.CH0SB4 = 0;											// CH0[+] = AIN[5]
  AD1CHS0bits.CH0SB3 = 0;
  AD1CHS0bits.CH0SB2 = 1;
  AD1CHS0bits.CH0SB1 = 0;
  AD1CHS0bits.CH0SB0 = 1;
  //For Sample-A
  AD1CHS0bits.CH0NA = 0;											// Input[-] = Vref[-]
  AD1CHS0bits.CH0SA4 = 0;											// CH0[+] = AIN[5]
  AD1CHS0bits.CH0SA3 = 0;
  AD1CHS0bits.CH0SA2 = 1;
  AD1CHS0bits.CH0SA1 = 0;
  AD1CHS0bits.CH0SA0 = 1;
 
  // ADC1 Input Scan Select AN[31..16] (0=Disable,1=Enable)
  AD1CSSH = 0x0000;													// Disable AN[31..16]

  // ADC1 Input Scan Select AN[15..00] (0=Disable,1=Enable)
  AD1CSSL = 0x0000;													// Disable AN[15..00]
  AD1CSSLbits.CSS5 = 1;												// Enable Scan AN5 Only

  // ADC1 Port Config AN[31..16] Pin Config (0=Analog,1=Digital)
  AD1PCFGH = 0xFFFF;												// AN[31..16] = Digital Mode

  // ADC1 Port Config AN[15..00] Pin Config (0=Analog,1=Digital)
  AD1PCFGL = 0xFFFF;												// AN[15..00] = Digital Mode
  AD1PCFGLbits.PCFG5 = 0;											// AN5 = Analog Only

  // ADC1 Interrupt Control
  IEC0bits.AD1IE = 0;												// Disable ADC interrupt 
  AD1CON1bits.ADON = 1;												// Turn-ON ADC Function
  /* End of Config ADC */

  // Wait Power-On Delay //
  delay(1000000);

  // UART[1] Print String //
  sprintf(uart1_buf,"ET-dsPIC33WEB-V1.0 : Demo ADC on UART1\n\r"); 	// Print Message String 
  print_uart1();
  sprintf(uart1_buf,"Demo ADC[0] 12Bit Mode on AN5(RB5) Pin\n\r");	// Print Message String 
  print_uart1(); 

  // Strat Read ADC Continue //
  while(1)															// Loop Continue
  {
    AD1CON1bits.SAMP = 0; 											// Clear SAMP Bit to Start Conversion 
    while(!AD1CON1bits.DONE);										// Wait ADC Sampling Complete
	adc_buff = ADCBUF0;												// Read ADC Result						

    // Display ADC[0] Result on UART1
    sprintf(uart1_buf,"\rADC[0] = "); 								// Print Message String 
    print_uart1();
    
    // Display Result ADC0    	

    //sprintf(uart1_buf,"%4d \r ",adc_buff);						// 4 Digit Decimal[0..4096] Display

    adc_volt = adc_buff * (3.3 / 4096.0);	  						// Volt = ADC Result x [3.3V / 2048]		
    sprintf(uart1_buf,"%1.1f Volt.\r ",adc_volt);					// 4 Digit Decimal[0..3.3V] Display
    print_uart1();
  } 
}	  

/****************************/
/* Write Character To UART1 */
/****************************/
void putchar1 (char c)  
{
  if (c == '\n')  													// If Line Feed(LF)
  {
    U1TXREG = 0x0D;                        							// Auto Add CR(LF+CR)
    while (!U1STAbits.TRMT);										// Wait Send Complete
    U1STAbits.TRMT = 0;												// Reset Status
  }
  U1TXREG = c;														// Send This Char
  while (!U1STAbits.TRMT);											// Wait Send Complete
  U1STAbits.TRMT = 0;												// Reset Status
}

/****************************/
/* Get character From UART1 */
/****************************/
char getchar1 ()  
{
  char c;
  while (!U1STAbits.URXDA);											// Wait UART1 Ready
  c = U1RXREG;														// Read UART1 Data
  U1STAbits.URXDA = 0;												// Reset Status
  return (c);
}

/*************************/
/* Print String to UART1 */
/*************************/
void print_uart1(void)
{
  char *p;															// Pointer Buffer
  p = uart1_buf;													// UART1 Buffer
 
  do 																// Get char & Print Until null
  {
    putchar1(*p); 													// Write char to UART1
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}




