;********************************************************************
;
; Author        : ADI - Apps            www.analog.com/MicroConverter
;
; Date          : Oct 2003
;
; File          : WDtimer.asm
;
; Hardware      : ADuC847
;
; Description   : Demonstrates use of the on-chip watchdog timer.
;                In normal operation, WD timer is refreshed by code.
;                In a runaway code condition (which can be simulated 
;                here by pressing the INT0 button on the eval board)
;                code fails to refresh WD bits, the LED stops toggling
;                before the  WD timer generates a
;                reset after a user selected time-out period (from
;                15.6ms to 2000ms) has elapsed. The time-out period
;                in this routine is 2.0s for visual clarity.
;        
;                After a reset the light blinks at a slower rate.
;                Future watchdog resets can be enabled in the same 
;                way. A hard reset will clear the WDS bit and the 
;                quicker flashes will occur again. 
;        
;                
;
;********************************************************************

$MOD847                         ; Use 8052 predefined symbols

LED     EQU     P3.4            ; P3.4 drives red LED on eval board
ERROR   EQU     F0              ; the 'ERROR' flag is used here to
                                ; simulate an erroneous command that
                                ; sends code into an unknown state

;____________________________________________________________________
                                                  ; BEGINNING OF CODE
CSEG

ORG 0000h

        JMP     MAIN            ; jump to main program
;____________________________________________________________________
                                    ; EXTERNAL INTERRUPT VECTOR SPACE
ORG 0003h ;  (INT0 ISR)

        SETB    ERROR           ; simulate an error condition..
                                ; ..when INT0 button is pressed
        RETI


;____________________________________________________________________
                                                       ; MAIN PROGRAM
ORG 0060h                       ; Start at address above interrupts

MAIN:
        JB      WDS, WDRESET 
        MOV     R0, #01h        ; this will blink the LED at 12Hz
        JMP     START

WDRESET: MOV    R0, #06h        ; this will blink the LED at 2Hz

; Enable external interupt to trigger simulated error condition...

START:  SETB    IT0             ; make INT0 edge triggered
        SETB    EX0             ; enable INT0 (button on eval board)
        SETB    EA

; Configure the Watchdog timer. It should be configured like this,
; with the global interrupts turned off and setting WDWR to allow
; writing to WDCON. 
        CLR     EA
        SETB    WDWR
        MOV     WDCON, #72h     ; Enable Watchdog timer to cause
                                ; -2.0 second timeout period
      			        ; -enable WDIR bit to generate
      			        ;  a reset and not an interrupt
	SETB    EA              ; set global interrupts again

;  from this point forward, watchdog bits must be refreshed every
;  2.0 seconds or less.  if they are not, watchdog timer will
;  generate a reset.

        CLR     ERROR           ; simulate error free operation

; The below loop represents normal code execution...

FLASH:  MOV     A, R0
        CALL    DELAY           ; delay  x R0
	CPL     LED             ; blink (complement) the red LED

        CLR     EA              ; refresh watchdog timer 
	SETB    WDWR
	SETB    WDE 	 
	SETB    EA      

        JNB     ERROR, FLASH    ; jump if 'ERROR' flag is not set

; The below endless loop represents run-away code execution...

        CLR     LED             ; turn LED off during runaway code 
        JMP     $               ; this endless loop is used to
                                ; represent an unknown state of
                                ; program execution

;  program will sit in the above endless loop until the watchdog
;  period (2000ms) has elapsed, at which time a reset will be
;  generated by the watchdog timer, thereby recovering the chip to
;  resume normal code execution.


;____________________________________________________________________
                                        
DELAY:					

		MOV	R2,A		; Acc holds delay variable
 DLY0:		MOV	R3,#0FFh	; Set up delay loop0
 DLY1:		MOV	R4,#0FFh	; Set up delay loop1
		DJNZ	R4,$		; Dec R4 & Jump here until R2 is 0
		DJNZ	R3,DLY1         ; Dec R3 & Jump DLY1 until R1 is 0
		DJNZ	R2,DLY0		; Dec R2 & Jump DLY0 until R0 is 0
		RET			; Return from subroutine

;____________________________________________________________________

END
