


/***********************************************************************************************
 *                                                                                             *
 *            Example_4. Control ISD2548 Record & Play & Skip (Push Button Mode)               *
 *                                                                                             *
 ***********************************************************************************************
 * Target MCU        : PIC18F8722(ET-BASE PIC8722)                                             *
 *                   : X-TAL : 10 MHz                                                          *
 *                   : CPU Clock = 40 MHz (x4)                                                 *                                                                  *
 * Editor-Compiler   : CCS Version 4.120                                                       *
 *                                                                                             *
 * Port Interface    :                                                                         *
 *                                                                                             *
 *            SPI    :     # MCU-PIC18F8722 #           ET-Mini ISD2548                        *
 *                         -------------------          ---------------                        *
 *                                                                                             *                                   
 *                          RD0            connect        DIN                                  *                
 *                          RD1            connect        SCK                                  *
 *                          RD2            connect        STB                                  *               
 *                          RD3(IN)        connect        EOM (Must Pull Up)                   *               
 *                          GND            connect        ENA                                  *             
 *                          NO USE         connect        OVF                                  *            
 *                                                                                             *
 *                                                                                             *
 *                                    5V                                                       *
 *                                  --+--                                                      *
 *                                    |                                                        *
 *                                    \                                                        *
 *                                    / 10K                                                    *
 *                                    \          ET-Mini ISD2548       5V                      *
 *                                    /      +-------------------+   --+--                     *
 *                                    |      |                   |     |                       *
 *                           RD0 -----|----->|DIN            +5V +-----+      /|               *
 *                           RD1 -----|----->|SCK                |          /  |               *
 *                           RD2 -----|----->|STR            SP+ +--------||   |               *
 *                           RD3 <----+------|EOM                |        || SP|               *
 *                                           |               SP- +--------||   |               *
 *                                           |                   |          \  |               *
 *                               +-----------|ENA            GND +-----+      \|               *
 *                             __|__         |                   |   __|__                     *
 *                              ---          |                   |    ---                      *
 *                               -        ---|VOF                |     -                       *
|*                                           |                   |       _ --                  *
 *                                           |               MIC+|------| |                    *
 *                                           |                  -|------|_|                    *
 *                                           +-------------------+         --                  *
  *                                                    | |     Set Jp1 : to GND                *
 *                                                     | |         Jp2 : to MIC or ANA IN      *
 *                                                     | |                                     *
 *                                                     \ /                                     *
 *  Dtermine Address ISD2548 For Push Button Mode:                                             *
 *                                                 A0(M0)   =  X (Skip) : Data Bit0            *
 *                                                 A5..A1   =  Logic 0  : Data Bit5..1         *
 *                                                 A8..A6   =  Logic 1  : Data Bit8..6         *
 *                                                CE,PD,P/R =  X        : Data Bit11..9        *
 *                                                                                             *
 *                                       When X = Logic 0 or Logic 1  Follow Control           *
 *                                                                                             *
 *                                                                                             *
 *      SWITCH    :   # MCU-PIC18F8722 #            # SW. for User COntrol Play& Record#       *
 *                    ------------------             -----------------------------------       * 
 *                       RJ4(IN)          connect         SW. Record Message                   *
 *                       RJ5(IN)          connect         SW. Play Message All                 *
 *                       RJ6(IN)          connect         SW. Skip Message                     *
 *                       RJ7(IN)          connect         SW. Reset                            *
 *                                                                                             *
 *                       5V                                                                    *
 *                      --+--                                                                  *
 *                        |                                                                    *
 *                        +-------+-------+-------+                                            *  
 *                        |       |       |       |                                            *
 *                        /       /       /       /                                            *
 *                        \       \       \       \                                            *
 *                        /10K    /10K    /10K    / 10K                                        *
 *                        \       \       \       \                                            *
 *              (Record)  |       |       |       |                                            *
 *         RJ4 <----------+       |       |       |                                            *
 *              (Play)    |       |       |       |                                            *
 *         RJ5 <------------------+       |       |                                            *
 *              (Skip-MSG)|       |       |       |                                            *
 *         RJ6 <--------------------------+       |                                            *
 *              (Reset)   |       |       |       |                                            *
 *         RJ7 <----------------------------------+                                            *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                   SW.4\   SW.5\   SW.6\  SW.7 \                                             *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                      __|__   __ __   __|__   __|__                                          *
 *                       -_-     -_-     -_-     -_-                                           *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 * Last Update     : 17/Junly/2012                                                             *  
 * Create By       : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                      *
 *                                                                                             *
 ***********************************************************************************************/

#include <18F8722.h>
#include <stdio.h>

#fuses H4,NOLVP,NOWDT,NOPROTECT,NOSTVREN         //Setup MCU
#use delay (clock = 40000000)                    //delay Clock = (ms)

#use fast_io(ALL)                                //Stop Set direction Auto for All Port  by Compiler  



 /****************************************************
  **             Function Initial Port              **
  ****************************************************/

 void Init_Port(void)
  {
     set_tris_D(0x08)   ;    //Set RD0..RD2 = Output For SPI Control ISD2548,RD3=Input for EOM
     set_tris_J(0xFF)   ;    //Set RJ0..R7  = Input  For Switch SW.2-SW.7 

  }



/***************************************************************                                                          
 **                        Function Switch                    **
 ***************************************************************/

//-------------- Check Release SW.4-SW.7 -----------

void Check_Sw()
 {
   while((INPUT_J() & 0xF0)!= 0xF0) {;}  //Check Release Sw.4-Sw.7

   delay_ms(50)        ;
 }

//---------------- Read SW.4-SW.7 (PE4-PE7)-----------------

int8 Rd_Sw()
 {
  int8 sw                  ;

   sw = INPUT_J()          ;  //Read RJ4-RJ7
   sw = ~sw                ;  //Invert data Read
   sw = sw & 0xF0          ;  //Mark bit RJ4..RJ7 (SW4-SW7)
   return sw               ;  //Return Value Press Switch


 }

/***************************************************************
 **                Function SPI Seny data 12 bit              **
 ***************************************************************
 **          RD0 = SDO                                        **
 **          RD1 = SCK                                        **
 **          RD2 = STROB                                      **   
 **                                                           **
 ***************************************************************/

//--------------- Define Pin -----------------------


//--------------- SDO-High(Data-Out) ---------------

#define     Sdo_Hi()          OUTPUT_HIGH(PIN_D0)    ;          //RD0= SDO:1
                          
//--------------- SDO-Low(Data-OUT) ----------------

#define     Sdo_Lo()          OUTPUT_LOW(PIN_D0)     ;         //RD0= SDO:0


//--------------- Sck-High(Clock) ---------------

#define     Sck_Hi()          OUTPUT_HIGH(PIN_D1)   ;          //RD1= SCK:1
                          
//--------------- Sck-Low(Clock) ----------------

#define     Sck_Lo()         OUTPUT_LOW(PIN_D1)     ;          //RD1= SCK:0


//--------------- STR-High(Strob) ---------------

#define     Str_Hi()          OUTPUT_HIGH(PIN_D2)  ;          //RD2= Str:1
                          
//--------------- STR-Low(Strob) ----------------

#define     Str_Lo()          OUTPUT_LOW(PIN_D2)   ;          //RD2= Str:0



//----------------- Write Data to ISD2548 12-bit -------------------

void Spi_Wr(unsigned int16 DWord)              
 {
    unsigned int16 Bit ;                   

     Str_Lo()                              ;    //Start Strob 595 to Low

     for(Bit=0;Bit<12;Bit++)                    //Loop write data 12 Bit
      {      
         Sck_Lo()                          ;    //Start 595 Clock  Low

      //--------- write Data(RD0) ---------

        if((DWord & 0x800)== 0x800)
         {
          Sdo_Hi()                       ;    //Set bit data(RD0) = 1
         }
        else 
         {
          Sdo_Lo()                       ;    //Clear bit data(RD0) = 0
         } 
      
        Sck_Hi()                         ;    //Set 595 Clock High for Shift data bit

        DWord <<= 1                      ;    //Shift Next Bit Data        
   
     }

     Str_Hi()                            ;   //Set Strob 595 to High for Out data parallel 


  }



/***************************************************************
 **                    FUNCTION ISD2548                       **
 ***************************************************************
 **  - Memory = 320 Address                                   **
 **  - Time Record Max = 48 secound                           **
 **                                                           **
 ***************************************************************/


//-------- Define Data Control ISD2548 ------------

#define Button_Mode       0x1C0      // Set Button Mode   A5..A1(Bit5..1)=0;  A8..A6(bit8..bit6)=1 

#define ISD_Reset         0xE00      // Set Reset ISD2548 CE,PD,P/R(bit11..9) = 1  ; A0/MO(bit0) = 0   
#define ISD_Boot_CLo      0x200      // Boot ISD2548  CE=0,PD=0,P/R=1
#define ISD_Boot_CHi      0xA00      // Boot ISD2548  CE=1;PD=0,P/R=1

#define Skip_On_CLo       0x201      // Skip CE=0,PD=0,P/R=1 ;A0/M0=1 
#define Skip_On_CHi       0xA01      // Skip CE=1,PD=0,P/R=1 ;A0/M0=1 
#define Skip_Off          0xA00      // Skip CE=1,PD=0,P/R=1 ;A0/M0=0 

#define Play_CLo          0x200      // Play : CE=0,PD=0,P/R=1
#define Play_CHi          0xA00      // Play : CE=1,PD=0,P/R=1

#define Rec_CLo           0x000      // Record : CE=0,PD=0,P/R=0
#define Rec_CHi           0x800      // Record : CE=1,PD=0,P/R=0




//----------------- Initial ISD2548 ---------------

 void Init_ISD2548()
  {
    
    Spi_Wr(Button_Mode|ISD_Reset)      ;  //Set M0 = 0;P/R=1;PD=1;CE=1
   
    //-Initial boot ISD2548-

    delay_ms(100)                     ;

    Spi_Wr(Button_Mode|ISD_Boot_CHi)  ; //Set M0 = 0;P/R=1;PD=0;CE=1
    
  }


//---------------- Read Status EOM : End OF Message ---------------- 

int8 Rd_Eom()
 {
  int8 eom                 ;
       
  eom  = INPUT(PIN_D3)     ; //Read RD3 : 0 = end of message , Nomal = 1   
  return eom               ;  
 }



//------------------- Check Status EOM -----------------
void Chk_Eom()
 {
  int8 em ;

  em = Rd_Eom();  

  while(em)                // Check Signal EOM to 0 : EOF = 0 :Exit Loop(for End message) 
  {
   em = Rd_Eom();
   
   if((Rd_Sw()) == 0x80)   //Check Press  SW Reset (RJ7)
    {
     em = 0 ;
    Init_ISD2548() ;      //Reset ISD2548 for Start Play & Record at Address 0x000
    }
  }   
 }

//------------ Skip Mesaage  ------------

void Skip_Msg(void)
 { 
      
   Spi_Wr(Button_Mode|Skip_On_CLo)   ; 
   Spi_Wr(Button_Mode|Skip_On_CHi)   ;
   Spi_Wr(Button_Mode|Skip_Off)   ;  

 }

//------------ Play 1 Message -----------

void Play_Msg()
 {

   Spi_Wr(Button_Mode|Play_CLo)   ;
   Spi_Wr(Button_Mode|Play_CHi)   ;
    
 }


//---------------- Record 1 Message -------------------

void Record()
 {
  
   Spi_Wr(Button_Mode|Rec_CLo)   ;
   Spi_Wr(Button_Mode|Rec_CHi)   ;
    
 }
 
//----------------Stop Record Message -------------------

void Stop_Rec()
 {   
    Spi_Wr(Button_Mode|Rec_CLo)   ;
    Spi_Wr(Button_Mode|Rec_CHi)   ;

 }

/**********************************************************************
 **                                                                  **
 **                         main Program                             **
 **                                                                  **
 **********************************************************************/
 
void main() 
{
   unsigned int8 sw  ;
 
   Init_Port()       ;
   Init_ISD2548()    ; 

 //--------------  Start  --------------

  while(TRUE)
   { 
     sw = Rd_Sw() ;   //Read Switch

     switch(sw)
      {                   
        case 0x10 :  //SW.4(RJ4) = Record Message
                   
                    Record()               ;  //Start Rec. 
                               
                    Check_Sw()             ;  //Check Release Switch Rec.

                    Stop_Rec()             ;  //Stop Message Record 

                    Chk_Eom()              ;  //Check EOM=0  Ready for  Next Message Record                    
            break ;

        case 0x20 :  //SW.5(RJ5) = Play Message
                
                     Play_Msg()            ;   //Play 1 Message

                     Check_Sw()            ;   //Check Release Switch
                    
                     Chk_Eom()             ;   //Check EOM=0  Ready for  Next Play Message           
             
             break ;

        case 0x40 : //SW.6(RJ6) = Skip Mesage
                               
                     Skip_Msg()            ;  //Skip 1 Message
            
                     Check_Sw()            ;  //Check Release Switch
             break ;


        case 0x80: //SW.7(RJ7) = Reset Address Pointer ISD2548 to Address 0x000
                               
                    Init_ISD2548()        ;  //Reset ISD2548 for Start Play & Record at Address 0x000
            
                    Check_Sw()            ;  //Check Release Switch            
              

             break ;

    }   //switch 
   }    //while
}      //End


