

 /**********************************************************************************************
  *                                                                                            *
  *             Example_4. Control ISD2548 Record & Play & Skip (Push Button Mode)             *
  *                                                                                            *
  **********************************************************************************************
  *						                                                                       * 
  *  MCU      : AT89C51RE2(Use CP-JR51RE2 V1.0 Board)                                          *
  *  Compiler : Keil C51 (V8.05 a)                                                             *
  *												                                               *
  *  PORT Interface   :										                                   *
  *              																			   *
  *             SPI   :  # MCU-51RE2 #  		   	         #ET-MINI ISD2548#                 *
  *                      -------------                        ----------------                 *                             
  *                       P1.0(Out)         connect            DIN                             *                
  *                       P1.1(Out)         connect            SCK                             *
  *                       P1.2(Out)         connect            STB                             *               
  *                       P1.3(IN)          connect            EOM (Must Pull Up)              *               
  *                       GND               connect            ENA                             *             
  *                       NO USE            connect            OVF                             *            
  *                                                                                            *
 *                                                                                             *
 *                                                                                             *
 *                                    5V                                                       *
 *                                  --+--                                                      *
 *                                    |                                                        *
 *                                    \                                                        *
 *                                    / 10K                                                    *
 *                                    \          ET-Mini ISD2548       5V                      *
 *                                    /      +-------------------+   --+--                     *
 *                                    |      |                   |     |                       *
 *                           P1.0 ----|----->|DIN            +5V +-----+      /|               *
 *                           P1.1 ----|----->|SCK                |          /  |               *
 *                           P1.2 ----|----->|STR            SP+ +--------||   |               *
 *                           P1.3 <---+------|EOM                |        || SP|               *
 *                                           |               SP- +--------||   |               *
 *                                           |                   |          \  |               *
 *                               +-----------|ENA            GND +-----+      \|               *
 *                             __|__         |                   |   __|__                     *
 *                              ---          |                   |    ---                      *
 *                               -        ---|VOF                |     -                       *
|*                                           |                   |       _ --                  *
 *                                           |               MIC+|------| |                    *
 *                                           |                  -|------|_|                    *
 *                                           +-------------------+         --                  *
  *                                                    | |     Set Jp1 : to GND                *
 *                                                     | |         Jp2 : to MIC or ANA IN      *
 *                                                     | |                                     *
 *                                                     \ /                                     *
 *  Dtermine Address ISD2548 For Push Button Mode:                                             *
 *                                                 A0(M0)   =  X (Skip) : Data Bit0            *
 *                                                 A5..A1   =  Logic 0  : Data Bit5..1         *
 *                                                 A8..A6   =  Logic 1  : Data Bit8..6         *
 *                                                CE,PD,P/R =  X        : Data Bit11..9        *
 *                                                                                             *
 *                                       When X = Logic 0 or Logic 1  Follow Control           *
 *                                                                                             *
 *                                                                                             *
 *      SWITCH    :   # MCU-51RE2 #                 # SW. for User COntrol Play& Record#       *
 *                    -------------                  -----------------------------------       * 
 *                       P3.4(IN)          connect         SW. Record Message                  *
 *                       P3.5(IN)          connect         SW. Play Message All                *
 *                       P3.6(IN)          connect         SW. Skip Message                    *
 *                       P3.7(IN)          connect         SW. Reset                           *
 *                                                                                             *
 *                       5V                                                                    *
 *                      --+--                                                                  *
 *                        |                                                                    *
 *                        +-------+-------+-------+                                            *  
 *                        |       |       |       |                                            *
 *                        /       /       /       /                                            *
 *                        \       \       \       \                                            *
 *                        /10K    /10K    /10K    / 10K                                        *
 *                        \       \       \       \                                            *
 *              (Record)  |       |       |       |                                            *
 *         P3.4 <---------+       |       |       |                                            *
 *              (Play)    |       |       |       |                                            *
 *         P3.5 <-----------------+       |       |                                            *
 *              (Skip-MSG)|       |       |       |                                            *
 *         P3.6 <-------------------------+       |                                            *
 *              (Reset)   |       |       |       |                                            *
 *         P3.7 <---------------------------------+                                            *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                   SW.4\   SW.5\   SW.6\  SW.7 \                                             *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                      __|__   __ __   __|__   __|__                                          *
 *                       -_-     -_-     -_-     -_-                                           *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 * Last Update     : 17/Junly/2012                                                             *  
 * Create By       : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                      *
 *                                                                                             *
 ***********************************************************************************************/
  
  						
#include <at89c51re2.h>
#include <stdio.h>


/********************************************
 **             Function Delay             **
 ********************************************/

 //--------- delay ----------
void delay(int count)          
 {
    int i,j;
    for(i=0;i<=count;i++)
    for(j=0;j<=count;j++)	 ;
 }

 

/*************************************************
 **				Function Switch                 **
 *************************************************/

//-------------- Check press SW.4-SW.7 -----------

void Check_Sw()
 {
    P3 = 0xFF         ;
	while((P3&0xF0) != 0xF0){;}	  //Check SW. P3[4..7] Release All 
	delay(50)         ;

 }

//---------------- Read SW.4-SW.7 -----------------

unsigned char Rd_Sw()
 {
    unsigned char sw         ;

    P3 = 0xFF                ;
	sw = P3                  ;	 //Read press SW. P3[4..7]
	sw = (~sw)& 0xF0         ;	 //Mark P3[4..7] and invert data (Normal Value Not press SW. is logic 1)

	return sw                ;

 }


/***************************************************************
 **                Function SPI Seny data 12 bit              **
 ***************************************************************
 **          P1.0 = SDO                                       **
 **          P1.1 = SCK                                       **
 **          P1.2 = STROB                                     **   
 **          Sent First bit High                              **
 ***************************************************************/

//--------------- Define Pin -----------------------


//--------------- SDO-High(Data-Out) ---------------

#define     Sdo_Hi()      P1_0 = 1            //P1.0= SDO:1

                          
//--------------- SDO-Low(Data-OUT) ----------------

#define     Sdo_Lo()      P1_0 = 0            //P1.0= SDO:0



//--------------- Sck-High(Clock) ---------------

#define     Sck_Hi()      P1_1 = 1            //P1.1= SCK:1
                          
//--------------- Sck-Low(Clock) ----------------

#define     Sck_Lo()      P1_1 = 0           //P1.1= SCK:0


//--------------- STR-High(Strob) ---------------

#define     Str_Hi()      P1_2 = 1            //P1.2= Str:1
                          
//--------------- STR-Low(Strob) ----------------

#define     Str_Lo()      P1_2 = 0            //P1.2= Str:0



//----------------- Write Data to ISD2548 12-bit -------------------

void Spi_Wr(unsigned int DWord)			  	
 {
    unsigned int Bit ;				 	   

     Str_Lo()                             ;  //Start Strob 595 to Low

     for(Bit=0;Bit<12;Bit++)                 //Loop write data 12 Bit
      {		
	   	Sck_Lo()                          ;	 //Start 595 Clock  Low

		//--------- write Data(P1.0) ---------

		if((DWord & 0x800)== 0x800)
		    Sdo_Hi()                     ;   //Set bit data(P1.0) = 1
		else 
		    Sdo_Lo()                     ;	 //Clear bit data(P1.0) = 0
		   	 
		
		Sck_Hi()                         ;   //Set 595 Clock High for Shift data bit

		DWord <<= 1                      ;	 //Shift Next Bit Data        
	
	  }

    Str_Hi()                             ;   //Set Strob 595 to High for Out data parallel 


  }

/************************************************************
 **                    FUNCTION ISD2548	                   **
 **                                                        **
 **  -Memory = 320 Address								   ** 
 **  -Time Record Max = 48 secound					       **
 **                                                        **
 ************************************************************/


//-------- Define Data Control ISD2548 ------------

#define Button_Mode       0x1C0      // Set Button Mode   A5..A1(Bit5..1)=0;  A8..A6(bit8..bit6)=1 

#define ISD_Reset         0xE00      // Set Reset ISD2548 CE,PD,P/R(bit11..9) = 1  ; A0/MO(bit0) = 0   
#define ISD_Boot_CLo      0x200      // Boot ISD2548  CE=0,PD=0,P/R=1
#define ISD_Boot_CHi      0xA00      // Boot ISD2548  CE=1;PD=0,P/R=1

#define Skip_On_CLo       0x201      // Skip CE=0,PD=0,P/R=1 ;A0/M0=1 
#define Skip_On_CHi       0xA01      // Skip CE=1,PD=0,P/R=1 ;A0/M0=1 
#define Skip_Off          0xA00      // Skip CE=1,PD=0,P/R=1 ;A0/M0=0 

#define Play_CLo          0x200      // Play : CE=0,PD=0,P/R=1
#define Play_CHi          0xA00      // Play : CE=1,PD=0,P/R=1

#define Rec_CLo           0x000      // Record : CE=0,PD=0,P/R=0
#define Rec_CHi           0x800      // Record : CE=1,PD=0,P/R=0





//----------------- Initial ISD2548 ---------------

 void Init_ISD2548()
  {
    
    Spi_Wr(Button_Mode|ISD_Reset)      ;  //Set M0 = 0;P/R=1;PD=1;CE=1
   
    //-Initial boot ISD2548-

    delay(100)                        ;

    Spi_Wr(Button_Mode|ISD_Boot_CHi)  ; //Set M0 = 0;P/R=1;PD=0;CE=1
    
  }


//---------------- Read Status EOM : End OF Message ---------------- 
unsigned char Rd_Eom()
{
  unsigned char eom        ;
       
  P1_3 = 1        ; //Set Port P1.3 = input
  eom  = P1_3     ; //Read P1.3 : 0 = end of message , Nomal = 1   
  return eom      ;  
}



//------------------- Check Status EOM -----------------

void Chk_Eom()
 {
  unsigned char em ;

  em = Rd_Eom();  

  while(em)                // Check Signal EOM to 0 : EOF = 0 :Exit Loop(for End message) 
  {
   em = Rd_Eom();
	  
   if((Rd_Sw()) == 0x80)   //Check Press  SW Reset (P3.7)
    {	
     em = 0 ;
    Init_ISD2548() ;       //Reset ISD2548 for Start Play & Record at Address 0x000
    }
  }   
 }


//------------ Skip Mesaage  ------------

void Skip_Msg(void)
 {     
   Spi_Wr(Button_Mode|Skip_On_CLo)   ; 
   Spi_Wr(Button_Mode|Skip_On_CHi)   ;
   Spi_Wr(Button_Mode|Skip_Off)      ;  

 }

//------------ Play 1 Message -----------

void Play_Msg()
 {

   Spi_Wr(Button_Mode|Play_CLo)   ;
   Spi_Wr(Button_Mode|Play_CHi)   ;
    
 }


//---------------- Record 1 Message -------------------

void Record()
 {
  
   Spi_Wr(Button_Mode|Rec_CLo)   ;
   Spi_Wr(Button_Mode|Rec_CHi)   ;
    
 }
 
//----------------Stop Record Message -------------------

void Stop_Rec()
 {   
    Spi_Wr(Button_Mode|Rec_CLo)   ;
    Spi_Wr(Button_Mode|Rec_CHi)   ;

 }


/************************************************************************************
 **                                                                                **
 ** ###################### ------------ MAIN Program ----------- ################# **
 **                                                                                **
 ************************************************************************************/

main()
 {
    unsigned char sw  ;

 //CKCON0   = 0xFE                    ;   //Set MCU 12 Clock Mode
   BMSEL    = 0x00                    ;   //Select Bank 0+Command Bank for Area Flash
 
   CKCON0 = 0x01                      ;   //div 6 clk mode  X2=1
   AUXR   = 0x10                      ;   //Set XRAM = 1792 Byte

   Init_ISD2548()     ; 

 //--------------  Start  --------------

  while(1)
   { 
     sw = Rd_Sw()     ;                      //Read Switch

     switch(sw)
      {                   
        case 0x10 :  //SW.4(P3.4) = Record Message
                   
                    Record()               ;  //Start Rec. 
                               
                    Check_Sw()             ;  //Check Release Switch Rec.

                    Stop_Rec()             ;  //Stop Message Record 

                    Chk_Eom()              ;  //Check EOM=0  Ready for  Next Message Record                    
            break ;

        case 0x20 :  //SW.5(P3.5) = Play Message
                
                     Play_Msg()            ;   //Play 1 Message

                     Check_Sw()            ;   //Check Release Switch
                    
                     Chk_Eom()             ;   //Check EOM=0  Ready for  Next Play Message           
             
             break ;

        case 0x40 : //SW.6(P3.6) = Skip Mesage
                               
                     Skip_Msg()            ;  //Skip 1 Message
            
                     Check_Sw()            ;  //Check Release Switch
             break ;


        case 0x80: //SW.7(P3.7) = Reset Address Pointer ISD2548 to Address 0x000
                               
                    Init_ISD2548()        ;  //Reset ISD2548 for Start Play & Record at Address 0x000
            
                    Check_Sw()            ;  //Check Release Switch                         

             break ;

    }   //switch 
   }    //while
}      //End


