

/***********************************************************************************************
 *                                                                                             *
 *             Example_4. Control ISD2548 Record & Play & Skip (Push Button Mode)              *
 *                                                                                             *
 ***********************************************************************************************
 *                                                                                             *
 * Target MCU     : ATMEL AVR-MEGA128 (ET-BASE AVR ATmega128 r3)                               *
 *       	      : X-TAL : 16 MHz                                                             * 
 *                                                                                             *
 * Editor         : AVR Studio 4.15 Build 623                                                  *
 * Compiler       : WinAVR-20081205                                                            *
 *                                                                                             *
 * Port Interface :                                                                            *
 *                                                                                             *
 *         SPI    :        # MCU-MEGA128 #           ET-Mini ISD2548                           *
 *                         --------------            ---------------                           *
 *                                                                                             *                                   
 *                          PC0            connect        DIN                                  *                
 *                          PC1            connect        SCK                                  *
 *                          PC2            connect        STB                                  *               
 *                          PC3(IN)        connect        EOM (Must Pull Up)                   *               
 *                          GND            connect        ENA                                  *             
 *                          NO USE         connect        OVF                                  *            
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *                        (Use Internal Pull up in AVR)                                        *
 *                                   5V                                                        *
 *                                  --+--                                                      *
 *                                    |                                                        *
 *                                    \                                                        *
 *                                    / 10K                                                    * 
 *                                    \                                                        *
 *                                    /         ET-Mini ISD2548       5V                       *
 *                                   |       +-------------------+   --+--                     *
 *                                   |       |                   |     |                       *
 *                        PC0 -------|------>|DIN            +5V +-----+      /|               *
 *                        PC1 -------|------>|SCK                |          /  |               *
 *                        PC2 -------|------>|STR            SP+ +--------||   |               *
 *                        PC3(In)<---+-------|EOM                |        || SP|               *
 *                                           |               SP- +--------||   |               *
 *                                           |                   |          \  |               *
 *                               +-----------|ENA            GND +-----+      \|               *
 *                             __|__         |                   |   __|__                     *
 *                              ---          |                   |    ---                      *
 *                               -        ---|VOF                |     -                       *
 *                                           |                   |       __+--                 *
 *                                           |               MIC+|------|  |                   *
 *                                           |                  -|------|__|                   *  
 *                                           +-------------------+         +--                 *
 *                                                                                             *
 *                                                    | |     Set Jp1 : to GND                 *
 *                                                    | |         Jp2 : to MIC or ANA IN       *
 *                                                    | |                                      *
 *                                                    \ /                                      *
 *  Dtermine Address ISD2548 For Push Button Mode:                                             *
 *                                                 A0(M0)   =  X (Skip) : Data Bit0            *
 *                                                 A5..A1   =  Logic 0  : Data Bit5..1         *
 *                                                 A8..A6   =  Logic 1  : Data Bit8..6         *
 *                                                CE,PD,P/R =  X        : Data Bit11..9        *
 *                                                                                             *
 *                                       When X = Logic 0 or Logic 1  Follow Control           *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *      SWITCH    :   # MCU-MEGA128 #               # SW. for User COntrol Play& Record#       *
 *                    ------------------             -----------------------------------       * 
 *                       PE4(IN)          connect         SW. Record                           *
 *                       PE5(IN)          connect         SW. Play                             *
 *                       PE6(IN)          connect         SW. Skip Message                     *
 *                       PE7(IN)          connect         SW. Reset                            *
 *                                                                                             *
 *                      5V                                                                     *
 *                     --+--                                                                   *
 *                       |                                                                     *
 *                       +-------+-------+-------+                                             *  
 *                       |       |       |       |                                             *
 *                       /       /       /       /                                             *
 *                       \       \       \       \                                             *
 *                       /10K    /10K    /10K    / 10K                                         *
 *                       \       \       \       \                                             *
 *              (Record)  |       |       |       |                                            *
 *         PE4 <----------+       |       |       |                                            *
 *              (Play)    |       |       |       |                                            *
 *         PE5 <------------------+       |       |                                            *
 *              (Skip-MSG)|       |       |       |                                            *
 *         PE6 <--------------------------+       |                                            *
 *              (Reset)   |       |       |       |                                            *
 *         PE7 <----------------------------------+                                            *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                   SW.4\   SW.5\   SW.6\  SW.7 \                                             *
 *                        |       |       |       |                                            *
 *                        |       |       |       |                                            *
 *                      __|__   __ __   __|__   __|__                                          *
 *                       -_-     -_-     -_-     -_-                                           *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 * Last Update     : 14/Junly/2012                                                               *  
 * Create By       : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                      *
 *                                                                                             *
 ***********************************************************************************************/




#include <stdio.h>
#include <avr/io.h>                  //define port
#include <compat/deprecated.h>       //call function sbi(port,bit),cbi(port,bit) = Set,Clear bit
#include <avr/pgmspace.h>            //for progrm data to Area Flash

#define F_CPU 16000000UL             //CPU Clock Frequency (Hz)
#include <util/delay.h>              //call function delay


/*******************************************
 **            Function Delay             ** 
 *******************************************/

                                     
//--------- Function Delay us --------------           


 void delay_us(unsigned int cnt)
  {
    for(;cnt>0;cnt--)
       _delay_us(1);
  }


//--------- Function Delay ms -------------- 

 void delay_ms(unsigned int cnt)
  {
    for(;cnt>0;cnt--)
       _delay_ms(1);
  }



/*****************************************
 **        Function Initial Port        **
 *****************************************/

 void Init_Port(void)
  {
   
   DDRC  = 0x07     ;        //Set PC0..PC2:Output For SPI Control ISD2548 ;PC3=Input
     
   DDRE  = 0x00     ;        //Set PE(0..7) = Input for Switch  
    

 
   //--- Set default output Port ----
 
   PORTC = 0x08     ;        //PC3 = Input Pull Up


  }



/***************************************************************                                                           ++
 **                        Function Switch                    **
 ***************************************************************/

//-------------- Check Release SW.4-SW.7 -----------

void Check_Sw()
 {
   while((PINE & 0xF0)!= 0xF0) {;}  //Check Release Sw.4-Sw.7

   delay_ms(50)        ;
 }

//---------------- Read SW.4-SW.7 (PE4-PE7)-----------------

char Rd_Sw()
 {
  char sw                  ;

   sw = PINE               ;  //Read PE4-PE7
   sw = ~sw                ;  //Invert data Read
   sw = sw & 0xF0          ;  //Mark bit PE4..PE7 (SW4-SW7)
   return sw               ;  //Return Value Press Switch


 }





/***************************************************************
 **                Function SPI Seny data 12 bit              **
 ***************************************************************
 **          PC0 = SDO                                        **
 **          PC1 = SCK                                        **
 **          PC2 = STROB                                      **   
 **                                                           **
 ***************************************************************/

//--------------- Define Pin -----------------------


//--------------- SDO-High(Data-Out) ---------------

#define     Sdo_Hi()          sbi(PORTC,0)            //PC0= SDO:1
                          
//--------------- SDO-Low(Data-OUT) ----------------

#define     Sdo_Lo()         cbi(PORTC,0)            //PC0= SDO:0



//--------------- Sck-High(Clock) ---------------

#define     Sck_Hi()          sbi(PORTC,1)            //PC1= SCK:1
                          
//--------------- Sck-Low(Clock) ----------------

#define     Sck_Lo()         cbi(PORTC,1)            //PC1= SCK:0



//--------------- STR-High(Strob) ---------------

#define     Str_Hi()          sbi(PORTC,2)            //PC2= Str:1
                          
//--------------- STR-Low(Strob) ----------------

#define     Str_Lo()          cbi(PORTC,2)            //PC2= Str:0



//----------------- Write Data to ISD2548 12-bit -------------------

void Spi_Wr(unsigned int DWord)			  	
 {
    unsigned int Bit ;				 	   

     Str_Lo()                             ;  //Start Strob 595 to Low

     for(Bit=0;Bit<12;Bit++)                 //Loop write data 12 Bit
      {		
	   	Sck_Lo()                          ;	 //Start 595 Clock  Low

		//--------- write Data(PC0) ---------

		if((DWord & 0x800)== 0x800)
		    Sdo_Hi()                     ;   //Set bit data(PC0) = 1
		else 
		    Sdo_Lo()                     ;	 //Clear bit data(PC0) = 0
		   	 
		
		Sck_Hi()                         ;   //Set 595 Clock High for Shift data bit

		DWord <<= 1                      ;	 //Shift Next Bit Data        
	
	  }

    Str_Hi()                             ;   //Set Strob 595 to High for Out data parallel 


  }


/***************************************************************
 **                    FUNCTION ISD2548                       **
 **                                                           **
 **  - Memory = 320 Address                                   **
 **  - Time Record Max = 48 secound                           **
 **                                                           **
 ***************************************************************/

 
//-------- Define Data Control ISD2548 ------------

#define Button_Mode       0x1C0      // Set Button Mode   A5..A1(Bit5..1)=0;  A8..A6(bit8..bit6)=1 

#define ISD_Reset         0xE00      // Set Reset ISD2548 CE,PD,P/R(bit11..9) = 1  ; A0/MO(bit0) = 0   
#define ISD_Boot_CLo      0x200      // Boot ISD2548  CE=0,PD=0,P/R=1
#define ISD_Boot_CHi      0xA00      // Boot ISD2548  CE=1;PD=0,P/R=1

#define Skip_On_CLo       0x201      // Skip CE=0,PD=0,P/R=1 ;A0/M0=1 
#define Skip_On_CHi       0xA01      // Skip CE=1,PD=0,P/R=1 ;A0/M0=1 
#define Skip_Off          0xA00      // Skip CE=1,PD=0,P/R=1 ;A0/M0=0 

#define Play_CLo          0x200      // Play : CE=0,PD=0,P/R=1
#define Play_CHi          0xA00      // Play : CE=1,PD=0,P/R=1

#define Rec_CLo           0x000      // Record : CE=0,PD=0,P/R=0
#define Rec_CHi           0x800      // Record : CE=1,PD=0,P/R=0


//----------------- Initial ISD2548 ---------------

 void Init_ISD2548()
  {
    
    Spi_Wr(Button_Mode|ISD_Reset)	 ;  //Set M0 = 0;P/R=1;PD=1;CE=1
   
    //-Initial boot ISD2548-

    delay_ms(100)                    ;

	Spi_Wr(Button_Mode|ISD_Boot_CHi) ; //Set M0 = 0;P/R=1;PD=0;CE=1
    
  }


//---------------- Read Status EOM : End OF Message ---------------- 

char Rd_Eom()
{
  char eom                 ;
       
  eom  = PINC              ; //Read PC3 : 0 = end of message , Nomal = 1   
  eom  = eom & 0x08        ;
  return eom               ;  
}



//------------------- Check Status EOM -----------------
void Chk_Eom()
 {
  char em ;

  em = Rd_Eom();  

  while(em)                // Check Signal EOM to 0 : EOF = 0 :Exit Loop(for End message) 
  {
   em = Rd_Eom();
   
   if((Rd_Sw()) == 0x80)   //Check Press  SW Reset (PE7)
    {
     em = 0 ;
	 Init_ISD2548() ;      //Reset ISD2548 for Start Play & Record at Address 0x000
    }
  }
  
   
 }



//------------ Skip Mesaage  ------------

void Skip_Msg(void)
 { 
      
   Spi_Wr(Button_Mode|Skip_On_CLo)	; 
   Spi_Wr(Button_Mode|Skip_On_CHi)	;
   Spi_Wr(Button_Mode|Skip_Off)	;  

 }



//------------ Play 1 Message -----------

void Play_Msg()
 {

   Spi_Wr(Button_Mode|Play_CLo)	;
   Spi_Wr(Button_Mode|Play_CHi)	;
    
 }


//---------------- Record 1 Message -------------------

void Record()
 {
  
   Spi_Wr(Button_Mode|Rec_CLo)	;
   Spi_Wr(Button_Mode|Rec_CHi)	;
    
 }
 
//----------------Stop Record Message -------------------

void Stop_Rec()
 {   
    Spi_Wr(Button_Mode|Rec_CLo)	;
    Spi_Wr(Button_Mode|Rec_CHi)	;

 }






/**********************************************************************
 **                                                                  **
 **                         main Program                             **
 **                                                                  **
 **********************************************************************/
 
 int main(void) 
 {
   unsigned char sw  ;
 
   Init_Port()       ;
   Init_ISD2548()    ; 

 //--------------  Start  --------------


  while(1)
   { 
     sw = Rd_Sw() ;   //Read Switch

     switch(sw)
      {
                     
        case 0x10 :  //SW.4(PE4) = Record Message
                   
				     Record()   ;  //Start Rec. 
                  				 
                     Check_Sw() ;  //Check Release Switch Rec.

					 Stop_Rec() ;  //Stop Message Record 

					 Chk_Eom()  ;  //Check EOM=0  Ready for  Next Message Record

                     
            break ;


         case 0x20 :  //SW.5(PE5) = Play Message
                
                     Play_Msg() ;   //Play 1 Message

			         Check_Sw() ;   //Check Release Switch
                    
                     Chk_Eom()  ;   //Check EOM=0  Ready for  Next Play Message 
				  
             
             break ;


         case 0x40 : //SW.6(PE6) = Skip Mesage
                               
                     Skip_Msg() ;  //Skip 1 Message
				
				     Check_Sw() ;  //Check Release Switch
                  

             break ;


          case 0x80: //SW.7(PE7) = Reset Address Pointer ISD2548 to Address 0x000
                               
                    Init_ISD2548() ;  //Reset ISD2548 for Start Play & Record at Address 0x000
				
				    Check_Sw()     ;  //Check Release Switch				
				  

             break ;

     }  //switch 
   }    //while
}      //End

 


