

/***********************************************************************************************
 *                                                                                             *
 *                        Example 3. Read Ticket Machine (Address Mode)                        *
 *                                                                                             *
 *                      - Determine Read Message Ticket Number  since '01-99'                  *
 *                      - Determine Number counter service =  6 Counter                        *                    
 *                                                                                             *
 ***********************************************************************************************
 *                                                                                             *
 * Target MCU     : ATMEL AVR-MEGA128 (ET-BASE AVR ATmega128 r3)                               *
 *       	      : X-TAL : 16 MHz                                                             * 
 *                                                                                             *
 * Editor         : AVR Studio 4.15 Build 623                                                  *
 * Compiler       : WinAVR-20081205                                                            *
 *                                                                                             *
 * Port Interface :                                                                            *
 *                                                                                             *
 *                                                                                             *
 *         SPI    :        # MCU-MEGA128 #           ET-Mini ISD2548                           *
 *                         --------------            ---------------                           *
 *                                                                                             *                                   
 *                          PC0            connect        DIN                                  *                
 *                          PC1            connect        SCK                                  *
 *                          PC2            connect        STB                                  *               
 *                          PC3(IN)        connect        EOM (Must Pull Up)                   *               
 *                          GND            connect        ENA                                  *             
 *                          NO USE         connect        OVF                                  *            
 *                                                                                             *
 *                                                                                             *
 *                       (Use Internal Pull up in AVR)                                         *
 *                                    5V                                                       *
 *                                  --+--                                                      *
 *                                    |                                                        *
 *                                    \                                                        *
 *                                    / 10K                                                    *
 *                                    \                                                        *
 *                                    /                                                        *
 *                                    |                                                        *
 *                                    |                                                        *
 *                                    |          ET-Mini ISD2548       5V                      *
 *                                    |      +-------------------+   --+--                     *
 *                                    |      |                   |     |                       *
 *                           PC0 -----|----->|DIN            +5V +-----+      /|               *
 *                           PC1 -----|----->|SCK                |          /  |               *
 *                           PC2 -----|----->|STR            SP+ +--------||   |               *
 *                           PC3 <----+------|EOM                |        || SP|               *
 *                                           |               SP- +--------||   |               *
 *                                           |                   |          \  |               *
 *                               +-----------|ENA            GND +-----+      \|               *
 *                             __|__         |                   |   __|__                     *
 *                              ---          |                   |    ---                      *
 *                               -        ---|VOF                |     -                       *
 *                                           |                   |                             *
 *                                           |                MIC|---                          *
 *                                           +-------------------+                             *
 *                                            Set Jp1 : to GND                                 *
 *                                                Jp2 : No use (Don't care)                    *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *         LCD     :     # MCU-MEGA128 #                   # LCD 2x16 DISPLAY #                *
 *                       ---------------                   --------------------                *
 *                                                                                             *
 *                          PG4            Connect            RS(OUT)                          *
 *                          PD7            Connect            EN(OUT)                          *  
 *                          PG0-PG3        Connect            Data 4 bit High(Out)             *
 *                                                                                             * 
 *        5V                                                                                   *
 *      --+--                                                                                  *
 *        |                                                                                    *
 *        +-----------------+                                                                  *
 *        |                 |                                                                  *
 *       ---                |                                                                  *
 *       |\|                |                                                                  *
 *    VR |/|<---------------|--+                                                               *
 *   10K ---                |  |  PG4     PD7            PG0 PG1 PG2 PG3                       *
 *        |                 |  |   ^       ^              ^   ^   ^   ^                        *
 *        |                 |  |   |       |              |   |   |   |                        *
 *        +--------------+--|--|---|---+---|--+--+--+--+  |   |   |   |                        *
 *      __|__            |  |  |   |   |   |  |  |  |  |  |   |   |   |                        *
 *       ---             |  |  |   |   |   |  |  |  |  |  |   |   |   |                        *
 *        -             1| 2| 3|  4|  5| 6 | 7| 8| 9|10|11|12 | 13| 14|                        *
 *                     __|__|__|___|___|___|__|__|__|__|__|___|___|___|_                       *
 *                    |  G Vcc Vo RS  R/W  E  D0 D1 D2 D3 D4  D5  D6 D7 |                      *
 *                    |    _________________________________________    |                      *
 *                    |   |                                         |   |                      *
 *                    |   |               Module                    |   |                      *
 *                    |   |         LCD 2x16 Charecter              |   |                      *  
 *                    |   |_________________________________________|   |                      *
 *                    |                                                 |                      *
 *                    |_________________________________________________|                      *
 *                                                                                             *       
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *                                                                                             *
 *      SWITCH    :   # MCU-MEGA128 #               # SW. for User Control Play& Record#       *
 *                    ----------------               -----------------------------------       * 
 *                       PE0,PE1           Not use        Pull-Up 5V                           *
 *                       PE2(IN)          connect         SW. Select Message UP+               *
 *                       PE3(IN)          connect         SW. Select Message Down-             *
 *                       PE4(IN)          connect         SW. Record Message                   *
 *                       PE5(IN)          connect         SW. Play Message All                 *
 *                       PE6(IN)          connect         SW. STOP Play Message All            *
 *                       PE7(IN)          connect         SW. Play 1 Message at Select         *
 *                                                                                             *
 *                     5V                                                                      *
 *                   --+--                                                                     *
 *                     |                                                                       *
 *                     +-------+-------+-------+-------+-------+                               *  
 *                     |       |       |       |       |       |                               *
 *                     /       /       /       /       /       /                               *
 *                     \       \       \       \       \       \                               *
 *                     /10K    /10K    /10K    /10K    /10K    /10K                            *
 *                     \       \       \       \       \       \                               *
 *      (Counter CH1)  |       |       |       |       |       |                               *
 *         PE2 <-------+       |       |       |       |       |                               *
 *      (Counter CH2)  |       |       |       |       |       |                               *
 *         PE3 <---------------+       |       |       |       |                               *
 *      (Counter CH3)  |       |       |       |       |       |                               *
 *         PE4 <-----------------------+       |       |       |                               *
 *      (Counter CH4)  |       |       |       |       |       |                               *
 *         PE5 <-------------------------------+       |       |                               *
 *      (Counter CH5)  |       |       |       |       |       |                               *
 *         PE6 <---------------------------------------+       |                               *
 *      (Counter CH6)  |       |       |       |       |       |                               *
 *         PE7 <---------------------------------------|-------+                               *
 *                     |       |       |       |       |       |                               *
 *                     |       |       |       |       |       |                               *
 *                SW.2\   SW.3\   SW.4\   sw.5\   sw.6\   SW7 \                                *
 *                     |       |       |       |       |       |                               *
 *                     |       |       |       |       |       |                               *
 *                   --+--   --+--   --+--   --+--   --+--   --+--                             *
 *                     G       G       G       G       G       G                               *
 *                                                                                             *
 *                                                                                             *
 *  Before Operate  :     Use Example_1 for Record Message by 1 Message Use 9 address          * 
 *                                                                                             *
 *                       Message 02 Record : Tee-chong     (Start Address 9 )                  *
 *                          "    03   "    : Bor-ri-kan    (Start Address 18)                  *
 *                          "    04   "    : Ka            (Start Address 27)                  *
 *                          "    05   "    : 1             (Start Address 36)                  *
 *                          "    06   "    : 2             (Start Address 45)                  *
 *                          "    07   "    : 3             (Start Address 54)                  *
 *                          "    08   "    : 4             (Start Address 63)                  *
 *                          "    09   "    : 5             (Start Address 72)                  *
 *                          "    10   "    : 6             (Start Address 81)                  * 
 *                          "    11   "    : 7             (Start Address 90)                  *
 *                          "    12   "    : 8             (Start Address 99)                  *
 *                          "    13   "    : 9             (Start Address 108)                 * 
 *                          "    14   "    : 10            (Start Address 117)                 *
 *                          "    15   "    : ed            (Start Address 126)                 *
 *                          "    16   "    : 20            (Start Address 135)                 *
 *                          "    17   "    : 30            (Start Address 144)                 * 
 *                          "    18   "    : 40            (Start Address 153)                 *
 *                          "    19   "    : 50            (Start Address 162)                 *
 *                          "    20   "    : 60            (Start Address 171)                 * 
 *                          "    21   "    : 70            (Start Address 180)                 *
 *                          "    22   "    : 80            (Start Address 189)                 *
 *                          "    23   "    : 90            (Start Address 198)                 * 
 *                          "    24   "    : Chen-may-lak  (Start Address 207)                 *            
 *                                                                                             *
 * Last Update     : 12/Junly/2012                                                             *  
 * Create By       : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                      *
 *                                                                                             *
 ***********************************************************************************************/



#include <stdio.h>
#include <avr/io.h>                     //define port
#include <compat/deprecated.h>          //call function sbi(port,bit),cbi(port,bit) = Set,Clear bit
#include <avr/pgmspace.h>               //for progrm data to Area Flash

#define F_CPU 16000000UL                //CPU Clock Frequency (Hz)
#include <util/delay.h>                 //call function delay


int add_nm1,add_nm2,add_ch,addr       ;

char number = 0,fag_nm=0,nh,nl        ;


/*****************************************
 **         Function Delay              **
 *****************************************/


 //------------ Delay us ------------

 void delay_us(unsigned int cnt)
  {
    for(;cnt>0;cnt--)
       _delay_us(1);
  }


//------------- Delay ms -------------

 void delay_ms(unsigned int cnt)
  {
    for(;cnt>0;cnt--)
       _delay_ms(1);
  }


/*****************************************
 **        Function Initial Port        **
 *****************************************/

 void Init_Port(void)
  {
   
   DDRC  = 0x07     ;        //Set PC0..PC2:Output For SPI Control ISD2548 ;PC3:Input
   DDRE  = 0x00     ;        //Set PE(0..7) = Input for Switch     
   DDRD  = 0xFF     ;        //Set PD(0..7) = output for Control LCD
   DDRG  = 0xFF     ;        //Set PG(0..7) = output for data LCD

 //---- Set default output Port ----
 
   PORTC = 0x08     ;        //PC3 = Input Pull Up
   PORTD = 0x00     ;
   PORTG = 0x00     ;

  }



/**************************************************************
 **                                                          **
 **                         FUNCTION LCD                     **
 **                                                          **
 **************************************************************/ 


//---------------- Define LCD Control Pin ------------------

#define Ena_Hi()     OUTPUT_HIGH(PIN_H3) ;         //RH3 = ENA:1
#define Ena_Lo()     OUTPUT_LOW(PIN_H3)  ;         //RH3 = ENA:0


//-------------- Set LCD Control Pin ------------------

#define ENA_HI()     sbi(PORTD,7)    //PD7 = ENA:1
#define ENA_LO()     cbi(PORTD,7)    //PD7 = ENA:0



//--------------- write instruction LCD ------------------      


void Write_Cmm(unsigned char cmm)
{
  
  PORTG = (cmm>>4) & 0x0F    ;   //Sent cmm 4 bit High,RS:PG4 = 0 
  ENA_HI()                   ;    
  delay_us(250)              ;
  ENA_LO()                   ;    
  delay_us(250)              ;

  PORTG = cmm & 0x0F         ;   //Sent cmm 4 bit Low,RS:PG4 = 0
  ENA_HI()                   ;   
  delay_us(250)              ;
  ENA_LO()                   ;   
  delay_us(250)              ;
}


 //--------------- write data lcd ------------------

void Write_Data(unsigned char dat)
{
  
  PORTG = ((dat>>4) & 0x0F)|0x10  ;   //Sent data 4 bit High,RS:PG4 = 1 
  ENA_HI()                        ;    
  delay_us(250)                   ;
  ENA_LO()                        ;    
  delay_us(250)                   ;

  PORTG = (dat & 0x0F)|0x10       ;   //Sent data 4 bit Low,RS:PG4 = 1
  ENA_HI()                        ;   
  delay_us(250)                   ;
  ENA_LO()                        ;   
  delay_us(250)                   ;
}


 //------------------- Clear Display LCD ------------------------


void clr_display(void)
 {
  Write_Cmm(0x01) ;
 }



//----------------------- Clear Line 1 LCD ---------------------------
 

void Clear_L1()
 {
  char k ;

   Write_Cmm(0x80|0x00)     ;  //Start address lcd = 0x4B Line 2
   for(k=0;k<16;k++)   
     Write_Data(0x20)       ;  //Plot Space to lcd 
 }


//---------------- initial lcd --------------------

void Init_Lcd(void)
{
  Write_Cmm(0x33);	        //Command Control 
  Write_Cmm(0x32);
  Write_Cmm(0x28);
  Write_Cmm(0x0C);  	    //Set Display On/Off:Display On,Not Show Cursor and not blink Cursor
  Write_Cmm(0x06);  	    //Set Entry Mode :Increment Address,Cursor Shift Right
    
  Write_Cmm(0x1C);          //Set Cursor Or Display Shift : Shift Right Charecter
  Write_Cmm(0x28);      	//Function Set: set interface data 4bit,Set Display LCD 2 Line ,Set Charecter Size 5x7
  Write_Cmm(0x80|0x00);		//Set Address Begin 0x00 (Line1 Charecter1)
  clr_display();
}

 
/********************************************** 
 *          Function Convert Ascii            *
 **********************************************/

void Conv_Ascii(char num)
 {
   nh = (num/10)+0x30     ;
   if(nh == 0x30)            // Check nh = '0' Give nh= space
     nh = 0x20            ;  // Not display '0' on LCD
   
   nl = (num%10)+0x30     ;
   
 }


/****************************************************************
 **                Function SPI Sent Data 12 Bit               **                                                  
 ****************************************************************                                          
 **   PC0 = SDO                                                **
 **   PC1 = SCK                                                **
 **   PC2 = STROB                                              **
 **                                                            **
 ****************************************************************/

 //--------------- Define Pin Control SPI -----------------

//--------------- SDO-High(Data-Out) ---------------

#define     Sdo_Hi()          sbi(PORTC,0)            //PC0= SDO:1
                          
//--------------- SDO-Low(Data-OUT) ----------------

#define     Sdo_Lo()         cbi(PORTC,0)            //PC0= SDO:0


//--------------- Sck-High(Clock) ---------------

#define     Sck_Hi()          sbi(PORTC,1)            //PC1= SCK:1
                          
//--------------- Sck-Low(Clock) ----------------

#define     Sck_Lo()         cbi(PORTC,1)            //PC1= SCK:0



//--------------- STR-High(Strob) ---------------

#define     Str_Hi()          sbi(PORTC,2)            //PC2= Str:1
                          
//--------------- STR-Low(Strob) ----------------

#define     Str_Lo()          cbi(PORTC,2)            //PC2= Str:0




//----------------- Write Data to ISD2548 12-bit -------------------

void Spi_Wr(unsigned int DWord)			  	
 {
    unsigned int Bit ;				 	   

     Str_Lo()                            ;  //Start Strob 595 to Low

     for(Bit=0;Bit<12;Bit++)                 //Loop write data 12 Bit
      {		
	   	Sck_Lo()                         ;	 //Start 595 Clock  Low

		//--------- write Data(PC0) ---------

		if((DWord & 0x800)== 0x800)
		    Sdo_Hi()                     ;   //Set bit data(PC0) = 1
		else 
		    Sdo_Lo()                     ;	 //Clear bit data(PC0) = 0
		   	 
		
		Sck_Hi()                         ;   //Set 595 Clock High for Shift data bit

		DWord <<= 1                      ;	 //Shift Next Bit Data        
	
	  }

    Str_Hi()                             ;   //Set Strob 595 to High for Out data parallel 


  }



/***************************************************************
 **                   FUNCTION ISD2548                        **
 ***************************************************************
 **  - Memory = 320 Address                                   **
 **  - Time Record Max = 48 secound                           **
 **  - 1 Address   = Time Record 0.15 secound                 **
 **                                                           **
 ***************************************************************/


//-------- Define Data Control ISD2548 ------------

#define PL_CHi            0xA00      //CE,P/R=1;PD=0
#define PL_CLo            0x200      //P/R=1;CE,PD=0
#define REC_CHi           0x800      //CE=1;PD,P/R=0
#define REC_CLo           0x000      //CE,PD,P/R =0
#define STOP              0xE00      //CE,PD,P/R = 1 


//---------------- Read Status EOM : End OF Message ---------------- 

char Rd_Eom()
{
  char eom                 ;
       
  eom  = PINC              ; //Read PC3 : 0 = end of message , Nomal = 1   
  eom  = eom & 0x08        ;
  return eom               ;  
}



//------------------- Check Status EOM -----------------
void Chk_Eom()
 {
  while(Rd_Eom()){;}   // Check Signal EOM to 0 : EOF = 0 :Exit Loop(for End message) 
  while(!Rd_Eom()){;}  // Check Signal EOM to 1 : EOF = 1 :Exit Loop (for Ready Begine Play   Next Message)
 }


//------------ Set Address for Play or Record(Addtess 0-320) ------------

void Set_Add(int add)
 { 

   addr = add & 0x1FF   ;  //Mark data Bit9-11 for write  Pin P/R,PD,CE 
   Spi_Wr(addr|STOP)	;  //Write Pin Address A0-A8 and Pin Contr0l To ISD2548  

 }



//-------------------- Play 1 Message ---------------------

void Play_Back()
 {

   addr = addr &0x01FF  ;  //Mark data Control Bit9-11 Pin P/R,PD,CE for  Play
   Spi_Wr(addr|PL_CLo)	;  //Write Pin Address A0-A8 and Pin Control for Play To ISD2548(Pin CE =0) 

   addr = addr &0x01FF  ;
   Spi_Wr(addr|PL_CHi)	;  //Write Pin CE = 1

 }


//----------------- Initial ISD2548 ---------------

 void Init_ISD2548()
  {

    Spi_Wr(STOP)	;     
    
  }


//*************************************************
//**        Function PLAY MESSAGE Ticket         **
//*************************************************
 
void Msg_Number()
 {
     //--------------- Play Message #1 "Chen-My-Lak"   --------------

      Set_Add(207)                  ;  //Set Address for Play Message  #1 
      Play_Back()                   ;  //Play "Chen-My-Lak"            
      Chk_Eom()                     ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)


     if(fag_nm==1)     //Play Number 2 Message   
       {
         Set_Add(add_nm1)           ;  //Set Address for Play  Message at 1
         Play_Back()                ;  //Play Message Number            
         Chk_Eom()                  ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)
        
         Set_Add(add_nm2)           ;  //Set Address for Play  Message at 2
         Play_Back()                ;  //Play Message Number            
         Chk_Eom()                  ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)   
      }
     else      //Play Number 1 Message
      {
        Set_Add(add_nm1)            ;  //Set Address for Play  Message at 1
        Play_Back()                 ;  //Play Message Number            
        Chk_Eom()                   ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)
      }
     

    //------------------- Play Message #2,#3 "Tee-Chong","Bor-ri-kan" ---------------------

      Set_Add(9)                    ;  //Set Address for Play Message  #2 
      Play_Back()                   ;  //Play "Tee-Chong"            
      Chk_Eom()                     ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)

      Set_Add(18)                   ;  //Set Address for Play Message  #3 
      Play_Back()                   ;  //Play "Bor-ri-kan"            
      Chk_Eom()                     ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)


      Set_Add(add_ch)               ;  //Set Address for Play Message  Number Channel(1-6)
      Play_Back()                   ;  //Play Number Channel "1-6"            
      Chk_Eom()                     ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)


    //------------------- Play Message #4 "ka" ---------------------

      Set_Add(27)                   ;  //Set Address for Play Message  #4 
      Play_Back()                   ;  //Play "ka"            
      Chk_Eom()                     ;  //waite EOM chang '1' to '0'(EOF) and '0' to '1'(Ready begin New Message)

 }
      

//***********************************************************
//**           Function Calculate Address Message          **
//***********************************************************

void Cal_Addr_Meg()
{
     number++                               ;       //Counter Number 1-99

   if(number>99)
     number = 1                             ;      //Clear Counter Number = 1 

    //----------------- Fine Address Message -----------------------

    if(((number>=1)&&(number<=10))||(number==20)||(number==30)||(number==40)||
       (number==50)||(number==60)||(number==70)||(number==80)||(number==90))     //Check for Play 1 message 
     {
       fag_nm = 0                            ;    //Set Flag for platy 1 Message
                     
       if((number>=1)&&(number<=10))              //Check Counter 1-10            
         add_nm1   =     (number*9)+27       ;    //Keep Address for Play Message '1-10' (27 = Address Start for Messsage '1')
       else                                       //if Number Counter = 20,30,40,50,60,70,80,90
         add_nm1   =  ((number/10)*9)+117    ;    //Keep Address for Message('20,30,40,50,60,70,80,90')                                       
     }
    else   //Check for Play 2 Message
     {
       fag_nm  = 1                           ;    //Set Flag for platy 2 Message
                     

      //---Set address Message 1 -----
                                              
      if((number/10)==1)                          //If Number  11-19
        add_nm1 = 117                        ;    //Keep Address Message '10'
      else                                        //if Number 21-29,31-39,41-49,51-59,61-69,71-79,81-89,91-99
        add_nm1 = ((number/10)*9)+117        ;    //Calcilate and Set Address Message '20,30,40...90'

      //---Set address Message 2 -----

       if((number%10)==1)                        //if Number%10 = 1
          add_nm2 = 126                      ;   //Keep Address message 'ed'       
       else                                      //if Number%10 > 1   (2-9)
         add_nm2 = ((number%10)*9)+27        ;   //Keep Address message '2-9'
     }
}


//********************************************
//**             Function Switch            **
//********************************************

//-------------- Check press SW.2-SW.7 -----------

void Check_Sw()
 {
   while((PINE&0xFC) != 0xFC){;}   //Check SW. PE[2..7] Release All 
   delay_ms(50)            ;
 }


//---------------- Read SW.2-SW.7 (PE2-PE7)-----------------

char Rd_Sw()
 {
  char sw                  ;
   sw = PINE               ;      //Read press SW. PE[2..7]
   sw = (~sw) & 0xFC       ;      //Mark PE[2..7] and invert data (Normal Value Not press SW. is logic 1)

   return sw               ;

 }






/**********************************************************************
 **                                                                  **
 **                         Main Program                             **
 **                                                                  **
 **********************************************************************/
 
int main(void) 
 {
   unsigned char   key,lp                  ;
 
   unsigned char str0[9]  = {"COUNTER="}   ;
   unsigned char str1[9]  = {"NUMBER ="}   ;
   
   Init_Port()       ;
   Init_ISD2548()    ; 
   Init_Lcd()        ;


 //--------------  Start  --------------

  Write_Cmm(0x80|0x00)            ;  //Start address lcd = 0x01 Line 1
  for(lp=0;lp<8;lp++)
    Write_Data(str0[lp])          ;  //Print "COUNTER" to LCD
 
  Write_Cmm(0x80|0x40)            ;  //Start address lcd = 0x43 Line 2
  for(lp=0;lp<8;lp++)
    Write_Data(str1[lp])          ;  //Print "NUMBER" to LCD


  while(1)
   {
     delay_ms(10); 

     key = Rd_Sw() ;   

     switch(key)
      {
        case 0x04:   //SW2-IN2 -------> COUNTER-1 Call 
                  
                  add_ch = 36               ;   //Keep address Message '1' for Channel
 
                  Cal_Addr_Meg()            ;  //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                    
                   Write_Cmm(0x80|0x0B)     ;  //Start address lcd = 0x0B  Line 1
                   Write_Data('1')          ;  //Show Channel '1' On LCD

                   Conv_Ascii(number)       ;  //Convert Ascii for Number Show to LCD Line2
                   Write_Cmm(0x80|0x4A)     ;  //Start address lcd = 0x4A  Line 2
                   Write_Data(nh)           ;  //Show Number Colum High On LCD
                   Write_Data(nl)           ;  //Show Number Colum Low On LCD
 
                   Msg_Number()             ;  //Call Play Sound Message
                  
                   Check_Sw()               ;  //Check Press Repeat   SW.
              break ;

         case 0x08:   //SW3-IN3 -------> COUNTER-2 Call


                   add_ch = 45              ;  //Keep address Message '2' for Channel

                   Cal_Addr_Meg()           ;  //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                   
                   Write_Cmm(0x80|0x0B)     ;  //Start address lcd = 0x0B  Line 1
                   Write_Data('2')          ;  //Show Channel '2' On LCD

                   Conv_Ascii(number)       ;  //Convert Ascii for Number Show to LCD Line2
                   Write_Cmm(0x80|0x4A)     ;  //Start address lcd = 0x4A  Line 2
                   Write_Data(nh)           ;  //Show Number Colum High On LCD
                   Write_Data(nl)           ;  //Show Number Colum Low On LCD

                   Msg_Number()             ;  //Call Play Sound Message
                  
                   Check_Sw()               ;  //Check Press Repeat   SW.

              break ;

     
          case 0x10:   //SW4-IN4 -------> COUNTER-3 Call


                    add_ch = 54              ;   //Keep address Message '3' for Channel

                    Cal_Addr_Meg()           ;   //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                   
                    Write_Cmm(0x80|0x0B)     ;  //Start address lcd = 0x0B  Line 1
                    Write_Data('3')          ;  //Show Channel '3' On LCD

                    Conv_Ascii(number)       ;  //Convert Ascii for Number Show to LCD Line2
                    Write_Cmm(0x80|0x4A)     ;  //Start address lcd = 0x4A  Line 2
                    Write_Data(nh)           ;  //Show Number Colum High On LCD
                    Write_Data(nl)           ;  //Show Number Colum Low On LCD

                    Msg_Number()             ;  //Call Play Sound Message
                  
                    Check_Sw()               ;  //Check Press Repeat   SW.

                break ;

           case 0x20:   //SW5-IN5 -------> COUNTER-4 Call


                    add_ch = 63              ;   //Keep address Message '4' for Channel

                    Cal_Addr_Meg()           ;  //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                   
                   Write_Cmm(0x80|0x0B)      ;  //Start address lcd = 0x0B  Line 1
                   Write_Data('4')           ;  //Show Channel '4' On LCD

                   Conv_Ascii(number)        ;  //Convert Ascii for Number Show to LCD Line2
                   Write_Cmm(0x80|0x4A)      ;  //Start address lcd = 0x4A  Line 2
                   Write_Data(nh)            ;  //Show Number Colum High On LCD
                   Write_Data(nl)            ;  //Show Number Colum Low On LCD

                   Msg_Number()              ;  //Call Play Sound Message
                  
                   Check_Sw()                ;  //Check Press Repeat   SW.

                break ;

           case 0x40:   //SW6-IN6 -------> COUNTER-5 Call


                    add_ch = 72              ;   //Keep address Message '5' for Channel

                    Cal_Addr_Meg()           ;  //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                   
                   Write_Cmm(0x80|0x0B)      ;  //Start address lcd = 0x0B  Line 1
                   Write_Data('5')           ;  //Show Channel '5' On LCD

                   Conv_Ascii(number)        ;  //Convert Ascii for Number Show to LCD Line2
                   Write_Cmm(0x80|0x4A)      ;  //Start address lcd = 0x4A  Line 2
                   Write_Data(nh)            ;  //Show Number Colum High On LCD
                   Write_Data(nl)            ;  //Show Number Colum Low On LCD

                   Msg_Number()              ;  //Call Play Sound Message
                  
                   Check_Sw()                ;  //Check Press Repeat   SW.

                 break ;

            case 0x80:   //SW6-IN6 -------> COUNTER-6 Call


                    add_ch = 81               ;   //Keep address Message '6' for Channel

                    Cal_Addr_Meg()            ;  //Call Calculate Address Message

                        //------------Show Number On LCD ----------------
                   
                   Write_Cmm(0x80|0x0B)       ;  //Start address lcd = 0x0B  Line 1
                   Write_Data('6')            ;  //Show Channel '6' On LCD

                   Conv_Ascii(number)         ;  //Convert Ascii for Number Show to LCD Line2
                   Write_Cmm(0x80|0x4A)       ;  //Start address lcd = 0x4A  Line 2
                   Write_Data(nh)             ;  //Show Number Colum High On LCD
                   Write_Data(nl)             ;  //Show Number Colum Low On LCD

                   Msg_Number()               ;  //Call Play Sound Message
                  
                   Check_Sw()                 ;  //Check Press Repeat   SW.

           break ;

    }     //switch

   }
}      //End

 

