/*******************************************/
/* Example Program For ET-BASE51 AC3 V1.0  */
/* MCU      : AT89C51AC3(XTAL=29.4912 MHz) */
/*          : Frequency Bus = 58.9824 MHz  */
/* Compiler : Keil C51 (V7.50)             */
/* Write By : Eakachai Makarn(ETT CO.,LTD.)*/
/* Examples : Demo Software SPI (AT45DB16) */
/*******************************************/
/*******************************************/
/* AT89C51AC3 Interface to AT45DB16
/* (Used 3V to 5V Logic Shifter)
/*  1.  ET-BASE51 AC3 (AT89C51AC3)
/*  2.  ET-MINI LOGIC SHIFTER (74LCX245)
/*  3.  ET-MINI SPI DATA FLASH (AT45DB16)
/* -> AT89C51AC3  --> [74LCX245] -> AT45DB16
/* -> P4.0(I/O)   --> [A0 -> B0] -> CS#
/* -> P4.1(I/O)   <-- [B1 <- A1] <- BUSY#
/* -> P4.2(I/O)   <-- [B2 <- A2] <- SDO
/* -> P4.3(I/O)   --> [A3 -> B3] -> SCK
/* -> P4.4(I/O)   --> [A4 -> B4] -> SDI
/*******************************************/
// Demo ET-MINI SPI DATA FLASH(AT45DB16)

/* Include  Section */
#include <reg52.h>														// Standard 8052 SFR : File
#include <stdio.h>                										// For printf I/O functions

/* AT89C51AC3 SFR */
sfr CKCON = 0x8F;														// Clock Control
sfr SPCR  = 0xD4;														// SPI Config
sfr SPSR  = 0xD5;														// SPI Status
sfr SPDR  = 0xD6;														// SPI Data

// Define Software SPI Pin Signal //
sfr P4    = 0xC0;														// Port P4
sbit AT45DB16_CS = P4^0;												// CS Data Flash
sbit AT45DB16_BUSY = P4^1;												// Busy
sbit AT45DB16_SDO = P4^2;												// Software Pin MISO
sbit AT45DB16_SCK = P4^3;												// Software Pin SCK 
sbit AT45DB16_SDI = P4^4;												// Software Pin MOSI 

/* User Define Function */
void SPI_WriteByte(unsigned char DataByte);								// Write 1-Byte to SPI
unsigned char SPI_ReadByte(void);										// Read 1-Byte From SPI
void SPI_Delay(void);				   									// SPI Delay Clock

/*--------------------------------------------
The main C function.  Program execution Here 
---------------------------------------------*/
void main()
{ 
  unsigned char My_Data;
  unsigned int j,k; 

  CKCON = 0x01;															// Initial X2 Mode (BUS Clock = 58.9824 MHz) 
 
  /* Initial MCS51 Serial Port */
  TMOD &=  0x0F;  														// Reset old Timer1 Mode Config
  TMOD |=  0x20;  														// Update Timer1 = 8 Bit Auto Reload             
  SCON  =  0x50;  														// Serial Port Mode 1 (N,8,1)
  ES    =  0;     														// Disable Serial Interupt
  ET1   =  0;     														// Disable Timer1 Interrupt
  PCON &=  0x7F;  														// SMOD1 = 0 (Disable Double Baudrate)                  
  TH1   =  0xF0;  														// Setup Timer1 Baudrate 9600BPS / 58.9824 MHz
  TL1   =  0xF0;
  TR1   =  1;     														// Start Timer1 Generate Baudrate                      
  TI    =  1;     														// Set TI to send First char of UART
  
  // Initial Start Sugnal                  
  AT45DB16_CS = 1;														// Disable Chips Select
  AT45DB16_BUSY = 1;				
  AT45DB16_SDO = 1;
  AT45DB16_SDI = 0;
  AT45DB16_SCK = 0;

  /* Print Message to RS232 */
  printf ("Demo Software SPI Read/Write AT45DB16 Page-0\n");  

  while(1)																// Loop Continue   
  {
    // Demo Read Main Page-0
    printf ("\n\rStart Read AT45DB161 Data Page-0\n");

    // Start Read Main Page-0
    AT45DB16_CS = 0;               										// Enable AT45DB161 Interface
                    
    // 8 Bit Command Main Memory Page Read 
    // D2H (SPI Mode-3 Command Code)       
    SPI_WriteByte(0xD2);  												// Command = Main Memory Page Read
                                        
    // 24 Bit Address For Read Data 
    // XX+[PA11..PA0]+[BA9..BA0]    
    // Page = 0
    // Byte = 0
    // 24 Bit Address = 00+[000000000000]+[0000000000]
    SPI_WriteByte(0x00);  // 00+PA[11..6]
    SPI_WriteByte(0x00);  // PA[5..0]+BA[8]
    SPI_WriteByte(0x00);  // BA[7..0]
                    
    // 32 Bit Dummy Data 
    SPI_WriteByte(0xFF); 												// 32 Bit Dummy Data
    SPI_WriteByte(0xFF);
    SPI_WriteByte(0xFF);
    SPI_WriteByte(0xFF);
                                      
    // Start Read Page Data 
    // AT45DB161 = 528 Byte 
    for (j = 0; j < 16; j++)											// 16 Line x 33 Byte = 528 Byte
    {
      for (k = 0; k < 33; k++)				   							// 33 Byte / Line
      {
        My_Data = SPI_ReadByte();										// Read 1 Byte From AT45DB16	  
        printf(" %2X",(unsigned int)My_Data);	
      }
	  printf("\n\r");													// New Line
    }
      
    AT45DB16_CS = 1;              										// Disable AT45DB161 Interface
    // End of Read Main Page-0

    // Demo Write Main Page-0
    printf ("\n\rStart Write AT45DB161 Data Page-0\n");
	printf ("Enter Start Data For Write...");
	My_Data = getchar(); 												// Wait Start Data

    // Start Write Main Page-0
    AT45DB16_CS = 0;           											// Enable AT45DB161 Interface
                    
    // 8 Bit Command Main Memory Page Write 
    // 82H (SPI Mode-3 Command Code)       
    SPI_WriteByte(0x82);                            					// Command = Main Memory Page Write
                                        
    // 24 Bit Address For Write Data   
    // XX+[PA11..PA0]+[BA9..BA0]    
    // Page = 0
    // Byte = 0
    // 24 Bit Address = 00+[000000000000]+[0000000000]
    SPI_WriteByte(0x00);  // 00+PA[11..6]
    SPI_WriteByte(0x00);  // PA[5..0]+BA[8]
    SPI_WriteByte(0x00);  // BA[7..0]
    
    // Start Write Page Data 
    // AT45DB161 = 528 Byte 
    for (j = 0; j < 528; j++)											// Write 528 Byte / Page
    {    
      SPI_WriteByte(My_Data);											// Write 1 Byte to AT45DB16	      
	  My_Data++;
    }

    AT45DB16_CS = 1;                									// Disable AT45DB161 Interface                    

	while(AT45DB16_BUSY!=1);											// Wait Until Busy = 1
    // End of Write Main Page-0

  }																		// Loop Continue
  
}

/***********************/
/* Write 1 Byte to SPI */
/***********************/
void SPI_WriteByte(unsigned char DataByte)								// Write Byte to SPI
{
  unsigned char Bit;													// Bit Counter

  for (Bit = 0; Bit < 8; Bit++)				   							// 8 Bit Write
      {
	    AT45DB16_SDI = (bit)((DataByte & 0x80) >> 7);					// MSB First of Data Bit(7..0)
		AT45DB16_SCK = 1;		   										// Strobe Bit Data

		SPI_Delay();								    				// Delay Clock

		AT45DB16_SCK = 0;  												// Next Clock
		DataByte <<= 1;	 												// Next Bit Data
      }
}

/*************************/
/* Read 1 Byte From SPI0 */
/* Buff = SPI_ReadByte() */
/*************************/
unsigned char SPI_ReadByte(void)										// Read Byte From SPI
{
  unsigned char Bit;													// Bit Counter
  unsigned char Result;									   				// Result
  Result = 0x00;

  for (Bit = 0; Bit < 8; Bit++)				   							// 8 Bit Write Dummy
    {
	  AT45DB16_SDI = 1; 												// Default SDI Logic		
	  AT45DB16_SDO = 1;													// Read SDO Pin(Input)

	  // Get Result 1 Bit (MSB <- LSB)
	  Result <<= 1;														// Logic "0"

	  if (AT45DB16_SDO)													// If SDO = "1"						
	    {
	      Result |= 0x01;   											// Logic "1"
	    }

	  AT45DB16_SCK = 1;		   											// Strobe Bit Data

	  SPI_Delay();								    					// Delay Clock

	  AT45DB16_SCK = 0;  												// Next Clock		
    }
  return(Result);							   							// Result Byte
}

                									
/**************************/
/* Delay SPI Clock Signal */
/**************************/
void SPI_Delay(void)
{
  int x;  																// Short Delay Counter
  x++;
  x++;
}




