;/*************************************/;
;/* Examples For : ET-MINI DATA FLASH */;
;/* Hardware     : ET-BASE51AC3 V1    */;
;/* Controller   : AT89C51AC3         */;
;/* Run X-TAL    : 29.4912 MHz        */;
;/*              : Run X2 Mode        */;
;/* System Clock : 58.9824 MHz        */;
;/* Assembler    : SXA51.EXE          */;
;/* File Name    : DEMO2.ASM          */;
;/* Function     : Demo Hardware SPI  */;
;/*              : Interface AT45DB16 */;
;/*************************************/;
;/ AT89C51AC3 Interface to AT45DB16
;/ (Used 3V to 5V Logic Shifter)
;/  1.  ET-BASE51 AC3 (AT89C51AC3)
;/  2.  ET-MINI LOGIC SHIFTER (74LCX245)
;/  3.  ET-MINI SPI DATA FLASH (AT45DB16)
;/ -> AT89C51AC3  --> [74LCX245] -> AT45DB16
;/ -> P4.0(I/O)   --> [A0 -> B0] -> CS#
;/ -> P4.1(I/O)   <-- [B1 <- A1] <- BUSY#
;/ -> P4.2(MISO)  <-- [B2 <- A2] <- SDO
;/ -> P4.3(SCK)   --> [A3 -> B3] -> SCK
;/ -> P4.4(MOSI)  --> [A4 -> B4] -> SDI
;/*************************************/;
;
;/* AT89C51AC3 Clock Control */;
AUXR                EQU     8EH
CKCON0              EQU     8FH
CKCON1              EQU     9FH
IEN0                EQU     0A8H
;
;/* AT89C51AC3 SPI Register */;
SPCON               EQU     0D4H                            ; SPI Config
SPSCR               EQU     0D5H                            ; SPI Status
SPDAT               EQU     0D6H                            ; SPI Data
;
;/* Data Flash Interface */;
P4                  EQU     0C0H                            ; AT89C51AC3 Port-P4
AT45DB161_CS_PIN    EQU     P4.0                            ; Port I/O
FLASH_BUSY_PIN      EQU     P4.1                            ; Port I/O
AT45DB161_SDO_PIN   EQU     P4.2
AT45DB161_SCK_PIN   EQU     P4.3
AT45DB161_SDI_PIN   EQU     P4.4

                    ORG     20H
FLASH_ADDR:         DS      4                               ; PA[11..8]+PA[7..0]+BA[8]+BA[7..0]
STACK:              DS      30                              ; Stack 30 Byte

;/* ERAM = 2048 Byte = 000H-7FFH */;
FLASH_DATA          EQU     0000H                           ; Flash Page Data = XRAM:0000H

                    ORG     0000H                           ; Reset Vector of CPU
                    ;
MAIN:               MOV     SP,#STACK                       ; Initial Stack
                    MOV     CKCON0,#00000001B               ; 6 Clock Mode(X2 Mode)
                    ;
                    ORL     AUXR,#00000001B                 ; ALE Active By MOVX,MOVC Only
                    ANL     AUXR,#11111101B                 ; Enable MOVX Accress XRAM
                    ANL     AUXR,#11100011B                 ; XRAM Size = 2048 Byte(xxx101xx)
                    ORL     AUXR,#00010100B
                    ;
                    ;/* Initial MCS51 Serial Port */;
                    ; Timer1 Baudrate (29.4912 MHz x 2 )
                    ; Reload = 256 - [58.9824 MHz / (384 x Baud)]
                    ; 1200   = 0x80(256-128=128)
                    ; 2400   = 0xC0(256-94=192)
                    ; 4800   = 0xE0(256-32=224)
                    ; 9600   = 0xF0(256-16=240)
                    ; 19200  = 0xF8(256-8=248)
                    ; 38400  = 0xFC(256-4=252)
                    ANL     TMOD,#00001111B                 ; Reset Timer1 Mode Config
                    ORL     TMOD,#00100000B                 ; Timer1 = 8Bit-Auto Reload
                    MOV     SCON,#01010000B                 ; Serial Port Mode 1 (N,8,1)
                    CLR     ES                              ; Disable Serial Interupt
                    CLR     ET1                             ; Disable Timer1 Interrupt
                    ANL     PCON,#01111111B                 ; SMOD1 = 0 (Disable Double Baudrate)
                    MOV     A,#240                          ; Setup Baudrate 9600BPS
                    MOV     TH1,A
                    MOV     TL1,A
                    SETB    TR1                             ; Start Timer1 Generate Baudrate
                    ;
                    ;/* Initial SPI = Master Mode
                    ;/* Shift Data  = Rising Edge(MSB First)
                    ;/* SPI Clock Rate  = Fosc/16(MAX)
                    ANL     SPSCR,#11111011B                ; UARTM = 0 = MSB First
                    ANL     SPSCR,#11111101B                ; SPTEIE = 0 = Disable SPIE Interrupt
                    ANL     SPSCR,#11111110B                ; MODFIE = 0 = Disable MODF Interrupt

                    ANL     SPCON,#00000000B                ; Reset all SPCON Bit Control
                    ANL     SPCON,#01111100B                ; SPR2:SPR1:SPR00 = 0:0:0 = Reset Prescale
                    ORL     SPCON,#10000010B                ; SPR2:SPR1:SPR00 = 1:1:0 = SCK is Fosc/128
                    ANL     SPCON,#11110011B                ; CPOL:CPHA  = 0:0 = SCK Rising Edge Shift Data
                    ORL     SPCON,#00010000B                ; MSTR = 1 = SPI Master Mode
                    ORL     SPCON,#00100000B                ; SDIS = 1 = Disable SS# Pin
                    ORL     SPCON,#01000000B                ; SPEN = 1 = SPI Enable
                    MOV     A,SPDAT                         ; Read For Reset SPIF
                    ;
                    SETB    AT45DB161_CS_PIN                ; Disable Chips Select
                    SETB    FLASH_BUSY_PIN
                    ;
                    ;/* Start Fill 528 Byte Flash Buffer */;
                    LCALL   PRINT_SER                       ; Print Message
                    DB      0DH,0AH,0AH
                    DB      'Demo Hardware SPI Read/Write AT45DB161'
                    DB      0DH,0AH,00H
                    LJMP    START_READ_PAGE                 ; Goto Read Page-0
                    ;
START_WRITE_PAGE:   LCALL   PRINT_SER
                    DB      0DH,0AH
                    DB      'Enter Start Data For Write...',00H
                    ;
                    MOV     DPTR,#FLASH_DATA                ; Strat Page Data Buffer
                    LCALL   RX_BYTE                         ; Wait Start Data
                    MOV     B,A                             ; Save Start Data
                    ;
                    MOV     R2,#0                           ; Byte of Page Counter
                    MOV     R3,#0
                    ;
WRITE_NEXT_BYTE:    MOV     A,B                             ; Get Data For Write
                    MOVX    @DPTR,A                         ; Fill Buffer
                    INC     B                               ; Next Byte Data
                    INC     DPTR                            ; Next Byte Pointer
                    ;
                    MOV     A,R3                            ; Byte Counter + 1
                    ADD     A,#1
                    MOV     R3,A
                    MOV     A,R2
                    ADDC    A,#0
                    MOV     R2,A
                    ;
                    ;/* Verify 528 Byte Fill Complete */;
                    MOV     A,R2                            ; Verify Last Record
                    CJNE    A,#HIGH(528),WRITE_NEXT_BYTE    ; Repeat Fill 528 Byte
                    MOV     A,R3
                    CJNE    A,#LOW(528),WRITE_NEXT_BYTE
                    ;
                    ;/* Start Write Page-0 */;
                    MOV     FLASH_ADDR+0,#0                 ; Page=0/4095
                    MOV     FLASH_ADDR+1,#0
                    MOV     FLASH_ADDR+2,#0                 ; Byte=0/527
                    MOV     FLASH_ADDR+3,#0
                    LCALL   FLASH_WRITE_PAGE
                    ;
                    ;/* Start Read Page-0 */;
START_READ_PAGE:    LCALL   PRINT_SER                       ; Print Message
                    DB      0DH,0AH
                    DB      'Start Read AT45DB161 Data'
                    DB      0DH,0AH,00H
                    ;
                    MOV     FLASH_ADDR+0,#0                 ; Page=0/4095
                    MOV     FLASH_ADDR+1,#0
                    MOV     FLASH_ADDR+2,#0                 ; Byte=0/527
                    MOV     FLASH_ADDR+3,#0
                    LCALL   FLASH_READ_PAGE                 ; Read Page
                    ;
                    ;/* Start Print 528 Byte Data to RS232 */;
                    MOV     DPTR,#FLASH_DATA                ; Start Flash Data Buffer
                    MOV     R2,#16                          ; 16 Line x 33 Byte
                    ;
NEXT_LINE_READ:     MOV     R3,#33                          ; 33 Byte / Line
                    ;
NEXT_BYTE_READ:     MOVX    A,@DPTR                         ; Read Flash Data
                    LCALL   TX_2ASC                         ; Show = Hex Data
                    ;
                    MOV     A,#20H                          ; Space
                    LCALL   TX_BYTE
                    ;
                    INC     DPTR                            ; Next Byte Pointer
                    DJNZ    R3,NEXT_BYTE_READ               ; Print 16 Byte / Line
                    ;
                    MOV     A,#0DH                          ; Enter
                    LCALL   TX_BYTE
                    MOV     A,#0AH                          ; Line Feed
                    LCALL   TX_BYTE
                    ;
                    DJNZ    R2,NEXT_LINE_READ
                    ;
                    LJMP    START_WRITE_PAGE                ; Goto Write Page-0


;/*****************/;
;/* SPI Send Byte */;
;/* Input : ACC   */;
;/*****************/;
;
SPI_SEND_BYTE:      MOV     SPDAT,A                         ; Send Data to SPI
                    ;
WAIT_SPI_SEND:      MOV     A,SPSCR                         ; Read SPI Status
                    JNB     ACC.7,WAIT_SPI_SEND             ; Wait SPIF = 1 = SPI Send Complete

                    MOV     A,SPDAT                         ; Read For Reset SPIF
                    RET

;/*****************/;
;/* SPI Read Byte */;
;/* Output : ACC  */;
;/*****************/;
;
SPI_READ_BYTE:      MOV     SPDAT,#0FFH                     ; Send Dummy Data to SPI
                    ;
WAIT_SPI_READ:      MOV     A,SPSCR                         ; Read SPI Status
                    JNB     ACC.7,WAIT_SPI_READ             ; Wait SPIF = 1 = SPI Send Complete

                    MOV     A,SPDAT                         ; Read Byte From SPI
                    RET

;/************************************/;
;/* Read AT45DB161B Main Memory Page */;
;/* Input  : FLASH_ADDR+[0..3]       */;
;/*        : FLASH_ADDR+0=PA[11..08] */;
;/*        : FLASH_ADDR+1=PA[07..00] */;
;/*        : FLASH_ADDR+2=BA[09..08] */;
;/*        : FLASH_ADDR+3=BA[07..00] */;
;/* Output : FLASH_DATA+[0..528]     */;
;/************************************/;
;
FLASH_READ_PAGE:    PUSH    DPH                             ; Save Register DPTR
                    PUSH    DPL
                    ;
                    CLR     AT45DB161_CS_PIN                ; Enable AT45DB161 Interface
                    ;
                    ;/* 8 Bit Command Main Memory Page Read */;
                    ;/* D2H (SPI Mode-3 Command Code)       */;
                    MOV     A,#0D2H                         ; Command = Main Memory Page Read
                    LCALL   SPI_SEND_BYTE
                    ;
                    ;/* 24 Bit Address For Read Data */;
                    ;/* XX+[PA11..PA0]+[BA9..BA0]   */;
                    MOV     DPH,FLASH_ADDR+0                ; Save  PA[11..08]
                    MOV     DPL,FLASH_ADDR+1                ; Save  PA[07..00]
                    ;
                    MOV     A,DPL                           ; Get   PA[7..0]
                    RLC     A                               ; Shift PA7 to Carry
                    MOV     DPL,A                           ; Save  PA[6..0]+X
                    MOV     A,DPH                           ; Get   XXXX+PA[11..8]
                    RLC     A                               ; Add   XXX+PA[11..8]+PA[7]
                    MOV     DPH,A                           ; Save  XXX+PA[11..8]+PA[7]
                    ;
                    MOV     A,DPL                           ; Get   PA[6..0]+X
                    RLC     A                               ; Shift PA6 to Carry
                    MOV     A,DPH                           ; Get   XXX+PA[11..8]+PA[7]
                    RLC     A                               ; Add   XX+PA[11..8]+PA[7..6]
                    ;
                    ;/* Send 1st Address(XX+PA[11..6])
                    LCALL   SPI_SEND_BYTE                   ; Send  XX+PA[11..6]
                    ;
                    MOV     DPH,FLASH_ADDR+1                ; PA[7..0]
                    MOV     DPL,FLASH_ADDR+2                ; XXXXXX+BA[9..8]
                    ANL     DPL,#00000011B                  ; 000000+BA[9..8]
                    ;
                    MOV     A,DPH                           ; Get PA[7..0]
                    RLC     A                               ; PA[6..0]+X
                    RLC     A                               ; PA[5..0]+XX
                    ANL     A,#11111100B                    ; PA[5..0]+00
                    ORL     A,DPL                           ; Add PA[5..0]+BA[9..8]
                    ;
                    ;/* Send 2nd Address(PA[5..0]+BA[9..8])
                    LCALL   SPI_SEND_BYTE                   ; Send  PA[6..0]+BA[8]
                    ;
                    MOV     A,FLASH_ADDR+3                  ; Get   BA[7..0]
                    ;
                    ;/* Send 3rd Address(BA[7..0])
                    LCALL   SPI_SEND_BYTE                   ; Send  BA[7..0]
                    ;
                    ;/* 32 Bit Dummy Data */;
                    MOV     A,#0FFH                         ; 32 Bit Dummy Data
                    LCALL   SPI_SEND_BYTE
                    MOV     A,#0FFH
                    LCALL   SPI_SEND_BYTE
                    MOV     A,#0FFH
                    LCALL   SPI_SEND_BYTE
                    MOV     A,#0FFH
                    LCALL   SPI_SEND_BYTE
                    ;
                    ;/* Start Read Page Data */;
                    ;/* AT45DB161 = 528 Byte */;
                    ANL     AUXR,#11111101B                 ; Enable MOVX Accress XRAM
                    MOV     DPTR,#FLASH_DATA                ; Start Buffer Save Data
                    MOV     R2,#0                           ; Byte Counter
                    MOV     R3,#0
FLASH_READ_PG0:     LCALL   SPI_READ_BYTE                   ; Read SPI Data 1 Byte
                    MOVX    @DPTR,A                         ; Save Data to Buffer
                    INC     DPTR                            ; Next Data Buffer
                    ;
                    MOV     A,R3                            ; Next Record Count
                    ADD     A,#1
                    MOV     R3,A
                    MOV     A,R2
                    ADDC    A,#0
                    MOV     R2,A
                    ;
                    ;/* Verify 528 Byte Read Complete */;
                    MOV     A,R2                            ; Verify Last Record
                    CJNE    A,#HIGH(528),FLASH_READ_PG0     ; Repeat Read 528 Byte
                    MOV     A,R3
                    CJNE    A,#LOW(528),FLASH_READ_PG0
                    ;
                    SETB    AT45DB161_CS_PIN                ; Disable AT45DB161 Interface
                    ;
                    POP     DPL                             ; Restore Regsiter DPTR
                    POP     DPH
                    RET

;/*************************************/;
;/* Write AT45DB161B Main Memory Page */;
;/* Input  : FLASH_ADDR+[0..3]        */;
;/*        : FLASH_ADDR+0=PA[11..08]  */;
;/*        : FLASH_ADDR+1=PA[07..00]  */;
;/*        : FLASH_ADDR+2=BA[09..08]  */;
;/*        : FLASH_ADDR+3=BA[07..00]  */;
;/*        : FLASH_DATA+[0..528]      */;
;/*************************************/;
;
FLASH_WRITE_PAGE:   PUSH    DPH                             ; Save Register DPTR
                    PUSH    DPL
                    ;
                    CLR     AT45DB161_CS_PIN                ; Enable AT45DB161 Interface
                    ;
                    ;/* 8 Bit Command Main Memory Page Write */;
                    ;/* 82H (SPI Mode-3 Command Code)        */;
                    MOV     A,#082H                         ; Command = Main Memory Page Write
                    LCALL   SPI_SEND_BYTE
                    ;
                    ;/* 24 Bit Address For Write Data */;
                    ;/* XX+[PA11..PA0]+[BA9..BA0]   */;
                    MOV     DPH,FLASH_ADDR+0                ; Save  PA[11..08]
                    MOV     DPL,FLASH_ADDR+1                ; Save  PA[07..00]
                    ;
                    MOV     A,DPL                           ; Get   PA[7..0]
                    RLC     A                               ; Shift PA7 to Carry
                    MOV     DPL,A                           ; Save  PA[6..0]+X
                    MOV     A,DPH                           ; Get   XXXX+PA[11..8]
                    RLC     A                               ; Add   XXX+PA[11..8]+PA[7]
                    MOV     DPH,A                           ; Save  XXX+PA[11..8]+PA[7]
                    ;
                    MOV     A,DPL                           ; Get   PA[6..0]+X
                    RLC     A                               ; Shift PA6 to Carry
                    MOV     A,DPH                           ; Get   XXX+PA[11..8]+PA[7]
                    RLC     A                               ; Add   XX+PA[11..8]+PA[7..6]
                    ;
                    ;/* Send 1st Address(XX+PA[11..6])
                    LCALL   SPI_SEND_BYTE                   ; Send  XX+PA[11..6]
                    ;
                    MOV     DPH,FLASH_ADDR+1                ; PA[7..0]
                    MOV     DPL,FLASH_ADDR+2                ; XXXXXX+BA[9..8]
                    ANL     DPL,#00000011B                  ; 000000+BA[9..8]
                    ;
                    MOV     A,DPH                           ; Get PA[7..0]
                    RLC     A                               ; PA[6..0]+X
                    RLC     A                               ; PA[5..0]+XX
                    ANL     A,#11111100B                    ; PA[5..0]+00
                    ORL     A,DPL                           ; Add PA[5..0]+BA[9..8]
                    ;
                    ;/* Send 2nd Address(PA[5..0]+BA[9..8])
                    LCALL   SPI_SEND_BYTE                   ; Send  PA[6..0]+BA[8]
                    ;
                    MOV     A,FLASH_ADDR+3                  ; Get   BA[7..0]
                    ;
                    ;/* Send 3rd Address(BA[7..0])
                    LCALL   SPI_SEND_BYTE                   ; Send  BA[7..0]
                    ;
                    ;/* Start Send Page Data */;
                    ;/* AT45DB161 = 528 Byte */;
                    ANL     AUXR,#11111101B                 ; Enable MOVX Accress XRAM
                    MOV     DPTR,#FLASH_DATA                ; Start Buffer Save Data
                    MOV     R2,#0                           ; Byte Counter
                    MOV     R3,#0
FLASH_WRITE_PG0:    MOVX    A,@DPTR                         ; Read Data From Buffer
                    LCALL   SPI_SEND_BYTE                   ; Send Data 1 Byte
                    INC     DPTR                            ; Next Data Buffer
                    ;
                    MOV     A,R3                            ; Next Record Count
                    ADD     A,#1
                    MOV     R3,A
                    MOV     A,R2
                    ADDC    A,#0
                    MOV     R2,A
                    ;
                    ;/* Verify 528 Byte Write Complete */;
                    MOV     A,R2                            ; Verify Last Record
                    CJNE    A,#HIGH(528),FLASH_WRITE_PG0    ; Repeat Write 528 Byte
                    MOV     A,R3
                    CJNE    A,#LOW(528),FLASH_WRITE_PG0
                    ;
                    SETB    AT45DB161_CS_PIN                ; Disable AT45DB161 Interface
                    ;
                    ;/* Wait Page Write Complete */;
FLASH_WRITE_BUSY:   JNB     FLASH_BUSY_PIN,FLASH_WRITE_BUSY ; Wait Flash Write Complete
                    ;
                    POP     DPL                             ; Restore Regsiter DPTR
                    POP     DPH
                    RET

;/*************************/;
;/* Send 1-Byte to RS-232 */;
;/* Input   : ACC         */;
;/* Output  : Serial Port */;
;/*************************/;
;
TX_BYTE:            MOV     SBUF,A                          ; Put Data to Send
                    JNB     TI,$                            ; Wait TX Data Ready
                    CLR     TI
                    RET

;/****************************/;
;/* Receive Data From RS-232 */;
;/* Input   :  Serial Port   */;
;/* Output  :  ACC           */;
;/****************************/;
;
RX_BYTE:            JNB     RI,RX_BYTE                      ; Wait RX Data Ready
                    CLR     RI
                    MOV     A,SBUF                          ; Get RX Data
                    RET

;/************************/;
;/* Convert ASCII to HEX */;
;/* Input  : ACC (ASCII) */;
;/* Output : ACC (HEX)   */;
;/************************/;
;
ASC_HEX:            PUSH    PSW
                    CJNE    A,#41H,ASC_HEX1         ; "A"
ASC_HEX1:           JC      ASC_HEX3                ; 30H-39H Jump
ASC_HEX2:           ADD     A,#09H                  ; A-F(ASCII) ==> A-F(Hex)
ASC_HEX3:           ANL     A,#00001111B            ; Convert ASCII to HEX
                    POP     PSW
                    RET

;/************************/;
;/* Send 2 byte of ASCII */;
;/* Input : ACC (HEX)    */;
;/* Exam. : A = A3H      */;
;/*       : Send 41H,33H */;
;/************************/;
;
TX_2ASC:            LCALL   HEX_2ASC
                    LCALL   TX_BYTE
                    MOV     A,B
                    LCALL   TX_BYTE
                    RET

;/****************************/;
;/* Convert 1-HEX to 2-ASCII */;
;/* Input  : ACC (Hex)       */;
;/* Output : ACC (High-Byte) */;
;/*        : B   (Low-Byte)  */;
;/* Exam A = 41H => 34H,31H  */;
;/****************************/;
;
HEX_2ASC:           PUSH    ACC
                    LCALL   HEX_ASC
                    MOV     B,A                     ; B = Lo byte
                    POP     ACC
                    SWAP    A
                    LCALL   HEX_ASC                 ; A = High Byte
                    RET

;/*****************************/;
;/* Convert Hex(0-F) to ASCII */;
;/* Input  : ACC (HEX code)   */;
;/* Output : ACC (ASCII code) */;
;/*****************************/;
;
HEX_ASC:            PUSH    PSW
                    CLR     CY
                    ANL     A,#00001111B
                    CJNE    A,#09H,HEX_ASC1
                    SETB    CY
HEX_ASC1:           JNB     CY,HEX_ASC2             ; > 0AH
                    ORL     A,#30H                  ; < 0AH
                    SJMP    HEX_ASC3
HEX_ASC2:           SUBB    A,#09H                  ; > 0AH
                    ORL     A,#40H
HEX_ASC3:           POP     PSW
                    RET

;/******************************/;
;/* Print Data to Serial Port  */;
;/* Usage    : LCALL PRINT_SER */;
;/*          : DB   'xxxx',00  */;
;/******************************/;
;
PRINT_SER:          POP     DPH
                    POP     DPL
PRN_SER1:           CLR     A
                    MOVC    A,@A+DPTR
                    CJNE    A,#00H,PRN_SER2
                    LJMP    PRN_SER3
PRN_SER2:           LCALL   TX_BYTE
                    INC     DPTR
                    LJMP    PRN_SER1
PRN_SER3:           PUSH    DPL
                    PUSH    DPH
                    RET

                    END


