/**********************************************/
/* Example Program For ET-AVR STAMP ATMEGA64  */
/* MCU      : ATMEGA64(XTAL=16 MHz)           */
/*          : Frequency Bus = 16 MHz          */
/* Compiler : CodeVisionAVR 1.24.8d           */
/* Write By : Adisak Choochan(ETT CO.,LTD.)   */
/* Function : MP3(VS1200D) Play Wave File     */
/**********************************************/
// ATMEGA64 Interface to VS1002D (Used 3V to 5V Logic Shifter) 
//  1.  ET-AVR STAMP ATMEGA64
//  2.  ET-MINI LOGIC SHIFTER (74LCX245)
//  3.  ET-MINI MP3 (VS1002D)
// -> ATMEGA64   --> [74LCX245] -> VS1002D(MP3) 
// -> PB5(I/O)   --> [A0 -> B0] -> RES#         
// -> PB4(I/O)   <-- [B1 <- A1] <- DREQ         
// -> PB3(MISO)  <-- [B2 <- A2] <- SO           
// -> PB1(SCK)   --> [A3 -> B3] -> SCLK         
// -> PB2(MOSI)  --> [A4 -> B4] -> SI           
// -> PB0(I/O)   --> [A5 -> B5] -> XCS#         
// -> PB6(I/O)   --> [A6 -> B6] -> XDCS/BSYNC   

// Interface VS1002D By SPI Function
/* ATMEGA64 Interface to VS1002D */
/* -> ATMEGA64   --> VS1002D(MP3)*/
/* -> PB5(I/O)   --> RES#        */
/* -> PB4(I/O)   <-- DREQ        */
/* -> PB3(MISO)  <-- SO          */
/* -> PB1(SCK)   --> SCLK        */
/* -> PB2(MOSI)  --> SI          */
/* -> PB0(I/O)   --> XCS#        */
/* -> PB6(I/O)   --> XDCS/BSYNC  */

/* Include  Section */
#include <mega64.h>                                                     // ATmega64 MCU
#include "sound0.c"                                                     // Sound = Zero  (Wave File)
#include "sound1.c"			                        // Sound = One   (Wave File)
#include "sound2.c"			                        // Sound = Two   (Wave File)

// Define For VS1002D SCI Register 
#define  SCI_MODE		0x00  	                        // Mode Control
#define  SCI_STATUS		0x01                                // Status
#define  SCI_BASS	       	0x02	                        // Built-In Bass Enhancer
#define  SCI_CLOCKF		0x03	                        // Clock Frequency + Double
#define  SCI_DECODE_TIME	0x04	                        // Decode Time in Second
#define  SCI_AUDATA		0x05	                        // Misc. Audio Data
#define  SCI_WRAM		0x06	                        // RAM Write
#define  SCI_WRAMADDR	            0x07	                        // Base Address For RAM Write
#define  SCI_HDAT0		0x08	                        // Stream Header Data0
#define  SCI_HDAT1		0x09	                        // Stream Header Data1
#define  SCI_AIADDR	  	0x0A	                        // Start Address of Application.
#define  SCI_VOL	            0x0B	                        // Volume Control
#define  SCI_AICTRL0		0x0C	                        // Application Control Register. 0
#define  SCI_AICTRL1		0x0D	                        // Application Control Register. 1
#define  SCI_AICTRL2		0x0E	                        // Application Control Register. 2
#define  SCI_AICTRL3        	0x0F	                        // Application Control Register. 3	

/*********************************/
/* ATMEGA64 Interface to VS1002D */
/* -> ATMEGA64   --> VS1002D(MP3)*/
/* -> PB5(I/O)   --> RES#        */
/* -> PB4(I/O)   <-- DREQ        */
/* -> PB3(MISO)  <-- SO          */
/* -> PB1(SCK)   --> SCLK        */
/* -> PB2(MOSI)  --> SI          */
/* -> PB0(I/O)   --> XCS#        */
/* -> PB6(I/O)   --> XDCS/BSYNC  */
/*********************************/

// Define VS1002D PinIO Interface Mask Bit 
#define  VS1002D_DREQ_MASK          0x10  			// DREQ  = PB4 (00010000)      
#define  VS1002D_XCS_HIGH()  	PORTB |= 0b00000001	            // XCS#(PB0)  = '1'  
#define  VS1002D_XCS_LOW()  	PORTB &= 0b11111110 		// XCS#(PB0)  = '0'
#define  VS1002D_RES_HIGH()  	PORTB |= 0b00100000		// RES#(PB5)  = '1' 
#define  VS1002D_RES_LOW()  	PORTB &= 0b11011111		// RES#(PB5)  = '0'
#define  VS1002D_XDCS_HIGH() 	PORTB |= 0b01000000		// XDCS(PB6)  = '1' 
#define  VS1002D_XDCS_LOW() 	PORTB &= 0b10111111		// XDCS(PB6)  = '0'
#define  VS1002D_BSYNC_HIGH() 	PORTB |= 0b01000000		// BSYNC(PB6) = '1' 
#define  VS1002D_BSYNC_LOW() 	PORTB &= 0b10111111		// BSYNC(PB6) = '0'
// End of Define For VS1002D

/* User Define Function */
// Start of VS1002D Function
void VS1002D_Initial();
void VS1002D_HW_Reset(void);
void VS1002D_SW_Reset(void);
void VS1002D_Setup_Volume(unsigned char Left,unsigned char Right);	
void VS1002D_Write_Zero(unsigned char count);
void VS1002D_Write_SCI(unsigned char SCI_Reg,unsigned int SCI_Data);
//unsigned int VS1002D_Read_SCI(unsigned char SCI_Reg);
void SPI_WriteByte(unsigned char DataByte);
//unsigned char SPI_ReadByte(void);
char VS1002D_Wait_DREQ_Ready(void);
void VS1002D_Write_SDI(unsigned char SDI_Data);
void delay(unsigned long int);			            // Delay Function
// End of VS1002D Function

/***************************/
/* Start Main program Here */
/***************************/
void main (void)
{
     unsigned int  Sound_Pointer;	   			// Wave File Read Data Pointer
     unsigned int i;
     
     DDRB = 0b01100111;                                                 // PB[7,4,3] = Input,PB[6,5,2,1,0] = Output     
     VS1002D_Initial();					// Initial MP3 Player 
      
     // Loop Continue Play Sound [0] -> [1] -> [2] //
     while(1)
     {  
       // Sound "Zero"   
       VS1002D_XDCS_LOW();	  			// Start SDI Function
       Sound_Pointer = 0;											
       while(Sound_Pointer < sound0_size)			// Repeat Write "Sound0.c" File
       {
         while(VS1002D_Wait_DREQ_Ready());			// Wait VS1002D Ready		      	  
         for (i=0; i<32; i++)
         {
           VS1002D_Write_SDI(sound0[Sound_Pointer]);		// Write Data to SPI0(VS1002D)                  
           Sound_Pointer++;	  	    	  
           if (Sound_Pointer == sound0_size)	     
           break;	     
         }
       }							          
       delay(100000);					// Delay Before Next Sound

       // Sound "One"
       VS1002D_XDCS_LOW();	  			// Start SDI Function
       Sound_Pointer = 0;											
       while(Sound_Pointer < sound1_size)			// Repeat Write "Sound1.c" File
       {
         while(VS1002D_Wait_DREQ_Ready());			// Wait VS1002D Ready		      	  
         for (i=0; i<32; i++)
         {
           VS1002D_Write_SDI(sound1[Sound_Pointer]);		// Write Data to SPI0(VS1002D)                  
           Sound_Pointer++;	  	    	  
           if (Sound_Pointer == sound1_size)	     
           break;	     
         }					          
       }  
       delay(100000);					// Delay Before Next Sound

       // Sound "Two"
       VS1002D_XDCS_LOW();	  			// Start SDI Function
       Sound_Pointer = 0;											
       while(Sound_Pointer < sound2_size)			// Repeat Write "Sound2.c" File
       {
         while(VS1002D_Wait_DREQ_Ready());			// Wait VS1002D Ready		      	  
         for (i=0; i<32; i++)
         {
           VS1002D_Write_SDI(sound2[Sound_Pointer]);		// Write Data to SPI0(VS1002D)                  
           Sound_Pointer++;	  	    	  
           if (Sound_Pointer == sound2_size)	     
           break;	     
         }					          
       }  
       delay(100000);					// Delay Before Next Sound
     }  
}    

/****************************/
/* Initial VS1002D Function */
/****************************/
void VS1002D_Initial()			                        // Initial VS1002D
{  
  unsigned char dummy;
  
  /* Initial GPIO Signal Interface VS1002D */
  PORTB = 0xFF;
  VS1002D_RES_HIGH();             				// RES# = High
  VS1002D_XCS_HIGH();             				// XCS# = High
  VS1002D_XDCS_HIGH();            				// XDCS = High
  
  // SPI initialization
  // SPI Type: Master
  // SPI Clock Rate: 4000.000 kHz
  // SPI Clock Phase: Cycle Half
  // SPI Clock Polarity: Low
  // SPI Data Order: MSB First
  // VS1200D Receive SPI Clock = 6.144 MHz(MAX)
  SPCR=0x51;
  SPSR=0x00;
 
  dummy = SPDR;                 				// Read For Reset SPIF
      
  /* Initial VS1002D Function */
  VS1002D_HW_Reset();					// VS1002D Hardware Reset
  VS1002D_SW_Reset();					// VS1002D Software Reset
  delay(1000); 
}

/**************************/
/* VS1002D Hardware Reset */
/**************************/
void VS1002D_HW_Reset(void)				// Active VS1002D Hardware Reset
{
  VS1002D_RES_LOW();					// Active  VS1002D RES# Pin
  delay(10000);
  VS1002D_RES_HIGH();					// Release VS1002D RES# Pin
  delay(10000);
}

/**************************/
/* VS1002D Software Reset */
/**************************/

void VS1002D_SW_Reset(void)			   	// VS1002D Software Reset
{                                       
  while(VS1002D_Wait_DREQ_Ready());	
  VS1002D_Write_SCI(SCI_MODE,0x0804); 	                        // Active Software Reset + SPI New Mode	 

  /* Wait 100mS After Reset Complete */                        
  delay(10000);					// 100mS Delay

  while(VS1002D_Wait_DREQ_Ready());	
  VS1002D_Write_SCI(SCI_CLOCKF,0x9800);  			// VS1002 Clock = 12.288 MHz + Double Clock

  while(VS1002D_Wait_DREQ_Ready());
  VS1002D_Write_SCI(SCI_AUDATA,48000);  			// Set Sampling Rate 8KHz

  while(VS1002D_Wait_DREQ_Ready());
  VS1002D_Write_Zero(2048);		  		// Reset All Memory (2048)	

  while(VS1002D_Wait_DREQ_Ready());                     
  VS1002D_Setup_Volume(0,0);				// Set Volume = Maximum
}

/************************/
/* VS1002D Setup Volume */
/* Left,Right = 0...255	*/
/************************/
void VS1002D_Setup_Volume(unsigned char Left,unsigned char Right)
{
  unsigned int Regval;

  Regval   =  Right;   					// Get Right Volume
  Regval  += (unsigned int)Left<<8 ;			// Get Left Volume
  while(VS1002D_Wait_DREQ_Ready());	
  VS1002D_Write_SCI(SCI_VOL,Regval);			// Setup VS1002D Volume
}

/************************/
/* Send Zero to VS1002D */
/************************/
void VS1002D_Write_Zero(unsigned char count)		  	// Write Zero to VS1002D
{
  VS1002D_XDCS_LOW();
  do {
       VS1002D_Write_SDI(0x00);
       count--;
     }
     while(count);
  VS1002D_XDCS_HIGH();  
}

/************************/
/* VS1002 Write Command */
/************************/
void VS1002D_Write_SCI(unsigned char SCI_Reg,unsigned int SCI_Data)
{                                     
  VS1002D_XCS_LOW();                  			// Enable VS1002D SPI
 
  SPI_WriteByte(0x02);                			// SCI Write Command Code
  SPI_WriteByte(SCI_Reg);                			// SCI Address For Write
  SPI_WriteByte((SCI_Data >> 8)& 0xFF);  			// Data Byte High
  SPI_WriteByte(SCI_Data & 0xFF);  				// Data Byte Low
  	
  VS1002D_XCS_HIGH();                  			// Disable VS1002D SPI
}   

/************************/
/* VS1002D Read Command */
/************************/
//unsigned int VS1002D_Read_SCI(unsigned char SCI_Reg)
//{
//  unsigned int SCI_Data;   				// Dummy Read
//
//  VS1002D_XCS_LOW();                  			// Enable VS1002D SPI
// 	
//  SPI_WriteByte(0x03);        				// SCI Read Command Code
//  SPI_WriteByte(SCI_Reg);       				// SCI Address For Read
//  SCI_Data  = (unsigned int)SPI_ReadByte()<<8;                          // Read Data 1 Byte (High Byte)
//  SCI_Data += SPI_ReadByte();  				// Read Data 1 Byte (Low Byte)
//                                          
//  VS1002D_XCS_HIGH();                  			// Disable VS1002D SPI
//  return SCI_Data;                         			// Return SCI Data
//}  

/************************/
/* Write 1 Byte to SPI0 */
/************************/
void SPI_WriteByte(unsigned char DataByte)			// Write Byte to SPI0
{
  unsigned char Dummy;

  SPDR = DataByte;					// Send Byte to SPI
  while((SPSR & 0x80)!= 0x80){;;}				// Wait SPIF = 1     
  Dummy = SPDR;					// Read For Reset SPIF	   
}

/*************************/
/* Read 1 Byte From SPI0 */
/* Buff = SPI_ReadByte() */
/*************************/
//unsigned char SPI_ReadByte(void)				// Read Byte From SPI0
//{
//  unsigned char DataByte;				// Dummy
//
//  SPDR = 0xFF;					// Send Dummy Byte to SPI
//  while((SPSR & 0x80)!= 0x80){;;}				// Wait SPIF = 1   
//  DataByte = SPDR;					// Read 1 Byte SPI Data 
//
//  return (DataByte);					// Return Result
//}

/**************************/
/* Wait VS1002 DREQ Ready */
/* Wait DREQ = 1(FIFO OK) */
/**************************/
char VS1002D_Wait_DREQ_Ready(void)				// Get VS1002D DREQ Pin Status
{
  unsigned char DREQ_Status;				// DREQ Status Read

  DREQ_Status = PINB;					// Read DREQ Signal Status

  DREQ_Status &= VS1002D_DREQ_MASK;			            // Verify DREQ Pin Status 
  if(DREQ_Status == VS1002D_DREQ_MASK)  			// Check DREQ Pin Status
  {    
    return 0;					// VS1002D Busy Status
  }
  else
  {   
    return 1;					// VS1002D Ready Status
  }
}
 
/**********************/
/* VS1002D Write Data */
/**********************/
void VS1002D_Write_SDI(unsigned char SDI_Data)
{
  SPI_WriteByte(SDI_Data);  				// Send 1 Byte to SDI(VS1002D)
}

/*******************************************/
/* Long Delay Time Function(1..4294967295) */
/*******************************************/
void delay(unsigned long int i)
{
  while(i > 0) {i--;}					// Loop Decrease Counter	
  return;
}
