/***********************************************/
/* Examples For "CP-JR ARM7 USB-LPC2148" Board */
/* Target MCU  : Philips ARM7-LPC2148          */
/*       	   : X-TAL : 12.00 MHz             */
/*             : Run Speed 60 MHz (With PLL)   */
/*             : PLL Setup = M(5),P(2)		   */
/*             : VPB Clock = CPU Clock = 60MHz */
/* Keil Editor : uVision3 V3.03a               */
/* Compiler    : Keil CARM V2.50a		       */
/* Create By   : Eakachai Makarn(WWW.ETT.CO.TH)*/
/* Last Update : 17/May/2006                   */
/* Function    : MP3(VS1002D) WAV Player Demo  */
/***********************************************/
// Wave File Store in Flash Memory
// 0x00000 - 0x03FFF = Monitor Program(16KB)
// 0x04000 - 0x80000 = Wave File Area(496KB)

// 0x04000 - 0x07FFF = Zero (16KB)
// 0x08000 - 0x0BFFF = One  (16KB)
// 0x0C000 - 0x0FFFF = Two  (16KB)
// 0x10000 - 0x13FFF = Three(16KB)
// 0x14000 - 0x17FFF = Four (16KB)
// 0x18000 - 0x1BFFF = Five (16KB)
// 0x1C000 - 0x1FFFF = Six  (16KB)
// 0x20000 - 0x23FFF = Seven(16KB)
// 0x24000 - 0x27FFF = Eight(16KB)
// 0x28000 - 0x2BFFF = Nine (16KB)
// 0x2C000 - 0x2FFFF = Ten  (16KB)

// Interface VS1002D By SPI0 Function
// LPC2148     <--->  VS1002D
// P0.2(GPIO)  <----  DREQ 
// P0.3(GPIO)  ---->  RES#
// P0.4(SCK)   ---->  SCLK
// P0.5(MISO)  <----  SO
// P0.6(MOSI)  ---->  SI
// P0.7(GPIO)  ---->  XCS#
// P0.8(GPIO)  ---->  XDCS/BSYNC

#include <LPC214x.H>                       						// LPC2148 MPU Register
#include <math.h>

// Define For Wave File Header Structure 
// 36 Byte Header of WAVE File Format
// -> 4 bytes "RIFF"
// -> 4 bytes number of bytes following this field (size of sound file + 36)
// -> 4 bytes "WAVE"
// -> 4 bytes "fmt " (last character is a space)
// -> 4 bytes 16 (size of the format chunk)
// -> 2 bytes 1 (data format code, 1 = PCM)
// -> 2 bytes 1 (number of channels)
// -> 4 bytes samples per second
// -> 4 bytes number of bytes sent per second (same as previous, since it's played as mono)
// -> 2 bytes 2 (number of bytes in a sample)
// -> 2 bytes 16 ("alignment": number of bits per sample)
//4 bytes "data"
//4 bytes number of bytes of wave data (should be even for 16-bit samples)

typedef struct wav_hdr 
{
  unsigned int   magic;		 		// 2 Byte
  unsigned int   length;			// 2 Byte
  unsigned int   chunk_type;		// 2 Byte
  unsigned int   chunk_format;		// 2 Byte
  unsigned int   chunk_length;		// 2 Byte
  unsigned short format;			// 2 Byte
  unsigned short channels;			// 2 Byte
  unsigned int   sample_rate;		// 2 Byte
  unsigned int   bytes_per_second;	// 2 Byte
  unsigned short sample_size;		// 2 Byte
  unsigned short precision;			// 2 Byte
  unsigned int   chunk_data;		// 2 Byte
  unsigned int   data_length;		// 2 Byte
};

// Data for Record & Block handling 
#define FIRSTBLOCK 1          									// The 1st free block - application is in Block 0 
#define LASTBLOCK  11											// 0..10 Voice Play
#define BLOCKSIZE  0x4000l   									// Size of each block - 16KB                       
#define SIZE_OF_BOOTLOADER 0x3000;  							// Size of the Bootloader that is in the last flash Sector 

signed short *p_record;       									// pointer to the wave sample                     
unsigned int eot;             									// end of track                                   
unsigned int blockindex;      									// current Block                                  
struct wav_hdr* p_wav_hdr;    									// Wave file header                              

// Define For VS1002D SCI Register 
#define  SCI_MODE		   		0x00  							// Mode Control
#define  SCI_STATUS		   		0x01							// Status
#define  SCI_BASS	       		0x02							// Built-In Bass Enhancer
#define  SCI_CLOCKF		   		0x03							// Clock Frequency + Double
#define  SCI_DECODE_TIME	   	0x04							// Decode Time in Second
#define  SCI_AUDATA		   		0x05							// Misc. Audio Data
#define  SCI_WRAM		   		0x06							// RAM Write
#define  SCI_WRAMADDR	   		0x07							// Base Address For RAM Write
#define  SCI_HDAT0		   		0x08							// Stream Header Data0
#define  SCI_HDAT1		   		0x09							// Stream Header Data1
#define  SCI_AIADDR	  	   		0x0A							// Start Address of Application.
#define  SCI_VOL			   	0x0B							// Volume Control
#define  SCI_AICTRL0		   	0x0C							// Application Control Register. 0
#define  SCI_AICTRL1		   	0x0D							// Application Control Register. 1
#define  SCI_AICTRL2		   	0x0E							// Application Control Register. 2
#define  SCI_AICTRL3        	0x0F							// Application Control Register. 3

// Define VS1002D PinIO Interface Mask Bit 
#define  VS1002D_DREQ_PIN 		0x00000004   					// P0.2 (0000 0000 0000 0000 0000 0000 0000 0x00)
#define  VS1002D_RES_PIN		0x00000008   					// P0.3 (0000 0000 0000 0000 0000 0000 0000 x000)
#define  VS1002D_SCLK_PIN		0x00000010						// P0.4 (0000 0000 0000 0000 0000 0000 000x 0000)
#define  VS1002D_SO_PIN			0x00000020						// P0.5 (0000 0000 0000 0000 0000 0000 00x0 0000)
#define  VS1002D_SI_PIN			0x00000030						// P0.6 (0000 0000 0000 0000 0000 0000 0x00 0000)
#define  VS1002D_XCS_PIN  		0x00000080   					// P0.7 (0000 0000 0000 0000 0000 0000 x000 0000)
#define  VS1002D_XDCS_PIN 		0x00000100   					// P0.8 (0000 0000 0000 0000 0000 000x 0000 0000)
#define  VS1002D_BSYNC_PIN 		0x00000100   					// P0.8 (0000 0000 0000 0000 0000 000x 0000 0000)

#define  VS1002D_DREQ_DIR()		IO0DIR &= ~VS1002D_DREQ_PIN		// DREQ  = Input  <-- DREQ(VS1002D)
#define  VS1002D_RES_DIR()		IO0DIR |= VS1002D_RES_PIN		// RES#  = Output --> RES#(VS1002D)
#define  VS1002D_SCLK_(DIR)		IO0DIR |= VS1002D_SCLK_PIN		// SCLK  = Output --> SCLK(VS1002D)
#define  VS1002D_SO_DIR()		IO0DIR &= ~VS1002D_SO_PIN		// SO    = Input  <-- SO(VS1002D)
#define  VS1002D_SI_DIR()		IO0DIR |= VS1002D_SI_PIN		// SI    = Output --> SI(VS1002D)
#define  VS1002D_XCS_DIR()		IODIR0 |= VS1002D_XCS_PIN		// XCS#  = Output --> XCS#(VS1002D)
#define  VS1002D_XDCS_DIR()		IODIR0 |= VS1002D_XDCS_PIN      // XDCS  = Output --> XDCS(VS1002D)

#define  VS1002D_XCS_HIGH()  	IOSET0  = VS1002D_XCS_PIN		// XCS#  = '1' 
#define  VS1002D_XCS_LOW()  	IOCLR0  = VS1002D_XCS_PIN		// XCS#  = '0'
#define  VS1002D_RES_HIGH()  	IOSET0  = VS1002D_RES_PIN		// RES#  = '1' 
#define  VS1002D_RES_LOW()  	IOCLR0  = VS1002D_RES_PIN		// RES#  = '0'
#define  VS1002D_XDCS_HIGH() 	IOSET0  = VS1002D_XDCS_PIN		// XDCS  = '1' 
#define  VS1002D_XDCS_LOW() 	IOCLR0  = VS1002D_XDCS_PIN		// XDCS  = '0'
#define  VS1002D_BSYNC_HIGH() 	IOSET0  = VS1002D_BSYNC_PIN		// BSYNC = '1' 
#define  VS1002D_BSYNC_LOW() 	IOCLR0  = VS1002D_BSYNC_PIN		// BSYNC = '0'
// End of Define For VS1002D

/* Pototype  Section */
void VS1002D_Initial();
void VS1002D_HW_Reset(void);
void VS1002D_SW_Reset(void);
void VS1002D_Setup_Volume(unsigned char Left,unsigned char Right);	
void VS1002D_Write_Zero(unsigned char count);
void VS1002D_Write_SCI(unsigned char SCI_Reg,unsigned int SCI_Data);
unsigned int VS1002_Read_SCI(unsigned char SCI_Reg);
void SPI_WriteByte(unsigned char DataByte);
unsigned char SPI_ReadByte(void);
char VS1002D_Wait_DREQ_Ready(void);
void VS1002D_Write_SDI(unsigned char SDI_Data);
void delay(unsigned long int);									// Delay Function

int main (void) 
{        
  unsigned int data;	
  unsigned int c;  	
  unsigned int blockaddress;

  VS1002D_Initial();											// Initial MP3 Player
  blockindex = FIRSTBLOCK; 										// Start First Voice Play

  // Loop Continue Play Voice
  while(1)
  {  
    blockaddress = (blockindex * BLOCKSIZE);     				// calculate the absolute memory address of the block
    p_wav_hdr = (void*) blockaddress;                         	// Set Pointer to Wave/RIFF Structure New Block 
    p_record = (signed short*)(void*) blockaddress;    			// Start 1st Byte of File 
    eot = (unsigned int)(blockaddress + p_wav_hdr->length); 	// Get the end of the file from either the header or 
  
    VS1002D_XDCS_LOW();	  										// Start SDI Function
    while(p_record < eot) 
    {
      while(VS1002D_Wait_DREQ_Ready());							// Wait VS1002D Ready					
      for(c=0;c<16;c++)											// 32 Byte Transfer to SPI0
	  {
	    data = *p_record++;				  						// Read 2 Byte From Buffer
	    VS1002D_Write_SDI(data&0xFF);							// Write LSB to SPI0
	    VS1002D_Write_SDI((data>>8)&0xFF);	              		// Write MSB to SPI0
	  }					
    }

    delay(10000);												// Delay Before Next Voice
    if (blockindex < LASTBLOCK) 								// Verify Last Voice Play
    { 
      blockindex++; 							 				// Next Voice Play
    } 
    else 
    { 
      blockindex = FIRSTBLOCK; 									// Restart First Voice Play
    }
  }
}

/****************************/
/* Initial VS1002D Function */
/****************************/
void VS1002D_Initial()					  						// Initial VS1002D
{
  unsigned char i, Dummy;				  				
  
  /* Initial SPI0 Interface VS1002D */  
  // Initial SPI0 Pin Connect on P0[4] to P0[6]
  // P0.4(SCK)   ---->  SCLK
  // P0.5(MISO)  <----  SO
  // P0.6(MOSI)  ---->  SI
  PINSEL0 &= 0xFFFF00FF;                                		// Reset GPIO Pin Control Status Bit[15:8] 
  PINSEL0 |= 0x00000100;										// Bit[09:08] = 01 = P0.4 is SCK(SPI0)  
  PINSEL0 |= 0x00000400;										// Bit[11:10] = 01 = P0.5 is MISO(SPI0)  
  PINSEL0 |= 0x00001000;										// Bit[13:12] = 01 = P0.6 is MOSI(SPI0)                                  
  
  // Initial SPI0 Function Interface to 74HC595
  // SPI0 Clock Counter Register
  S0SPCCR = 0x000000FF;	  										// SPI-0 Clock = PCLK / 256
															
  // SPI0 Control Register
  S0SPCR &= 0x000000F7;											// CPHA = 0 = Rising Clock Shift Data 
  S0SPCR &= 0x000000EF;                                 		// CPOL = 0 = Normal Clock "0",Active = "1"
  S0SPCR |= 0x00000020;											// MSTR = 1 = Master SPI
  S0SPCR &= 0xFFFFFFBF;                                 		// LSBF = 0 = MSB First
  S0SPCR &= 0x0000007F;											// SPIE = 0 = Disable SPI Interrupt

  for ( i = 0; i < 8; i++ )
  {
    Dummy = S0SPDR; 											// Clear RxFIFO 
  }

  /* Initial GPIO Signal Interface VS1002D */
  VS1002D_RES_HIGH();											// Active RES# = High
  VS1002D_XCS_HIGH();											// Active XCS# = High
  VS1002D_XDCS_HIGH(); 											// Active XDCS = High
  VS1002D_DREQ_DIR();											// Setup DREQ Direction
  VS1002D_RES_DIR();											// Setup RES# Direction
  VS1002D_XCS_DIR();											// Setup XCS# Direction
  VS1002D_XDCS_DIR();											// Setup XDCS Direction

  /* Initial VS1002D Function */
  VS1002D_HW_Reset();											// VS1002D Hardware Reset
  VS1002D_SW_Reset();											// VS1002D Software Reset
  delay(1000); 
}

/**************************/
/* VS1002D Hardware Reset */
/**************************/
void VS1002D_HW_Reset(void)										// Active VS1002D Hardware Reset
{
  VS1002D_RES_LOW();											// Active  VS1002D RES# Pin
  delay(500000);
  VS1002D_RES_HIGH();											// Release VS1002D RES# Pin
  delay(500000);
}

/**************************/
/* VS1002D Software Reset */
/**************************/

void VS1002D_SW_Reset(void)			   							// VS1002D Software Reset
{                                       
  VS1002D_Write_SCI(SCI_MODE,0x0804); 							// Active Software Reset + SPI New Mode	 
  while(VS1002D_Wait_DREQ_Ready());
  /* Wait 100mS After Reset Complete */                        
  delay(500000);												// 100mS Delay

  VS1002D_Write_SCI(SCI_CLOCKF,0x9800);  						// VS1002 Clock = 12.288 MHz + Double Clock
  while(VS1002D_Wait_DREQ_Ready());

  VS1002D_Write_SCI(SCI_AUDATA,48000);  						// Set Sampling Rate 8KHz
  while(VS1002D_Wait_DREQ_Ready());

  VS1002D_Write_Zero(1250);		  								// Reset All Memory	(1048)	
  while(VS1002D_Wait_DREQ_Ready());                   
  
  VS1002D_Setup_Volume(0,0);									// Set Volume = Maximum 	
  while(VS1002D_Wait_DREQ_Ready());  
}

/************************/
/* VS1002D Setup Volume */
/* Left,Right = 0...255	*/
/************************/
void VS1002D_Setup_Volume(unsigned char Left,unsigned char Right)
{
  unsigned int Regval;

  Regval   =  Right;   											// Get Right Volume
  Regval  += (unsigned int)Left<<8 ;							// Get Left Volume
  VS1002D_Write_SCI(SCI_VOL,Regval);							// Setup VS1002D Volume
}

/************************/
/* Send Zero to VS1002D */
/************************/

void VS1002D_Write_Zero(unsigned char count)		  			// Write Zero to VS1002D
{
  VS1002D_XDCS_LOW();
  do {
       VS1002D_Write_SDI(0x00);
       count--;
     }
     while(count);
  VS1002D_XDCS_HIGH();  
}

/************************/
/* VS1002 Write Command */
/************************/
void VS1002D_Write_SCI(unsigned char SCI_Reg,unsigned int SCI_Data)
{                                     
  VS1002D_XCS_LOW();                  							// Enable VS1002D SPI
 
  SPI_WriteByte(0x02);                							// SCI Write Command Code
  SPI_WriteByte(SCI_Reg);                						// SCI Address For Write
  SPI_WriteByte((SCI_Data >> 8)& 0xFF);  						// Data Byte High
  SPI_WriteByte(SCI_Data & 0xFF);  								// Data Byte Low
  	
  VS1002D_XCS_HIGH();                  							// Disable VS1002D SPI
}   

/************************/
/* VS1002D Read Command */
/************************/
unsigned int VS1002_Read_SCI(unsigned char SCI_Reg)
{
  unsigned int SCI_Data;   										// Dummy Read

  VS1002D_XCS_LOW();                  							// Enable VS1002D SPI
 	
  SPI_WriteByte(0x03);        									// SCI Read Command Code
  SPI_WriteByte(SCI_Reg);       								// SCI Address For Read
	                               
  SCI_Data  = SPI_ReadByte();                     				// Read Data 1 byte (Low Byte)
  SCI_Data += (unsigned int)SPI_ReadByte()<<8;  				// Read Data 1 byte (Hig Byte)
                                          
  VS1002D_XCS_HIGH();                  							// Disable VS1002D SPI
  return SCI_Data;                         						// Return SCI Data
}  

/************************/
/* Write 1 Byte to SPI0 */
/************************/
void SPI_WriteByte(unsigned char DataByte)						// Write Byte to SPI0
{
  unsigned char Dummy;

  S0SPDR = DataByte;											// Send SPI to 74HC595 Output(Toggle Logic Drive)
  while((S0SPSR & 0x80)!= 0x80){;;}								// Wait SPIF = 1 (SPI Send Complete)  
  Dummy = S0SPDR; 												// Clear RxFIFO 
}

/*************************/
/* Read 1 Byte From SPI0 */
/* Buff = SPI_ReadByte() */
/*************************/
unsigned char SPI_ReadByte(void)								// Read Byte From SPI0
{
  unsigned char DataByte;									   	// Dummy

  S0SPDR = 0xFF;												// Write Dummy For Generate SCK Read MISO   
  while (S0SPSR & 0x10);  										// Wait until the Busy bit is Clear
  DataByte = S0SPDR;											// Read Byte From SPI0
  return (DataByte);										   	// Return Result
}

/**************************/
/* Wait VS1002 DREQ Ready */
/* Wait DREQ = 1(FIFO OK) */
/**************************/
char VS1002D_Wait_DREQ_Ready(void)								// Get VS1002D DREQ Pin Status
{
  unsigned long busy_status;									// Busy Status Read
  
  busy_status = (IO0PIN & VS1002D_DREQ_PIN);					// Read DREQ Pin Signal 
  if(busy_status == VS1002D_DREQ_PIN)  							// Read & Check DREQ Pin Status
  {    
    return 0;													// VS1002D Busy Status
  }
  else
  {   
    return 1;													// VS1002D Ready Status
  }
}
 
/**********************/
/* VS1002D Write Data */
/**********************/
void VS1002D_Write_SDI(unsigned char SDI_Data)
{
  SPI_WriteByte(SDI_Data);  									// Send 1 Byte to SDI(VS1002D)
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}									// Loop Decrease Counter	
}





